/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import java.io.File;
import java.util.Map;
import java.util.HashMap;
import java.util.Locale ;
import java.util.Set;

/**
 * FileMap は、ファイルを読み取って、キー情報から、ファイルへのリンクを作成するための
 * 情報を返します。
 * ファイルそのものは、指定のディレクトリをすべて読み取り、拡張子以外の部分を、キーとして
 * 登録します。(キーは大文字に統一されます。)
 * 実際のファイルの拡張子は、リンク作成時の処理で付与されます。
 * 例えば、HELPファイルを、XXXX.html や、XXXX.htm 、XXXX.pdf など、色々な形態で作成した
 * 場合でも、キーとしては、XXXX で存在チェックをかけることができるようになります。
 *
 * ファイルは、一旦すべて読み取ってメモリ上で管理されます。
 * ディレクトリの再読取が必要な場合は、オブジェクトを再作成する必要があります。
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public final class FileMap {
	private final Map<String,String> map = new HashMap<String,String>();

	/**
	 * すでに読み取った Set オブジェクトを指定して、ファイルマップを構築します。
	 *
	 * これは、ServletContext を利用した、META-INF/resources からの読み取り対応になります。
	 * 一覧を取得するのは、ServletContext 関連の実装が必要になるため、fukurou では
	 * java の一般的なオブジェクトである Set を処理するだけとします。
	 *
	 * ファイル名は、dir を削除した残りで構築します。フォルダ階層を含みます。
	 * Mapのキーは、フォルダ階層を含まない、ファイル名のみとします。
	 * つまり、フォルダ階層を持ってリソースを用意しておいても、キーとしては、
	 * ファイル名のみを使用します。
	 *
	 * @og.rev 5.5.4.2 (2012/07/13) 新規作成
	 *
	 * @param  dir ディレクトリ
	 * @param  resourcePaths リソースパス
	 * @throws IllegalArgumentException 引数の dir や、resourcePaths が、null の場合
	 */
	public FileMap( final String dir , final Set<?> resourcePaths ) throws IllegalArgumentException {
		if( resourcePaths == null || dir == null ) {
			String errMsg = "指定のリソースは、存在しません。[" + dir + "]";
			throw new IllegalArgumentException( errMsg );
		}

		int len = dir.length() ;

		for( Object path : resourcePaths ) {
			String fname  = String.valueOf( path ).substring( len );	// ファイル名
			String upkey  = fname.toUpperCase( Locale.JAPAN ) ;			// 大文字化されたファイル名(Mapのキー)

			int idx = fname.lastIndexOf( '.' );
			if( idx >= 0 ) {
				map.put( upkey.substring( 0,idx ), fname );
			}
			else {
				map.put( upkey, fname );
			}
		}
	}

	/**
	 * 読み取るディレクトリを指定して、ファイルマップを構築します。
	 *
	 * このディレクトリは、OSに対する物理アドレスになります。
	 *
	 * @og.rev 5.5.4.2 (2012/07/13) makeFileMap() を直接コンストラクターとして使用
	 *
	 * @param  dir ディレクトリ
	 * @throws IllegalArgumentException 引数の dir が存在しないか、ディレクトリ出ない場合。
	 */
	public FileMap( final String dir ) throws IllegalArgumentException {
		File directory = new File( dir );
		if( ! directory.exists() ) {
			String errMsg = "指定のディレクトリは、存在しません。[" + directory + "]";
			throw new IllegalArgumentException( errMsg );
		}

		if( ! directory.isDirectory() ) {
			String errMsg = "指定のキーは、ディレクトリではありません。[" + directory + "]";
			throw new IllegalArgumentException( errMsg );
		}

		for( String fname : directory.list() ) {
			String upkey  = fname.toUpperCase( Locale.JAPAN ) ;

			int idx = upkey.lastIndexOf( '.' );
			if( idx >= 0 ) {
				map.put( upkey.substring( 0,idx ), fname );
			}
			else {
				map.put( upkey, fname );
			}
		}
	}

	/**
	 * 指定のキーのファイルが存在しているかどうかを返します。
	 * 存在している場合は、true , 存在していない場合は、false になります。
	 *
	 * @param   key 指定のキー
	 *
	 * @return	存在しているかどうか(true:存在する/false:存在しない)
	 * @throws  IllegalArgumentException キーが指定されていません。
	 */
	public boolean exists( final String key ) {
		if( key == null ) {
			String errMsg = "キーが指定されていません。" ;
			throw new IllegalArgumentException( errMsg );
		}

		return map.containsKey( key.toUpperCase( Locale.JAPAN ) );
	}

	/**
	 * キーに対応したファイル名を返します。
	 * 指定のキーに対するファイル名が存在しない場合は、null を返します。
	 *
	 * @param   key 指定のキー
	 *
	 * @return	ファイル名(ディレクトリパスは含まず)
	 */
	public String getFilename( final String key ) {
		if( key == null ) {
			return null ;
	//		String errMsg = "キーが指定されていません。" ;
	//		throw new IllegalArgumentException( errMsg );
		}

		return map.get( key.toUpperCase( Locale.JAPAN ) );
	}
}
