/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.automata.nfa;

import java.io.IOException;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Stack;

import net.morilib.automata.NFA;
import net.morilib.automata.NFAState;
import net.morilib.automata.RegexMetaCharacter;
import net.morilib.nina.NinaParseException;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/20
 */
public class NFAParser<T, A, B> {

	/* S -> R
	 * R -> R '|' E
	 * R -> E
	 * E -> E F
	 * E -> F
	 * F -> F l
	 * F -> c
	 * F -> '(' R ')' */
	private static enum S {
		S01,
		R01, R02, R03, R04,
		E02,
		F01, F02, F03, F04, F05, F06
	}

	private static enum N {
		NTR, NTE, NTF
	}

	private Stack<Object> syn;
	private Iterator<T> rd;
	private S etat;
	private T lookahead;
	private N nonterm;

	NFAParser(Iterator<T> t) throws IOException {
		rd = t;
		etat = S.S01;
		syn = new Stack<Object>();
		next();
	}

	private void next() throws IOException {
		lookahead = rd.hasNext() ? rd.next() : null;
	}

	private void shift() throws IOException {
		syn.push(etat);  syn.push(lookahead);
		next();
		nonterm = null;
	}

	private boolean eqlook(Object o) {
		return lookahead != null ? lookahead.equals(o) : o == null;
	}

	private boolean eqnt(N o) {
		return nonterm != null && nonterm.equals(o);
	}

	private Object pop() {
		Object o;

		o = syn.pop();  etat = (S)syn.pop();
		return o;
	}

	private void goTo(N s, Object a) {
		nonterm = s;
		syn.push(etat);  syn.push(a);
	}

	private boolean eqclo() {
		return (eqlook(RegexMetaCharacter.REPETATION_ZERO) ||
				eqlook(RegexMetaCharacter.REPETATION_ONE)  ||
				eqlook(RegexMetaCharacter.OPTION));
	}

	private boolean eqalt() {
		return (eqlook(null) ||
				eqlook(RegexMetaCharacter.ALTERNATION) ||
				eqlook(RegexMetaCharacter.BLOCK_END));
	}

	/**
	 * 
	 * @param t
	 * @return
	 * @throws IOException
	 */
	public static<T, A, B> NFA<T, A, B> parse(
			Iterator<T> t) throws IOException {
		NFAParser<T, A, B> p = new NFAParser<T, A, B>(t);
		NFAObject<T, A, B> o;

		while((o = p.step()) == null) {
			// do nothing
		}
		return o;
	}

	/**
	 * 
	 * @param s
	 * @return
	 */
	public static NFA<Object, NFAState, Void> parse(final String s) {
		Iterator<Object> i = new Iterator<Object>() {

			private int l = 0;

			@Override
			public boolean hasNext() {
				return l < s.length();
			}

			@Override
			public Object next() {
				if(l >= s.length()) {
					throw new NoSuchElementException();
				}

				switch(s.charAt(l++)) {
				case '*':  return RegexMetaCharacter.REPETATION_ZERO;
				case '+':  return RegexMetaCharacter.REPETATION_ONE;
				case '?':  return RegexMetaCharacter.OPTION;
				case '|':  return RegexMetaCharacter.ALTERNATION;
				case '(':  return RegexMetaCharacter.BLOCK_BEGIN;
				case ')':  return RegexMetaCharacter.BLOCK_END;
				default:   return Integer.valueOf(s.charAt(l - 1));
				}
			}

			@Override
			public void remove() {
				throw new UnsupportedOperationException();
			}

		};

		try {
			return NFAParser.<Object, NFAState, Void>parse(i);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	@SuppressWarnings("unchecked")
	NFAObject<T, A, B> step() throws IOException {
		NFAObject<T, A, B> a, b;
		Object o;

		switch(etat) {
		case S01:
			/* S -> *R
			 * R -> *R '|' E
			 * R -> *E
			 * E -> *E F
			 * E -> *F
			 * F -> *F l
			 * F -> *c
			 * F -> *'(' R ')' */
			if(eqnt(N.NTR)) {
				etat = S.R01;
			} else if(eqnt(N.NTE)) {
				etat = S.R04;
			} else if(eqnt(N.NTF)) {
				etat = S.F01;
			} else if(eqlook(RegexMetaCharacter.BLOCK_BEGIN)) {
				shift();  etat = S.F02;
			} else if(!eqalt()) {
				shift();  etat = S.F03;
			} else {
				throw new NinaParseException();
			}
			break;
		case R01:
			/* S -> R*
			 * R -> R *'|' E */
			if(eqlook(null)) {
				return NFAAccept.newInstance(
						(NFAObject<T, A, B>)pop(), null);
			} else if(eqlook(RegexMetaCharacter.ALTERNATION)) {
				shift();  etat = S.R02;
			}
			break;
		case R02:
			/* R -> R '|' *E
			 * E -> *E F
			 * E -> *F
			 * F -> *F l
			 * F -> *c
			 * F -> *'(' R ')' */
			if(eqnt(N.NTE)) {
				etat = S.R03;
			} else if(eqnt(N.NTF)) {
				etat = S.F01;
			} else if(eqlook(RegexMetaCharacter.BLOCK_BEGIN)) {
				shift();  etat = S.F02;
			} else if(!eqalt()) {
				shift();  etat = S.F03;
			} else {
				throw new NinaParseException();
			}
			break;
		case R03:
			/* R -> R '|' E*
			 * E -> E *F
			 * F -> *F l
			 * F -> *c
			 * F -> *'(' R ')' */
			if(eqnt(N.NTF)) {
				etat = S.E02;
			} else if(eqalt()) {
				b = (NFAObject<T, A, B>)pop();
				pop();
				a = (NFAObject<T, A, B>)pop();
				a = NFAAlternative.newInstance(a, b);
				goTo(N.NTR, a);
			} else if(eqlook(N.NTF)) {
				shift();  etat = S.F01;
			} else if(eqlook(RegexMetaCharacter.BLOCK_BEGIN)) {
				shift();  etat = S.F02;
			} else {
				shift();  etat = S.F03;
			}
			break;
		case R04:
			/* R -> E*
			 * E -> E *F
			 * E -> *F
			 * F -> *F l
			 * F -> *c
			 * F -> *'(' R ')' */
			if(eqnt(N.NTF)) {
				etat = S.E02;
			} else if(eqlook(RegexMetaCharacter.BLOCK_BEGIN)) {
				shift();  etat = S.F02;
			} else if(eqalt()) {
				goTo(N.NTR, pop());
			} else {
				shift();  etat = S.F03;
			}
			break;
		case E02:
			/* E -> E F*
			 * F -> F *l */
			if(eqclo()) {
				shift();  etat = S.F06;
			} else {
				b = (NFAObject<T, A, B>)pop();
				a = (NFAObject<T, A, B>)pop();
				a = NFAConcatenation.newInstance(a, b);
				goTo(N.NTE, a);
			}
			break;
		case F01:
			/* E -> F*
			 * F -> F *l */
			if(eqclo()) {
				shift();  etat = S.F06;
			} else {
				goTo(N.NTE, pop());
			}
			break;
		case F02:
			/* F -> '(' *R ')'
			 * R -> *R '|' E
			 * R -> *E
			 * E -> *E F
			 * E -> *F
			 * F -> *F l
			 * F -> *c
			 * F -> *'(' R ')' */
			if(eqnt(N.NTR)) {
				etat = S.F04;
			} else if(eqnt(N.NTE)) {
				etat = S.R04;
			} else if(eqnt(N.NTF)) {
				etat = S.F01;
			} else if(eqlook(RegexMetaCharacter.BLOCK_BEGIN)) {
				shift();  etat = S.F04;
			} else if(!eqalt()) {
				shift();  etat = S.F03;
			} else {
				throw new NinaParseException();
			}
			break;
		case F03:
			/* F -> c* */
			a = SingleObjectNFA.newInstance((T)pop());
			goTo(N.NTF, a);
			break;
		case F04:
			/* F -> '(' R *')'
			 * R -> R *'|' E */
			if(eqlook(RegexMetaCharacter.ALTERNATION)) {
				shift();  etat = S.R02;
			} else if(eqlook(RegexMetaCharacter.BLOCK_END)) {
				shift();  etat = S.F05;
			} else {
				throw new NinaParseException();
			}
			break;
		case F05:
			/* F -> '(' R ')'* */
			pop();  a = (NFAObject<T, A, B>)pop();  pop();
			goTo(N.NTF, a);
			break;
		case F06:
			/* F -> F l* */
			o = pop();
			a = (NFAObject<T, A, B>)pop();
			if(o.equals(RegexMetaCharacter.REPETATION_ZERO)) {
				a = NFARepetition.newInstance(a, true);
			} else if(o.equals(RegexMetaCharacter.REPETATION_ONE)) {
				a = NFARepetition.newInstance(a, false);
			} else if(o.equals(RegexMetaCharacter.OPTION)) {
				a = NFAOptional.newInstance(a);
			}
			goTo(N.NTF, a);
			break;
		}
		return null;
	}

}
