/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.declare;

import static org.maru.core.util.ConditionUtil.verifyNotNull;
import static org.maru.dog.core.ClassChecker.checkClassType;
import static org.maru.core.util.ConditionUtil.verifyState;
import static org.maru.core.util.ConditionUtil.isNotNull;
import static org.maru.core.util.ConditionUtil.isNull;
import static org.maru.dog.core.Syntax.checkSyntax;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.maru.core.Key;
import org.maru.core.KeyGen;
import org.maru.dog.DefinitionException;
import org.maru.dog.bind.BindingConfiguration;
import org.maru.dog.bind.OverridableBindingConfig;
import org.maru.dog.core.Configuration;
import org.maru.dog.core.Operator;

public final class BindingDeclarator implements DeclarativeBindingDeclarator,
        BoundPointDeclarator, BridgeBindingDeclarator, BindPointDeclarator,
        ConverterFlagDeclarator, TerminativeBindingDeclarator {

    private static final String ERROR_ARGUMENT_IS_NULL = "The argument is null";

    private static final String ERROR_DOSE_NOT_START_DECLARATION = "Dose not start declaration.";

    private boolean isDefining = false;

    private Key<?> currentDeclarationKey = null;

    private BindingConfiguration currentBindingConfig = null;

    final Map<Key<?>, List<Configuration>> configurationsMap = new HashMap<Key<?>, List<Configuration>>();

    private final List<Operator> syntaxList = new ArrayList<Operator>();

    @Override
    public BoundPointDeclarator declareFor(Class<?> target) {
        if (isDefining) {
            throw new DefinitionException("The other definition is not complete.");
        }
        isDefining = true;
        verifyNotNull(target, ERROR_ARGUMENT_IS_NULL);
        checkClassType(target);

        return declareFor(KeyGen.getKey(target));
    }

    private BoundPointDeclarator declareFor(Key<?> key) {
        currentDeclarationKey = key;
        if (configurationsMap.containsKey(key)) {
                throw new DefinitionException("The same declaration has already existed.");
        }
        List<Configuration> configurations = new ArrayList<Configuration>();
        configurationsMap.put(key, configurations);
        return this;
    }

    @Override
    public BridgeBindingDeclarator bound(String name) {
        checkState(isDefining, ERROR_DOSE_NOT_START_DECLARATION);
        verifyNotNull(name, ERROR_ARGUMENT_IS_NULL);
        putPreviousConfigurationIntoMap();
        this.currentBindingConfig = new OverridableBindingConfig();
        this.currentBindingConfig.setBoundName(name);
        this.syntaxList.add(Operator.BOUND);
        return this;
    }

    @Override
    public BindPointDeclarator from(Class<?> fromClass) {
        checkState(isDefining, ERROR_DOSE_NOT_START_DECLARATION);
        verifyNotNull(fromClass, ERROR_ARGUMENT_IS_NULL);
        checkClassType(fromClass);
        confirmCurrentBindingConfigIsNotNull();
        this.currentBindingConfig.setInputClass(fromClass);
        this.syntaxList.add(Operator.FROM);
        return this;
    }

    @Override
    public ConverterFlagDeclarator bind(String name) {
        checkState(isDefining, ERROR_DOSE_NOT_START_DECLARATION);
        verifyNotNull(name, ERROR_ARGUMENT_IS_NULL);
        confirmCurrentBindingConfigIsNotNull();
        this.currentBindingConfig.setBindName(name);
        this.syntaxList.add(Operator.BIND);
        return this;
    }

    @Override
    public BoundPointDeclarator converterOff() {
        checkState(isDefining, ERROR_DOSE_NOT_START_DECLARATION);
        confirmCurrentBindingConfigIsNotNull();
        this.currentBindingConfig.setConverterOff(true);
        this.syntaxList.add(Operator.CONVERTER_OFF);
        return this;
    }

    @Override
    public DeclarativeBindingDeclarator end() {
        checkState(isDefining, ERROR_DOSE_NOT_START_DECLARATION);
        putPreviousConfigurationIntoMap();
        this.currentDeclarationKey = null;
        this.currentBindingConfig = null;
        this.isDefining = false;
        return this;
    }

    private void putPreviousConfigurationIntoMap() {
        if (isNotNull(this.currentBindingConfig)) {
            confirmSyntax();
            List<Configuration> configurations = this.configurationsMap.get(this.currentDeclarationKey);
            verifyNotNull(configurations);
            if (configurations.contains(currentBindingConfig)) {
                    throw new DefinitionException("The Same definition has already existed.");
            }
            configurations.add(currentBindingConfig);
        }
    }

    private static void checkState(boolean isDefining, String message) {
        verifyState(isDefining, new DefinitionException(message));
    }

    private void confirmSyntax() {
        Operator[] operators = (Operator[])this.syntaxList.toArray(new Operator[syntaxList.size()]);
        verifyState(checkSyntax(operators),
                new DefinitionException("Syntax error: " + Arrays.toString(operators) + " is illegal syntax."));
        this.syntaxList.clear();
    }

    private void confirmCurrentBindingConfigIsNotNull() {
        if (isNull(this.currentBindingConfig)) {
            throw new DefinitionException("Encounters syntax error.");
        }
    }

    public Map<Key<?>, List<Configuration>> getConfigurationMap() {
        return this.configurationsMap;
    }

}
