/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.maru.core.Key;
import org.maru.core.KeyGen;
import org.maru.core.type.GenericClassType;
import org.maru.dog.core.AbstractMarkedPointBuilder;
import org.maru.dog.core.Builder;
import org.maru.dog.core.Definition;
import org.maru.dog.core.MarkedPoint;
import org.maru.dog.core.Configuration;

import static org.maru.core.type.PrimitiveWrapperTypes.getWrapperType;
import static org.maru.core.type.PrimitiveWrapperTypes.isPrimitiveOrWrapperType;
import static org.maru.core.type.PrimitiveWrapperTypes.isPrimitiveType;
import static org.maru.core.util.ConditionUtil.isNotNull;

/**
 * builder class of {@link InstanceBinderImpl}
 */
public final class InstanceBinderBuilder<T, K> implements Builder {

    private final T target;
    private final K[] inputs;

    private final List<Configuration> configurations;

    private final InstanceBinder<T, K> instanceBinder;

    @SuppressWarnings("unchecked")
    public InstanceBinderBuilder(List<Configuration> configurations, T target, K... inputs) {
        this.target = target;
        this.inputs = inputs;
        this.configurations = configurations;
        Class<T> targetType = (Class<T>) target.getClass();
        Key<?> targetKey = KeyGen.getKey(targetType);
        this.instanceBinder = new InstanceBinderImpl<T, K>(targetKey, targetType);
    }

    public Builder build() {

        Map<Key<?>,Definition<K>> inputDefinitions = buildInputMarkedPoints();
        Definition<T> targetDefinition = buildTargetMarkedPoints();
        validateAndCreateInstanceBinder(targetDefinition, inputDefinitions);
        return this;

    }

    public InstanceBinder<T, K> getInstanceBinder() {
        return this.instanceBinder;
    }

    @SuppressWarnings("unchecked")
    private Map<Key<?>, Definition<K>>buildInputMarkedPoints() {
        Map<Key<?>,Definition<K>> definitions = new HashMap<Key<?>,Definition<K>>();
        for (K input : inputs) {
            Class<K> inputClass = (Class<K>) input.getClass();
            if (isPrimitiveOrWrapperType(inputClass)) {
                continue;
            }
            Key<?> key = KeyGen.getKey(inputClass);

            if (!BindingDefinitionCache.containsInputDefinition(key)) {
                AbstractMarkedPointBuilder<K> builder = new InputMarkedPointBuilder<K>(inputClass);
                builder.build();

                BindingDefinitionCache.applyInputDefinition(key, builder.getDefinition());
                if (!definitions.containsKey(key)) {
                    definitions.put(key,builder.getDefinition());
                }
            } else {
                Definition<K> definition = (Definition<K>) BindingDefinitionCache.getInputDefinition(key);
                definitions.put(key, definition);
            }
        }
        return definitions;
    }

    @SuppressWarnings("unchecked")
    private Definition<T> buildTargetMarkedPoints() {
        Key<?> key = KeyGen.getKey(target.getClass());
        Definition<T> definition = null;
        Class<T> targetType = (Class<T>) target.getClass();
        if (!BindingDefinitionCache.containsTargetDefinition(key)) {
            AbstractMarkedPointBuilder<T> builder = new TargetMarkedPointBuilder<T>(targetType);
            builder.build();
            definition = builder.getDefinition();

            BindingDefinitionCache.applyTargetDefinition(key, definition);
        } else {
            definition =  (Definition<T>) BindingDefinitionCache.getTargetDefinition(key);
        }
        return definition;
    }

    /**
     * Creates {@link InstanceBinderImpl}
     *
     * @param targetDefinedInfo
     * @param inputDefinitions
     * @return List of {@link BindingRelation}
     */
    private void validateAndCreateInstanceBinder(Definition<T> targetDefinition, Map<Key<?>,Definition<K>> inputDefinitions) {

        Key<?> targetTypeKey = KeyGen.getKey(targetDefinition.getType());
        for (Entry<Key<?>, MarkedPoint<T>> entry : ((BindingDefinition<T>)targetDefinition).getMarkedPoints().entrySet()) {
            Key<?> targetBoundKey = entry.getKey();
            TargetMarkedPoint<T> targetMarkedPoint = (TargetMarkedPoint<T>)entry.getValue();
            Class<?> inputClass = null;

            //  get binding configuration if it's defined.
            BindingConfiguration bindingConfiguration = searchBindingConfiguration(targetTypeKey, targetBoundKey);
            if (isNotNull(bindingConfiguration)) {
                inputClass = bindingConfiguration.getInputClass();
                Key<?> bindKey = isNotNull(bindingConfiguration.getBindName()) ? KeyGen.getKey(bindingConfiguration.getBindName()) : targetBoundKey;

                // if binding context is created, then go to next loop
                BindingContext<T, K> context = createBindingContext(targetMarkedPoint, bindKey, inputDefinitions, inputClass, bindingConfiguration);

                if (isNotNull(context)) {
                    MemberBinder<T, K> memberBinder = createMemberBinder(context);
                    instanceBinder.addMemberBinder(memberBinder);
                }
                continue;
            }

            inputClass = targetMarkedPoint.getInputClass();

            BindingContext<T, K> context = createBindingContext(targetMarkedPoint, targetBoundKey, inputDefinitions, inputClass, null);
            if (isNotNull(context)) {
                MemberBinder<T, K> memberBinder = createMemberBinder(context);
                instanceBinder.addMemberBinder(memberBinder);
            }

        }
    }

    /**
     * search and get defined {@link SetupInfo} if it's defined in LOCAL_CONFIGURATION_INFOS
     *
     * @param targetTypeKey
     * @param targetBoundKey
     * @return {@link SetupInfo}
     */
    private BindingConfiguration searchBindingConfiguration(Key<?> targetTypeKey, Key<?> targetBoundKey) {

        BindingConfiguration configuration = null;
        if (isNotNull(configurations) && configurations.size() > 0) {
            for (Configuration c : configurations) {
                Key<?> configKey = KeyGen.getKey(((BindingConfiguration)c).getBoundName());
                if (targetBoundKey.equals(configKey)) {
                    configuration = (BindingConfiguration)c;
                    break;
                }
            }
        }
        return configuration;
    }

    private static <T, K> MemberBinder<T, K> createMemberBinder(BindingContext<T, K> context) {
        Member member = context.getTargetMarkedPoint().getMember();

        if (member instanceof Field) {
            return  new FieldBinder<T, K>(context);
        } else if (member instanceof Method) {
            return new AccessorMethodBinder<T, K>(context);
        } else {
            throw new IllegalArgumentException("Fails to solve the field or method binder.");
        }

    }

    /**
     * Creates and returns {@link BindingContext}
     */
    @SuppressWarnings("unchecked")
    private static <T, K> BindingContext<T, K> createBindingContext(TargetMarkedPoint<T> targetMarkedPoint, Key<?> targetBoundKey,
            Map<Key<?>,Definition<K>> inputDefinitions, Class<?> inputClass, BindingConfiguration configuration) {

        BindingContext<T, K> bindingContext = null;
        if (isNotNull(inputClass)) {
            Key<?> inputClassKey = KeyGen.getKey(inputClass);
            if (inputDefinitions.containsKey(inputClassKey)) {
                Definition<?> inputDefinition = inputDefinitions.get(inputClassKey);

                if (((BindingDefinition<K>)inputDefinition).containsMarkedPoint(targetBoundKey)) {
                    InputMarkedPoint<K> inputMarkedPoint = (InputMarkedPoint<K>)((BindingDefinition<K>)inputDefinition).getMarkedPoint(targetBoundKey);

                    // type safe check.
                    verifyTypeSafe(inputMarkedPoint, targetMarkedPoint);

                    bindingContext = new BindingContext<T, K>(targetMarkedPoint, inputMarkedPoint, configuration);
                }
            }
        } else {
            int duplicateCount = 0;
            InputMarkedPoint<K> inputMarkedPoint = null;
            for (Entry<Key<?>, Definition<K>> inputEntry : inputDefinitions.entrySet()) {
                for (Entry<Key<?>, MarkedPoint<K>> definedEntry : ((BindingDefinition<K>) inputEntry.getValue()).getMarkedPoints().entrySet()) {
                    InputMarkedPoint<K> tmpInputMarkedPoint = (InputMarkedPoint<K>)definedEntry.getValue();
                    if (tmpInputMarkedPoint.getName().equals(targetMarkedPoint.getName())) {
                        if (duplicateCount == 0) {
                            inputMarkedPoint = tmpInputMarkedPoint;
                        }
                        duplicateCount++;
                    }
                }
            }

            if (duplicateCount > 1) {
                throw new IllegalArgumentException("The name [" + inputMarkedPoint.getName() + "] is duplicate.");
            }

            if (duplicateCount == 1) {
                // type safe check.
                verifyTypeSafe(inputMarkedPoint, targetMarkedPoint);

                bindingContext = new BindingContext<T, K>(targetMarkedPoint, inputMarkedPoint, configuration);
            }
        }
        return bindingContext;
    }

    /**
     * Verifies if input marked point and target marked point are functionally
     * equivalent.
     *
     * @param inputMarkedPoint
     *            input marked point.
     * @param targetMarkedPoint
     *            target marked point.
     */
    private static void verifyTypeSafe(InputMarkedPoint<?> inputMarkedPoint,
            TargetMarkedPoint<?> targetMarkedPoint) {
        if (isNotNull(inputMarkedPoint.getConverterDef().converterClass)) {
            verifyTypeSafeWithConverter(inputMarkedPoint, targetMarkedPoint);
        } else {
            verifyTypeSafeWithNoConverter(inputMarkedPoint, targetMarkedPoint);
        }
    }

    /*
     * type safe check method including converter method.
     */
    private static void verifyTypeSafeWithConverter(
            InputMarkedPoint<?> inputMarkedPoint, TargetMarkedPoint<?> targetMarkedPoint) {
        GenericClassType inputType = getInputGenericClassType(inputMarkedPoint);

        Type convertedValueType = inputMarkedPoint.getConverterDef().convertedValueType;
        GenericClassType parameter = GenericClassType.getGenericClassType(
                isPrimitiveType(convertedValueType) ? getWrapperType(convertedValueType) : convertedValueType);

        Type returnValueType = inputMarkedPoint.getConverterDef().converterMethod.getGenericReturnType();
        GenericClassType returnType = GenericClassType.getGenericClassType(
                isPrimitiveType(returnValueType) ? getWrapperType(returnValueType) : returnValueType);

        GenericClassType targetType = getTargetGenericClassType(targetMarkedPoint);

        checkTypeSafe(inputType, parameter, returnType, targetType);
    }

    /*
     * type safe check method not including converter method.
     */
    private static void verifyTypeSafeWithNoConverter(
            InputMarkedPoint<?> inputMarkedPoint, TargetMarkedPoint<?> targetMarkedPoint) {
        GenericClassType inputType = getInputGenericClassType(inputMarkedPoint);

        GenericClassType targetType = getTargetGenericClassType(targetMarkedPoint);

        checkTypeSafe(inputType, targetType);
    }

    /**
     * Gets {@link GenericClassType} in the input marked point.
     *
     * @param markedPoint
     *            input marked point.
     * @return {@link GenericClassType} object.
     */
    private static GenericClassType getInputGenericClassType(
            MarkedPoint<?> markedPoint) {
        GenericClassType genericClassType;
        if (markedPoint.getMember() instanceof Field) {
            Field field = (Field) markedPoint.getMember();
            Type type = field.getGenericType();
            genericClassType = GenericClassType.getGenericClassType(
                    isPrimitiveType(type) ? getWrapperType(type) : type);
        } else {
            Method method = (Method) markedPoint.getMember();
            Type type = method.getGenericReturnType();
            genericClassType = GenericClassType.getGenericClassType(
                    isPrimitiveType(type) ? getWrapperType(type) : type);
        }
        return genericClassType;
    }

    /**
     * Gets {@link GenericClassType} in the target marked point.
     *
     * @param markedPoint
     *            target marked point.
     * @return {@link GenericClassType} object.
     */
    private static GenericClassType getTargetGenericClassType(
            MarkedPoint<?> markedPoint) {
        GenericClassType genericClassType;
        if (markedPoint.getMember() instanceof Field) {
            Field field = (Field) markedPoint.getMember();
            Type type = field.getGenericType();
            genericClassType = GenericClassType.getGenericClassType(
                    isPrimitiveType(type) ? getWrapperType(type) : type);
        } else {
            Method method = (Method) markedPoint.getMember();
            Type type = method.getGenericParameterTypes()[0];
            genericClassType = GenericClassType.getGenericClassType(
                    isPrimitiveType(type) ? getWrapperType(type) : type);
        }
        return genericClassType;
    }

    private static void checkTypeSafe(GenericClassType inputType,
            GenericClassType parameter, GenericClassType returnType,
            GenericClassType targetType) {
        if (!(inputType.equals(parameter) && targetType.equals(returnType))) {
            throw new IllegalArgumentException("The inpout type ["
                    + inputType.getType()
                    + "] and the parameter type on converter method ["
                    + parameter.getType() + "], the target type ["
                    + targetType.getType()
                    + "] and the return type on converter method ["
                    + returnType.getType() + "] are not type safe.");
        }
    }

    private static void checkTypeSafe(GenericClassType inputType,
            GenericClassType targetType) {
        if (!inputType.equals(targetType)) {
            throw new IllegalArgumentException("The input type ["
                    + inputType.getType() + " and the target type ["
                    + targetType.getType() + "] are not type safe.");
        }
    }
}
