/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import static org.maru.core.util.ConditionUtil.isNotNull;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Type;

import org.maru.core.Key;
import org.maru.core.KeyGen;
import org.maru.core.annotation.Execute;
import org.maru.core.reflection.Clazz;
import org.maru.core.type.GenericClassType;
import org.maru.core.util.StringUtil;
import org.maru.dog.DefinitionException;
import org.maru.dog.annotation.Bind;
import org.maru.dog.converter.Converter;
import org.maru.dog.converter.ConverterIdentifier;
import org.maru.dog.core.AbstractMarkedPointBuilder;

import static org.maru.dog.util.MethodUtil.isGetterMethod;

final class InputMarkedPointBuilder<T> extends AbstractMarkedPointBuilder<T> {

    InputMarkedPointBuilder(Class<T> type) {
        super(type, new BindingDefinitionImpl<T>(type));
    }

    @Override
    public void makeMarkedPoint(Member member) {
        InputMarkedPoint<T> inputMarkedPoint = getInputMarkedPoint(member, target);
        if (isNotNull(inputMarkedPoint)) {
            String keyName = inputMarkedPoint.getName();
            Key<?> key = KeyGen.getKey(keyName);
            ((BindingDefinition<T>)definition).putMarkedPoint(key, inputMarkedPoint);
        }
    }

    private static <T> InputMarkedPoint<T> getInputMarkedPoint(Member member, Class<T> target) {
        InputMarkedPoint<T> inputMarkedPoint = null;
        if (member instanceof Field) {
            Field f = (Field) member;
            Bind bind = (Bind) f.getAnnotation(Bind.class);
            String keyName = f.getName();
            if (bind != null) {
                if (StringUtil.isNotEmpty(bind.name())) {
                    keyName = bind.name();
                }
            } else {
                return null;
            }

            // check if converter annotation is defined.
            Annotation converterAnnotation = checkAndGetConverterClassAnnotation(f);
            ConverterDef converterDef = new ConverterDef();
            if (converterAnnotation != null) {
                converterDef = getConverterClassInformation(converterAnnotation, f.getGenericType());
                converterDef.verifyConverter();
            }

            // create marked point
            inputMarkedPoint = createInputMarkedPointInstance(target, keyName, f, converterDef);

            return inputMarkedPoint;
        } else {

            Method m = (Method) member;
            Bind bind = (Bind) m.getAnnotation(Bind.class);

            // The only marked method is used to bind.
            if (bind != null) {
                String methodName = m.getName();

                // check whether or not the method is java beans accessor.
                if (isGetterMethod(methodName, m)) {

                    String keyName = null;

                    if (StringUtil.isNotEmpty(bind.name())) {
                            keyName = bind.name();
                    } else {
                        /*
                         * convert keyName from method name. Remove "get" or "set"
                         * from method name.
                         * If the name of the keyName starts with Upper case,
                         * change it to lower case.
                         */
                        keyName = canonicalizeMethodName(methodName);
                    }

                    if (StringUtil.isEmpty(keyName)) {
                        throw new IllegalArgumentException(
                                "Fail to make bind identifier. " +
                                "It might not be set the bind name, or gettter method name is illegal style");
                    }

                    // check if converter annotation is defined.
                    Annotation converterAnnotation = checkAndGetConverterClassAnnotation(m);
                    ConverterDef converterDef = new ConverterDef();
                    if (converterAnnotation != null) {
                        converterDef = getConverterClassInformation(converterAnnotation, m.getGenericReturnType());
                        converterDef.verifyConverter();
                    }

                    inputMarkedPoint = createInputMarkedPointInstance(target, keyName, m, converterDef);
                } else {
                    throw new DefinitionException(m.getName() + " method is not getter method following JavaBeans.");
                }
            }
        }
        return inputMarkedPoint;
    }

    private static <T> InputMarkedPoint<T> createInputMarkedPointInstance(Class<T> target,
            String name, Member member,
            ConverterDef converterDef) {
        return new InputMarkedPoint<T>(target, name, member, converterDef);
    }

    private static ConverterDef getConverterClassInformation(
            Annotation converterAnnotation, Type type) {
        ConverterDef converterDef = new ConverterDef();

        if (converterAnnotation instanceof Converter) {
            Converter converter = (Converter) converterAnnotation;

            if (converter != null) {
                converterDef.converterClass = converter.converterClass();
                String executeIdentifier = converter.execute();
                converterDef.converterMethod = checkAndGetConverterClassMethod(
                        executeIdentifier, converterDef.converterClass,
                        GenericClassType.getClassFromType(type));
            }
        } else {
            ConverterIdentifier converterIdentifier = converterAnnotation
                    .annotationType().getAnnotation(ConverterIdentifier.class);

            converterDef.converterClass = converterIdentifier.value();

            converterDef.converterMethod = checkAndGetConverterMethod(
                    converterAnnotation, converterDef.converterClass, type);
        }
        converterDef.converterAnnotation = converterAnnotation;

        return converterDef;
    }



    private static Method checkAndGetConverterClassMethod(
            String executeIdentifier, Class<?> converterClass, Class<?> type) {
        Method method = null;

        if (StringUtil.isNotEmpty(executeIdentifier)) {
            // gets the identified converter method.
            int duplicatingCount = 0;

            for (Method m : converterClass.getDeclaredMethods()) {
                Execute execute = m.getAnnotation(Execute.class);
                if (isNotNull(execute) && (execute.value().equals(executeIdentifier) || m.getName().equals(executeIdentifier))) {
                    method = m;
                    duplicatingCount++;
                }
            }

            if (duplicatingCount > 1) {
                throw new DefinitionException("[" + executeIdentifier + "] The name of executing identifier is duplicate.");
            }
        } else {
            method = Clazz.getDeclaredMethod(converterClass, "execute", type);

        }
        if (checkSimpleConverterMethodDefinition(method)) {
            return method;

        } else {
            throw new DefinitionException(
                    "The Definition of simple convert class [" + converterClass.getName() + "] is invalid.");
        }

    }

    private static boolean checkSimpleConverterMethodDefinition(Method method) {
        if (isNotNull(method)
                && !(method.getGenericReturnType().equals(void.class))
                        && method.getGenericParameterTypes().length == 1) {
            return true;
        }
        return false;
    }

    /**
     * checks number of converter class.<br>
     * Do not let you use multiple converter annotations on one field or method.
     *
     * @param member
     */
    private static Annotation checkAndGetConverterClassAnnotation(Member member) {
        int number = 0;
        Annotation converterAnnotation = null;
        if (member instanceof Field) {
            Field field = (Field) member;
            for (Annotation annotation : field.getAnnotations()) {
                if (isConverterAnnotation(annotation)) {
                    converterAnnotation = annotation;
                    number++;
                }
            }
        } else if (member instanceof Method) {
            Method method = (Method) member;

            for (Annotation annotation : method.getAnnotations()) {
                if (isConverterAnnotation(annotation)) {
                    converterAnnotation = annotation;
                    number++;
                }
            }
        }

        if (number > 1) {
            throw new IllegalArgumentException(
                    "Two converter annotations are specified on one attribute");
        }
        return converterAnnotation;
    }

    /**
     * Check if the annotation passed by the argument has {@link ConverterIdentifier}, then
     * return true or false.
     * @param annotation
     * @return true if annotation has {@link ConverterIdentifier}, otherwise false.
     */
    private static boolean isConverterAnnotation(Annotation annotation) {
        if (annotation instanceof Converter) {
            return true;
        } else {
            ConverterIdentifier converterIdentifier = annotation.annotationType().getAnnotation(ConverterIdentifier.class);
            if (isNotNull(converterIdentifier)) {
                return true;
            }
            return false;
        }
    }



    private static Method checkAndGetConverterMethod(
            Annotation converterAnnotation, Class<?> converterClass, Type type) {
        int count = 0;
        Method converterMethod = null;

        for (Method method : converterClass.getDeclaredMethods()) {
            ConverterIdentifier converterIdentifier = method.getAnnotation(ConverterIdentifier.class);

            if(converterIdentifier != null && converterIdentifier.value().equals(converterAnnotation.annotationType())) {
                converterMethod = method;
                count++;
            }
        }

        if (count > 1) {
            throw new IllegalArgumentException(
                    converterAnnotation.annotationType().getName() + " is not unique on the " + converterClass.getName());
        }

        if (isNotNull(converterMethod)
                && !(converterMethod.getGenericReturnType().equals(void.class))
                        && converterMethod.getGenericParameterTypes().length >= 1) {
            return converterMethod;

        } else {
            throw new DefinitionException(
                    "The Definition of the extended convert class [" + converterClass.getName() + "] is invalid.");
        }

    }


}
