/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import static org.maru.core.util.ConditionUtil.isNotNull;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import org.maru.core.Key;
import org.maru.core.KeyGen;
import org.maru.core.annotation.Value;
import org.maru.core.reflection.Clazz;
import org.maru.core.reflection.Fields;
import org.maru.core.reflection.Methods;
import org.maru.dog.core.ObjectFactory;

final class BindingFactoryImpl implements BindingFactory {

    @Override
    public <T, K> Object get(BindingContext<T, K> context, K input) {
        boolean converterOff = false;

        Member inputMember = context.getInputMarkedPoint().getMember();
        InputMarkedPoint<K> inputMarkedPoint = context.getInputMarkedPoint();
        BindingConfiguration configuration = context.getConfiguration();

        if (isNotNull(configuration)) {
            converterOff = configuration.isConverterOff();
        }

        List<Object> inputObjects = !converterOff ? getInputObjects(inputMarkedPoint) : new ArrayList<Object>();

        Object object = null;
        if (inputMember instanceof Field) {
            Field inputField = (Field) inputMember;
            object = Fields.get(inputField, input);
        }
        if (inputMember instanceof Method) {
            Method inputMethod = (Method) inputMember;
            object = Methods.invoke(inputMethod, input, null);
        }

        int valuePosition = context.getInputMarkedPoint().getConverterDef().identifierParameters.indexOf(KeyGen.getKey(Value.class));
        if (valuePosition < 0) {
            inputObjects.add(object);
        } else {
            inputObjects.add(valuePosition, object);
        }
        Object convertedValue = !converterOff ? getConvertedValue(inputMarkedPoint, inputObjects) : null;

        return convertedValue != null ? convertedValue : object;

    }

    private List<Object> getInputObjects(InputMarkedPoint<?> inputMarkedPoint) {
        List<Object> inputObjects = new ArrayList<Object>();
        Annotation converterAnnotation = inputMarkedPoint.getConverterDef().converterAnnotation;

        for (Key<?> attributeKey : inputMarkedPoint.getConverterDef().identifierParameters) {

            if (attributeKey.getType().equals(String.class)) {
                String attribute = (String) attributeKey.getInstance();
                Method m = Clazz.getDeclaredMethod(converterAnnotation.annotationType(), attribute, null);
                Object value = Methods.invoke(m, converterAnnotation, null);
                inputObjects.add(value);
            }

        }
        return inputObjects;
    }

    private <K> Object getConvertedValue(InputMarkedPoint<K> inputMarkedPoint, List<Object> inputObjects) {

        Object convertedValue = null;
        Class<?> converterClass = inputMarkedPoint.getConverterDef().converterClass;
        Method converterMethod = inputMarkedPoint.getConverterDef().converterMethod;

        if (isNotNull(converterClass) && isNotNull(converterMethod)) {
            Object[] inputObjectArray = inputObjects.toArray();

            Key<?> converterKey = KeyGen.getKey(converterClass);

            Object converter = ObjectFactory.newInstance(converterClass);
            ConverterCache.apply(converterKey, converter);
            convertedValue = Methods.invoke(converterMethod, converter, inputObjectArray);
        }
        return convertedValue;
    }

}
