/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.dog.bind;

import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.maru.core.Key;
import org.maru.core.MapFactory;
import org.maru.dog.core.Definition;

final class BindingDefinitionCache {

    private BindingDefinitionCache(){}

    /** The definition of the target object */
    private static final ConcurrentMap<Key<?>, Definition<?>> CACHED_TARGET_DEFINITIONS = MapFactory.createConcurrentSoftMap();

    /** The definition of the input objects */
    private static final ConcurrentMap<Key<?>, Definition<?>> CACHED_INPUT_DEFINITIONS  = MapFactory.createConcurrentSoftMap();

    public static Definition<?> getInputDefinition(Key<?> key) {
        return CACHED_INPUT_DEFINITIONS.get(key);
    }

    public static Definition<?> getTargetDefinition(Key<?> key) {
        return CACHED_TARGET_DEFINITIONS.get(key);
    }

    public static boolean containsTargetDefinition(Key<?> key) {
        return CACHED_TARGET_DEFINITIONS.containsKey(key);
    }

    public static boolean containsInputDefinition(Key<?> key) {
        return CACHED_INPUT_DEFINITIONS.containsKey(key);
    }

    public static Definition<?> applyTargetDefinition(Key<?> key, Definition<?> definition) {
        return CACHED_TARGET_DEFINITIONS.putIfAbsent(key, definition);
    }

    public static Definition<?> applyInputDefinition(Key<?> key, Definition<?> definition) {
        return CACHED_INPUT_DEFINITIONS.putIfAbsent(key, definition);
    }

    public static void clearAll() {
        Lock lock = new ReentrantLock();
        lock.lock();
        try {
            if (!CACHED_TARGET_DEFINITIONS.isEmpty()) {
                CACHED_TARGET_DEFINITIONS.clear();
            }
            if (!CACHED_INPUT_DEFINITIONS.isEmpty()) {
                CACHED_INPUT_DEFINITIONS.clear();
            }
        } finally {
            lock.unlock();
        }
    }

    public static Definition<?> removeTargetDefinition(Key<?> key) {
        return CACHED_TARGET_DEFINITIONS.remove(key);
    }

    public static Definition<?> removeInputDefinition(Key<?> key) {
        return CACHED_INPUT_DEFINITIONS.remove(key);
    }
}
