/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.core.type;

import java.io.Serializable;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;

/**
 * Generic class type wrapper.
 *
 */
public class GenericClassType implements Type,Serializable {

    /**
     *
     */
    private static final long serialVersionUID = -5400526810102785412L;
    private Type type;

    public GenericClassType(){
    }

    public GenericClassType(Type type) {
        this.type = getWrappedType(type);
    }

    public Type getType() {
        return type;
    }

    public void setType(Type type) {
        this.type = type;
    }

    public static GenericClassType getGenericClassType(Type type) {
        return new GenericClassType(type);
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((type == null) ? 0 : type.hashCode());
        return result;
    }

    public static Class<?> getClassFromType(Type type) {
        if (type instanceof Class<?>) {
            return WrappedClass.getClassFromType(type);
        } else if (type instanceof ParameterizedType) {
            return WrappedParameterizedType.getClassFromType(type);
        } else if (type instanceof GenericArrayType) {
            return WrappedGenericArrayType.getClassFromType(type);
        } else if (type instanceof TypeVariable<?>) {
            return WrappedTypeVariable.getClassFromType((TypeVariable<?>)type);
        } else {
            throw new IllegalArgumentException("Unexpected type : " + type);
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        GenericClassType other = (GenericClassType) obj;
        if (type == null) {
            if (other.type != null)
                return false;
        } else {
            if (type instanceof WrappedClass
                    && other.type instanceof WrappedClass) {
                WrappedClass type1 = (WrappedClass) type;
                WrappedClass type2 = (WrappedClass) other.type;
                return type1.equals(type2);
            } else if (type instanceof WrappedParameterizedType
                    && other.type instanceof WrappedParameterizedType) {
                WrappedParameterizedType type1 = (WrappedParameterizedType) type;
                WrappedParameterizedType type2 = (WrappedParameterizedType) other.type;
                return type1.equals(type2);
            } else if (type instanceof WrappedWildcardType
                    && other.type instanceof WrappedWildcardType) {
                WrappedWildcardType type1 = (WrappedWildcardType) type;
                WrappedWildcardType type2 = (WrappedWildcardType) other.type;
                return type1.equals(type2);
            } else if (type instanceof WrappedGenericArrayType
                    && other.type instanceof WrappedGenericArrayType) {
                WrappedGenericArrayType type1 = (WrappedGenericArrayType) type;
                WrappedGenericArrayType type2 = (WrappedGenericArrayType) other.type;
                return type1.equals(type2);
            } else if (type instanceof WrappedTypeVariable<?>
                    && other.type instanceof WrappedTypeVariable<?>) {
                WrappedTypeVariable<?> type1 = (WrappedTypeVariable<?>) type;
                WrappedTypeVariable<?> type2 = (WrappedTypeVariable<?>) other.type;
                return type1.equals(type2);
            } else if (type.equals(other.type)) {
                return true;
            }
        }
        return false;
    }

    /**
     * returns a real type.
     *
     * @param type
     * @return real type.
     */
    public static Type getWrappedType(Type type) {
        if (type instanceof Class<?>) {
            return new WrappedClass(type);
        } else if (type instanceof ParameterizedType) {
            return new WrappedParameterizedType((ParameterizedType) type);
        } else if (type instanceof WildcardType) {
            return new WrappedWildcardType((WildcardType) type);
        } else if (type instanceof GenericArrayType) {
            return new WrappedGenericArrayType((GenericArrayType) type);
        } else if (type instanceof TypeVariable<?>) {
            return new WrappedTypeVariable((TypeVariable<?>)type);
        } else {
            return type;
        }
    }

    public static GenericClassType[] getGenericClassTypes(Type[] types) {
        GenericClassType[] genericClassTypes = new GenericClassType[types.length];

        for (int i = 0; i < types.length; i++) {
            genericClassTypes[i] = getGenericClassType(types[i]);
        }
        return genericClassTypes;
    }
}
