/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.core.reflection;

import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

/**
 * Class Utility.
 *
 */
public final class Clazz {

    private Clazz() {}

    /**
     * Gets default constructor
     *
     * @param type
     *            target to get default constructor
     * @return default constructor or null.
     */
    public static <T> Constructor<?> getDefaultConstructor(Class<T> type) {
        for (Constructor<?> constructor : type.getDeclaredConstructors()) {
            if (constructor.getGenericParameterTypes().length == 0) {
                if (!constructor.isAccessible()) {
                    constructor.setAccessible(true);
                }
                return constructor;
            }
        }
        return null;
    }

    /**
     * Checks if class passed as parameter is an abstract class.
     *
     * @param type
     *            confirmed class.
     * @return true if it is abstract class, otherwise false.
     */
    public static <T> boolean isAbstractClass(Class<T> type) {
        if (Modifier.isAbstract(type.getModifiers())) {
            return true;
        }
        return false;
    }

    /**
     * Checks if class passed as a parameter is an interface.
     *
     * @param type
     *            checked class
     * @return true if it is interface, otherwise false.
     */
    public static <T> boolean isInterface(Class<T> type) {
        if (Modifier.isInterface(type.getModifiers())) {
            return true;
        }
        return false;
    }

    /**
     * Checks if class passed as a parameter is an annotation.
     *
     * @param type
     *            checked class
     * @return true if it is not annotation, otherwise false.
     */
    public static <A> boolean isAnnotationClass(Class<? extends A> type) {
        if (type.isAnnotation()) {
            return true;
        }
        return false;
    }

    /**
     * Checks if class passed as a parameter is an array class
     *
     * @param type
     *            checked class
     * @return true if it is array class, otherwise false
     */
    public static <T> boolean isArray(Class<T> type) {
        if (type.isArray()) {
            return true;
        }
        return false;
    }

    /**
     * Checks if class passed as a parameter is an instance of {@link Class}.
     *
     * @param type
     *            checked class
     * @return true if it is an instance of {@link Class}, otherwise false
     */
    public static <T> boolean isClassType(T type) {
        if (type instanceof Class<?>) {
            return true;
        }
        return false;
    }

    /**
     * Checks if an instance passed as a parameter is an annotation instance.
     *
     * @param type
     *            checked instance.
     * @return true if it's the annotation instance, otherwise false.
     */
    public static <A> boolean isAnnotation(A type) {
        if (type instanceof Annotation) {
            return true;
        }
        return false;
    }

    /**
     * Gets declared method from the target class passed at the first parameter.
     * @param target target class
     * @param name name
     * @param parameterTypes array of parameter type
     * @return method
     */
    public static Method getDeclaredMethod(Class<?> target, String name, Class<?>... parameterTypes) {
        try {
            return target.getDeclaredMethod(name, parameterTypes);
        } catch (SecurityException e) {
            throw new RuntimeException(e);
        } catch (NoSuchMethodException e) {
            throw new RuntimeException(e);
        }
    }

}
