/*
 * Copyright 2011 maru project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maru.core;

import java.lang.annotation.Annotation;
import java.lang.reflect.Type;

import org.maru.core.util.ConditionUtil;

/**
 * Key generator
 *
 *
 */
public final class KeyGen {

    private KeyGen() {
    }

    public static <T> Key<T> getKey(T value) {
        ConditionUtil.verifyNotNull(value);
        return new KeyImpl<T>(value, new ObjectComposer<T>(value));
    }

    @SuppressWarnings("unchecked")
    public static <T> Key<T> getKey(Class<T> type) {
        ConditionUtil.verifyNotNull(type);
        if (Annotation.class.isAssignableFrom(type)) {
            return new KeyImpl<T>(type, new AnnotationTypeComposer(
                    (Class<? extends Annotation>) type));
        }
        return new KeyImpl<T>(type, new ClassComposer<T>(type));
    }

    public static <A extends Annotation> Key<A> getKey(A annotation) {
        ConditionUtil.verifyNotNull(annotation);
        return new KeyImpl<A>(annotation, new AnnotationComposer<A>(annotation));
    }

    public interface Composer {

        <T> T getCompositeObject();

        Class<?> getType();

        Composer getTypeComposer();
    }

    /**
     * composer type enum.
     */
    enum ComposerType {
        OBJECT, CLASS, ANNOTATION, ANNOTATION_TYPE;
    }

    /**
     * Composer for the object type.
     */
    static class ObjectComposer<T> implements Composer {

        final T object;

        ObjectComposer(T object) {
            ConditionUtil.verifyNotNull(object, "The object is null");
            this.object = object;
        }

        @SuppressWarnings("unchecked")
        @Override
        public T getCompositeObject() {
            return object;
        }

        @Override
        public int hashCode() {
            return object != null ? object.hashCode() : 0;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (!(obj instanceof ObjectComposer))
                return false;
            ObjectComposer<?> other = (ObjectComposer<?>) obj;
            if (object == null) {
                if (other.object != null)
                    return false;
            } else if (object.getClass() != other.object.getClass()) {
                return false;
            } else if (!object.equals(other.object))
                return false;
            return true;
        }

        @Override
        public String toString() {
            return object.toString();
        }

        @Override
        public Class<?> getType() {
            return this.object.getClass();
        }

        @SuppressWarnings({ "unchecked", "rawtypes" })
        @Override
        public Composer getTypeComposer() {
            return new ClassComposer(object.getClass());
        }

    }

    /**
     * Composer for the class type.
     */
    static class ClassComposer<T> implements Composer {

        final Class<T> type;

        ClassComposer(Class<T> type) {
            ConditionUtil.verifyNotNull(type, "The type is null");
            this.type = type;
        }

        @SuppressWarnings("unchecked")
        @Override
        public Class<T> getCompositeObject() {
            return type;
        }

        @Override
        public int hashCode() {
            return type.hashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (!(obj instanceof ClassComposer))
                return false;
            ClassComposer<?> other = (ClassComposer<?>) obj;
            if (type == null) {
                if (other.type != null)
                    return false;
            } else if (!type.equals(other.type))
                return false;
            return true;
        }

        @Override
        public String toString() {
            return type.toString();
        }

        @Override
        public Class<T> getType() {
            return this.type;
        }

        @Override
        public Composer getTypeComposer() {
            return this;
        }

    }

    /**
     * Composer for the annotation.
     */
    static class AnnotationComposer<A extends Annotation> implements Composer {

        final A annotation;

        AnnotationComposer(A annotation) {
            ConditionUtil.verifyNotNull(annotation, "The annotation is null");
            this.annotation = annotation;
        }

        @SuppressWarnings("unchecked")
        @Override
        public A getCompositeObject() {
            return this.annotation;
        }

        @Override
        public Class<? extends Annotation> getType() {
            return this.annotation.annotationType();
        }

        @Override
        public int hashCode() {
            return annotation.hashCode();
        }

        @SuppressWarnings("unchecked")
        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (!(obj instanceof AnnotationComposer))
                return false;
            AnnotationComposer<A> other = (AnnotationComposer<A>) obj;
            if (annotation == null) {
                if (other.annotation != null)
                    return false;
            } else if (annotation.annotationType() != other.annotation
                    .annotationType()) {
                return false;
            } else if (!annotation.equals(other.annotation)) {
                return false;
            }

            return true;
        }

        @Override
        public String toString() {
            return this.annotation.toString();
        }

        @Override
        public Composer getTypeComposer() {
            return new AnnotationTypeComposer(annotation.annotationType());
        }

    }

    /**
     * Composer for the annotation type.
     */
    static class AnnotationTypeComposer implements Composer {

        final Class<? extends Annotation> annotationType;

        AnnotationTypeComposer(Class<? extends Annotation> annotationType) {
            ConditionUtil.verifyNotNull(annotationType, "The annotation type is null");
            this.annotationType = annotationType;
        }

        @SuppressWarnings("unchecked")
        @Override
        public Class<? extends Annotation> getCompositeObject() {
            return this.annotationType;
        }

        @Override
        public Class<?> getType() {
            return this.annotationType;
        }

        @Override
        public int hashCode() {
            return annotationType.hashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (!(obj instanceof AnnotationTypeComposer))
                return false;
            AnnotationTypeComposer other = (AnnotationTypeComposer) obj;
            if (annotationType == null) {
                if (other.annotationType != null)
                    return false;
            } else if (!annotationType.equals(other.annotationType))
                return false;
            return true;
        }

        @Override
        public String toString() {
            return this.annotationType.toString();
        }

        @Override
        public Composer getTypeComposer() {
            return this;
        }

    }

    /**
     * Implementation of Key interface.
     *
     */
    static class KeyImpl<T> implements Key<T> {

        final Class<?> type;
        final int hashCode;
        T instance = null;
        Composer composer = null;
        ComposerType composerType = null;

        KeyImpl(Class<T> type, Composer compoer) {
            this.type = type;
            this.composer = compoer;
            this.composerType = getComposerType(composer);
            this.hashCode = calculateHashCode(this.type, composerType, composer);

        }

        KeyImpl(T value, Composer composer) {
            this.instance = value;
            this.type = value.getClass();
            this.composer = composer;
            this.composerType = getComposerType(composer);
            this.hashCode = calculateHashCode(this.type, this.composerType,
                    this.composer);
        }

        @SuppressWarnings("unchecked")
        <A extends Annotation> KeyImpl(A annotation, Composer composer) {
            type = annotation.annotationType();
            instance = (T) annotation;
            this.composer = composer;
            this.composerType = getComposerType(composer);
            this.hashCode = calculateHashCode(this.type, composerType, composer);
        }

        private static ComposerType getComposerType(Composer composer) {
            if (composer instanceof ClassComposer) {
                return ComposerType.CLASS;
            } else if (composer instanceof ObjectComposer) {
                return ComposerType.OBJECT;
            } else if (composer instanceof AnnotationComposer) {
                return ComposerType.ANNOTATION;
            } else if (composer instanceof AnnotationTypeComposer) {
                return ComposerType.ANNOTATION_TYPE;
            } else {
                throw new IllegalArgumentException("Unexpected composer : "
                        + composer.getClass().getName());
            }
        }

        private static int calculateHashCode(Class<?> type,
                ComposerType composerType, Composer composer) {

            if (composerType.equals(ComposerType.CLASS)) {
                return type.hashCode() * 31 + composerType.hashCode();
            } else if (composerType.equals(ComposerType.OBJECT)) {
                return composer.hashCode() * 31 + composerType.hashCode();
            } else if (composerType.equals(ComposerType.ANNOTATION)) {
                return composer.hashCode() * 31 + composerType.hashCode();
            } else if (composerType.equals(ComposerType.ANNOTATION_TYPE)) {
                return composer.hashCode() * 31 + composerType.hashCode();
            } else {
                return 0;
            }
        }

        @Override
        public int hashCode() {
            return this.hashCode;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            KeyImpl<?> other = (KeyImpl<?>) obj;
            if (!(this.composer.equals(other.composer)
                    && this.composerType.equals(other.composerType) && this.type.equals(other.type))) {
                return false;
            }
            return true;
        }

        @Override
        public Type getType() {
            return this.type;
        }

        @Override
        public Composer getComposer() {
            return this.composer;
        }

        @Override
        public T getInstance() {
            return instance;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("KeyImpl [");
            sb.append("composer=" + composer.toString() + ", ");
            sb.append("composerType=" + composerType.toString() + ", ");
            sb.append("hashCode=" + hashCode + ", ");
            sb.append("instance=" + instance + ", ");
            sb.append("type=" + type + "]");
            return sb.toString();

        }

    }
}
