/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import plus.util.Escape;
import plus.util.NumHelper;

/**
 * Lexical Analyzer - Token.
 *
 * @author kunio himei.
 */
abstract class Token extends Comment {

    /**
     * 変数名を表わす正規表現.
     */
    private static final String rxNAME = "[A-Za-z_][A-Za-z_0-9.$]*";

    /**
     * 変数名を表わす正規表現.
     */
    private static final LexRegx
            rxName = new LexRegx("^(`" + rxNAME + "`|" + rxNAME + "|[$]).*");

    /**
     * 数値を表わす正規表現.
     */
    private static final LexRegx rxNumber = new LexRegx("^("
            + "(?:0[Xx][0-9A-Fa-f_]+|" + "0[Bb][01_]+|"
            + "[0-9][0-9_]*[.]?[0-9_]*|" + "[.][0-9_]+)"
            + "(?:[eE][+-]?[0-9_]+)?" + ").*");

    /**
     * 文字列(single quotes)を表わす正規表現. '..'
     */
    private static final LexRegx rxString_SingleQuote = new LexRegx(
            "^('[^'\\\\]*(?:\\\\.[^'\\\\]*)*').*");

    /**
     * 文字列を表わす正規表現. * "[^"\]*(?:\\.[^"\]*)*"
     */
    private static final LexRegx rxString_DoubleQuote = new LexRegx(
            "^(\"[^\"\\\\]*(?:\\\\.[^\"\\\\]*)*\").*");

    /**
     * 行注釈を表わす正規表現.
     */
    private static final LexRegx rxComment = new LexRegx(
            "^(\\s*)(#.*|//.*|/\\*.*?\\*/)(.*)"); // #.. //.. /*..*/

    /**
     * 注釈(Annotation)を表わす正規表現 - 最短一致.
     */
    private static final LexRegx rxAnnotation = new LexRegx(
            "^(@[\\w]+([(].*?[)])?).*");

    //////////////////////////////////////////////////////////////////////
    /**
     * 演算子を表わす正規表現.
     * <p>
     * リダイレクト:  <, >, >>, |&
     * インクリメント・デクリメント演算子	++, –-
     * 算術演算子:   +, -, *, /, %, POWER **     - Arithmetic operators
     * シフト演算子:  <<, >>, >>>(符号無し)       - Shift operator
     * ビット演算子:  AND &, OR |, XOR ^,  NOT ~ - Bit operator
     * 比較演算子:   !=, ==、 <、 <=、 >、 >=  - Comparison operator
     * <p>
     * 論理演算子:   &, &&, |, ||, !, ^
     * 条件演算子:   ?
     * <p>
     * NOTE ビット否定('~'NOT)は、正規表現適合と競合するため'`~'とする.
     * REMIND Regex.が誤動作するため、長い物から記述する.
     */
    private static final LexRegx
            rxTokens = new LexRegx("^(" + // NOTE REMIND 全ての演算子☆☆
            "[!]?~|" + // 正規表現適合
            ">>|" + // アペンドパイプ
            "[|]&|" + // 双方向パイプ
            "[!=]={1,2}|[<>]=?|" + // 比較演算子
            "&{1,2}|[|]{1,2}|[!^?]|" + // 論理演算子(boolean)

            "[+][+]|--|" + // inc/dec演算子
            "([-+/%]|[*]{1,2})=?|" + // 算術演算子 post'='
            "(<<|>{2,3})=?|" + // シフト演算子 post'='
            "([&|^]|`~)=?|" + // ビット演算子 post'=' REMIND '`~'
            "[.]{1,2}|" + // Function Value.
            ".).*"); // 1文字のキーワードを定義なしで利用するためここで食べる.

    /**
     * get Next Token.
     */
    @Override
    Object nextToken() {
        if (rxAnnotation.find(super.yyText)) { // Annotation 生成☆
            String a = rxAnnotation.group(1);
            dropMatchToken(a);
//            System.err.println("Lex.Annotation: " + a);
            super.yyAnnotation.add(a);
            return new Node.Annotation(a);

        } else if (rxComment.find(super.yyText)) { // 注釈 (Comment)
            doComment();
            return super.yyText.isEmpty() ? Advance.YYEOL : nextToken(); // recursive call

        } else if (rxNumber.find(super.yyText)) { // 数値
            String a = rxNumber.group(1);
            dropMatchToken(a);
            String x = a.replace("_", "");
            return new Term.NUMBER(x,
                    NumHelper.normalise(NumHelper.doubleValue(x)));

        } else if (rxString_DoubleQuote.find(super.yyText)) { // 文字列★ "..."
            String a = rxString_DoubleQuote.group(1);
            dropMatchToken(a);
            String x = a.substring(1, a.length() - 1);
            return new Term.STRING(Escape.nopFilter(x)); // "nop"

        } else if (rxString_SingleQuote.find(super.yyText)) { // 文字列★ '...'
            String a = rxString_SingleQuote.group(1);
            dropMatchToken(a);
            String x = a.substring(1, a.length() - 1);
            return new Term.STRING(Escape.encodeString(x, "\"")); // ダブルクォート文字列化.

        } else if (rxName.find(super.yyText)) { // 変数名
            String a = rxName.group(1);
            dropMatchToken(a);
            String x = Operator.changeOperator(a); // 読み替え
            if ((1 < x.length())
                    && ((0 <= x.indexOf('.')) || (0 <= x.indexOf('$')))) {
                return new Term.BOXING(x); // 名前に[.$]を含む

            } else if (Operator.isOperator(x)) { // 演算子
                return x;

            } else if (Keyword.contains(x)) { // 予約語
                return Keyword.apply(x);

            } else {
                Symbols.createType(x);
                return new Node.NAME(Keyword.SyyNAME, x); // 変数
            }
        } else {

            String a = LexRegx.isNativeRegexp(super.yyText); // 正規表現★
            if ((0 >= super.yyEnableANAREG) && (null != a)) { // Regex解析は許可されている
                dropMatchToken(a);
                String x = a.substring(1, a.length() - 1);
                return new Term.REGEXP(Escape.nopFilter(x)); // /nop/

            } else if (rxTokens.find(super.yyText)) { // 演算子
                String x = rxTokens.group(1);
                dropMatchToken(x);
                return Operator.changeOperator(x); // 読み替え☆

            } else {
                throw new IllegalStateException("Lex: " + super.yyText);
            }
        }
    }
}