/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.bench.target;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Benchmark Targets.
 * <p>
 * <pre><code>
 * // case.1
 *  　while (0 <= --i && ' ' >= s.charAt(i))
 *      ; // do nothing
 *     ↓
 * // case.2
 *  　while (0 <= --i)
 *      if (' ' < s.charAt(i)) break; </code></pre>
 * <p>
 * case.1,2 while の複合条件を条件式 2つに分けてもバイトコードは、同じです.<br>
 * case.1,2, Even if the compound condition of while is divided into
 * two conditional expressions, the bytecode is still.
 * </p>
 *
 * @author kunio himei.
 */
public class Trim {

    private static final Pattern RIGHT_TRIM_CLASSIC = Pattern.compile("[\\s]+$");
    private static final Pattern RIGHT_TRIM_REGEX = Pattern.compile("[\0- ]+$");

    /*
     * 1. Right trim (REGEX).
     */
    public static String rTrimRegex01Classic(String s) {
        Matcher m = RIGHT_TRIM_CLASSIC.matcher(s);
        if (m.find()) return s.substring(0, m.start());
        return s;
    }

    /*
     * 2. Right trim New. (REGEX).
     */
    public static String rTrimRegex02New(String s) {
        Matcher m = RIGHT_TRIM_REGEX.matcher(s);
        if (m.find()) return s.substring(0, m.start());
        return s;
    }

    //////////////////////////////////////////////////////////////////////
    // left Trim Parts. (CharSequence)
    private static int leftTrimParts(CharSequence s, int end) {
        int start = 0;
        while (start < end && ' ' >= s.charAt(start))
            start++;
        return start;
    }

    // right Trim Parts. (CharSequence)
    private static int rightTrimParts(CharSequence s, int start, int end) {
        while (start <= --end)
            if (' ' < s.charAt(end)) break;
        return end + 1;
    }

    //////////////////////////////////////////////////////////////////////
    /*
     * 4. trim. (String)
     */
    public static String trim04String(String s) {
        int length = s.length();
        int start = leftTrimParts(s, length);
        int end = rightTrimParts(s, start, length);
        return end < length || 0 < start ? s.substring(start, end) : s;
    }

    /*
     * 5. trim. (CharSequence)
     */
    public static String trim05CharSeq(CharSequence s) {
        int length = s.length();
        int start = leftTrimParts(s, length);
        int end = rightTrimParts(s, start, length);
        if (end < length || 0 < start) s = s.subSequence(start, end);
        return (s instanceof String) ? (String) s : s.toString();
    }

    /**
     * 6. Right trim. (String)
     */
    public static String rTrim06String(String s) {
        int length = s.length();
        int end = rightTrimParts(s, 0, length);
        return end < length ? s.substring(0, end) : s;
    }

    /*
     * 7. Right trim. (StringBuilder)
     */
    @SuppressWarnings("UnusedReturnValue")
    public static StringBuilder rTrim07Builder(StringBuilder sb) {
        int length = sb.length();
        int end = rightTrimParts(sb, 0, length);
        if (end < length) sb.setLength(end);
        return sb;
    }
}
/*
    // ORIGINAL: String#trim
    public static String trim(byte[] value) {
        int len = value.length;
        int st = 0;
        while ((st < len) && ((value[st] & 0xff) <= ' ')) {
            st++;
        }
        while ((st < len) && ((value[len - 1] & 0xff) <= ' ')) {
            len--;
        }
        return ((st > 0) || (len < value.length)) ?
            newString(value, st, len - st) : null;
    }
 */