/*
 * Copyright (C) 2009 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package org.awk4j.bench;

import java.util.Date;

/**
 * Benchmark Var.2.
 * <p>
 * ・テストケース - Future case
 * ・ベンチマーク - Benchmark
 * ・クルー - Crew
 * ・ドライバ - Driver
 * ・ラッパー - Wrapper
 * ・ターゲット - Target
 *
 * @author kunio himei.
 */
public class Bench {

    private static final String PATTERN = "^^);"; // LATIN1
    private static final String PREFIX = " ".repeat(10);
    private static final String SUFFIX_256 = " ".repeat(256);
    public static final String BENCH_DATA = PREFIX + PATTERN + SUFFIX_256;
    public static final String ANS_RIGHT = PREFIX + PATTERN;
    public static final String ANS_BOTH = PATTERN;

    private static final String PATTERN_UTF = "※";
    //    private static final String PREFIX_UTF = " ".repeat(10); // (素の状態)
    private static final String PREFIX_UTF = "\u3000".repeat(10); // 全角空白(桁数を調整)
    public static final String BENCH_UTF = PREFIX_UTF + PATTERN_UTF + SUFFIX_256;
    public static final String ANS_BOTH_UTF = PATTERN_UTF;

    private static final long NANO = 1000_000; // 1ms = 10^6 ns.
    private static final int BENCHMARK_REPEAT_COUNT = 12; // times

    public static long driver_loop_time; // (ns)
    private static int cooling_time; // Main.cool or Main.ice (ms)

    private static long active_time; // active time (ms)
    private static long warmup_time; // warmup time (ms)
    private static long target_time; // Target time (ms)
    private static long driver_time; // Driver time (ms)

    private static boolean isLoopByCounter; // Driver, loops with counter
    private static int JIT_completion_position; // JIT completion position
    private static int JIT_completion_limit; // JIT completion position (Internal use)

    /*
     * Warm up - ウオームアップ.
     */
    private static void warmup() {
        long start = System.currentTimeMillis();
        final int prefetch = Main.prefetch;
        if (prefetch > 0) {
            active_time = System.currentTimeMillis();
            JIT_completion_position = prefetch;
            long JIT_mini_time = Long.MAX_VALUE;

            for (int i = 1; i <= prefetch; i++) { // Warmup Target
                long JIT_target_time = warmupHandler();

                if (JIT_mini_time > JIT_target_time) { // Find the minimum value
                    JIT_mini_time = JIT_target_time;
                    JIT_completion_position = i;
                }
                if (i % 1000 == 0 && Main.msg)
                    System.out.print("warm-up #" + i + " min:" + JIT_mini_time + "(ns)\n");
            }
            // Set the compilation completion position
            JIT_completion_limit = Math.min(prefetch, JIT_completion_position + 2000);
            target_time = System.currentTimeMillis() - active_time;
            if (Main.msg) {
                System.out.print("warm-up #" + JIT_completion_position + " completion\n");
            }
            ice();
            isLoopByCounter = true; // Driver, loops with counter
            driver_time = crewHandler(Main.driverTime);
            isLoopByCounter = false;
        }
        warmup_time = System.currentTimeMillis() - start;
    }

    // Driver is Complete.
    public static boolean isDriverComplete(int count) {
        return isLoopByCounter && count > JIT_completion_limit;
    }

    // クールダウン - cool down
    static void cool() {
        sleep(cooling_time);
    }

    // 強制冷却 - Forced cooling
    private static void ice() {
        sleep(Main.ice);
    }

    private static void sleep(int millis) {
        try {
            if (millis > 0) {
                active_time += millis;
                Thread.sleep(millis);
            }
        } catch (Throwable e) {
            throw new IllegalStateException(e);
        }
    }

    //////////////////////////////////////////////////////////////////////
    // ベンチマーク - benchmark
    static void bench() {
        long start = System.currentTimeMillis();
        initialize();
        if (Main.warmup) {
            cooling_time = 0; // do nothing
            warmup();
        }
        cooling_time = Main.cool; // cool down

        for (int i = 1; i <= BENCHMARK_REPEAT_COUNT; i++) {
            ice();
            long crew_time = crewHandler(Main.driverTime);
            if (Main.msg)
                System.out.print("bench #" + i + ' ' + crew_time + "(ms)\n");

            for (int k = 1; k < ANSWER.length; k++)
                System.err.println('\t' + ANSWER_NAME[k] + '\t' + ANSWER[k]);
            System.err.println();
        }
        double elapsed = (System.currentTimeMillis() - start) / 1000.;

        System.err.print("\tbenchTrim: [" + Main.title + "], -Framework: " + Main.framework +
                ", (" + elapsed + " sec.)\n" +
                "\t-priority: " + Main.priority + ", (" + new Date() + ")\n" +
                "\t-driverTime: " + Main.driverTime + " ms per driver (Adjust the benchmark time)\n" +
                "\t-warmup: " + Main.warmup + ", -msg: " + Main.msg +
                ", (" + warmup_time + " ms)\n" +
                "\t  -prefetch Target & Driver: " + Main.prefetch + " times (>" +
                JIT_completion_position + " required)\n" +
                "\t  (Target: " + target_time + " ms, Driver: " +
                driver_time + " ms, active time)\n" +
                "\t-cool down: " + Main.cool + " ms per driver\n" +
                "\t-ice: " + Main.ice + " ms per benchmark\n");
    }

    //////////////////////////////////////////////////////////////////////
    // Crew handling
    private static long warmupHandler() {
        long start = System.nanoTime();
        if (Main.majorVersion == FW04Strip) { // strip
            F04Strip.warmupTarget();
        } else if (Main.majorVersion == FW05Template) { // template
            F05Template.warmupTarget();
        } else {
            F01Classic.warmupTarget(); // others
        }
        return System.nanoTime() - start;
    }

    //////////////////////////////////////////////////////////////////////
    // Crew handling
    @SuppressWarnings("SameParameterValue")
    private static long crewHandler(int driverTime) {
        active_time = System.currentTimeMillis();
        driver_loop_time = driverTime * NANO;
        boolean isGenerics = Main.minorVersion != 0f;
        if (Main.majorVersion == FW01Classic) { // classic
            F01Classic.crew(ANSWER);
        } else if (Main.majorVersion == FW02Function) { // function
            if (isGenerics)
                F02Function.GenericsCrew(ANSWER); // 2.1
            else
                F02Function.crew(ANSWER);
        } else if (Main.majorVersion == FW03Future) { // future
            if (isGenerics)
                F03Future.GenericsCrew(ANSWER); // 3.1
            else
                F03Future.crew(ANSWER);
        } else if (Main.majorVersion == FW04Strip) { // strip
            F04Strip.crew(ANSWER);
        } else if (Main.majorVersion == FW05Template) { // template
            F05Template.crew(ANSWER);
        } else {
            throw new IllegalArgumentException("Framework: " + Main.framework);
        }
        return System.currentTimeMillis() - active_time;
    }

    //////////////////////////////////////////////////////////////////////
    private static final int FW01Classic = 1;
    private static final int FW02Function = 2;
    private static final int FW03Future = 3;
    private static final int FW04Strip = 4;
    private static final int FW05Template = 5;

    private static void initialize() {
        if (Main.majorVersion == FW04Strip) // strip
            ANSWER_NAME = ANSWER_NAME_STRIP;
        else if (Main.majorVersion == FW05Template) { // template
            ANSWER_NAME = ANSWER_NAME_TEMPLATE;
        }
    }

    private static final String[] ANSWER_NAME_CLASSIC = { // for #1-3
            null,
            "1. regex.Classic",
            "2. regex.New",
            "3. Trim.Original",
            "4. Trim.String",
            "5. Trim.CharSeq",
            "6. rTrim.String",
            "7. rTrim.Builder"};

    private static final String[] ANSWER_NAME_STRIP = { // for strip
            null,
            "1. strip.Original",
            "2. strip.Challenger",
            "3. strip.Join",
            "4. Trim.Original",
            "5. Trim.String",
            "6. rTrim.String",
            "7. rTrim.Builder"};

    private static final String[] ANSWER_NAME_TEMPLATE = { // for template
            null,
            "1. strip.Original",
            "2. strip.Challenger",
            "3. strip.Join",
            "4. empty",
            "5. empty",
            "6. empty",
            "7. rTrim.Builder"};

    private static String[] ANSWER_NAME = ANSWER_NAME_CLASSIC;

    private static final int[] ANSWER = new int[ANSWER_NAME.length];
}