/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package plus.gen

import groovy.transform.CompileStatic
import plus.lex.Keyword
import plus.lex.Node
import plus.lex.Term
import plus.reflect.Reflection

/**
 * Code Generator - Invoke.
 *
 * @author kunio himei.
 */
@CompileStatic
abstract class GenInvoke extends GenFunc {

    /**
     * Import実装 - called from Invoke, Try-Catch, _is(Node.Bc).
     * REMIND - Groovyは、Pathの一部を指定することはできない.
     */
    static Object applyImport(String obj) {
        String claz = "." + obj // Path[.Class]
        for (path in sTree.userImport)
            if (path.endsWith(claz))
                return path
        for (path in sTree.preImport)
            if (path.endsWith(claz))
                return path
        return claz
    }

    /** Invoke.
     */
    String invoke(Node.Invoke e) {
        Tuple2 rr = getClass(e.obj)
        String claz = rr.claz
        def hasMethod = { String obj ->
            e.isMethod || 0 < e.args.length
                    || Reflection.hasMethod(obj, e.name, e.args.length)
        }
        def hasField = { String obj ->
            !e.isMethod && Reflection.hasField(obj, e.name)
        }
        Object[] args = evalArgs(e.args)
        String sarg = mkString(args, '', ', ', '')
        String carg = mkStringNil(args, ', ', ', ', '')
        String rs                                       // ^ 先頭が、カンマ.
        if ((Keyword.SyyFVal == e.id) ||
                (Keyword.SyyFValFuture == e.id)) { // Function Value
            String func = (Keyword.SyyFVal == e.id) ?
                    "Function" : "ConcurrentFunction" // Concurrent Function Value.
            if (exists(claz))
                rs = "new ${func}(${claz}, '${e.name}'${carg})"
            else if (!e.name.contains('$'))
                rs = "new ${func}(this, '${e.name}'${carg})" // ルート関数(インライン以外)
            else {
                String simpleName = simpleFunctionName(e.name)
                Node.Func fn = functionMap.get(e.name)
                if (null == fn) { // name: [Caller$]name.
                    System.err.println("invoke name: " + e.name)
                    System.err.println("functionMap: " + functionMap.keySet())
                    throw new IllegalStateException(e as String)
                }

                if (isClosure(fn)) // クロージャ関数
                    rs = "new LocalFunction('${e.name}', '${simpleName}'(${sarg}))"
                else // ローカル関数
                    rs = "new ${func}(this, '${e.name}'${carg})"
            }
        } else if (Keyword.SyyNEW == e.id) { // new
            rs = "new ${claz}(${sarg})"

        } else if (Keyword.SymHAS == e.id) { // has
            rs = "_has(${claz}, '${e.name}'${carg})"

        } else if (Keyword.SymINVOKE == e.id) { // invoke
            if (isNil(claz)) { // クラスは空.
                rs = "_invoke(null, '${e.name}'${carg})"
            } else if (_has(claz)) { // クラスは存在する.
                if (hasMethod(claz)) {
                    rs = "${claz}.${e.name}(${sarg})" // メソド.
                } else if (hasField(claz)) {
                    rs = "${claz}.${e.name}" // プロパティ.
                } else {
                    rs = "${claz}(${sarg})" // ？
                }
            } else {  // クラスは存在しない！
                // NOTE クラスを取得できない場合は、()指定が必須！
                // - 抜本的な解決には、動的解析が必須.
                String call =
                        (isNil(e.name)) ? claz : claz + '.' + e.name
                if (hasMethod(rr.info)) {
                    rs = "${call}(${sarg})" // メソド.
                } else {
                    rs = "${call}" // プロパティ.
                }
            }
        } else {
            throw new IllegalStateException(e.class.getName() + " " + e)
        }
        return rs
    }

    //* クラス名、付加情報を返す.
    private Tuple2 getClass(Object obj) {
        switch (obj) {
            case Term.BOXING: // BOXING
                Term.BOXING x = obj as Term.BOXING
                return new Tuple2(x.value, x.value)

            case Term.NUMBER: // 数値
                Term.NUMBER x = obj as Term.NUMBER
                return new Tuple2(x.value, x.value.class.getName())

            case Term.YyValue: // 文字列, 正規表現
                Term.YyValue x = obj as Term.YyValue
                return new Tuple2(x.toString(), "".class.getName())

            case Node.YyVariable: // 変数参照
                Node.YyVariable x = obj as Node.YyVariable
                return new Tuple2(mkVerName(x), null)
        }
        Object x = eval(obj)
        return new Tuple2(x, x) // 評価値.
    }

    /**
     * タプル戻り値.
     */
    static class Tuple2 {
        final String claz // Class Name.
        final String info // Additional information.

        Tuple2(Object claz, Object additional) {
            this.claz = claz
            this.info = additional
        }
    }
}