/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.runtime;

import org.jetbrains.annotations.Nullable;

import java.util.HashMap;
import java.util.Map;

import plus.concurrent.ArrayAccessor;
import plus.concurrent.AtomicNumber;
import plus.concurrent.FutureMap;
import plus.concurrent.Record;
import plus.util.NumHelper;

/**
 * Built-in - AWK 組み込み変数.
 *
 * @author kunio himei.
 */
@SuppressWarnings("unused")
public enum BuiltInVar implements ArrayAccessor<Object, Object> {
    /* 変数を変更した場合は、lex.Symbols#PREDEF を修正すること.
     */
    _$(new Record()), // ☆レコード配列 - '_$' Groovy Requirements.
    $(new Record()), // ☆レコード配列
    @SuppressWarnings({"rawtypes"})
    ENVIRON(new FutureMap()), // ☆環境変数
    ARGV(new Record()), // コマンド行引数配列 (★後で'Argument'に置換する)
    ARGC(), // 配列 ARGV の要素数.
    ARGIND(), // 配列 ARGV のインデックス.
    CONVFMT("%.6g"), // 数値から文字列への変換書式.
    ERRNO(""), // ★例外発生理由を表わす文字列 (gawk拡張変数かつ未実装).
    FIELDWIDTHS(""), // ★空白で区切られたフィールド長のリスト (gawk拡張変数かつ未実装).
    FILENAME(""), // 現在の入力ファイル名.
    FNR(), // 現在のファイルの入力レコード数.
    FS(" "), // 入力フィールドセパレータ.
    IGNORECASE(), // 英大小文字の区別指示子.
    NF(), // 現在の入力レコードのフィールド数.
    NR(), // 現在までに入力したレコード数の合計.
    OFMT("%.6g"), // 数値の出力書式.
    OFS(" "), // 出力フィールドセパレータ.
    //  ORS(System.getProperty("line.separator")), // ★プラットフォーム依存の改行.
    ORS("\n"), // プラットフォーム非依存の改行.
    @SuppressWarnings("rawtypes")
    PROCINFO(new FutureMap()), // ★gawk固有機能の実装用配列(未実装).
    RESULT(""), // ☆gsub,sub 関数での置換結果の文字列.
    RLENGTH(), //  match で正規表現にマッチした文字列の長さ.
    RS("\n"), // 入力レコード区切子.
    RSTART(), // match() で正規表現にマッチした文字列の開始位置.
    RT(""), // RS にマッチし入力レコード区切子となった入力テキスト.
    SUBSEP("\034"); // ☆多次元配列の添字区切り文字.

    //* ------------------------------------------------------------------------
    //* 初期化.
    //* ------------------------------------------------------------------------
    public static void reInitialize() { // for Android.
        CONVFMT.put("%.6g"); // 数値から文字列への変換書式.
        FNR.put(0); // 現在のファイルの入力レコード数.
        FS.put(" "); // 入力フィールドセパレータ.
        IGNORECASE.put(0); // 英大小文字の区別指示子.
        NR.put(0); // 現在までに入力したレコード数の合計.
        OFMT.put("%.6g"); // 数値の出力書式.
        OFS.put(" "); // 出力フィールドセパレータ.
        ORS.put("\n"); // プラットフォーム非依存の改行.
        RS.put("\n"); // 入力レコード区切子.
        SUBSEP.put("\034"); // ☆多次元配列の添字区切り文字.
    }

    private static final Map<String, BuiltInVar> REVERCE_MAP =
            new HashMap<>((BuiltInVar.values().length));
    /*
     * TypeForGroovy.
     */
    private static final int NumberType = 0;
    private static final int StringType = 1;
    private static final int ObjectType = 2;
    private static final int ArrayType = 3;

    //* Enumの初期化順序 {} -> () -> static.
    static {
        // Enumのリバース配列を構築.
        for (BuiltInVar x : BuiltInVar.values()) {
            REVERCE_MAP.put(x.name(), x);
        }
        // ENVIRON - Linuxスタイルに変更.
        for (Map.Entry<String, String> x : System.getenv().entrySet()) {
            ENVIRON.putAt(x.getKey(),
                    x.getValue().replace('\\', '/'));
        }
        _$.arr = $.arr; // エイリアス化.
    }

    //* コンテナ.
    private final BuiltInVar thisENUM; // この ENUM オブジェクト.
    private final int type; // このオブジェクトのタイプ.
    private final boolean enableAlias; // エイリアスを返すかどうか.
    private final AtomicNumber atom = new AtomicNumber(); // 数値オブジェクト.
    private transient ArrayAccessor<Object, Object> arr; // 配列オブジェクト.
    private volatile Object obj; // 文字列などその他オブジェクト.

    /*
     * コンテナを構築する.
     */
    BuiltInVar() {
        this.thisENUM = this;
        this.enableAlias = true;
        this.type = NumberType;
    }

    BuiltInVar(ArrayAccessor<Object, Object> x) {
        this.thisENUM = this;
        this.enableAlias = false;
        this.type = ArrayType;
        this.arr = x;
    }

    BuiltInVar(Object x) {
        this.thisENUM = this;
        this.enableAlias = true;
        this.type = ((x instanceof String) ? StringType : ObjectType);
        this.obj = x;
    }

    /*
     * 指定された名前の変数オブジェクトを返す.
     */
    @Nullable
    public static BuiltInVar forName(String name) {
        return REVERCE_MAP.get(name);
    }

    //* エイリアスを返す - for Groovy.
    public static String getAlias(String name) {
        BuiltInVar var = forName(name);
        if (null != var) {
            if ("$".equals(name)) return _$.name(); // レコード変数
            if (var.enableAlias) return "$" + name; // 組込み変数
        }
        return name;
    }

    //* ------------------------------------------------------------------------
    //* Getter / Setter.
    //* ------------------------------------------------------------------------
    //* コンテナの変数値を取得する.
    public Object get() {
        switch (this.type) {
            case NumberType:
                if (this.thisENUM == NF) { // NFを同期する.
                    $.getAt(0);
                    if (NF.intValue() > 0)
                        $.getAt(1);
                }
                return this.atom.intValue();
            case ArrayType:
                return this.arr;
        }
        return this.obj;
    }

    /**
     * コンテナに変数値を設定する.
     */
    public void put(final Object value) {
        switch (this.type) {
            case NumberType:
                this.atom.setAtom(NumHelper.doubleValue(value));
                break;
            case StringType:
                this.obj = RunHelper.toString(null, value);
                break;
            case ArrayType:
                //noinspection unchecked
                this.arr = (ArrayAccessor<Object, Object>) value;
            default:
                this.obj = value;
        }
    }

    /**
     * Numerical calculations - AtomicNumber を呼び出す.
     */
    @SuppressWarnings("UnusedReturnValue")
    public Object calculate(String op, final Object right) {
        if (this.type == NumberType)
            return this.atom.calculate(op, right);
        put(right);
        return right;
    }

    //* ------------------------------------------------------------------------
    //* Array Getter / Setter - for Groovy.
    //* ------------------------------------------------------------------------
    @Override
    public Object getAt(Object index) {
        if (null != index)
            return this.arr.getAt(index);
        return get();
    }

    @Override
    public void putAt(Object index, Object value) {
        if (null != index)
            this.arr.putAt(index, value);
        else put(value);
    }

    public int intValue() {
        return this.atom.intValue();
    }

    @Override
    public String toString() {
        return get().toString();
    }
}