/*
 * Copyright (C) 2010 awk4j - http://awk4j.sourceforge.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed reader the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.io;

import org.jetbrains.annotations.NotNull;

import java.io.InterruptedIOException;
import java.io.Reader;
import java.io.Writer;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * [%device%] Piped Writer.
 * <p>
 * The class to which this annotation is applied is thread-safe.
 *
 * @author kunio himei.
 */
final class PipedWriter extends Writer {

    /**
     * [%enclosure%] 空の文字配列.
     */
    static final char[] ENPTY_CHAR_ARRAY = {};
    /**
     * 入力ｽﾄﾘｰﾑのバッファサイズの初期値.
     */
    private static final int INITIAL_INPUT_BUFFERSIZE = IoConstants.DEFAULT_CHAR_BUFFER_SIZE;
    /**
     * このストリームを読み込む待機時間 (milliseconds).
     */
    private static final long WAIT_TIME = 10L;
    /**
     * Queue.
     */
    private final LinkedBlockingQueue<char[]> queue = new LinkedBlockingQueue<>();
    /**
     * このｽﾄﾘｰﾑのリーダ.
     */
    private final Reader reader = new Tr();
    /**
     * このｽﾄﾘｰﾑは閉じているか?.
     */
    private volatile boolean hasClose;
    /**
     * 入力ｽﾄﾘｰﾑのバッファサイズ(可変長).
     */
    private volatile int inputBufferSize = INITIAL_INPUT_BUFFERSIZE;

    /**
     * このストリームを閉じる.
     */
    @Override
    public void close() {
        this.hasClose = true;
    }

    /**
     * 何もしない.
     */
    @Override
    public void flush() { // 何もしない
    }

    /**
     * @return このｽﾄﾘｰﾑのリーダを返す.
     */
    public Reader getReader() {
        return this.reader;
    }

    /**
     * このストリームから文字列を取得する.
     *
     * @param maxlength 取得する文字列の最大長
     * @return 取得した文字列の配列
     * @throws InterruptedIOException 待機中に割り込みが発生した
     */
    char[] readBuffer(int maxlength) throws InterruptedIOException {
        if (maxlength > this.inputBufferSize) {
            this.inputBufferSize = maxlength;
        }
        char[] arr = this.queue.poll();
        while (!this.hasClose && (null == arr)) {
            try {
                // Thread.sleep(1); // Busy-waiting anti-pattern
                arr = this.queue.poll(WAIT_TIME, TimeUnit.MILLISECONDS);

            } catch (InterruptedException e) { // 待機中に割り込みが発生した
                this.hasClose = true;
                Thread.currentThread().interrupt(); // (InterruptedException)
                throw new InterruptedIOException(e.getMessage());
            }
        }
        return arr;
    }

    /**
     * 文字の配列の一部を書き込む.
     *
     * @param cbuf 転送先バッファ
     * @param off  文字の格納開始オフセット
     * @param len  読み込む文字の最大数
     */
    @Override
    public void write(@NotNull char[] cbuf, int off, int len) {
        int xoff = off;
        int xlen = len;
        while (0 < xlen) { // 現在の入力バッファサイズに分割する
            int x = Math.min(this.inputBufferSize, xlen);
            char[] arr = new char[x];
            System.arraycopy(cbuf, xoff, arr, 0, x);
            this.queue.offer(arr);
            xoff += x;
            xlen -= x;
        }
    }

    /**
     * このストリームのリーダ.
     */
    final class Tr extends Reader {

        /**
         * このストリームのバッファ.
         */
        private char[] buffer = ENPTY_CHAR_ARRAY;

        /**
         * 何もしない.
         */
        @Override
        public void close() { // 何もしない
        }

        /**
         * 配列の一部に文字を読み込む (バッファが空の場合はブロックする).
         *
         * @param cbuf 転送先バッファ
         * @param off  文字の格納開始オフセット
         * @param len  読み込む文字の最大数
         * @return 読み込んだ文字数、 ストリームの終わりに達した場合は -1
         * @throws InterruptedIOException 待機中に割り込みが発生した
         */
        @Override
        public synchronized int read( // SYNC.
                                      @NotNull char[] cbuf, int off, int len)
                throws InterruptedIOException {
            char[] buf = this.buffer; // pop
            while ((null != buf) && (0 == buf.length)) {
                buf = readBuffer(len);
            }
            if (null != buf) {
                int xlen = buf.length;
                int x = Math.min(xlen, len);
                System.arraycopy(buf, 0, cbuf, off, x); // これを返す
                xlen -= x;
                char[] arr;
                if (0 < xlen) {
                    arr = new char[xlen];
                    System.arraycopy(buf, x, arr, 0, xlen); // 次回に取得する
                } else {
                    arr = ENPTY_CHAR_ARRAY;
                }
                this.buffer = arr; // push
                return x;
            }
            this.buffer = null; // push
            return -1;
        }
    }
}