/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.eval;

import plus.lex.Node;

import java.util.concurrent.atomic.AtomicInteger;

/**
 * Logging.
 *
 * @author kunio himei.
 */
final class Log {

    private static final int logMax = 16; // ログの最大行数.
    private static final Node.Root[] logBuf = new Node.Root[logMax]; // ログバッフア.
    private static final AtomicInteger pointer = new AtomicInteger(); // ログの書き込み位置.
    private transient int lineNumber; // ログの先頭の行番号.

    /**
     * Snap shot.
     */
    public String snap() {
        final StringBuilder sb = new StringBuilder();
        sb.append("--------------------------------------------------------------------------------\n");
        final int p = pointer.get();
        for (int i = 0; logMax > i; i++) {
            final Node.Root x = logBuf[(p + i) % logMax];
            if (null != x) {
                sb.append('[').append(x.linenumber).append("]\t");
                sb.append(x.getClass().getName()).append(": ").append(x)
                        .append('\n');
            }
        }
        return sb.toString();
    }

    //* 行番号を返す.
    public int getLineNumber() {
        return lineNumber;
    }

    //* 行番号を設定する.
    public void setLineNumber(final int x) {
        lineNumber = x;
    }

    //* ロギング.
    public void log(final Node.Root node) {
        int p;
        do {
            p = pointer.get();
        } while (!pointer.compareAndSet(p, (p + 1) % logMax));
        logBuf[p] = node;
        lineNumber = node.linenumber;
    }
}