/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.concurrent;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import plus.runtime.RunHelper;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Future Map の実装 - キーや値はnullにすることはできない.
 * The class to which this annotation is applied is thread-safe.
 *
 * @param <K> Key
 * @param <V> Value
 * @author Kunio Himei.
 */
//* NOTE Nil値が、漏れることがある.アノテーションが怪しいため、<K, Object>とする.
@SuppressWarnings("unused")
public class FutureMap<K, V> extends ConcurrentHashMap<K, Object>
        implements ArrayAccessor<K, Object> {

    private static final String NUL = "\u0080"; //☆ PATCH 値の変更は不可

    private static final int initialCapacity = 19;

    public FutureMap() {
        super(initialCapacity);
    }

    public FutureMap(int capacity) {
        super(capacity);
    }

    public FutureMap(Map<K, V> map) {
        super(map);
    }

    //* Nil -> null.
    private Object nil2(Object value) { //
        return (NUL == value) ? null : value;
    }

    //* ------------------------------------------------------------------------
    //* 堅牢化.
    //* ------------------------------------------------------------------------
    @Override
    public final Object get(Object key) { //
        return nil2(super.get(arrayKey(key)));
    }

    @Override
    public final Object put(@NotNull K key, @Nullable Object value) {
        Object x = (null == value) ? NUL : value;
        return super.put(arrayKey(key), x);
    }

    @Override
    public final boolean containsKey(Object key) { //
        return super.containsKey(arrayKey(key));
    }

    @Override
    public final Object remove(@NotNull Object key) { //
        return nil2(super.remove(arrayKey(key)));
    }

    //* ------------------------------------------------------------------------
    //* Array Getter / Setter - for Groovy.
    //* ------------------------------------------------------------------------
    @Override
    public Object getAt(Object key) { //
        return this.get(key);
    }

    @Override
    public void putAt(Object key, Object value) { //
        this.put(arrayKey(key), value);
    }

    /**
     * 配列アクセスキー作成 (浮動小数点は、'CONVFMT'で変換).
     */
    public K arrayKey(Object key) {
        //noinspection unchecked
        return (K) RunHelper.toString(null, key);
    }
}