/*
 * Copyright (C) 2006 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.awt;

import plus.reflect.Listener;

import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;

/**
 * [%wrapper%] java.awt.event.MouseAdapter ラッパの実装.
 * <p>
 * The class to which this annotation is applied is thread-safe.
 *
 * @author kunio himei.
 */
public final class Mouse extends MouseAdapter {

    /**
     * ☆ Mouse のラッパオブジェクトを構築する.
     */
    public Mouse(Listener a, Listener b) {
        append(new Listener[]{a, b});
    }

    /**
     * Mouse のラッパオブジェクトを構築する.
     */
    @Deprecated
    Mouse(Listener[] listeners) {
        append(listeners);
    }

    /**
     * Mouse のラッパオブジェクトを更新する.
     */
    private static void append(Listener[] listeners) {
        for (Listener x : listeners) {
            Enum.valueOf(Me.class, x.getName()).setLstener(x);
        }
    }

    /**
     * メソドがオーバライドされていれば TRUE を返す.
     */
    private static boolean isEnable(Me name, MouseEvent e) {
        Listener listener = name.getLstener();
        if (null != listener) {
            listener.apply(e);
            return false;
        }
        return true;
    }

    /**
     * コンポーネント上でマウスボタンをクリック したときに呼び出される (MouseListener).
     */
    @Override
    public void mouseClicked(MouseEvent e) {
        if (isEnable(Me.mouseClicked, e)) {
            super.mouseClicked(e);
        }
    }

    /**
     * コンポーネント上でマウスのボタンを押してドラッグすると呼び出される (MouseMotionListener).
     */
    @Override
    public void mouseDragged(MouseEvent e) {
        if (isEnable(Me.mouseDragged, e)) {
            super.mouseExited(e);
        }
    }

    /**
     * コンポーネントにマウスが入ると呼び出される (MouseListener).
     */
    @Override
    public void mouseEntered(MouseEvent e) {
        if (isEnable(Me.mouseEntered, e)) {
            super.mouseEntered(e);
        }
    }

    /**
     * コンポーネントからマウスが出ると呼び出される (MouseListener).
     */
    @Override
    public void mouseExited(MouseEvent e) {
        if (isEnable(Me.mouseExited, e)) {
            super.mouseExited(e);
        }
    }

    /**
     * ボタンを押さずに、マウスカーソルをコンポーネント上に移動すると呼び出される (MouseMotionListener).
     */
    @Override
    public void mouseMoved(MouseEvent e) {
        isEnable(Me.mouseMoved, e);
    }

    /**
     * コンポーネント上でマウスボタンが押されると呼び出される (MouseListener).
     */
    @Override
    public void mousePressed(MouseEvent e) {
        if (isEnable(Me.mousePressed, e)) {
            super.mousePressed(e);
        }
    }

    /**
     * コンポーネント上でマウスボタンが離されると呼び出される (MouseListener).
     */
    @Override
    public void mouseReleased(MouseEvent e) {
        if (isEnable(Me.mouseReleased, e)) {
            super.mouseReleased(e);
        }
    }

    /**
     * マウスホイールが回転すると呼び出される (MouseWheelListener).
     */
    @Override
    public void mouseWheelMoved(MouseWheelEvent e) {
        isEnable(Me.mouseWheelMoved, e);
    }

    /**
     * マウスイベントリスナ.
     */
    private enum Me {
        /**
         * コンポーネント上でマウスボタンをクリック (押してから離す) したときに呼び出さされる.
         */
        mouseClicked,
        /**
         * コンポーネント上でマウスのボタンを押してドラッグすると呼び出される.
         */
        mouseDragged,
        /**
         * コンポーネントにマウスが入ると呼び出される.
         */
        mouseEntered,
        /**
         * コンポーネントからマウスが出ると呼び出される.
         */
        mouseExited,
        /**
         * ボタンを押さずに、マウスカーソルをコンポーネント上に移動すると呼び出される .
         */
        mouseMoved,
        /**
         * コンポーネント上でマウスボタンが押されると呼び出される .
         */
        mousePressed,
        /**
         * コンポーネント上でマウスボタンが離されると呼び出される .
         */
        mouseReleased,
        /**
         * マウスホイールが回転すると呼び出される.
         */
        mouseWheelMoved;

        /**
         * リスナ.
         */
        private transient Listener listener;

        /**
         * listener リスナを返す.
         */
        Listener getLstener() {
            return this.listener;
        }

        /**
         *
         */
        void setLstener(Listener value) {
            this.listener = value;
        }
    }
}