/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.spawn;

import plus.io.Io;
import plus.io.IoHelper;
import plus.spawn.system.UtilInterface;

import java.io.*;
import java.util.Arrays;
import java.util.List;

/**
 * [%command%] execute command.
 * <p>
 * Usage: Exec command<br>
 * </p>
 *
 * @author kunio himei.
 */
public final class Exec extends OutputStreamWriter implements UtilInterface {

    /**
     * サブプロセスの終了値の初期値(終了値は未設定).
     */
    private static final int CSTAT_UNDEF_CODE = 256;
    /**
     * 入力ｽﾄﾘｰﾑのバッファサイズ.
     */
    private static final int DEFAULT_BYTE_BUFFER_SIZE = 1024 * 8;
    /**
     * 入力ｽﾄﾘｰﾑの文字バッファサイズ (DEFAULT_BYTE_BUFFER_SIZE / 2).
     */
    private static final int DEFAULT_CHAR_BUFFER_SIZE = 1024 * 4;
    /**
     * USAGE.
     */
    private static final String USAGE = "Usage: Exec command";
    /**
     * サブプロセスの出力タスクの完了を待機する時間 (milliseconds).
     */
    private static final long WAIT_TIME = 500L;
    /**
     * サブプロセスの標準出力ストリーム.
     */
    private final Writer out;
    /**
     * このサブプロセス.
     */
    private final Process process;
    /**
     * サブプロセスからの出力ストリーム.
     */
    private final InputStream[] readers;
    /**
     * サブプロセスの出力タスク.
     */
    private final Thread[] tasks;
    /**
     * サブプロセスの終了値.
     */
    private volatile int statuscode = CSTAT_UNDEF_CODE;

    /**
     * Execute.
     */
    private Exec(Process proces, Writer output) {
        super(proces.getOutputStream());
        this.process = proces;
        InputStream ot = proces.getInputStream();
        InputStream er = proces.getErrorStream();
        this.readers = new InputStream[]{ot, er};
        this.tasks = new Thread[]{new Thread(new Tout(ot, output)),
                new Thread(new Terr(er))};
        this.out = output;
    }

    /**
     * サブプロセスへのライタを取得する.
     */
    public static Exec getWriter(List<String> command, Writer output)
            throws IOException {
        System.err.println('`' + command.toString().replace("\t", "\\t"));
        ProcessBuilder pb = new ProcessBuilder(command);
        // Android 環境では ENVIRON の変更は不可!
        Process proc = pb.start();
        Exec me = new Exec(proc, output);
        me.start();
        return me;
    }

    /**
     *
     */
    public static void main(String[] args) throws IOException {
        if ((0 != args.length) && args[0].startsWith("--help")) {
            System.out.println(USAGE);
        } else {
            UtilInterface me = getWriter(Arrays.asList(args),
                    new OutputStreamWriter(System.out));
            if (me.hasInput()) {
                IoHelper.copyline(Io.STDIN, (Writer) me);
            }
            me.close();
        }
    }

    /**
     * このストリームを閉じる.
     * <p>
     * {@literal InterruptedIOException} - 待機中に割り込みが発生した
     */
    @Override
    public void close() throws IOException {
        try {
            super.close(); // サブプロセスへの標準入力を閉じる
            this.statuscode = this.process.waitFor();
            join(WAIT_TIME); // 出力ライタの終了を待つ
        } catch (InterruptedException e) { // 待機中に割り込みが発生した
            Thread.currentThread().interrupt(); // (InterruptedException)
            throw new InterruptedIOException(e.getMessage());
        } finally {
            try {
                this.process.destroy();
                Io.close(this.readers); // サブプロセスからの入力を強制的に閉じる
                join(0);
            } catch (InterruptedException e) { // 待機中に割り込みが発生した
                Thread.currentThread().interrupt(); // ひき逃げ
                // throw new InterruptedIOException(e.getMessage());
            } finally {
                Io.close(this.out);
                System.err.flush();
            }
        }
    }

    /**
     * サブプロセスの終了値を返す.
     */
    @Override
    public int exitValue() {
        return this.statuscode;
    }

    /**
     *
     */
    @Override
    public boolean hasInput() {
        return false;
    }

    /**
     * スレッドの終了を待機する.
     */
    private void join(long millis) throws InterruptedException {
        for (Thread x : this.tasks) {
            if (x.isAlive()) {
                x.join(millis);
            }
        }
    }

    /**
     * サブプロセスのライタを起動.
     */
    void start() {
        this.tasks[0].start();
        this.tasks[1].start();
    }

    /**
     * 標準エラー出力ライタの実装.
     */
    private static class Terr implements Runnable {

        /**
         * 入出力バッファ.
         */
        private final byte[] buf = new byte[DEFAULT_BYTE_BUFFER_SIZE];

        /**
         * 入力ストリーム.
         */
        private final InputStream in;

        /**
         * ストリームを構築する.
         */
        Terr(InputStream input) {
            this.in = input;
        }

        /**
         * run.
         */
        @Override
        public synchronized void run() { // SYNC.
            try {
                for (int len = 0; 0 <= len; ) {
                    len = this.in.read(this.buf);
                    if (0 < len) {
                        System.err.write(this.buf, 0, len);
                    }
                }
            } catch (IOException e) { // ひき逃げ
                System.err.println("\t Process.err: " + e);
            }
        }
    }

    /**
     * 標準出力ライタの実装.
     */
    private static class Tout implements Runnable {

        /**
         * 入出力バッファ.
         */
        private final char[] buf = new char[DEFAULT_CHAR_BUFFER_SIZE];

        /**
         * 入力ストリーム.
         */
        private final Reader in;

        /**
         * 出力ストリーム.
         */
        private final Writer out;

        /**
         * ストリームを構築する.
         */
        Tout(InputStream input, Writer output) {
            this.in = new InputStreamReader(input);
            this.out = output;
        }

        /**
         * run.
         */
        @Override
        public synchronized void run() { // SYNC.
            // int outlen = 0; // (logging)
            try {
                for (int len = 0; 0 <= len; ) {
                    len = this.in.read(this.buf);
                    if (0 < len) {
                        this.out.write(this.buf, 0, len);
                        this.out.flush();
                        // outlen += len; // (logging)
                    }
                } // System.err.println("\t　Process.out: " + outlen);
            } catch (IOException e) { // ひき逃げ
                System.err.println("\t Process.out: " + e);
            }
        }
    }
}