/*
 * Copyright (C) 2010 awk4j - https://ja.osdn.net/projects/awk4j/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package plus.lex;

import org.jetbrains.annotations.Nullable;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * RegExp - 正規表現パターンオブジェクト.
 *
 * @author kunio himei.
 */
final class LexRegx {

    //* コンパイル済みの正規表現.
    private final Pattern pattern;

    //* 文字シーケンスのマッチ操作を行うエンジン.
    private Matcher matcher;

    /**
     * 正規表現オブジェクトを構築.
     */
    LexRegx(String regex) {
        this.pattern = Pattern.compile(regex);
    }

    /**
     * トークンが正規表現なら正規表現を、以外なら null を返す.
     */
    @Nullable
    public static String isNativeRegexp(String s) {
        if ('/' == s.charAt(0)) {
            int len = s.length() - 1;
            int inCharClass = 0, inParen = 0;
            for (int ix = 0; len > ix++; ) {
                char c = s.charAt(ix);
                if ('\\' == c) {
                    ix++;
                } else if ('[' == c) {
                    inCharClass++;
                } else if ((']' == c) && (0 < inCharClass)) {
                    inCharClass--;
                } else if ('(' == c) {
                    inParen++;
                } else if (')' == c) {
                    if ((0 >= inParen) && (0 >= inCharClass)) {
                        return null;
                    }
                    inParen--;
                } else if ('/' == c) {
                    return s.substring(0, ix + 1);
                }
            }
        }
        return null;
    }

    /**
     * 指定された入力とこのパターンをマッチする正規表現エンジンを作成する.
     */
    public boolean find(Object s) {
        if (s instanceof CharSequence) {
            this.matcher = this.pattern.matcher((CharSequence) s);
            return this.matcher.find();
        }
        return false;
    }

    /**
     * 前方参照された入力部分シーケンスを返す.
     */
    public String group(int i) {
        assert this.matcher.groupCount() >= i;
        return this.matcher.group(i);
    }

    /**
     * このオブジェクトの文字列表現を返す.
     */
    @Override
    public String toString() {
        return (null == this.matcher) ?
                this.pattern.toString() : this.matcher.toString();
    }
}