/**
 * MultipartStripper.java
 * Created on 2003/06/05 
 * Copyright (c) by es , All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND 
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED 
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY 
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF 
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER 
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package net.prixdami.alexis.multipart;

import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletRequest;

import net.prixdami.alexis.model.MimeData;

/**
 * form-Muitipart data splitter
 * @author es
 * @version 1.06
 */
public class MultipartStripper {
    private static final int BYTE_TO_ARRAY = 0;
    private static final int BYTE_TO_FILE = 1;

    private final String boundaryKey = "boundary=";
    private final String contentKey = "Content-Disposition:";
    private final String fileKey = "filename=\"";
    private final String paramKey = "name=\"";
    private String encoding = "JISAutoDetect";
    private String path = "/data/";
    private boolean eof2abort = true;
    private int mode = BYTE_TO_ARRAY;

    public MultipartStripper() {
    }

    public MultipartStripper(String encoding) {
        this.encoding = encoding;
    }

    public MultipartStripper(int mode, String path) {
        this.mode = mode;
        this.path = path;
    }

    public MultipartStripper(String encoding, int mode, String path) {
        this.encoding = encoding;
        this.mode = mode;
        this.path = path;
    }

    public void setAbortMode(boolean flag) {
        eof2abort = flag;
    }

    public Map strip(ServletRequest request) throws IOException {
        return strip(request, -1);
    }

    public Map strip(ServletRequest request, int maxsize) throws IOException {
        return strip(request.getInputStream(), request.getContentType(), request.getContentLength(), maxsize);
    }

    public Map strip(InputStream inputStream, String contentType, int contentLength) throws IOException {
        return strip(inputStream, contentType, contentLength, -1);
    }

    public Map strip(InputStream inputStream, String contentType, int contentLength, int limit) throws IOException {

        MultipartInputStream in = new MultipartInputStream(inputStream);
        Map returnValue = new HashMap();
        boolean eof = false;
        byte[] buffer = new byte[512];

        String boundary = getBoundary(contentType);
        if (boundary == null) {
            /**@todo hOƍ */
            throw new IOException("this data isn't MultiPart request.");
        }
        if (contentLength > limit && limit > 0) {
            /**@todo hOƍ */
            throw new IOException("request is too huge.");
        }

        String currentLine;
        while (!eof) {
            currentLine = getLine(in, buffer);
            if (currentLine == null) {
                eof = true;
                continue;
            }

            if (currentLine.indexOf(boundary) != -1) {
                currentLine = getLine(in, buffer);
                if (currentLine == null) {
                    eof = true;
                    continue;
                }
            }
            if (currentLine.lastIndexOf(contentKey) == -1) {
                /**@todo hOƍ */
                throw new IOException("can't strip request.[Content-Disposition]");
            }
            String paramName = getToken(currentLine, paramKey);
            String filename = getToken(currentLine, fileKey);

            try {
                if (filename == null) {
                    getLine(in, buffer);

                    byte[] rawParam = (byte[]) getBytes(in, boundary, BYTE_TO_ARRAY);
                    String param = new String(rawParam, 0, rawParam.length, encoding);
                    returnValue.put(paramName, param);
                } else if (filename.equals("")) {
                    for (int i = 0; i < 3; i++) {
                        getLine(in, buffer);
                    }
                } else {
                    String mimeType = getMimeType(getLine(in, buffer));
                    getLine(in, buffer);

                    MimeData data = null;
                    switch (mode) {
                        case BYTE_TO_ARRAY :
                            byte[] rawParam = (byte[]) getBytes(in, boundary, mode);
                            data = new MimeData(rawParam, mimeType, filename);
                            returnValue.put(paramName, data);
                            break;
                        case BYTE_TO_FILE :
                            File rawFile = (File) getBytes(in, boundary, mode);
                            if (rawFile.length() == 0) {
                                rawFile.delete();
                            } else {
                                int pos = filename.lastIndexOf(".");
                                String ext = "";
                                if (pos != -1) {
                                    ext = filename.substring(pos).toLowerCase();
                                    //20030920 fix by v1.06
                                    if (ext.endsWith(".jsp") || ext.endsWith(".htm") || ext.endsWith(".html")) {
                                        ext = ".txt";
                                    }
                                    //
                                    File newFile = new File(rawFile.getCanonicalPath() + ext);
                                    rawFile.renameTo(newFile);
                                    rawFile = newFile;
                                }

                                data = new MimeData(rawFile, mimeType, filename);
                                returnValue.put(paramName, data);
                            }
                            break;
                        default :
                            throw new IllegalArgumentException("wrong data store mode :" + mode);
                    }
                }
            } catch (EOFException ex) {
                if (eof2abort) {
                    throw ex;
                }
            }
        }
        return returnValue;
    }

    private Object getBytes(MultipartInputStream in, String boundary, int flag) throws IOException {
        switch (flag) {
            case BYTE_TO_ARRAY :
                return readByte(in, boundary, null);
            case BYTE_TO_FILE :
                File f = new File(path + Long.toString(new Date().getTime()));
                try {
                    f = (File) readByte(in, boundary, f);
                } catch (EOFException ex) {
                    f.delete();
                    throw ex;
                }
                return f;
            default :
                throw new IllegalArgumentException("wrong data store flag :" + flag);
        }
    }

    private Object readByte(MultipartInputStream in, String boundary, File file) throws IOException {
        OutputStream byteArray = null;
        byte[] buffer = new byte[4096 * 4];
        int maxRead = buffer.length - boundary.length() - 2;

        boolean eof = false;
        boolean abort = false;

        if (file == null) {
            byteArray = new ByteArrayOutputStream();
        } else {
            byteArray = new BufferedOutputStream(new FileOutputStream(file), buffer.length);
        }

        int fullcount = 0;
        while (!eof) {
            int readCount = 0;
            int currentPos = 0;

            do {
                readCount = in.readLine(buffer, currentPos, maxRead - currentPos);
                if (readCount == -1) {
                    eof = true;
                    abort = true;
                }
                if (readCount >= boundary.length()) {
                    eof = true;
                    for (int i = 0; i < boundary.length(); i++) {
                        if (boundary.charAt(i) != buffer[currentPos + i]) {
                            eof = false;
                            break;
                        }
                    }
                }
                if (!eof) {
                    currentPos += readCount;
                    fullcount += readCount;
                } else {
                    currentPos -= 2;
                }
            } while (!eof && maxRead > currentPos);

            //20030918 fix by v1.05
            if (abort) {
                currentPos -= 2;
                for (int i = 1; i < currentPos; i++) {
                    if (boundary.charAt(boundary.length() - i) == buffer[currentPos - i]) {
                        abort = false;
                    } else {
                        abort = true;
                        break;
                    }
                }
            } else {
                byteArray.write(buffer, 0, currentPos);
            }
            //
        }
        if (abort) {
            byteArray.close();
            byteArray = null;
            throw new EOFException("EOF found.");
        }

        if (file == null) {
            int size = ((ByteArrayOutputStream) byteArray).size();
            byte[] returnValue = new byte[size];
            System.arraycopy(((ByteArrayOutputStream) byteArray).toByteArray(), 0, returnValue, 0, size);
            byteArray.close();
            byteArray = null;
            return returnValue;
        } else {
            byteArray.flush();
            byteArray.close();
            byteArray = null;
            return file;
        }
    }

    private String getLine(MultipartInputStream in, byte[] buffer) throws java.io.IOException {
        int currentLength = in.readLine(buffer, 0, buffer.length);
        if (currentLength == -1) {
            return null;
        }
        String line = new String(buffer, 0, currentLength - 2, encoding);
        return line;
    }

    private String getBoundary(String line) {
        int index = line.lastIndexOf(boundaryKey);
        if (index == -1)
            return null;
        String boundary = line.substring(index + boundaryKey.length());
        boundary = "--" + boundary;
        return boundary;
    }

    private String getMimeType(String line) {
        String typeKey = "Content-Type: ";
        int index = line.lastIndexOf(typeKey);
        if (index == -1)
            return null;
        String attr = line.substring(index + typeKey.length());
        return attr;
    }

    private String getToken(String line, String key) {
//		String[] params = Utils.split(line, ";");
		String[] params = line.split(";");
        for (int i = 0; i < params.length; i++) {
            if (params[i].indexOf(key) != -1) {
                line = params[i];
                break;
            }
        }

        int startpos = line.lastIndexOf(key);
        int endpos = line.lastIndexOf("\"");
        if (startpos == -1)
            return null;
        String attr = line.substring(startpos + key.length(), endpos);
        return attr;
    }

    //-----------------------------------------------------------------------------
    private class MultipartInputStream extends InputStream {

        private InputStream _in;

        private MultipartInputStream() {
        }

        public MultipartInputStream(InputStream in) {
            _in = in;
        }

        public int read() throws java.io.IOException {
            return _in.read();
        }

        public int read(byte[] b) throws java.io.IOException {
            return _in.read(b);
        }

        public int read(byte[] b, int off, int len) throws java.io.IOException {
            return _in.read(b, off, len);
        }

        public int readLine(byte[] b, int off, int len) throws IOException {
            if (len <= 0) {
                return 0;
            }
            int count = 0;
            int i;
            while ((i = read()) != -1) {
                b[off++] = (byte) i;
                count++;
                if (i == 10 || count == len) {
                    break;
                }
            }

            return count <= 0 ? -1 : count;
        }

        public void close() throws java.io.IOException {
            _in.close();
        }

    }
}