////////////////////////////////////////////////////////////////
// aaoh.kaze
//   Copyright (C) 2008 - 2011 AAOh inc.
// 
// Licensed under the MIT license
//   http://www.opensource.org/licenses/mit-license.php
////////////////////////////////////////////////////////////////

/**
 * @fileOverview aaoh.kaze
 * @author       $Author: fujishiro $
 */

/**
 * namespace。
 *   aaoh.kaze.Type    (static)
 *   aaoh.kaze.Gadget  (static)
 *   aaoh.kaze.Element (static)
 *   aaoh.kaze.Ajax
 *   aaoh.kaze.JSON
 *   aaoh.kaze.Storage
 *   aaoh.kaze.Clipboard
 *   aaoh.kaze.Sound
 *   aaoh.kaze.Timer
 *   aaoh.kaze.Locale
 */

(function () {
/**
 * Type。
 *
 * @namespace
 */
aaoh.kaze.Type = {
	MOBILE_WIDTH:  799,
	MOBILE_HEIGHT: 599,

	/**
	 * Sidebar Gadgetの判定。
	 *
	 * @return {Boolean} Sidebar Gadgetだったらtrueを返す
	 */
	isSidebar: function() {
		var flgGadget = false;

		if (aaoh.kaze.FLG_WINDOW == 0 && !!window.System) {
			flgGadget = true;
		}

		return flgGadget;
	},

	/**
	 * Opera Widgetの判定。
	 *
	 * @return {Boolean} Opera Widgetだったらtrueを返す
	 */
	isOpera: function() {
		var flgGadget = false;

		if (aaoh.kaze.FLG_WINDOW == 0 && !!window.widget && !!window.opera) {
			flgGadget = true;
		}

		return flgGadget;
	},

	/**
	 * Dashboard Widgetの判定。
	 *
	 * @return {Boolean} Dashboard Widgetだったらtrueを返す
	 */
	isDashboard: function() {
		var flgGadget = false;

		if (
			aaoh.kaze.FLG_WINDOW == 0
			&& !!window.widget && !!window.widget.setCloseBoxOffset
		) {
			flgGadget = true;
		}

		return flgGadget;
	},

	/**
	 * AIRの判定。
	 *
	 * @return {Boolean} AIRだったらtrueを返す
	 */
	isAIR: function() {
		var flgGadget = false;

		if (aaoh.kaze.FLG_WINDOW == 0 && !!window.air) {
			flgGadget = true;
		}

		return flgGadget;
	},

	/**
	 * SBM Widgetの判定。
	 *
	 * @return {Boolean} SBM Widgetだったらtrueを返す
	 */
	isSBM: function() {
		var flgGadget = false;

		if (
			aaoh.kaze.FLG_WINDOW == 0 
			&& !!window.widget && !!window.widget.calendar
		) {
			flgGadget = true;
		}

		return flgGadget;
	},

	/**
	 * iGoogle Gadgetの判定。
	 *
	 * @return {Boolean} iGoogle Gadgetだったらtrueを返す
	 */
	isIGoogle: function() {
		var flgGadget = false;

		//if (aaoh.kaze.FLG_WINDOW == 0 && !!window._IG_Prefs)
		if (aaoh.kaze.FLG_WINDOW == 0 && !!window.gadgets) {
			flgGadget = true;
		}

		return flgGadget;
	},

	/**
	 * Google Gadgetの判定。
	 *
	 * @return {Boolean} Google Gadgetだったらtrueを返す
	 */
	isGoogle: function() {
		var flgGadget = false;

		if (aaoh.kaze.FLG_WINDOW == 1) {
			flgGadget = true;
		}

		return flgGadget;
	},

	/**
	 * Yahoo Widgetの判定。
	 *
	 * @return {Boolean} Yahoo Widgetだったらtrueを返す
	 */
	isYahoo: function() {
		var flgGadget = false;

		if (aaoh.kaze.FLG_WINDOW == 2) {
			flgGadget = true;
		}

		return flgGadget;
	},

	/**
	 * Yahoo Widget(mac)の判定。
	 *
	 * @return {Boolean} Yahoo Widgetだったらtrueを返す
	 */
	isYahooMac: function() {
		var flgGadget = false;

		if (aaoh.kaze.Type.isYahoo() && system.platform == 'macintosh') {
			flgGadget = true;
		}

		return flgGadget;
	},

	/**
	 * 通常ブラウザの判定。
	 *
	 * @return {Boolean} 通常ブラウザだったらtrueを返す
	 */
	isOthers: function() {
		var flgGadget = true;

		if (
			aaoh.kaze.FLG_WINDOW != 0
			//|| (!!window.System || !!window.widget || !!window.air || !!window._IG_Prefs)
			|| (!!window.System || !!window.widget || !!window.air || !!window.gadgets)
		) {
			flgGadget = false;
		}

		return flgGadget;
	},

	/**
	 * IEかの判定。
	 *
	 * @return {Boolean} IEだったらtrueを返す
	 */
	isIE: function() {
		var flgGadget = false;

		if (
			aaoh.kaze.FLG_WINDOW == 0
			&& !!window.attachEvent && !window.opera
		) {
			flgGadget = true;
		}

		return flgGadget;
	},

	/**
	 * モバイルかの判定。
	 *
	 * @return {Boolean} モバイルだったらtrue
	 */
	isMobile: function() {
		if (aaoh.kaze.FLG_WINDOW != 0 || !window.screen) {
			return false;
		}

		if (
			screen.width <= aaoh.kaze.Type.MOBILE_WIDTH
			|| screen.height <= aaoh.kaze.Type.MOBILE_HEIGHT
		) {
			return true;
		}

		if (!!window.widget && !!window.opera) {
			if (
				!!window.parent.ui
				&& !!window.parent.ui.device
				&& !!window.parent.ui.device.media
			) {
				var mediaType = window.parent.ui.device.media.type;
				if (mediaType == 'handheld' || mediaType == 'tv') {
					return true;
				}
			}

			if (
				!!window.parent.Control
				&& !!window.parent.Control.getCurrentDevice()
			) {
				var mediaType = window.parent.Control.getCurrentDevice().media;
				if (mediaType == 'handheld' || mediaType == 'tv') {
					return true;
				}
			}
		} // if (!!window.widget && !!window.opera)

		return false;
	}
}
})();

(function () {
/**
 * Gadget。
 *
 * @namespace
 */
aaoh.kaze.Gadget = {
	ERROR_NM: "errorLocation",

	/**
	 * エレメントの取得。
	 *
	 * @param  {String} id id名
	 * @return {Object}    エレメント
	 */
	$: function(id) {
		if (id == null) {
			throw new Error('id is null');
		}

		if (typeof id != 'string') {
			return id;
		}

		var element = null;
		if (aaoh.kaze.Type.isGoogle()) {
			//element = view.children.item(id);
			try {
				element = eval(id);
			} catch (ex) {
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			element = widget.getElementById(id);

		} else {
			element = document.getElementById(id);
		}

		return element;
	},

	/**
	 * 継承を行います。
	 *
	 * @param  {Object} parent  親オブジェクト
	 * @param  {Object} current 自身オブジェクト
	 * @return {Object}         継承した自身オブジェクト
	 */
	extend: function(parent, current) {
		if (parent == null) {
			throw new Error('parent is null');

		} else if (current == null) {
			throw new Error('current is null');
		}

		function parent_() {};
		parent_.prototype = parent.prototype;

		current.prototype                       = new parent_();
		current.prototype.__super__             = parent.prototype;
		current.prototype.__super__.constructor = parent;
		current.prototype.constructor           = current;

		current.prototype.__super__.constructor();
		return current;
	},

	/**
	 * URLをブラウザにて開きます。
	 *
	 * @param {String} url URL
	 */
	openURL: function(url) {
		if (url == null) {
			throw new Error('url is null');
		}

		if (aaoh.kaze.Type.isSidebar()) {
			System.Shell.execute(url);

		} else if (aaoh.kaze.Type.isAIR()) {
			air.navigateToURL(new air.URLRequest(url));

		} else if (aaoh.kaze.Type.isGoogle()) {
			framework.openUrl(url);

		} else if (aaoh.kaze.Type.isYahoo()) {
			openURL(url);

		} else if (
			aaoh.kaze.Type.isIGoogle()
			|| aaoh.kaze.Type.isOthers()
		) {
			window.open(url);

		} else {
			widget.openURL(url);
		}
	},

	/**
	 * 表示の設定。
	 *
	 * @param {Boolean} value 設定値
	 */
	setVisible: function(value) {
		if (value == null) {
			throw new Error('value is null');
		}

		if (aaoh.kaze.Type.isAIR()) {
			if (value) {
				window.nativeWindow.activate();
				document.body.style.visibility = 'visible';

			} else {
				window.nativeWindow.close();
				document.body.style.visibility = 'hidden';
			}

		} else if (aaoh.kaze.Type.isGoogle()) {
			// 何もしない

		} else if (aaoh.kaze.Type.isYahoo()) {
			widget.visible = value;

		} else {
			if (value) {
				document.body.style.visibility = 'visible';

			} else {
				document.body.style.visibility = 'hidden';
			}
		} // if (aaoh.kaze.Type.isAIR())
	},

	/**
	 * 表示の取得。
	 *
	 * @return {Boolean} 表示されていたらtrueを返す
	 */
	isVisible: function() {
		var value = false;

		if (aaoh.kaze.Type.isAIR()) {
			value = window.nativeWindow.active;

		} else if (aaoh.kaze.Type.isGoogle()) {
			value = true;

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = widget.visible;

		} else {
			if (
				!!document.body.style.visibility
				&& document.body.style.visibility == 'visible'
			) {
				value = true;
			}
		}

		return value;
	},

	/**
	 * 表示の設定。
	 *
	 * @param {Boolean} value 設定値
	 */
	setDisplay: function(value) {
		if (value == null) {
			throw new Error('value is null');
		}

		if (aaoh.kaze.Type.isAIR()) {
			if (value) {
				window.nativeWindow.activate();
				document.body.style.display = '';

			} else {
				window.nativeWindow.close();
				document.body.style.display = 'none';
			}

		} else if (aaoh.kaze.Type.isGoogle()) {
			aaoh.kaze.Gadget.setVisible(value);

		} else if (aaoh.kaze.Type.isYahoo()) {
			aaoh.kaze.Gadget.setVisible(value);

		} else {
			if (value) {
				document.body.style.display = '';

			} else {
				document.body.style.display = 'none';
			}
		} // if (aaoh.kaze.Type.isAIR())
	},

	/**
	 * 表示の取得。
	 *
	 * @return {Boolean} 表示されていたらtrueを返す
	 */
	isDisplay: function() {
		var value = false;

		if (aaoh.kaze.Type.isAIR()) {
			value = aaoh.kaze.Gadget.isVisible();

		} else if (aaoh.kaze.Type.isGoogle()) {
			value = aaoh.kaze.Gadget.isVisible();

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = aaoh.kaze.Gadget.isVisible();

		} else {
			if (
				!!document.body.style.display
				&& document.body.style.display != 'none'
			) {
				value = true;
			}
		} // if (aaoh.kaze.Type.isAIR())

		return value;
	},

	/**
	 * サイズを設定します。
	 *
	 * @param {Number} width   幅
	 * @param {Number} height  高さ
	 */
	setSize: function(width, height) {
		if (isNaN(width)) {
			throw new Error('width is null, height is ' + height);

		} else if (isNaN(height)) {
			throw new Error('height is null, width is ' + width);
		}

		if (aaoh.kaze.Type.isIGoogle()) {
			//if (!!window._IG_AdjustIFrameHeight)
			if (!!gadgets.window && !!gadgets.window.adjustHeight) {
				//_IG_AdjustIFrameHeight(height);
				gadgets.window.adjustHeight(height);
			}

		} else if (aaoh.kaze.Type.isGoogle()) {
			view.width  = width;
			view.height = height;

		} else if (aaoh.kaze.Type.isYahoo()) {
			//widget.width  = width;
			//widget.height = height;
			var win = widget.getElementsByTagName('window');
			win.width  = width;
			win.height = height;

		} else if (aaoh.kaze.Type.isOthers()) {
			// 何もしない

		} else {
			window.resizeTo(width, height);
			document.body.style.width  = width  + 'px';
			document.body.style.height = height + 'px';
		} // if (aaoh.kaze.Type.isIGoogle())
	},

	/**
	 * 位置を設定します。
	 *
	 * @param {Number} x X座標
	 * @param {Number} y Y座標
	 */
	setPosition: function(x, y) {
		if (isNaN(x)) {
			throw new Error('x is null, y is ' + y);

		} else if (isNaN(y)) {
			throw new Error('y is null, x is ' + x);
		}

		if (aaoh.kaze.Type.isAIR()) {
			window.nativeWindow.x = x;
			window.nativeWindow.y = y;

		} else {
			// その他はなにもしない
		}
	},

	/**
	 * 位置の取得。
	 *
	 * @return {JSON} 表示位置
	 */
	getPosition: function() {
		var position = {
			x: 0,
			y: 0
		};

		if (aaoh.kaze.Type.isAIR()) {
			if (window.nativeWindow.closed) {
				// 閉じられている
				return null;
			}

			position.x = window.nativeWindow.x;
			position.y = window.nativeWindow.y;

		} else {
			position = null;
		}

		return position;
	},

	/**
	 * 閉じる処理。
	 */
	close: function() {
		if (aaoh.kaze.Type.isSidebar()) {
			System.Gadget.close();

		} else if (aaoh.kaze.Type.isAIR()) {
			var exitingEvent = new air.Event(air.Event.EXITING, false, true);
			air.NativeApplication.nativeApplication.dispatchEvent(exitingEvent);
			air.NativeApplication.nativeApplication.exit();

		} else if (aaoh.kaze.Type.isYahoo()) {
			widget.visible = false;

		} else if (aaoh.kaze.Type.isIGoogle() || aaoh.kaze.Type.isGoogle()) {
			// 何もしない

		} else {
			window.close();
		}
	},

	/**
	 * メッセージ表示が行えるか。
	 *
	 * @return {Boolean} メッセージ表示が行えるときにはtrueを返す
	 */
	isConfirm: function() {
		if (
			aaoh.kaze.Type.isSidebar()
			|| aaoh.kaze.Type.isDashboard() || aaoh.kaze.Type.isAIR()
			|| aaoh.kaze.Type.isSBM()
		) {
			return false;
		}

		return true;
	},

	/**
	 * メッセージ表示を行います。
	 *
	 * @param {String}   msg      メッセージ
	 * @param {Function} callable OKボタン押下ファンクション
	 * @param {Object}   callObj  呼び出し元オブジェクト
	 */
	confirm: function(msg, callable, callObj) {
		if (msg == null) {
			throw new Error('msg is null');

		} else if (!callable) {
			throw new Error('callable is null, msg is ' + msg);
		}
		if (callObj == null) {
			callObj = this;
		}
		//if (!aaoh.kaze.Gadget.isConfirm()) {
		//	throw new Error("can't confitm");
		//}

		if (aaoh.kaze.Type.isOpera()) {
			widget.showNotification(msg, callable);

		} else if (aaoh.kaze.Type.isGoogle()) {
			var flgConf = view.confirm(msg);
			if (flgConf) {
				callable.call(callObj);
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			var ttlOk     = widget.getLocalizedString('TTL_OK');
			var ttlCancel = widget.getLocalizedString('TTL_CANCEL');
			if (ttlOk == null) {
				ttlOk = 'OK';
			}
			if (ttlCancel == null) {
				ttlCancel = 'Cancel';
			}

			var flgConf = alert(msg, ttlOk, ttlCancel);
			if (flgConf == 1) {
				callable.call(callObj);
			}

		} else if (aaoh.kaze.Type.isIGoogle() || aaoh.kaze.Type.isOthers()) {
			var flgConf = confirm(msg);
			if (flgConf) {
				callable.call(callObj);
			}
		} // if (aaoh.kaze.Type.isOpera())

		// その他は何もしない
	},

	/**
	 * ファイル選択。
	 * Adobe AIR用のfilterName, filterExtは複数指定できます。
	 *
	 * @param {String}   filter     フィルタ
	 * @param {String}   filterName フィルタ名(Adobe AIR用)
	 * @param {String}   filterExt  フィルタリング拡張子(Adobe AIR用)
	 * @param {Function} callable   ファイル選択ファンクション
	 * @param {Object}   callObj    呼び出し元オブジェクト
	 */
	browseForFile: function(filter, filterName, filterExt) {
		if (filter == null) {
			throw new Error('filter is null, filterName is ' + filterName);

		} else if (filterName == null) {
			throw new Error('filterName is null, filter is ' + filter);

		} else if (filterExt == null) {
			throw new Error('filterExt is null, filter is ' + filter);

		} else if (arguments.length < 5) {
			throw new Error('callable is null, filter is ' + filter);
		}

		var callable = arguments[arguments.length - 2];
		var callObj  = arguments[arguments.length - 1];
		if (!callable) {
			throw new Error('callable is null, filter is ' + filter);
		}
		if (callObj == null) {
			callObj = this;
		}

		var selectFile = null;
		if (aaoh.kaze.Type.isSidebar()) {
			var shellItem = System.Shell.chooseFile(true, filter, '', '');
			if (shellItem != null) {
				selectFile = shellItem.path;
			}

			if (selectFile != null && selectFile != '') {
				callable.call(callObj, selectFile);
			}

		} else if (aaoh.kaze.Type.isAIR()) {
			var filterSize = (arguments.length - 3) / 2;
			var filterList = new window.runtime.Array();
			for (var cnt = 0; cnt < filterSize; cnt++) {
				var filter = new air.FileFilter(
					arguments[cnt * 2 + 1], arguments[cnt * 2 + 2]
				);
				filterList.push(filter);
			}

			var docDir = air.File.documentsDirectory;
			docDir.browseForOpen(filter, filterList);
			docDir.addEventListener(air.Event.SELECT, function(event) {
				var selectFile_ = event.target.nativePath;
				callable.call(callObj, selectFile_);
			});

		} else if (aaoh.kaze.Type.isGoogle()) {
			selectFile = framework.BrowseForFile(filter);
			if (selectFile != null && selectFile != '') {
				callable.call(callObj, selectFile);
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			// konで設定を記述するので何もしない

		} else {
			// 何もしない
		} // if (aaoh.kaze.Type.isSidebar())
	},

	/**
	 * 読み込み完了の設定。
	 *
	 * @param {Function} callable 読み込み完了通知ファンクション
	 */
	setOnLoad: function(callable) {
		if (!callable) {
			throw new Error('callable is null');
		}

		if (aaoh.kaze.Type.isAIR()) {
			air.NativeApplication.nativeApplication.addEventListener(
				air.InvokeEvent.INVOKE , callable
			);

		} else if (aaoh.kaze.Type.isIGoogle()) {
			//_IG_RegisterOnloadHandler(callable);
			gadgets.util.registerOnLoadHandler(callable)

		} else if (aaoh.kaze.Type.isGoogle()) {
			view.onopen = callable;

		} else if (aaoh.kaze.Type.isYahoo()) {
			widget.onLoad = callable;

		} else {
			window.onload = callable;
		} // if (aaoh.kaze.Type.isAIR())
	},

	/**
	 * 処理終了の設定。
	 *
	 * @param {Function} callable 処理終了通知ファンクション
	 */
	setUnLoad: function(callable) {
		if (!callable) {
			throw new Error('callable is null');
		}

		if (aaoh.kaze.Type.isAIR()) {
			air.NativeApplication.nativeApplication.addEventListener(
				air.Event.EXITING, callable
			);

		} else if (aaoh.kaze.Type.isGoogle()) {
			view.onclose = callable;

		} else if (aaoh.kaze.Type.isYahoo()) {
			widget.onUnload = callable;

		} else {
				window.onunload = callable;
		} // if (aaoh.kaze.Type.isAIR())
	},

	/**
	 * onDockイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnDock: function(callable) {
		if (!callable) {
			throw new Error('callable is null');
		}

		if (aaoh.kaze.Type.isSidebar()) {
			System.Gadget.onDock = callable;

		} else if (aaoh.kaze.Type.isGoogle()) {
			view.ondock = callable;
		}

		// その他はなにもしない
	},

	/**
	 * unDockイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnUnDock: function(callable) {
		if (!callable) {
			throw new Error('callable is null');
		}

		if (aaoh.kaze.Type.isSidebar()) {
			System.Gadget.onUndock = callable;

		} else if (aaoh.kaze.Type.isGoogle()) {
			view.onundock = callable;
		}

		// その他はなにもしない
	},

	/**
	 * イベントの取得。
	 *
	 * @return {Object} イベントオブジェクト
	 */
	getEvent: function() {
		var event = null;

		if (aaoh.kaze.Type.isYahoo()) {
			event = system.event;

		} else if (aaoh.kaze.Type.isGoogle()) {
			event = view.event;

		} else {
			event = window.event;
		}

		return event;
	},

	/**
	 * デバッグ情報の表示を行います。
	 *
	 * @param {String} msg メッセージ
	 */
	debug: function(msg) {
		if (msg == null) {
			throw new Error('msg is null');
		}

		if (aaoh.kaze.Type.isOpera()) {
			opera.postError(msg);

		} else if (aaoh.kaze.Type.isAIR()) {
			air.trace(msg);

		} else if (aaoh.kaze.Type.isGoogle()) {
			gadget.debug.trace(msg);

		} else if (aaoh.kaze.Type.isYahoo()) {
			log(msg);

		} else {
			var errorDiv = document.getElementById(aaoh.kaze.Gadget.ERROR_NM);

			if (errorDiv != null) {
				var errMsg = errorDiv.innerHTML;
				if (errMsg != null && errMsg != '') {
					errMsg = errMsg + '<br />';
				}
				errorDiv.innerHTML = errMsg + msg;

			} else {
				if (aaoh.kaze.Type.isIGoogle() || aaoh.kaze.Type.isOthers()) {
					alert(msg);

				} else {
					// 何もしない
				}
			} // if (errorDiv != null)
		} // if (aaoh.kaze.Type.isOpera())
	}
}
})();

(function () {
/**
 * Element。
 *
 * @namespace
 */
aaoh.kaze.Element = {
	/**
	 * 値の設定。
	 *
	 * @param {Object} element エレメント
	 * @param {String} value   設定値
	 */
	setValue: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);

		} else if (value == null) {
			throw new Error('value is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			if (
				element.tagName == 'checkbox'
				|| element.tagName == 'edit'
				|| element.tagName == 'progressbar'
				|| element.tagName == 'scrollbar'
				|| element.tagName == 'combobox'
			) {
				element.value = value;

			} else if (
				element.tagName == 'view'
				|| element.tagName == 'button'
			) {
				element.caption = value;

			} else {
				aaoh.kaze.Element.setInnerValue(element, value);
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.data = value;

		} else {
			if (element.value != undefined && element.tagName != 'BUTTON') {
				element.value = value;

			} else {
				aaoh.kaze.Element.setInnerValue(element, value);
			}
		} // if (aaoh.kaze.Type.isGoogle())
	},

	/**
	 * 値の取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {String}         設定値
	 */
	getValue: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = null;
		if (aaoh.kaze.Type.isGoogle()) {
			if (
				element.tagName == 'checkbox'
				|| element.tagName == 'edit'
				|| element.tagName == 'progressbar'
				|| element.tagName == 'scrollbar'
				|| element.tagName == 'combobox'
			) {
				value = element.value;

			} else if (
				element.tagName == 'view'
				|| element.tagName == 'button'
			) {
				value = element.caption;

			} else {
				value = aaoh.kaze.Element.getInnerValue(element);
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = element.data;

		} else {
			if (element.value != undefined) {
				value = element.value;

			} else {
				value = aaoh.kaze.Element.getInnerValue(element);
			}
		} // if (aaoh.kaze.Type.isGoogle())

		return value;
	},

	/**
	 * 値の設定。
	 *
	 * @param {Object} element エレメント
	 * @param {String} value   設定値
	 */
	setInnerValue: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);

		} else if (value == null) {
			throw new Error('value is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			if (
				element.tagName == 'label'
				|| element.tagName == 'checkbox'
				|| element.tagName == 'a'
			) {
				element.innerText = value;

			} else {
				element.children.item(0).innerText = value;
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			aaoh.kaze.Element.setValue(element, value);

		} else {
			element.innerHTML = value;
		} // if (aaoh.kaze.Type.isGoogle())
	},

	/**
	 * 値の取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {String}         設定値
	 */
	getInnerValue: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = null;
		if (aaoh.kaze.Type.isGoogle()) {
			if (
				element.tagName == 'label'
				|| element.tagName == 'checkbox'
				|| element.tagName == 'a'
			) {
				value = element.innerText;

			} else {
				value = element.children.item(0).innerText;
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			aaoh.kaze.Element.getValue(element);

		} else {
			value = element.innerHTML;
		} // if (aaoh.kaze.Type.isGoogle())

		return value;
	},

	/**
	 * サイズを設定します。
	 *
	 * @param {Object} element エレメント
	 * @param {Number} width   幅
	 * @param {Number} height  高さ
	 */
	setSize: function(element, width, height) {
		if (element == null) {
			throw new Error(
				'element is null, width is ' + width + ', height is' + height
			);

		} else if (isNaN(width)) {
			throw new Error(
				'width is null, element is ' + element.id + ', height is ' + height
			);

		} else if (isNaN(height)) {
			throw new Error(
				'height is null, element is ' + element.id + ', width is ' + width
			);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.width  = width;
			element.height = height;

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.width  = width;
			element.height = height;

		} else {
			element.style.width  = width  + 'px';
			element.style.height = height + 'px';
		} // if (aaoh.kaze.Type.isGoogle())
	},

	/**
	 * サイズを設定します。
	 *
	 * @param  {Object} element エレメント
	 * @return {JSON}           サイズ
	 */
	getSize: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}
		var size = {
			width:  0,
			height: 0
		};

		if (aaoh.kaze.Type.isGoogle()) {
			size.width  = element.width;
			size.height = element.height;

		} else if (aaoh.kaze.Type.isYahoo()) {
			size.width  = element.width;
			size.height = element.height;

		} else {
			var width_  = element.style.width;
			var height_ = element.style.height;
			if (width_ != null) {
				width_ = width_.replace('px', '');
				width_ = Number(width_);

			} else {
				width_ = 0;
			}
			if (height_ != null) {
				height_ = height_.replace('px', '');
				height_ = Number(height_);

			} else {
				height_ = 0;
			}

			size.width  = width_;
			size.height = height_;
		} // if (aaoh.kaze.Type.isGoogle())

		return size;
	},

	/**
	 * 位置を設定します。
	 *
	 * @param {Object} element エレメント
	 * @param {Number} x       X座標
	 * @param {Number} y       Y座標
	 */
	setPosition: function(element, x, y) {
		if (element == null) {
			throw new Error(
				'element is null, x is ' + x + ', y is' + y
			);

		} else if (isNaN(x)) {
			throw new Error(
				'x is null, element is ' + element.id + ', y is ' + y
			);

		} else if (isNaN(y)) {
			throw new Error(
				'y is null, element is ' + element.id + ', x is ' + x
			);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.x = x;
			element.y = y;

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.hOffset = x;
			element.vOffset = y;

		} else {
			element.style.left = x  + 'px';
			element.style.top  = y + 'px';
		} // if (aaoh.kaze.Type.isGoogle())
	},

	/**
	 * 位置の取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {JSON}   表示位置
	 */
	getPosition: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}
		var position = {
			x: 0,
			y: 0
		};

		if (aaoh.kaze.Type.isGoogle()) {
			position.x = element.x;
			position.y = element.y;

		} else if (aaoh.kaze.Type.isYahoo()) {
			position.x = element.hOffset;
			position.y = element.vOffset;

		} else {
			var x_ = element.style.left;
			var y_ = element.style.top;
			if (x_ != null) {
				x_ = x_.replace('px', '');
				x_ = Number(x_);

			} else {
				x_ = 0;
			}
			if (y_ != null) {
				y_ = y_.replace('px', '');
				y_ = Number(y_);

			} else {
				y_ = 0;
			}

			position.x = x_;
			position.y = y_;
		} // if (aaoh.kaze.Type.isGoogle())

		return position;
	},

	/**
	 * 値の設定。
	 *
	 * @param {Object}  element エレメント
	 * @param {Boolean} value   設定値
	 */
	setChecked: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.value = value;

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.checked = value;

		} else {
			element.checked = value;
		}
	},

	/**
	 * 値の取得。
	 *
	 * @param  {Object}  element エレメント
	 * @return {Boolean}         設定値
	 */
	isChecked: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = false;
		if (aaoh.kaze.Type.isGoogle()) {
			value = element.value;

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = element.checked;

		} else {
			value = element.checked;
		}

		return value;
	},

	/**
	 * 値の設定。
	 *
	 * @param {Object}  element エレメント
	 * @param {Boolean} value   選択エレメント
	 */
	setSelected: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.selected = value;
			if (value) {
				element.parentElement.selectedItem = element;
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.checked = value;

		} else {
			element.selected = value;
		}
	},

	/**
	 * 値の取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {String}         設定値
	 */
	getSelected: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = null;
		if (aaoh.kaze.Type.isGoogle()) {
			var selItem = element.selectedItem;
			value = selItem.children.item(0).innerText;

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = element.data;

		} else {
			value = element.value;
		}

		return value;
	},

	/**
	 * 値の取得。
	 *
	 * @param  {Object}  element エレメント
	 * @return {Boolean}         設定値
	 */
	isSelected: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = false;
		if (aaoh.kaze.Type.isGoogle()) {
			value = element.selected;

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = element.checked;

		} else {
			value = element.selected;
		}

		return value;
	},

	/**
	 * ツールチップの設定。
	 *
	 * @param {Object} element エレメント
	 * @param {String} value   設定値
	 */
	setTooltip: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);

		} else if (value == null) {
			throw new Error('value is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.tooltip = value;

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.tooltip = value;

		} else {
			element.title = value;
		}
	},

	/**
	 * ツールチップの取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {String}         設定値
	 */
	getTooltip: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = null;
		if (aaoh.kaze.Type.isGoogle()) {
			value = element.tooltip;

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = element.tooltip;

		} else {
			value = element.title;
		}

		return value;
	},

	/**
	 * フォーカスの設定。
	 *
	 * @param {Object} element エレメント
	 */
	setFocus: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		element.focus();
	},

	/**
	 * テキスト選択の設定。
	 *
	 * @param {Object} element エレメント
	 */
	setTextSelect: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		if (aaoh.kaze.Type.isGoogle()) {
			// 何もしない

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.select(0, -1);

		} else {
			element.select();
		}
	},

	/**
	 * カーソルの設定。
	 *
	 * @param {Object} element エレメント
	 * @param {String} value   設定値
	 */
	setCursor: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);

		} else if (value == null) {
			throw new Error('value is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			switch (value) {
				case 'auto':
					break;
				case 'crosshair':
					value = 'cross';
					break;
				case 'default':
					value = 'arrow';
					break;
				case 'pointer':
					value = 'hand';
					break;
				case 'move':
					break;
				case 'e-resize':
					value = 'sizewe';
					break;
				case 'ne-resize':
					break;
				case 'nw-resize':
					value = 'sizenwse';
					break;
				case 'n-resize':
					break;
				case 'se-resize':
					break;
				case 'sw-resize':
					value = 'sizenesw';
					break;
				case 's-resize':
					value = 'sizens';
					break;
				case 'w-resize':
					break;
				case 'text':
					value = 'ibeam';
					break;
				case 'wait':
					//value = 'wait';
					break;
				case 'help':
					//value = 'help';
					break;

				default:
					//value = 'uparrow';
					//value = 'size';
					//value = 'sizeall';
					//value = 'no';
					//value = 'busy';
					break;
			}
			element.cursor = value;

		} else if (aaoh.kaze.Type.isYahoo()) {
			//element.style.cursor = value;
			// 何もしない

		} else {
			element.style.cursor = value;
		}
	},

	/**
	 * カーソルの取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {String}         設定値
	 */
	getCurdor: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = null;
		if (aaoh.kaze.Type.isGoogle()) {
			value = element.cursor;
			switch (value) {
				//case '':
				//	value = 'auto';
				//	break;
				case 'cross':
					value = 'crosshair';
					break;
				case 'arrow':
					value = 'default';
					break;
				case 'hand':
					value = 'pointer';
					break;
				//case '':
				//	value = 'move';
				//	break;
				case 'sizewe':
					value = 'e-resize';
					break;
				//case '':
				//	value = 'ne-resize';
				//	break;
				case 'sizenwse':
					value = 'nw-resize';
					break;
				//case '':
				//	value = 'n-resize';
				//	break;
				//case '':
				//	value = 'se-resize';
				//	break;
				case 'sizenesw':
					value = 'sw-resize';
					break;
				case 'sizens':
					value = 's-resize';
					break;
				//case '':
				//	value = 'w-resize';
				//	break;
				case 'ibeam':
					value = 'text';
					break;
				case 'wait':
					//value = 'wait';
					break;
				case 'help':
					//value = 'help';
					break;

				case 'uparrow':
				case 'size':
				case 'sizeall':
				case 'no':
				case 'busy':
				default:
					break;
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = element.style.cursor;

		} else {
			value = element.style.cursor;
		}

		return value;
	},

	/**
	 * 表示の設定。
	 *
	 * @param {Object}  element エレメント
	 * @param {Boolean} value   設定値
	 */
	setVisible: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.visible = value;

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.visible = value;

		} else {
			if (value) {
				element.style.visibility = 'visible';

			} else {
				element.style.visibility = 'hidden';
			}
		} // if (aaoh.kaze.Type.isGoogle())
	},

	/**
	 * 表示の取得。
	 *
	 * @param  {Object}  element エレメント
	 * @return {Boolean}         設定値
	 */
	isVisible: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = false;
		if (aaoh.kaze.Type.isGoogle()) {
			value = element.visible;

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = element.visible;

		} else {
			if (
				!!element.style.visibility
				&& element.style.visibility == 'visible'
			) {
				value = true;
			}
		} // if (aaoh.kaze.Type.isGoogle())

		return value;
	},

	/**
	 * 表示の設定。
	 *
	 * @param {Object}  element エレメント
	 * @param {Boolean} value   設定値
	 */
	setDisplay: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			aaoh.kaze.Element.setVisible(element, value);

		} else if (aaoh.kaze.Type.isYahoo()) {
			aaoh.kaze.Element.setVisible(element, value);

		} else {
			if (value) {
				element.style.display = '';

			} else {
				element.style.display = 'none';
			}
		} // if (aaoh.kaze.Type.isGoogle())
	},

	/**
	 * 表示の取得。
	 *
	 * @param  {Object}  element エレメント
	 * @return {Boolean}         設定値
	 */
	isDisplay: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = false;
		if (aaoh.kaze.Type.isGoogle()) {
			value = aaoh.kaze.Element.isVisible(element);

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = aaoh.kaze.Element.isVisible(element);

		} else {
			if (
				!element.style.display
				|| element.style.display != 'none'
			) {
				value = true;
			}
		} // if (aaoh.kaze.Type.isGoogle())

		return value;
	},

	/**
	 * 有効の設定。
	 *
	 * @param {Object}  element エレメント
	 * @param {Boolean} value   設定値
	 */
	setEnabled: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.enabled = value;

		} else if (aaoh.kaze.Type.isYahoo()) {
			// 何もしない

		} else {
			element.disabled = !value;
		}
	},

	/**
	 * 有効の取得。
	 *
	 * @param  {Object}  element エレメント
	 * @return {Boolean}         設定値
	 */
	isEnabled: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = false;
		if (aaoh.kaze.Type.isGoogle()) {
			value = element.enabled;

		} else if (aaoh.kaze.Type.isYahoo()) {
			// 何もしない

		} else {
			value = !element.disabled;
		}

		return value;
	},

	/**
	 * 表示色の設定。
	 *
	 * @param {Object} element エレメント
	 * @param {String} value   設定値
	 */
	setForeground: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);

		} else if (value == null) {
			throw new Error('value is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.color = value;

		} else {
			element.style.color = value;
		}
	},

	/**
	 * 表示色の取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {String}         設定値
	 */
	getForeground: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		if (aaoh.kaze.Type.isGoogle()) {
			value = element.color;

		} else {
			value = element.style.color;
		}

		return value;
	},

	/**
	 * 背景色の設定。
	 *
	 * @param {Object} element エレメント
	 * @param {String} value   設定値
	 */
	setBackground: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);

		} else if (value == null) {
			throw new Error('value is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.background = value;

		} else {
			element.style.backgroundColor = value;
		}
	},

	/**
	 * 背景色の取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {String}         設定値
	 */
	getBackground: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = null;
		if (aaoh.kaze.Type.isGoogle()) {
			value = element.background;

		} else {
			value = element.style.backgroundColor;
		}

		return value;
	},

	/**
	 * ボーダー色の設定。
	 *
	 * @param {Object} element エレメント
	 * @param {String} value   設定値
	 */
	setBorderColor: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);

		} else if (value == null) {
			throw new Error('value is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isGoogle() || aaoh.kaze.Type.isYahoo()) {
			// 何もしない

		} else {
			element.style.borderColor = value;
		}
	},

	/**
	 * ボーダー色の取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {String}         設定値
	 */
	getBorderColor: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = null;
		if (aaoh.kaze.Type.isGoogle() || aaoh.kaze.Type.isYahoo()) {
			// 何もしない

		} else {
			value = element.style.borderColor;
		}

		return value;
	},

	/**
	 * 画像の設定。
	 * Google Gadgetでは第3, 4引数を設定することにより
	 * overImage, downImageも設定できます。
	 * iGoogle Gadgetでは第3引数を設定することにより
	 * IE6での透過PNG対応を行うかの設定が行えます。
	 *
	 * @param {Object} element エレメント
	 * @param {String} value   設定値
	 */
	setImage: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);

		} else if (value == null) {
			throw new Error('value is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isIGoogle()) {
			var base = document.getElementsByTagName('base');
			if (base != null && base.length > 0) {
				value = base[0].href + value;
			}
			element.src = value;

			// IE6 透過PNG対応
			if (
				aaoh.kaze.Type.isIE()
				&& (arguments.length <= 2 || !!arguments[arguments.length - 1])
			) {
				aaoh.kaze.Element.setPNGImage(element, value);
			}

		} else if (aaoh.kaze.Type.isGoogle()) {
			if (element.tagName != 'img') {
				element.image = value;
				if (arguments.length > 2) {
					element.overImage = arguments[2];
				}
				if (arguments.length > 3) {
					element.downImage = arguments[3];
				}

			} else {
				element.src = value;
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.src = 'Resources/' + value;

		} else {
			element.src = value;

			// IE6 透過PNG対応
			if (
				aaoh.kaze.Type.isIE()
				&& (arguments.length <= 2 || !!arguments[arguments.length - 1])
			) {
				aaoh.kaze.Element.setPNGImage(element, value);
			}
		} // if (aaoh.kaze.Type.isIGoogle())
	},

	/**
	 * 画像の設定(IE6 透過PNG対応)。
	 *
	 * @param {Object} element エレメント
	 * @param {String} value   設定値
	 */
	setPNGImage: function(element, value) {
		if (
			!!window.IEPNGFIX
			&& !!element.currentStyle && !!element.style
			&& value.lastIndexOf(".png") >= 0 
		) {
			IEPNGFIX.fix(element);
		}
	},

	/**
	 * 画像の取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {String}         設定値
	 */
	getImage: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = null;
		if (aaoh.kaze.Type.isGoogle()) {
			if (element.tagName != 'img') {
				value = element.image;

			} else {
				value = element.src;
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = element.src;
			value = value.replace('Resources/', '');

		} else {
			value = element.src;
		} // if (aaoh.kaze.Type.isGoogle())

		return value;
	},

	/**
	 * 背景画像の設定。
	 * Google Gadgetでは第3, 4引数を設定することにより
	 * overImage, downImageも設定できます。
	 * IE6での透過PNG対応を行うかの設定が行えます。
	 *
	 * @param {Object} element エレメント
	 * @param {String} value   設定値
	 */
	setBackgroundImage: function(element, value) {
		if (element == null) {
			throw new Error('element is null, value is ' + value);

		} else if (value == null) {
			throw new Error('value is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isIGoogle()) {
			var base = document.getElementsByTagName('base');
			if (base != null && base.length > 0) {
				value = base[0].href + value;
			}
			element.style.backgroundImage = 'url(' + value + ')';
			element.style.backgroundRepeat = 'no-repeat';

			// IE6 透過PNG対応
			if (
				aaoh.kaze.Type.isIE()
				&& (arguments.length <= 2 || !!arguments[arguments.length - 1])
			) {
				aaoh.kaze.Element.setPNGImage(element, value);
			}

		} else if (aaoh.kaze.Type.isGoogle()) {
			if (
				element.tagName == 'div'
				|| element.tagName == 'edit'
				|| element.tagName == 'scrollbar'
				|| element.tagName == 'listbox'
				|| element.tagName == 'item'
				|| element.tagName == 'combobox'
			) {
				element.background = value;

			} else {
				element.image = value;
				if (arguments.length > 2) {
					element.overImage = arguments[2];
				}
				if (arguments.length > 3) {
					element.downImage = arguments[3];
				}
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.style.backgroundImage = 'url(Resources/' + value + ')';
			element.style.backgroundRepeat = 'no-repeat';

		} else {
			element.style.backgroundImage = 'url(' + value + ')';
			element.style.backgroundRepeat = 'no-repeat';

			// IE6 透過PNG対応
			if (
				aaoh.kaze.Type.isIE()
				&& (arguments.length <= 2 || !!arguments[arguments.length - 1])
			) {
				aaoh.kaze.Element.setPNGImage(element, value);
			}
		}
	},

	/**
	 * 背景画像の取得。
	 *
	 * @param  {Object} element エレメント
	 * @return {String}         設定値
	 */
	getBackgroundImage: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = null;
		if (aaoh.kaze.Type.isGoogle()) {
			if (
				element.tagName == 'div'
				|| element.tagName == 'edit'
				|| element.tagName == 'scrollbar'
				|| element.tagName == 'listbox'
				|| element.tagName == 'item'
				|| element.tagName == 'combobox'
			) {
				value = element.background;

			} else {
				value = element.image;
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = element.style.backgroundImage;
			value = value.replace('url(Resources/', '');
			value = value.replace(')', '');

		} else {
			value = element.style.backgroundImage;
			value = value.replace('url(', '');
			value = value.replace(')', '');
		}

		return value;
	},

	/**
	 * 透明度の設定。
	 *
	 * @param {Object} element エレメント
	 * @param {Number} opacity 透明度(0 - 100)
	 */
	setOpacity: function(element, opacity) {
		if (element == null) {
			throw new Error('element is null, opacity is ' + opacity);

		} else if (isNaN(opacity)) {
			throw new Error('opacity is null, element is ' + element.id);

		} else if (opacity < 0 || opacity > 100) {
			throw new Error(
				'out of value, '
				+ 'element is ' + element.id + ', '
				+ 'opacity is ' + opacity
			);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			var opacity_ = opacity * 2.55;
			element.opacity = opacity_; // 0 - 255

		} else if (aaoh.kaze.Type.isIE()) {
			element.style.filter = 'Alpha(opacity=' + opacity + ')'; // 0 - 100

		} else {
			var opacity_ = opacity * 0.01;
			element.style.opacity = opacity_; // 0.0 - 1.0
		} // if (aaoh.kaze.Type.isGoogle())
	},

	/**
	 * 透明度の取得。
	 *
	 * @param  {Object}  element エレメント
	 * @return {Number} 透明度(0 - 100)
	 */
	getOpacity: function(element) {
		if (element == null) {
			throw new Error('element is null');
		}

		var value = 0;
		if (aaoh.kaze.Type.isGoogle()) {
			var value_ = parseFloat(element.opacity, 10);
			value = value_ / 2.55;

		} else if (aaoh.kaze.Type.isIE()) {
			var value_ = null;

			var fillterValue = element.style.filter;
			var startIndex  = fillterValue.indexOf('opacity=');
			if (startIndex >= 0) {
				startIndex = startIndex + 'opacity='.length;
				var endIndex = fillterValue.indexOf(')', startIndex);

				if (endIndex >= 0) {
					value_ = fillterValue.substring(startIndex, endIndex);
				}
			}

			if (value_ != null) {
				value = parseInt(value_);
			}

		} else {
			var value_ = parseFloat(element.style.opacity, 10);
			value = value_ / 0.01;
		} // if (aaoh.kaze.Type.isGoogle())

		return parseInt(value, 10);
	},

	/**
	 * onclickイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnClick: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isYahoo()) {
			element.onClick = callable;

		} else {
			element.onclick = callable;
		}
	},

	/**
	 * onchangeイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnChange: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.onchange = callable;

		} else if (aaoh.kaze.Type.isYahoo()) {
			// 何もしない

		} else {
			if (element.tagName == 'SELECT') {
				element.onchange = callable;

			} else {
				return aaoh.kaze.Element.setOnClick(element, callable);
			}
		}
	},

	/**
	 * onkeyupイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnKeyUp: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isYahoo()) {
			element.onKeyUp = callable;

		} else {
			element.onkeyup = callable;
		}
	},

	/**
	 * onkeydownイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnKeyDown: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isYahoo()) {
			element.onKeyDown = callable;

		} else {
			element.onkeydown = callable;
		}
	},

	/**
	 * onkeypressイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnKeyPress: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isYahoo()) {
			element.onKeyPress = callable;

		} else {
			element.onkeypress = callable;
		}
	},

	/**
	 * onmousedownイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnMouseDown: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isYahoo()) {
			element.onMouseDown = callable;

		} else {
			element.onmousedown = callable;
		}
	},

	/**
	 * onmouseupイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnMouseUp: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isYahoo()) {
			element.onMouseUp = callable;

		} else {
			element.onmouseup = callable;
		}
	},

	/**
	 * onmouseoverイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnMouseOver: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isYahoo()) {
			element.onMouseEnter = callable;

		} else {
			element.onmouseover = callable;
		}
	},

	/**
	 * onmouseoutイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnMouseOut: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isYahoo()) {
			element.onMouseExit = callable;

		} else {
			element.onmouseout = callable;
		}
	},

	/**
	 * onfocusイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnFocus: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.onfocusin = callable;

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.onGainFocus = callable;

		} else {
			element.onfocus = callable;
		}
	},

	/**
	 * onblurイベントの追加。
	 *
	 * @param {Object}   element  エレメント
	 * @param {Function} callable イベント通知ファンクション
	 */
	setOnBlur: function(element, callable) {
		if (element == null) {
			throw new Error('element is null');

		} else if (!callable) {
			throw new Error('callable is null, element is ' + element.id);
		}

		if (aaoh.kaze.Type.isGoogle()) {
			element.onfocusout = callable;

		} else if (aaoh.kaze.Type.isYahoo()) {
			element.onLoseFocus = callable;

		} else {
			element.onblur = callable;
		}
	}
}
})();

(function () {
/**
 * Ajax。
 *
 * @constructor
 * @param {Object} callObj_ 呼び出し元オブジェクト
 */
aaoh.kaze.Ajax = function(callObj_) {
	this.callObj = callObj_;
	if (this.callObj == null) {
		this.callObj = this;
	}
}
aaoh.kaze.Ajax.prototype = {
	/**
	 * get通信を行います。
	 * できる限り同期を試みます。
	 *
	 * @param {String}   url      リクエストURL
	 * @param {Function} callable レスポンス受信ファンクション
	 */
	gets: function(url, callable) {
		if (url == null) {
			throw new Error('url is null');

		} else if (!callable) {
			throw new Error('callable is null, url is ' + url);
		}

		if (aaoh.kaze.Type.isAIR()) {
			var callObj_ = this.callObj;

			var request = new air.URLRequest(url);
			var loader  = new air.URLLoader();
			loader.addEventListener(air.Event.COMPLETE, function(event) {
				var loader_   = event.target;

				callable.call(callObj_, loader_.data);
			});
			loader.load(request);

		} else if (aaoh.kaze.Type.isIGoogle()) {
			//_IG_FetchContent(url, callable);
			var callObj_ = this.callObj;

			var params = {};
			params[gadgets.io.RequestParameters.CONTENT_TYPE]
				= gadgets.io.ContentType.JSON;
			params[gadgets.io.RequestParameters.METHOD]
				= gadgets.io.MethodType.GET;
			gadgets.io.makeRequest(url, function(obj) {
				callable.call(callObj_, obj.text);
			}, params);

		} else {
			try {
				if (!!window.netscape) {
					netscape.security.PrivilegeManager.enablePrivilege('UniversalBrowserRead');
				}
			} catch (ex) {
			}

			var res = ajax.gets(url);
			callable.call(this.callObj, res);
		} // if (aaoh.kaze.Type.isAIR())
	},

	/**
	 * get通信を行います。
	 * できる限り同期を試みます。
	 *
	 * @param {String}   url      リクエストURL
	 * @param {Function} callable レスポンス受信ファンクション
	 */
	getsForXML: function(url, callable) {
		if (url == null) {
			throw new Error('url is null');

		} else if (!callable) {
			throw new Error('callable is null, url is ' + url);
		}

		if (aaoh.kaze.Type.isAIR()) {
			var callObj_ = this.callObj;

			var request = new air.URLRequest(url);
			var loader  = new air.URLLoader();
			loader.addEventListener(air.Event.COMPLETE, function(event) {
				var loader_   = event.target;

				var resDoc = null;
				if (loader_.data != null) {
					try{
						var appXml = new DOMParser();
						resDoc = appXml.parseFromString(loader_.data, 'text/xml');

					} catch(ex) {
					}
				}

				callable.call(callObj_, resDoc);
			});
			loader.load(request);

		} else if (aaoh.kaze.Type.isIGoogle()) {
			//_IG_FetchXmlContent(url, callable);
			var callObj_ = this.callObj;

			var params = {};
			params[gadgets.io.RequestParameters.CONTENT_TYPE]
				= gadgets.io.ContentType.DOM;
			params[gadgets.io.RequestParameters.METHOD]
				= gadgets.io.MethodType.GET;
			gadgets.io.makeRequest(url, function(obj) {
				callable.call(callObj_, obj.data);
			}, params);

		} else {
			try {
				if (!!window.netscape) {
					netscape.security.PrivilegeManager.enablePrivilege('UniversalBrowserRead');
				}
			} catch (ex) {
			}

			var x = ajax.x();
			x.open('GET', url, false);
			x.send(null);
			var resXML = x.responseXML;
			var resDoc = null;
			if(resXML != null && resXML.documentElement != null) {
				resDoc = resXML.documentElement;
			}
			callable.call(this.callObj, resDoc);
		} // if (aaoh.kaze.Type.isAIR())
	}
}
})();

(function () {
/**
 * JSON。
 *
 * @constructor
 */
aaoh.kaze.JSON = function() {
	this.flgEnable = true;
	if (aaoh.kaze.FLG_WINDOW == 0 && !window.JSON) {
		this.flgEnable = false;
	}
}
aaoh.kaze.JSON.prototype = {
	/**
	 * 文字列JSONをJSONオブジェクトにパースします。
	 *
	 * @param  {String} strJSON  文字列JSON
	 * @return {Object}          JSONオブジェクト
	 */
	parse: function(txtJSON) {
		if (!this.flgEnable || txtJSON == null) {
			return null;
		}

		return JSON.parse(txtJSON);
	},

	/**
	 * JSONオブジェクトを文字列JSONに変換します。
	 *
	 * @param  {Object} jsonObj JSONオブジェクト
	 * @return {String}         文字列JSON
	 */
	stringify: function(jsonObj) {
		if (!this.flgEnable || !jsonObj) {
			return null;
		}

		return JSON.stringify(jsonObj);
	}
}
})();

(function () {
/**
 * Storage。
 *
 * @constructor
 */
aaoh.kaze.Storage = function() {
	this.pref = null;
	if (aaoh.kaze.Type.isIGoogle()) {
		//this.pref = new _IG_Prefs();
		this.pref = new gadgets.Prefs();
	}
}
aaoh.kaze.Storage.prototype = {
	/**
	 * 設定ファイルから情報を読み込みます。
	 *
	 * @param  {String} key 設定キー
	 * @return {String}     設定情報
	 */
	readString: function(key) {
		if (key == null) {
			throw new Error('key is null');
		}

		var value = null;
		if (aaoh.kaze.Type.isOthers()) {
			return null;

		} else if (aaoh.kaze.Type.isSidebar()) {
			value = System.Gadget.Settings.readString(key);

		} else if (aaoh.kaze.Type.isAIR()) {
			var dir  = air.File.applicationStorageDirectory;
			var file = dir.resolvePath(key + '.txt');

			if (file.exists) {
				var fileStream = new air.FileStream();
				fileStream.open(file, air.FileMode.READ);
				value = fileStream.readUTFBytes(fileStream.bytesAvailable);
				fileStream.close();
			}

		} else if (aaoh.kaze.Type.isIGoogle()) {
			// 型変換
			if (key.indexOf('is') == 0) {
				value = this.pref.getBool(key);

			} else if (key.indexOf('num') == 0) {
				value = this.pref.getInt(key);

			} else {
				value = this.pref.getString(key);
			}

		} else if (aaoh.kaze.Type.isGoogle()) {
			value = options.getValue(key);

		} else if (aaoh.kaze.Type.isYahoo()) {
			var prefObj = preferences[key];
			if (prefObj != null) {
				value = prefObj.value;
			}

			// 型変換
			if (key.indexOf('is') == 0) {
				if (value != null && value == '1') {
					value = true;

				} else {
					value = false;
				}

			} else if (key.indexOf('num') == 0) {
				if (value != null) {
					value = Number(value);
				}
			} // if (key.indexOf('is') == 0)

		} else {
			value = widget.preferenceForKey(key);
		} // if (aaoh.kaze.Type.isOthers())

		return value;
	},

	/**
	 * 設定ファイルに情報を読み込みます。
	 *
	 * @param  {String} key   設定キー
	 * @param  {String} value 設定情報
	 */
	writeString: function(key, value) {
		if (key == null) {
			throw new Error('key is null, value is ' + value);
		}

		if (aaoh.kaze.Type.isOthers()) {
			return;

		} else if (aaoh.kaze.Type.isSidebar()) {
			System.Gadget.Settings.writeString(key, value);

		} else if (aaoh.kaze.Type.isAIR()) {
			var dir  = air.File.applicationStorageDirectory;
			var file = dir.resolvePath(key + '.txt');

			var fileStream = new air.FileStream();
			fileStream.open(file, air.FileMode.WRITE);
			fileStream.writeUTFBytes(value);
			fileStream.close();

		} else if (aaoh.kaze.Type.isIGoogle()) {
			this.pref.set(key, value);

		} else if (aaoh.kaze.Type.isGoogle()) {
			options.putValue(key, value);

		} else if (aaoh.kaze.Type.isYahoo()) {
			var prefObj = preferences[key];
			if (prefObj != null) {
				// 型変換
				if (key.indexOf('is') == 0) {
					if (value) {
						value = '1';

					} else {
						value = '0';
					}
				}

				prefObj.value = value;
			}

		} else {
			widget.setPreferenceForKey(value, key);
		} // if (aaoh.kaze.Type.isOthers())
	}
}
})();

(function () {
/**
 * Clipboard。
 *
 * @constructor
 */
aaoh.kaze.Clipboard = function() {
}
aaoh.kaze.Clipboard.prototype = {
	/**
	 * Clipboardが行えるか。
	 *
	 * @return {Boolean} Clipboardが行えるときにはtrueを返す
	 */
	isClipboard: function() {
		if (
			aaoh.kaze.Type.isSidebar() || aaoh.kaze.Type.isDashboard()
			|| aaoh.kaze.Type.isAIR() || aaoh.kaze.Type.isYahoo()
			|| (aaoh.kaze.FLG_WINDOW == 0 && !!window.clipboardData)
		) {
			return true;
		}

		return false;
	},

	/**
	 * コピーを行います。
	 *
	 * @param {String} value コピーを行う文字列
	 */
	copy: function(value) {
		if (!this.isClipboard()) {
			throw new Error("can't copy");
		}
		if (value == null) {
			return;
		}

		if (aaoh.kaze.Type.isDashboard()) {
			widget.system("/bin/echo -n '" + value + "' | /usr/bin/pbcopy", null);

		} else if (aaoh.kaze.Type.isAIR()) {
			var copyObj = air.Clipboard.generalClipboard;
			copyObj.clear();
			copyObj.setData(air.ClipboardFormats.TEXT_FORMAT, value);

		} else if (aaoh.kaze.Type.isYahoo()) {
			system.clipboard = value;

		} else {
			clipboardData.setData('Text', value);
		}
	},

	/**
	 * ペーストを行います。
	 *
	 * @return {String} ペーストされた文字列
	 */
	paste: function() {
		if (!this.isClipboard()) {
			throw new Error("can't paste");
		}

		var value = null;
		if (aaoh.kaze.Type.isDashboard()) {
			value = widget.system('/usr/bin/pbpaste', null).outputString;

		} else if (aaoh.kaze.Type.isAIR()) {
			var pasteObj = air.Clipboard.generalClipboard;
			value = pasteObj.getData(air.ClipboardFormats.TEXT_FORMAT);

		} else if (aaoh.kaze.Type.isYahoo()) {
			value = system.clipboard;

		} else {
			value = clipboardData.getData('Text');
		}

		return value;
	}
}
})();

(function () {
/**
 * Sound。
 *
 * @constructor
 */
aaoh.kaze.Sound = function() {
	this.volume          = 50;
	this.flgRepeat       = false;
	this.vistaId         = null;
	this.airSoundChannel = null;
	this.googleAudioClip = null;
}
aaoh.kaze.Sound.prototype = {
	/**
	 * 音量の設定が行えるか。
	 *
	 * @return {Boolean} 音の設定が行えるときにはtrueを返す
	 */
	isVolume: function() {
		if (
			aaoh.kaze.Type.isAIR()
			|| aaoh.kaze.Type.isGoogle() || aaoh.kaze.Type.isYahoo()
		) {
			return true;
		}

		return false;
	},

	/**
	 * 音量の設定。
	 *
	 * @param {Number} volume 音量(0 - 100)
	 */
	setVolume: function(volume) {
		if (!this.isVolume()) {
			throw new Error("can't set volume");
		}
		if (isNaN(volume)) {
			throw new Error('volume is null');

		} else if (volume < 0 || volume > 100) {
			throw new Error('out of value, volume is ' + volume);
		}

		this.volume = volume;
	},

	/**
	 * 音量の取得。
	 *
	 * @return {Number} 音量(0 - 100)
	 */
	getVolume: function() {
		return this.volume;
	},

	/**
	 * リピート再生が行えるか。
	 *
	 * @return {Boolean} リピート再生が行えるときにはtrueを返す
	 */
	isRepeat: function() {
		if (aaoh.kaze.Type.isAIR()) {
			return true;
		}

		return false;
	},

	/**
	 * リピート再生の設定。
	 *
	 * @param {Boolean} flgRepeat リピート再生
	 */
	setRepeat: function(flgRepeat) {
		this.flgRepeat = flgRepeat;
	},

	/**
	 * リピート再生の取得。
	 *
	 * @return {Boolean} リピート再生を行うときにはtrueを返す
	 */
	getRepeat: function() {
		return this.flgRepeat;
	},

	/**
	 * ビープ音の再生が行えるか。
	 *
	 * @return {Boolean} ビープ音の再生が行えるときにはtrueを返す
	 */
	isBeep: function() {
		if (aaoh.kaze.Type.isSidebar() || aaoh.kaze.Type.isYahoo()) {
			return true;
		}

		return false;
	},

	/**
	 * ビープ音の再生。
	 */
	beep: function() {
		if (!this.isBeep()) {
			throw new Error("can't play");
		}

		if (aaoh.kaze.Type.isSidebar()) {
			System.Sound.beep();

		} else if (aaoh.kaze.Type.isYahoo()) {
			beep();
		} // if (aaoh.kaze.Type.isSidebar())
	},

	/**
	 * 音の再生が行えるか。
	 *
	 * @return {Boolean} 音の再生が行えるときにはtrueを返す
	 */
	isSound: function() {
		if (
			aaoh.kaze.Type.isSidebar() || aaoh.kaze.Type.isAIR()
			|| aaoh.kaze.Type.isGoogle() || aaoh.kaze.Type.isYahoo()
		) {
			return true;
		}

		return false;
	},

	/**
	 * 音を再生します。
	 *
	 * @param {String} file 再生ファイル
	 */
	play: function(file) {
		if (file == null) {
			throw new Error('file is null');
		}
		if (!this.isSound()) {
			throw new Error("can't play");
		}

		this.stop();
		if (aaoh.kaze.Type.isSidebar()) {
			// 音量の設定なし
			// リピート再生の設定なし
			//if (this.flgRepeat) {
			//	var fileItem = System.Shell.itemFromPath(file);
			//	var duration = fileItem.metadata('Duration');
			//	var interval = 0;
			//	if (duration != null && duration != '') {
			//		var durationList = duration.split(':');
			//		if (durationList.length >= 3) {
			//			interval = durationList[0] * 60 * 60 * 1000
			//				+ durationList[1] * 60 * 1000
			//				+ durationList[2] * 1000
			//		}
			//	}
			//	if (interval > 0) {
			//		this.vistaId = setInterval(
			//			'System.Sound.playSound("' + file + '");'
			//		,interval);
			//	}
			//} // if (this.flgRepeat)
			System.Sound.playSound(file);

		} else if (aaoh.kaze.Type.isAIR()) {
			var sound      = new air.Sound(new air.URLRequest(file));
			var soundTrans = new air.SoundTransform(this.volume / 100, 0); // 0 - 1
			var loops      = 0;

			if (this.flgRepeat) {
				//loops = Number.MAX_VALUE;
				loops = 10000;
			}
			this.airSoundChannel = sound.play(0, loops, soundTrans);

		} else if (aaoh.kaze.Type.isGoogle()) {
			this.googleAudioClip = framework.audio.open(file);
			//this.googleAudioClip.volume = (this.volume * 100) - 10000; // -10000 - 0
			var volume_ = -10000;
			if (this.volume > 0) {
				volume_ = (Math.atan(Math.PI * this.volume / (2 * 100)) -1) * 2000;
			}
			this.googleAudioClip.volume = volume_; // -10000 - 0

			//if (this.flgRepeat) {
			//	/** @private */
			//	this.googleAudioClip.onstatechange = function(googleAudioClip_, state) {
			//		if (state != gddSoundStateStopped) {
			//			return;
			//		}
			//		googleAudioClip_.currentPosition = 0;
			//		googleAudioClip_.play();
			//	}
			//}
			this.googleAudioClip.play();

		} else if (aaoh.kaze.Type.isYahoo()) {
			// リピート再生の設定なし
			system.volume = (16 * this.volume) / 100; // 0 - 16
			play(file, true);
		} // if (aaoh.kaze.Type.isSidebar())
	},

	/**
	 * 音を停止します。
	 */
	stop: function() {
		if (!this.isSound()) {
			throw new Error("can't stop");
		}

		if (aaoh.kaze.Type.isSidebar()) {
			if (this.vistaId != null) {
				clearInterval(this.vistaId);
				this.vistaId = null;
			}
			System.Sound.playSound('');

		} else if (aaoh.kaze.Type.isAIR()) {
			if (this.airSoundChannel != null) {
				this.airSoundChannel.stop();
				this.airSoundChannel = null;
			}

		} else if (aaoh.kaze.Type.isGoogle()) {
			if (this.googleAudioClip != null) {
				this.googleAudioClip.stop();
				this.googleAudioClip = null;
			}

		} else if (aaoh.kaze.Type.isYahoo()) {
			play(null, true);
		} // if (aaoh.kaze.Type.isSidebar())
	}
}
})();

(function () {
/**
 * Timer。
 *
 * @constructor
 * @param {Object} callObj_ 呼び出し元オブジェクト
 */
aaoh.kaze.Timer = function(callObj_) {
	this.callObj = callObj_;
	this.id      = null;
	this.timer   = null;
	if (this.callObj == null) {
		this.callObj = this;
	}

	if (aaoh.kaze.Type.isYahoo()) {
		this.timer = new Timer();
	}
}
aaoh.kaze.Timer.prototype = {
	/**
	 * タイマーを設定します。
	 *
	 * @param {Function} callable タイマー通知ファンクション
	 * @param {Number}   interval インターバル(msec)
	 */
	setInterval: function(callable, interval) {
		if (!callable) {
			throw new Error('callable is null, interval is ' + interval);

		} else if (isNaN(interval)) {
			throw new Error('interval is null');
		}

		if (this.id != null) {
			this.clearInterval();
		}

		if (aaoh.kaze.Type.isGoogle()) {
			var callObj_ = this.callObj;
			this.id = view.setInterval(function() {
				callable.call(callObj_);
			}, interval);

		} else if (aaoh.kaze.Type.isYahoo()) {
			var callObj_ = this.callObj;

			this.timer.interval     = interval / 1000; // sec
			/** @private */
			this.timer.onTimerFired = function(event) {
				callable.call(callObj_);
			};
			this.timer.ticking      = true;
			this.id                 = this.timer.id;

		} else {
			var callObj_ = this.callObj;
			this.id = setInterval(function() {
				callable.call(callObj_);
			}, interval);
		}
	},

	/**
	 * タイマーを設定します。
	 *
	 * @param {Function} callable タイマー通知ファンクション
	 * @param {Number}   timeout  インターバル(msec)
	 */
	setTimeout: function(callable, timeout) {
		if (!callable) {
			throw new Error('callable is null, timeout is ' + timeout);

		} else if (isNaN(timeout)) {
			throw new Error('timeout is null');
		}

		if (this.id != null) {
			this.clearTimeout();
		}

		if (aaoh.kaze.Type.isGoogle()) {
			var callObj_ = this.callObj;
			this.id = view.setTimeout(function() {
				callable.call(callObj_);
			}, timeout);

		} else if (aaoh.kaze.Type.isYahoo()) {
			var callObj_ = this.callObj;
			var timer_   = this.timer;

			this.timer.interval = timeout / 1000; // sec
			/** @private */
			this.timer.onTimerFired = function(event) {
				timer_.ticking = false;
				callable.call(callObj_);
			};
			this.timer.ticking = true;
			this.id            = this.timer.name;

		} else {
			var callObj_ = this.callObj;
			this.id = setTimeout(function() {
				callable.call(callObj_);
			}, timeout);
		} // if (aaoh.kaze.Type.isGoogle())
	},

	/**
	 * タイマーを停止します。
	 */
	clearInterval: function() {
		if (this.id == null) {
			return;
		}

		if (aaoh.kaze.Type.isGoogle()) {
			view.clearTimeout(this.id);
			this.id = null;

		} else if (aaoh.kaze.Type.isYahoo()) {
			this.timer.ticking = false;
			this.timer.reset();
			this.id = null;

		} else {
			clearTimeout(this.id);
			this.id = null;
		}
	},

	/**
	 * タイマーを停止します。
	 */
	clearTimeout: function() {
		if (this.id == null) {
			return;
		}

		if (aaoh.kaze.Type.isGoogle()) {
			view.clearTimeout(this.id);
			this.id = null;

		} else if (aaoh.kaze.Type.isYahoo()) {
			this.timer.ticking = false;
			this.timer.reset();
			this.id = null;

		} else {
			clearTimeout(this.id);
			this.id = null;
		}
	}
}
})();

(function () {
/**
 * Locale。
 *
 * @constructor
 */
aaoh.kaze.Locale = function() {
	this.DEFAULT_LANG  = 'ja';
	this.LANG_NAME     = 'localizedStrings';
	this.LANG_NAME_JS  = this.LANG_NAME + '.js';
	this.ROOT_LOCATION = null;
	if (aaoh.kaze.FLG_WINDOW == 0 && !!window.location) {
		this.ROOT_LOCATION = location.toString().substring(
			0, location.toString().lastIndexOf('/')
		) + '/';
	}

	this.pref = null;
	if (aaoh.kaze.Type.isIGoogle()) {
		//this.pref = new _IG_Prefs();
		this.pref = new gadgets.Prefs();
	}
}
aaoh.kaze.Locale.prototype = {
	/**
	 * ローカライズの設定。
	 *
	 * @param {String}   lang     言語名
	 * @param {Function} callable ローカライズ通知ファンクション
	 */
	setLocale: function(lang, callable) {
		if (lang == null) {
			throw new Error('lang is null');

		} else if (!callable) {
			throw new Error('callable is null, lang is ' + lang);
		}

		if (!(aaoh.kaze.Type.isOpera() || aaoh.kaze.Type.isOthers())) {
			return;
		}

		var jsObjList = document.getElementsByTagName('script');
		for (var cnt = 0; cnt < jsObjList.length; cnt++) {
			var jsSrc = jsObjList[cnt].src;
			if (jsSrc.lastIndexOf(this.LANG_NAME_JS) < 0) {
				continue;
			}

			var jsSrc = null;
			if (lang == this.DEFAULT_LANG) {
				jsSrc = this.LANG_NAME_JS;
			} else {
				jsSrc = lang + '/' + this.LANG_NAME_JS;
			}

			if (
				jsObjList[cnt].src == jsSrc
				|| jsObjList[cnt].src == this.ROOT_LOCATION + jsSrc
			) {
				break;
			}

			// 今のjsを削除
			var headObj = jsObjList[cnt].parentNode;
			headObj.removeChild(jsObjList[cnt]);

			// 新しいjsを追加
			var jsObj = document.createElement('script');
			jsObj.src    = jsSrc;
			jsObj.onload = callable;
			headObj.appendChild(jsObj);

			break;
		} // for (var cnt = 0; cnt < jsObjList.length; cnt++)
	},

	/**
	 * ローカライズされた文字列の取得。
	 *
	 * @param  {String} key キー
	 * @return {String}     ローカライズされた文字列
	 */
	getString: function(key) {
		return this.getLocalizedString(key, false);
	},

	/**
	 * ローカライズされた文字列の取得。
	 *
	 * @param  {String}  key    キー
	 * @param  {Boolean} isNull 値がなかった時にnullを返すときにはtrue
	 * @return {String}         ローカライズされた文字列
	 */
	getLocalizedString: function(key, isNull) {
		if (key == null) {
			throw new Error('key is null');
		}

		var msg = null;
		try {
			if (aaoh.kaze.Type.isAIR()) {
				msg = air.Localizer.localizer.getString(this.LANG_NAME, key);

			} else if (aaoh.kaze.Type.isIGoogle()) {
				msg = this.pref.getMsg(key);

			} else if (aaoh.kaze.Type.isYahoo()) {
				msg = widget.getLocalizedString(key);

			} else {
				msg = localizedStrings[key];
			}
		} catch(ex) {
		}

		if (msg == null && !isNull) {
			msg = key;
		}

		return msg;
	}
}
})();
