#!/usr/bin/env python
r"""
Pandoc filter to make sure that the resulting document is
as perfect as possible.

Pandoc filters work by reading from stdin and writing to stdout. Therefore
printing variables to debug is not possible. To debug, you can use a log file:

```python
with open("log.txt", "a") as f:
    f.write(f"Whathever you want to log")

# Instead of
print("Whathever you want to log")
```

The actions that the current filter performs are:

    - Replace all "!" in fdf parameter names with "." (fdf parameter names
    are located in divs with the class "fdfparamname").

    - Assign an ID to divs with the class "labelcontainer". This divs are
    generated by the \label latex command (because we redefine it like this,
    pandoc doesn't handle \label automatically). By assigning an ID to those
    divs, they can be targeted by links.

    - Convert internal links to use the :ref: role in reStructuredText. In this
    way, references work accross different rst files. That is, for example, we 
    don't need to parse the whole .tex into a single .rst file.

    - Remove the labelcontainer environment from display math environments. The problem here
    is that since we have redefined the \label command, it will create a labelcontainer environment
    in its place. Latex equations that had labels will also incorporate this change and will make
    mathjax fail with the error "Undefined environment labelcontainer". This filter removes
    the labelcontainer environment. Ideally, the label should be placed just outside the 
    math environment so that the equation can still be referenced, but I haven't managed to make
    that work. I tried replacing `Math()` with `Div(id=label, content=[Math()])` but this does
    not work because pandoc does not allow replacing an inline element (Math) with a block element (Div).
"""
from pandocfilters import toJSONFilter, Str, Div, Math

# Special character placeholders that we use to avoid pandoc from escaping them.
# We will replace them in post-processing.
special_chars = {
    "backslash": "PLACEHOLDERSIESTA_PANDOC_BACKSLASHPLACEHOLDER",
    "backtick": "PLACEHOLDERSIESTA_PANDOC_BACKTICKPLACEHOLDER",
}

def siesta_manual_filter(key, value, format, meta):
    
    if key == 'Div':
        # This is a pandoc div element.

        # The information contained in value (as retreived in the following line) is:
        #     - div_id (str): The ID of the div. If the div does not have an ID, div_id is "".
        #     - classes (list of str): A list of classes assigned to the div.
        #     - keyvals (list of tuples): A list of key-value pairs that are attributes of the div.
        #     - content (JSON): The content of the div, in pandoc JSON format.
        [[div_id, classes, keyvals], content] = value

        if "fdfparamname" in classes:
            # This is a div containing the name of a fdf parameter. 

            # Get the name of the parameter. The structure of content is:
            # [{'t': 'Para', 'c': [{'t': 'Str', 'c': 'ParameterName'}]}]
            name = content[0]['c'][0]['c']

            # If the name does not contain a "!" character, we don't need to replace it.
            # If we don't do this check, we end up recursing infinitely, because the returned
            # modified Div will be passed to the filter again.
            if "!" not in name:
                return
            
            # Replace the "!" with "." to make it a valid parameter name.
            content[0]['c'][0]['c'] = name.replace("!", ".")

            # Return a new div with the modified content.
            return Div([div_id, classes, keyvals], content)
        elif "fdfparamdefault" in classes:
            # if there is no default, substitute the content with "<none>"
            if len(content) == 0:
                content = [{'t': 'Para', 'c': [{'t': 'Str', 'c': '<none>'}]}]
                return Div([div_id, classes, keyvals], content)
        elif "fdfparamtype" in classes:
            # if there is no type, substitute the content with "Unknown"
            if len(content) == 0:
                content = [{'t': 'Para', 'c': [{'t': 'Str', 'c': 'Unknown'}]}]
                return Div([div_id, classes, keyvals], content)
        elif "fdfentrycontainerbody" in classes:
            # If there is no description, substitute the content with "-"
            if len(content) == 0:
                content = [{'t': 'Para', 'c': [{'t': 'Str', 'c': '-'}]}]
                return Div([div_id, classes, keyvals], content)
        elif "labelcontainer" in classes:
            # This is a div generated by the \label latex command.
            # We need to assign an ID to this div so that it can be targeted by links.

            # The div already contains an ID. We don't need to do anything.
            # By doing this check, we avoid recursing infinitely (see fdfparamname comments).
            if div_id:
                return
            
            # Get the name of the label. The structure of content is:
            # [{'t': 'Para', 'c': [{'t': 'Str', 'c': 'LabelName'}]}]
            # But the paragraph content might contain spaces. We then need to join
            # the different words incorporating real spaces.
            paragraph_content = content[0]['c']
            name = " ".join([c['c'] for c in paragraph_content if c['t'] != 'Space'])

            # Some further purification depending on the type of label.
            if name.startswith("fdfparam:"):
                name = name.lower().replace("!",".").replace(" ", "-")

            # Return a new div with the ID attribute set to the label name.
            # We don't remove the div content because sphinx does not render empty divs!!
            # We will hide the div with css.
            return Div([name, classes, keyvals], content)
        
    elif key == "Link":
        # This is a pandoc link element. We also 

        # We only modify the link if converting to rst.
        if format != "rst":
            return
        
        # The information contained in value (as retreived in the following line) is:
        #     - *_others: I don't know exactly the use of these.
        #     - keyvals (list of tuples): A list of key-value pairs that are attributes of link.
        #     - content (JSON): The content of the div, in pandoc JSON format.
        #     - *_: Contains the reference, but I don't care about this.
        [[*_others, keyvals], content, _] = value

        # Convert attributes to a dictionary for easier access.
        attrs = {k: v for k, v in keyvals}

        # For now we only modify links with reference-type "ref".
        # The "reference" attribute contains the target of the link.
        if attrs.get("reference-type") != "ref" or "reference" not in attrs:
            return
        
        # Get link target.
        ref = attrs["reference"]
        # Get text of the link. The link content has the following structure:
        # [{'t': 'Str', 'c': 'Text of the link'}].
        # When pandoc parses latex files, it converts \ref{sec:somesection} to a
        # link with the text "[sec:somesection]". Here we remove the square brackets.
        text = content[0]['c'].removeprefix("[").removesuffix("]")

        # Some further purification depending on the type of target.
        if ref.startswith("fdfparam:"):
            ref = ref.lower().replace("!",".")
            text = text.replace("!",".").removeprefix("fdfparam:")
            # References to options have an colon to separate the parameter name 
            # and the option name. Display a space instead.
            text = text.replace(":"," ")
        elif ref.startswith("sec:"):
            text = text.removeprefix("sec:")

        # Generate the link using the :ref: rst role.
        # It is very important that we escape any < or > characters in the text and ref, 
        # otherwise it will create an invalid link.
        # However, we can't use a backslash to escape them because pandoc will escape the
        # backslash itself (i.e. it will convert \< to \\<). We use a special character 
        # placeholder that we will replace in post-processing. Also, we must avoid that
        # backticks are escaped by pandoc.
        backslash = special_chars["backslash"]
        backtick = special_chars["backtick"]
        text = text.replace("<", f"{backslash}<").replace(">", f"{backslash}>")
        ref = ref.replace("<", f"{backslash}<").replace(">", f"{backslash}>")
        return Str(f":ref:{backtick}{text}<{ref}>{backtick}")
        
    elif key == "Math":
        # This is a math element.

        # The information contained in value (as retreived in the following line) is:
        #     - display (dict): {"t": "DisplayMath" | "InlineMath"}.
        #     - tex (str): The latex code of the math.
        [display, tex] = value

        # Only modify display math environments. These are the ones generated by the latex
        # \begin{equation} ... \end{equation}.
        if display['t'] != "DisplayMath":
            return
        
        # We loop until we remove all labelcontainer environments.
        while True:
            # Try to find the labelcontainer environment.
            start = tex.find("begin{labelcontainer}") 
            end = tex.find("end{labelcontainer}")

            # If not found, this is not a labeled equation, exit the loop
            if start == -1 or end == -1:
                break
            
            # Otherwise, extract the label.
            label = tex[start + len("begin{labelcontainer}") :end - 1].strip()

            # Get the tex without the labelcontainer environment.
            tex = tex[:start - 1] + tex[end + len("end{labelcontainer}"):]

        # Return the clean math environment. For now we don't use the label.
        return Math(display, tex)

            

if __name__ == "__main__":
    toJSONFilter(siesta_manual_filter)
