# Переменные окружения { #environment-variables }

/// tip | Совет

Если вы уже знаете, что такое «переменные окружения» и как их использовать, можете пропустить это.

///

Переменная окружения (также известная как «**env var**») - это переменная, которая живет **вне** кода Python, в **операционной системе**, и может быть прочитана вашим кодом Python (или другими программами).

Переменные окружения могут быть полезны для работы с **настройками** приложений, как часть **установки** Python и т.д.

## Создание и использование переменных окружения { #create-and-use-env-vars }

Можно **создавать** и использовать переменные окружения в **оболочке (терминале)**, не прибегая к помощи Python:

//// tab | Linux, macOS, Windows Bash

<div class="termy">

```console
// Вы можете создать переменную окружения MY_NAME с помощью
$ export MY_NAME="Wade Wilson"

// Затем её можно использовать в других программах, например
$ echo "Hello $MY_NAME"

Hello Wade Wilson
```

</div>

////

//// tab | Windows PowerShell

<div class="termy">

```console
// Создайте переменную окружения MY_NAME
$ $Env:MY_NAME = "Wade Wilson"

// Используйте её с другими программами, например
$ echo "Hello $Env:MY_NAME"

Hello Wade Wilson
```

</div>

////

## Чтение переменных окружения в python { #read-env-vars-in-python }

Так же существует возможность создания переменных окружения **вне** Python, в терминале (или любым другим способом), а затем **чтения их в Python**.

Например, у вас есть файл `main.py`:

```Python hl_lines="3"
import os

name = os.getenv("MY_NAME", "World")
print(f"Hello {name} from Python")
```

/// tip | Совет

Второй аргумент <a href="https://docs.python.org/3.8/library/os.html#os.getenv" class="external-link" target="_blank">`os.getenv()`</a> - это возвращаемое по умолчанию значение.

Если значение не указано, то по умолчанию оно равно `None`. В данном случае мы указываем `«World»` в качестве значения по умолчанию.

///

Затем можно запустить эту программу на Python:

//// tab | Linux, macOS, Windows Bash

<div class="termy">

```console
// Здесь мы еще не устанавливаем переменную окружения
$ python main.py

// Поскольку мы не задали переменную окружения, мы получим значение по умолчанию

Hello World from Python

// Но если мы сначала создадим переменную окружения
$ export MY_NAME="Wade Wilson"

// А затем снова запустим программу
$ python main.py

// Теперь она прочитает переменную окружения

Hello Wade Wilson from Python
```

</div>

////

//// tab | Windows PowerShell

<div class="termy">

```console
// Здесь мы еще не устанавливаем переменную окружения
$ python main.py

// Поскольку мы не задали переменную окружения, мы получим значение по умолчанию

Hello World from Python

// Но если мы сначала создадим переменную окружения
$ $Env:MY_NAME = "Wade Wilson"

// А затем снова запустим программу
$ python main.py

// Теперь она может прочитать переменную окружения

Hello Wade Wilson from Python
```

</div>

////

Поскольку переменные окружения могут быть установлены вне кода, но могут быть прочитаны кодом, и их не нужно хранить (фиксировать в `git`) вместе с остальными файлами, их принято использовать для конфигураций или **настроек**.

Вы также можете создать переменную окружения только для **конкретного вызова программы**, которая будет доступна только для этой программы и только на время ее выполнения.

Для этого создайте её непосредственно перед самой программой, в той же строке:

<div class="termy">

```console
// Создайте переменную окружения MY_NAME в строке для этого вызова программы
$ MY_NAME="Wade Wilson" python main.py

// Теперь она может прочитать переменную окружения

Hello Wade Wilson from Python

// После этого переменная окружения больше не существует
$ python main.py

Hello World from Python
```

</div>

/// tip | Совет

Подробнее об этом можно прочитать на сайте <a href="https://12factor.net/config" class="external-link" target="_blank">The Twelve-Factor App: Config</a>.

///

## Типизация и Валидация { #types-and-validation }

Эти переменные окружения могут работать только с **текстовыми строками**, поскольку они являются внешними по отношению к Python и должны быть совместимы с другими программами и остальной системой (и даже с различными операционными системами, такими как Linux, Windows, macOS).

Это означает, что **любое значение**, считанное в Python из переменной окружения, **будет `str`**, и любое преобразование к другому типу или любая валидация должны быть выполнены в коде.

Подробнее об использовании переменных окружения для работы с **настройками приложения** вы узнаете в [Расширенное руководство пользователя - Настройки и переменные среды](./advanced/settings.md){.internal-link target=_blank}.

## Переменная окружения `PATH` { #path-environment-variable }

Существует **специальная** переменная окружения **`PATH`**, которая используется операционными системами (Linux, macOS, Windows) для поиска программ для запуска.

Значение переменной `PATH` - это длинная строка, состоящая из каталогов, разделенных двоеточием `:` в Linux и macOS, и точкой с запятой `;` в Windows.

Например, переменная окружения `PATH` может выглядеть следующим образом:

//// tab | Linux, macOS

```plaintext
/usr/local/bin:/usr/bin:/bin:/usr/sbin:/sbin
```

Это означает, что система должна искать программы в каталогах:

* `/usr/local/bin`
* `/usr/bin`
* `/bin`
* `/usr/sbin`
* `/sbin`

////

//// tab | Windows

```plaintext
C:\Program Files\Python312\Scripts;C:\Program Files\Python312;C:\Windows\System32
```

Это означает, что система должна искать программы в каталогах:

* `C:\Program Files\Python312\Scripts`
* `C:\Program Files\Python312`
* `C:\Windows\System32`

////

Когда вы вводите **команду** в терминале, операционная система **ищет** программу в **каждой из тех директорий**, которые перечислены в переменной окружения `PATH`.

Например, когда вы вводите `python` в терминале, операционная система ищет программу под названием `python` в **первой директории** в этом списке.

Если она ее находит, то **использует ее**. В противном случае она продолжает искать в **других каталогах**.

### Установка Python и обновление `PATH` { #installing-python-and-updating-the-path }

При установке Python вас могут спросить, нужно ли обновить переменную окружения `PATH`.

//// tab | Linux, macOS

Допустим, вы устанавливаете Python, и он оказывается в каталоге `/opt/custompython/bin`.

Если вы скажете «да», чтобы обновить переменную окружения `PATH`, то программа установки добавит `/opt/custompython/bin` в переменную окружения `PATH`.

Это может выглядеть следующим образом:

```plaintext
/usr/local/bin:/usr/bin:/bin:/usr/sbin:/sbin:/opt/custompython/bin
```

Таким образом, когда вы набираете `python` в терминале, система найдет программу Python в `/opt/custompython/bin` (последний каталог) и использует ее.

////

//// tab | Windows

Допустим, вы устанавливаете Python, и он оказывается в каталоге `C:\opt\custompython\bin`.

Если вы согласитесь обновить переменную окружения `PATH`, то программа установки добавит `C:\opt\custompython\bin` в переменную окружения `PATH`.

```plaintext
C:\Program Files\Python312\Scripts;C:\Program Files\Python312;C:\Windows\System32;C:\opt\custompython\bin
```

Таким образом, когда вы набираете `python` в терминале, система найдет программу Python в `C:\opt\custompython\bin` (последний каталог) и использует ее.

////

Итак, если вы напечатаете:

<div class="termy">

```console
$ python
```

</div>

//// tab | Linux, macOS

Система **найдет** программу `python` в `/opt/custompython/bin` и запустит ее.

Это примерно эквивалентно набору текста:

<div class="termy">

```console
$ /opt/custompython/bin/python
```

</div>

////

//// tab | Windows

Система **найдет** программу `python` в каталоге `C:\opt\custompython\bin\python` и запустит ее.

Это примерно эквивалентно набору текста:

<div class="termy">

```console
$ C:\opt\custompython\bin\python
```

</div>

////

Эта информация будет полезна при изучении [Виртуальных окружений](virtual-environments.md){.internal-link target=_blank}.

## Вывод { #conclusion }

Благодаря этому вы должны иметь базовое представление о том, что такое **переменные окружения** и как использовать их в Python.

Подробнее о них вы также можете прочитать в <a href="https://en.wikipedia.org/wiki/Environment_variable" class="external-link" target="_blank">статье о переменных окружения на википедии</a>.

Во многих случаях не всегда очевидно, как переменные окружения могут быть полезны и применимы. Но они постоянно появляются в различных сценариях разработки, поэтому знать о них полезно.

Например, эта информация понадобится вам в следующем разделе, посвященном [Виртуальным окружениям](virtual-environments.md).
