/*
Bullet Continuous Collision Detection and Physics Library
Copyright (c) 2003-2009 Erwin Coumans  http://bulletphysics.org

This software is provided 'as-is', without any express or implied warranty.
In no event will the authors be held liable for any damages arising from the use of this software.
Permission is granted to anyone to use this software for any purpose, 
including commercial applications, and to alter it and redistribute it freely, 
subject to the following restrictions:

1. The origin of this software must not be misrepresented; you must not claim that you wrote the original software. If you use this software in a product, an acknowledgment in the product documentation would be appreciated but is not required.
2. Altered source versions must be plainly marked as such, and must not be misrepresented as being the original software.
3. This notice may not be removed or altered from any source distribution.
*/

#include "cbtStridingMeshInterface.h"
#include "LinearMath/cbtSerializer.h"

cbtStridingMeshInterface::~cbtStridingMeshInterface()
{
}

void cbtStridingMeshInterface::InternalProcessAllTriangles(cbtInternalTriangleIndexCallback* callback, const cbtVector3& aabbMin, const cbtVector3& aabbMax) const
{
	(void)aabbMin;
	(void)aabbMax;
	int numtotalphysicsverts = 0;
	int part, graphicssubparts = getNumSubParts();
	const unsigned char* vertexbase;
	const unsigned char* indexbase;
	int indexstride;
	PHY_ScalarType type;
	PHY_ScalarType gfxindextype;
	int stride, numverts, numtriangles;
	int gfxindex;
	cbtVector3 triangle[3];

	cbtVector3 meshScaling = getScaling();

	///if the number of parts is big, the performance might drop due to the innerloop switch on indextype
	for (part = 0; part < graphicssubparts; part++)
	{
		getLockedReadOnlyVertexIndexBase(&vertexbase, numverts, type, stride, &indexbase, indexstride, numtriangles, gfxindextype, part);
		numtotalphysicsverts += numtriangles * 3;  //upper bound

		///unlike that developers want to pass in double-precision meshes in single-precision Bullet build
		///so disable this feature by default
		///see patch http://code.google.com/p/bullet/issues/detail?id=213

		switch (type)
		{
			case PHY_FLOAT:
			{
				float* graphicsbase;

				switch (gfxindextype)
				{
					case PHY_INTEGER:
					{
						for (gfxindex = 0; gfxindex < numtriangles; gfxindex++)
						{
							unsigned int* tri_indices = (unsigned int*)(indexbase + gfxindex * indexstride);
							graphicsbase = (float*)(vertexbase + tri_indices[0] * stride);
							triangle[0].setValue(graphicsbase[0] * meshScaling.getX(), graphicsbase[1] * meshScaling.getY(), graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (float*)(vertexbase + tri_indices[1] * stride);
							triangle[1].setValue(graphicsbase[0] * meshScaling.getX(), graphicsbase[1] * meshScaling.getY(), graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (float*)(vertexbase + tri_indices[2] * stride);
							triangle[2].setValue(graphicsbase[0] * meshScaling.getX(), graphicsbase[1] * meshScaling.getY(), graphicsbase[2] * meshScaling.getZ());
							callback->internalProcessTriangleIndex(triangle, part, gfxindex);
						}
						break;
					}
					case PHY_SHORT:
					{
						for (gfxindex = 0; gfxindex < numtriangles; gfxindex++)
						{
							unsigned short int* tri_indices = (unsigned short int*)(indexbase + gfxindex * indexstride);
							graphicsbase = (float*)(vertexbase + tri_indices[0] * stride);
							triangle[0].setValue(graphicsbase[0] * meshScaling.getX(), graphicsbase[1] * meshScaling.getY(), graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (float*)(vertexbase + tri_indices[1] * stride);
							triangle[1].setValue(graphicsbase[0] * meshScaling.getX(), graphicsbase[1] * meshScaling.getY(), graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (float*)(vertexbase + tri_indices[2] * stride);
							triangle[2].setValue(graphicsbase[0] * meshScaling.getX(), graphicsbase[1] * meshScaling.getY(), graphicsbase[2] * meshScaling.getZ());
							callback->internalProcessTriangleIndex(triangle, part, gfxindex);
						}
						break;
					}
					case PHY_UCHAR:
					{
						for (gfxindex = 0; gfxindex < numtriangles; gfxindex++)
						{
							unsigned char* tri_indices = (unsigned char*)(indexbase + gfxindex * indexstride);
							graphicsbase = (float*)(vertexbase + tri_indices[0] * stride);
							triangle[0].setValue(graphicsbase[0] * meshScaling.getX(), graphicsbase[1] * meshScaling.getY(), graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (float*)(vertexbase + tri_indices[1] * stride);
							triangle[1].setValue(graphicsbase[0] * meshScaling.getX(), graphicsbase[1] * meshScaling.getY(), graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (float*)(vertexbase + tri_indices[2] * stride);
							triangle[2].setValue(graphicsbase[0] * meshScaling.getX(), graphicsbase[1] * meshScaling.getY(), graphicsbase[2] * meshScaling.getZ());
							callback->internalProcessTriangleIndex(triangle, part, gfxindex);
						}
						break;
					}
					default:
						cbtAssert((gfxindextype == PHY_INTEGER) || (gfxindextype == PHY_SHORT));
				}
				break;
			}

			case PHY_DOUBLE:
			{
				double* graphicsbase;

				switch (gfxindextype)
				{
					case PHY_INTEGER:
					{
						for (gfxindex = 0; gfxindex < numtriangles; gfxindex++)
						{
							unsigned int* tri_indices = (unsigned int*)(indexbase + gfxindex * indexstride);
							graphicsbase = (double*)(vertexbase + tri_indices[0] * stride);
							triangle[0].setValue((cbtScalar)graphicsbase[0] * meshScaling.getX(), (cbtScalar)graphicsbase[1] * meshScaling.getY(), (cbtScalar)graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (double*)(vertexbase + tri_indices[1] * stride);
							triangle[1].setValue((cbtScalar)graphicsbase[0] * meshScaling.getX(), (cbtScalar)graphicsbase[1] * meshScaling.getY(), (cbtScalar)graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (double*)(vertexbase + tri_indices[2] * stride);
							triangle[2].setValue((cbtScalar)graphicsbase[0] * meshScaling.getX(), (cbtScalar)graphicsbase[1] * meshScaling.getY(), (cbtScalar)graphicsbase[2] * meshScaling.getZ());
							callback->internalProcessTriangleIndex(triangle, part, gfxindex);
						}
						break;
					}
					case PHY_SHORT:
					{
						for (gfxindex = 0; gfxindex < numtriangles; gfxindex++)
						{
							unsigned short int* tri_indices = (unsigned short int*)(indexbase + gfxindex * indexstride);
							graphicsbase = (double*)(vertexbase + tri_indices[0] * stride);
							triangle[0].setValue((cbtScalar)graphicsbase[0] * meshScaling.getX(), (cbtScalar)graphicsbase[1] * meshScaling.getY(), (cbtScalar)graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (double*)(vertexbase + tri_indices[1] * stride);
							triangle[1].setValue((cbtScalar)graphicsbase[0] * meshScaling.getX(), (cbtScalar)graphicsbase[1] * meshScaling.getY(), (cbtScalar)graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (double*)(vertexbase + tri_indices[2] * stride);
							triangle[2].setValue((cbtScalar)graphicsbase[0] * meshScaling.getX(), (cbtScalar)graphicsbase[1] * meshScaling.getY(), (cbtScalar)graphicsbase[2] * meshScaling.getZ());
							callback->internalProcessTriangleIndex(triangle, part, gfxindex);
						}
						break;
					}
					case PHY_UCHAR:
					{
						for (gfxindex = 0; gfxindex < numtriangles; gfxindex++)
						{
							unsigned char* tri_indices = (unsigned char*)(indexbase + gfxindex * indexstride);
							graphicsbase = (double*)(vertexbase + tri_indices[0] * stride);
							triangle[0].setValue((cbtScalar)graphicsbase[0] * meshScaling.getX(), (cbtScalar)graphicsbase[1] * meshScaling.getY(), (cbtScalar)graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (double*)(vertexbase + tri_indices[1] * stride);
							triangle[1].setValue((cbtScalar)graphicsbase[0] * meshScaling.getX(), (cbtScalar)graphicsbase[1] * meshScaling.getY(), (cbtScalar)graphicsbase[2] * meshScaling.getZ());
							graphicsbase = (double*)(vertexbase + tri_indices[2] * stride);
							triangle[2].setValue((cbtScalar)graphicsbase[0] * meshScaling.getX(), (cbtScalar)graphicsbase[1] * meshScaling.getY(), (cbtScalar)graphicsbase[2] * meshScaling.getZ());
							callback->internalProcessTriangleIndex(triangle, part, gfxindex);
						}
						break;
					}
					default:
						cbtAssert((gfxindextype == PHY_INTEGER) || (gfxindextype == PHY_SHORT));
				}
				break;
			}
			default:
				cbtAssert((type == PHY_FLOAT) || (type == PHY_DOUBLE));
		}

		unLockReadOnlyVertexBase(part);
	}
}

void cbtStridingMeshInterface::calculateAabbBruteForce(cbtVector3& aabbMin, cbtVector3& aabbMax)
{
	struct AabbCalculationCallback : public cbtInternalTriangleIndexCallback
	{
		cbtVector3 m_aabbMin;
		cbtVector3 m_aabbMax;

		AabbCalculationCallback()
		{
			m_aabbMin.setValue(cbtScalar(BT_LARGE_FLOAT), cbtScalar(BT_LARGE_FLOAT), cbtScalar(BT_LARGE_FLOAT));
			m_aabbMax.setValue(cbtScalar(-BT_LARGE_FLOAT), cbtScalar(-BT_LARGE_FLOAT), cbtScalar(-BT_LARGE_FLOAT));
		}

		virtual void internalProcessTriangleIndex(cbtVector3* triangle, int partId, int triangleIndex)
		{
			(void)partId;
			(void)triangleIndex;

			m_aabbMin.setMin(triangle[0]);
			m_aabbMax.setMax(triangle[0]);
			m_aabbMin.setMin(triangle[1]);
			m_aabbMax.setMax(triangle[1]);
			m_aabbMin.setMin(triangle[2]);
			m_aabbMax.setMax(triangle[2]);
		}
	};

	//first calculate the total aabb for all triangles
	AabbCalculationCallback aabbCallback;
	aabbMin.setValue(cbtScalar(-BT_LARGE_FLOAT), cbtScalar(-BT_LARGE_FLOAT), cbtScalar(-BT_LARGE_FLOAT));
	aabbMax.setValue(cbtScalar(BT_LARGE_FLOAT), cbtScalar(BT_LARGE_FLOAT), cbtScalar(BT_LARGE_FLOAT));
	InternalProcessAllTriangles(&aabbCallback, aabbMin, aabbMax);

	aabbMin = aabbCallback.m_aabbMin;
	aabbMax = aabbCallback.m_aabbMax;
}

///fills the dataBuffer and returns the struct name (and 0 on failure)
const char* cbtStridingMeshInterface::serialize(void* dataBuffer, cbtSerializer* serializer) const
{
	cbtStridingMeshInterfaceData* trimeshData = (cbtStridingMeshInterfaceData*)dataBuffer;

	trimeshData->m_numMeshParts = getNumSubParts();

	//void* uniquePtr = 0;

	trimeshData->m_meshPartsPtr = 0;

	if (trimeshData->m_numMeshParts)
	{
		cbtChunk* chunk = serializer->allocate(sizeof(cbtMeshPartData), trimeshData->m_numMeshParts);
		cbtMeshPartData* memPtr = (cbtMeshPartData*)chunk->m_oldPtr;
		trimeshData->m_meshPartsPtr = (cbtMeshPartData*)serializer->getUniquePointer(memPtr);

		//	int numtotalphysicsverts = 0;
		int part, graphicssubparts = getNumSubParts();
		const unsigned char* vertexbase;
		const unsigned char* indexbase;
		int indexstride;
		PHY_ScalarType type;
		PHY_ScalarType gfxindextype;
		int stride, numverts, numtriangles;
		int gfxindex;
		//	cbtVector3 triangle[3];

		//	cbtVector3 meshScaling = getScaling();

		///if the number of parts is big, the performance might drop due to the innerloop switch on indextype
		for (part = 0; part < graphicssubparts; part++, memPtr++)
		{
			getLockedReadOnlyVertexIndexBase(&vertexbase, numverts, type, stride, &indexbase, indexstride, numtriangles, gfxindextype, part);
			memPtr->m_numTriangles = numtriangles;  //indices = 3*numtriangles
			memPtr->m_numVertices = numverts;
			memPtr->m_indices16 = 0;
			memPtr->m_indices32 = 0;
			memPtr->m_3indices16 = 0;
			memPtr->m_3indices8 = 0;
			memPtr->m_vertices3f = 0;
			memPtr->m_vertices3d = 0;

			switch (gfxindextype)
			{
				case PHY_INTEGER:
				{
					int numindices = numtriangles * 3;

					if (numindices)
					{
						cbtChunk* chunk = serializer->allocate(sizeof(cbtIntIndexData), numindices);
						cbtIntIndexData* tmpIndices = (cbtIntIndexData*)chunk->m_oldPtr;
						memPtr->m_indices32 = (cbtIntIndexData*)serializer->getUniquePointer(tmpIndices);
						for (gfxindex = 0; gfxindex < numtriangles; gfxindex++)
						{
							unsigned int* tri_indices = (unsigned int*)(indexbase + gfxindex * indexstride);
							tmpIndices[gfxindex * 3].m_value = tri_indices[0];
							tmpIndices[gfxindex * 3 + 1].m_value = tri_indices[1];
							tmpIndices[gfxindex * 3 + 2].m_value = tri_indices[2];
						}
						serializer->finalizeChunk(chunk, "cbtIntIndexData", BT_ARRAY_CODE, (void*)chunk->m_oldPtr);
					}
					break;
				}
				case PHY_SHORT:
				{
					if (numtriangles)
					{
						cbtChunk* chunk = serializer->allocate(sizeof(cbtShortIntIndexTripletData), numtriangles);
						cbtShortIntIndexTripletData* tmpIndices = (cbtShortIntIndexTripletData*)chunk->m_oldPtr;
						memPtr->m_3indices16 = (cbtShortIntIndexTripletData*)serializer->getUniquePointer(tmpIndices);
						for (gfxindex = 0; gfxindex < numtriangles; gfxindex++)
						{
							unsigned short int* tri_indices = (unsigned short int*)(indexbase + gfxindex * indexstride);
							tmpIndices[gfxindex].m_values[0] = tri_indices[0];
							tmpIndices[gfxindex].m_values[1] = tri_indices[1];
							tmpIndices[gfxindex].m_values[2] = tri_indices[2];
							// Fill padding with zeros to appease msan.
							tmpIndices[gfxindex].m_pad[0] = 0;
							tmpIndices[gfxindex].m_pad[1] = 0;
						}
						serializer->finalizeChunk(chunk, "cbtShortIntIndexTripletData", BT_ARRAY_CODE, (void*)chunk->m_oldPtr);
					}
					break;
				}
				case PHY_UCHAR:
				{
					if (numtriangles)
					{
						cbtChunk* chunk = serializer->allocate(sizeof(cbtCharIndexTripletData), numtriangles);
						cbtCharIndexTripletData* tmpIndices = (cbtCharIndexTripletData*)chunk->m_oldPtr;
						memPtr->m_3indices8 = (cbtCharIndexTripletData*)serializer->getUniquePointer(tmpIndices);
						for (gfxindex = 0; gfxindex < numtriangles; gfxindex++)
						{
							unsigned char* tri_indices = (unsigned char*)(indexbase + gfxindex * indexstride);
							tmpIndices[gfxindex].m_values[0] = tri_indices[0];
							tmpIndices[gfxindex].m_values[1] = tri_indices[1];
							tmpIndices[gfxindex].m_values[2] = tri_indices[2];
							// Fill padding with zeros to appease msan.
							tmpIndices[gfxindex].m_pad = 0;
						}
						serializer->finalizeChunk(chunk, "cbtCharIndexTripletData", BT_ARRAY_CODE, (void*)chunk->m_oldPtr);
					}
					break;
				}
				default:
				{
					cbtAssert(0);
					//unknown index type
				}
			}

			switch (type)
			{
				case PHY_FLOAT:
				{
					float* graphicsbase;

					if (numverts)
					{
						cbtChunk* chunk = serializer->allocate(sizeof(cbtVector3FloatData), numverts);
						cbtVector3FloatData* tmpVertices = (cbtVector3FloatData*)chunk->m_oldPtr;
						memPtr->m_vertices3f = (cbtVector3FloatData*)serializer->getUniquePointer(tmpVertices);
						for (int i = 0; i < numverts; i++)
						{
							graphicsbase = (float*)(vertexbase + i * stride);
							tmpVertices[i].m_floats[0] = graphicsbase[0];
							tmpVertices[i].m_floats[1] = graphicsbase[1];
							tmpVertices[i].m_floats[2] = graphicsbase[2];
						}
						serializer->finalizeChunk(chunk, "cbtVector3FloatData", BT_ARRAY_CODE, (void*)chunk->m_oldPtr);
					}
					break;
				}

				case PHY_DOUBLE:
				{
					if (numverts)
					{
						cbtChunk* chunk = serializer->allocate(sizeof(cbtVector3DoubleData), numverts);
						cbtVector3DoubleData* tmpVertices = (cbtVector3DoubleData*)chunk->m_oldPtr;
						memPtr->m_vertices3d = (cbtVector3DoubleData*)serializer->getUniquePointer(tmpVertices);
						for (int i = 0; i < numverts; i++)
						{
							double* graphicsbase = (double*)(vertexbase + i * stride);  //for now convert to float, might leave it at double
							tmpVertices[i].m_floats[0] = graphicsbase[0];
							tmpVertices[i].m_floats[1] = graphicsbase[1];
							tmpVertices[i].m_floats[2] = graphicsbase[2];
						}
						serializer->finalizeChunk(chunk, "cbtVector3DoubleData", BT_ARRAY_CODE, (void*)chunk->m_oldPtr);
					}
					break;
				}

				default:
					cbtAssert((type == PHY_FLOAT) || (type == PHY_DOUBLE));
			}

			unLockReadOnlyVertexBase(part);
		}

		serializer->finalizeChunk(chunk, "cbtMeshPartData", BT_ARRAY_CODE, chunk->m_oldPtr);
	}

	// Fill padding with zeros to appease msan.
	memset(trimeshData->m_padding, 0, sizeof(trimeshData->m_padding));

	m_scaling.serializeFloat(trimeshData->m_scaling);
	return "cbtStridingMeshInterfaceData";
}
