///////////////////////////////////////////////////////////////////////////////
//                                                         
// CegoPredDesc.cc
// ---------------
// Cego predicate structure class definition
//                                                 
// Design and Implementation by Bjoern Lemke               
//     
// (C)opyright 2000-2019 Bjoern Lemke
//
// IMPLEMENTATION MODULE
//
// Class: CegoPredDesc
//
// Description: The CegoPrdDesc class is a container class for the query predicate representation
//
// Status: CLEAN
//
///////////////////////////////////////////////////////////////////////////////

// CEGO INCLUDES
#include "CegoPredDesc.h"
#include "CegoCondDesc.h"
#include "CegoSelect.h"
#include "CegoXMLdef.h"
#include "CegoDatabaseFormater.h"

// POSIX INCLUDES
#include <string.h>
#include <stdlib.h>

CegoPredDesc::CegoPredDesc(char* buf, CegoDistManager *pGTM, int tabSetId)
{
    _pExpr1 = 0;
    _pExpr2 = 0;
    _pExpr3 = 0;
    _pC = 0;
    _pNotPred = 0;
    _isChecked = false;
    _pSelect = 0;
    _pMatcher = 0;
    decode(buf, pGTM, tabSetId);
}

CegoPredDesc::CegoPredDesc(Element* pPredElement, CegoDistManager *pGTM)
{
    _pExpr1 = 0;
    _pExpr2 = 0;
    _pExpr3 = 0;
    _pC = 0;
    _pNotPred = 0;
    _isChecked = false;
    _pSelect = 0;
    _pMatcher = 0;
    fromElement(pPredElement, pGTM);
}

CegoPredDesc::CegoPredDesc(CegoExpr *pExpr1, CegoExpr* pExpr2, const CegoComparison& comp)
{
    _pExpr1 = pExpr1;
    _pExpr2 = pExpr2;
    _pExpr3 = 0;
    _comp = comp;
    _pC = 0;
    _pNotPred = 0;
    _mode = CegoPredDesc::EXPRCOMP;
    _isChecked = false;
    _pSelect = 0;
    _pMatcher = 0;
}

CegoPredDesc::CegoPredDesc(CegoExpr *pExpr1, CegoExpr* pExpr2, CegoExpr* pExpr3 )
{
    _pExpr1 = pExpr1;
    _pExpr2 = pExpr2;
    _pExpr3 = pExpr3;
    _pC = 0;
    _pNotPred = 0;
    _mode = CegoPredDesc::BETWEEN;
    _isChecked = false;
    _pSelect = 0;
    _pMatcher = 0;
}

CegoPredDesc::CegoPredDesc(CegoExpr *pExpr, const Chain& pattern, bool isNegated)
{
    _pExpr1 = pExpr;
    _pExpr2 = 0;
    _pExpr3 = 0;
    _pattern = pattern;
    Chain sqlPattern = Chain("^") + pattern + Chain("$");
    sqlPattern.replaceAll(Chain("%"), Chain(".*"), sqlPattern);
    sqlPattern.replaceAll(Chain("_"), Chain("."), sqlPattern);

    _pMatcher = new Matcher(sqlPattern);
    _pMatcher->prepare();
    
    if ( isNegated )
	_mode = CegoPredDesc::ISNOTLIKE;
    else
	_mode = CegoPredDesc::ISLIKE;

    _isChecked = false;
    _pSelect = 0;
    _pC = 0;
    _pNotPred = 0;
}

CegoPredDesc::CegoPredDesc(const CegoPredDesc& p)
{
    *this = p;
}

CegoPredDesc::CegoPredDesc(CegoCondDesc *pC)
{
    _pExpr1 = 0;
    _pExpr2 = 0;
    _pExpr3 = 0;    
    _pC = pC;
    _pNotPred = 0;
    _isChecked = false;
    _pSelect = 0;
    _pMatcher = 0;
    _mode = CegoPredDesc::CONDITION;
}

CegoPredDesc::CegoPredDesc(CegoPredDesc *pNotPred)
{
    _pExpr1 = 0;
    _pExpr2 = 0;
    _pExpr3 = 0;    
    _pC = 0;
    _isChecked = false;
    _pSelect = 0;
    _pMatcher = 0;
    _pNotPred = pNotPred;
    _mode = CegoPredDesc::NOTPRED;
}

CegoPredDesc::CegoPredDesc(CegoSelect *pSelect)
{
    _pExpr1 = 0;
    _pExpr2 = 0;
    _pExpr3 = 0;
    _pSelect = pSelect;
    _pMatcher = 0;
    _pC = 0;
    _pNotPred = 0;
    _mode = CegoPredDesc::EXISTSCOMP;
    _isChecked = false;
}

CegoPredDesc::CegoPredDesc(CegoExpr *pExpr, ListT<CegoExpr*>& exprList, bool isNegated)
{
    _pExpr1 = pExpr;
    _pExpr2 = 0;
    _pExpr3 = 0;
    _exprList = exprList;
    _pSelect = 0;
    _pMatcher = 0;
    _pC = 0;
    _pNotPred = 0;
    if ( isNegated )
	_mode = CegoPredDesc::NOTIN;
    else
	_mode = CegoPredDesc::IN;
    _isChecked = false;
}

CegoPredDesc::CegoPredDesc(CegoExpr *pExpr, CegoSelect* pSelect, bool isNegated)
{
    _pExpr1 = pExpr;
    _pExpr2 = 0;
    _pExpr3 = 0;
    _pSelect = pSelect;
    _pMatcher = 0;
    _pC = 0;
    _pNotPred = 0;
    if ( isNegated )
	_mode = CegoPredDesc::NOTINSUB;
    else
	_mode = CegoPredDesc::INSUB;
    _isChecked = false;
}

CegoPredDesc::CegoPredDesc(CegoExpr* pExpr, bool isNull)
{
    _pExpr1 = pExpr;
    _pExpr2 = 0;
    _pExpr3 = 0;
    _pSelect = 0;
    _pMatcher = 0;
    _pC = 0;
    _pNotPred = 0;
    if  ( isNull )
	_mode = CegoPredDesc::NULLCOMP;
    else
    	_mode = CegoPredDesc::NOTNULLCOMP;
    _isChecked = false;
}

CegoPredDesc::~CegoPredDesc()
{
    if ( _pExpr1 )
    {
	delete _pExpr1;
    }
    if ( _pExpr2 )
    {
	delete _pExpr2;
    }
    if ( _pExpr3 )
    {
	delete _pExpr3;
    }
    if (_pC)
    {	
	delete _pC;
    }
    if (_pNotPred)
    {	
	delete _pNotPred;
    }
    if (_pSelect)
    {
	delete _pSelect;
    }   
    if (_pMatcher)
    {
	delete _pMatcher;
    }
}

void CegoPredDesc::setTabSetId(int tabSetId)
{
    if ( _pExpr1 )
    {
	_pExpr1->setTabSetId(tabSetId);
    }
    if ( _pExpr2 )
    {
	_pExpr2->setTabSetId(tabSetId);
    }
    if ( _pExpr3 )
    {
	_pExpr3->setTabSetId(tabSetId);
    }
    if (_pC)
    {	
	_pC->setTabSetId(tabSetId);
    }
    if (_pNotPred)
    {	
	_pNotPred->setTabSetId(tabSetId);
    }

    if (_pSelect)
    {
	_pSelect->setTabSetId(tabSetId);
    }
}
    
void CegoPredDesc::getPlanList(ListT<Element*>& planList)
{
    if (_pSelect)
    {
	planList.Insert( _pSelect->getPlan() );
    }
    if ( _pC )
    {
	_pC->getPlanList(planList);
    }
    if (_pNotPred)
    {
	_pNotPred->getPlanList(planList);
    }
    if ( _pExpr1 )
    {
	_pExpr1->getPlanList(planList);
    }
    if ( _pExpr2 )
    {
	_pExpr2->getPlanList(planList);
    }
    if ( _pExpr3 )
    {
	_pExpr3->getPlanList(planList);
    }
}

CegoPredDesc& CegoPredDesc::operator = ( const CegoPredDesc& p)
{
    _pExpr1 = p._pExpr1;
    _pExpr2 = p._pExpr2;
    _pExpr3 = p._pExpr3;
    _exprList = p._exprList;
    _pMatcher = p._pMatcher;
    _comp = p._comp;
    _pSelect = p._pSelect;
    _pC = p._pC;
    _pNotPred = p._pNotPred;
    _mode = p._mode;
    _isChecked = p._isChecked;
    return (*this);
}

void CegoPredDesc::analyzeSelect()
{
    if ( _pSelect )
	_pSelect->prepare();
    if ( _pC )
    {
	if ( _pC->Left() )
	    _pC->Left()->analyzeSelect();
	if ( _pC->Right() )
	    _pC->Right()->analyzeSelect();
    }
    if ( _pNotPred )
    {
	_pNotPred->analyzeSelect();
    }
}

CegoSelect* CegoPredDesc::getSelectQuery()
{
    return _pSelect;
}

void CegoPredDesc::getSelectQueryList(ListT<CegoSelect*>& queryList)
{   
    if ( _pSelect )
	queryList.Insert(_pSelect);

    if ( _pExpr1 )
    {
	_pExpr1->getSelectQueryList(queryList);
    }
    if ( _pExpr2 )
    {
	_pExpr2->getSelectQueryList(queryList);
    }
    if ( _pExpr3 )
    {
	_pExpr3->getSelectQueryList(queryList);
    }

    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->getSelectQueryList(queryList);
	pExpr = _exprList.Next();
    }   
    
    if ( _pC )
    {
	if ( _pC->Left() )
	    _pC->Left()->getSelectQueryList(queryList);
	if ( _pC->Right() )
	    _pC->Right()->getSelectQueryList(queryList);
    }
    if ( _pNotPred )
    {
	_pNotPred->getSelectQueryList(queryList);
    }
}

CegoCondDesc* CegoPredDesc::getCondition()
{
    return _pC;
}

CegoPredDesc* CegoPredDesc::getNotPred()
{
    return _pNotPred;
}

CegoComparison CegoPredDesc::getComparison()
{
    return _comp;
}

CegoPredDesc::CompMode CegoPredDesc::getMode()
{
    return _mode;
}

CegoExpr* CegoPredDesc::getExpr1()
{
    return _pExpr1;
}

CegoExpr* CegoPredDesc::getExpr2()
{
    return _pExpr2;
}

CegoExpr* CegoPredDesc::getExpr3()
{
    return _pExpr3;
}

ListT<CegoExpr*> CegoPredDesc::getExprList()
{
    return _exprList;
}

void CegoPredDesc::setBlock(CegoProcBlock* pBlock)
{
    if ( _pExpr1 )
    {
	_pExpr1->setBlock(pBlock);
    }
    if ( _pExpr2 )
    {
	_pExpr2->setBlock(pBlock);
    }
    if ( _pExpr3 )
    {
	_pExpr3->setBlock(pBlock);
    }

    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->setBlock(pBlock);
	pExpr = _exprList.Next();
    }   
    
    if ( _pSelect )
    {
	_pSelect->setProcBlock(pBlock);
    }
    if (_pC)
    {
	_pC->Left()->setBlock(pBlock);
	_pC->Right()->setBlock(pBlock);
    }
    if (_pNotPred)
    {
	_pNotPred->setBlock(pBlock);
    }
}

void CegoPredDesc::setChecked(bool val)
{
    _isChecked = val;
}

void CegoPredDesc::clearAttrCache()
{
    if ( _pExpr1 )
    {
	_pExpr1->clearAttrCache();
    }
    if ( _pExpr2 )
    {
	_pExpr2->clearAttrCache();
    }
    if ( _pExpr3 )
    {
	_pExpr3->clearAttrCache();
    }

    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->clearAttrCache();
	pExpr = _exprList.Next();
    }   
    
    if ( _pC )
    {
	if ( _pC->Left() )
	    _pC->Left()->clearAttrCache();
	if ( _pC->Right() )
	    _pC->Right()->clearAttrCache();
    }
    if ( _pNotPred )
    {
	_pNotPred->clearAttrCache();
    }
}

void CegoPredDesc::cleanUp()
{
    if ( _pExpr1 )
    {
	_pExpr1->cleanUp();
    }
    if ( _pExpr2 )
    {
	_pExpr2->cleanUp();
    }
    if ( _pExpr3 )
    {
	_pExpr3->cleanUp();
    }

    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->cleanUp();
	pExpr = _exprList.Next();
    }   
    
    if (_pNotPred)
    {	
	_pNotPred->cleanUp();
    }
    if (_pC)
    {
	_pC->cleanUp();
    }
    if (_pSelect)
    {
	_pSelect->cleanUp();
    }   
}

void CegoPredDesc::setCheckedRec(bool val)
{
    _isChecked = val;
    if ( _pC )
    {
	_pC->Left()->setCheckedRec(val);
	_pC->Right()->setCheckedRec(val);       
    }
    if ( _pNotPred )
    {
	_pNotPred->setChecked(val);
    }
}

bool CegoPredDesc::isChecked() const
{
    return _isChecked;
}

bool CegoPredDesc::hasOrCond() const
{
    if (_pC)
    {
	if ( _pC->getCondType() == CegoCondDesc::OR )
	    return true;
	else if ( _pC->getCondType() == CegoCondDesc::AND )
	    return _pC->Left()->hasOrCond() || _pC->Right()->hasOrCond();       
	else if ( _pC->getCondType() == CegoCondDesc::PRED )
	    return _pC->Left()->hasOrCond();
    }
    return false;
}

const Chain& CegoPredDesc::getPattern() const
{
    return _pattern;
}

bool CegoPredDesc::match(const CegoFieldValue& val) const
{
    if ( _pMatcher )
	return _pMatcher->match(val.valAsChain());
    else 
	throw Exception(EXLOC, "Invalid matcher in predicate");
}

void CegoPredDesc::encode(char *buf)
{
    char* pP = (char*)buf;
    
    memcpy( pP, &_mode, sizeof(CegoPredDesc::CompMode));
    pP = pP + sizeof(CegoPredDesc::CompMode);

    switch (_mode )
    {
    case CegoPredDesc::CONDITION:
    {
	_pC->encode(pP);
	pP = pP + _pC->getEncodingLength();
	break;
    }
    case CegoPredDesc::NOTPRED:
    {
	_pNotPred->encode(pP);
	pP = pP + _pNotPred->getEncodingLength();
	break;
    }
    case CegoPredDesc::EXPRCOMP:
    {	
	memcpy( pP, &_comp, sizeof(CegoComparison));
	pP = pP + sizeof(CegoComparison);
	
	_pExpr1->encode(pP);
	pP = pP + _pExpr1->getEncodingLength();

	_pExpr2->encode(pP);
	pP = pP + _pExpr2->getEncodingLength();

	break;
    }
    case CegoPredDesc::BETWEEN:
    {	
	_pExpr1->encode(pP);
	pP = pP + _pExpr1->getEncodingLength();
	
	_pExpr2->encode(pP);
	pP = pP + _pExpr2->getEncodingLength();

	_pExpr3->encode(pP);
	pP = pP + _pExpr3->getEncodingLength();

	break;
    }
    case CegoPredDesc::NULLCOMP:
    case CegoPredDesc::NOTNULLCOMP:
    {
	_pExpr1->encode(pP);
	pP = pP + _pExpr1->getEncodingLength();
	break;
    }
    case CegoPredDesc::EXISTSCOMP:
    {
	_pSelect->encode(pP);
	pP = pP + _pSelect->getEncodingLength();
	break;
    }
    case CegoPredDesc::INSUB:
    case CegoPredDesc::NOTINSUB:
    {	
	_pExpr1->encode(pP);
	pP = pP + _pExpr1->getEncodingLength();
	_pSelect->encode(pP);
	pP = pP + _pSelect->getEncodingLength();
	break;
    }
    case CegoPredDesc::IN:
    case CegoPredDesc::NOTIN:
    {	
	_pExpr1->encode(pP);
	pP = pP + _pExpr1->getEncodingLength();     

	int num = _exprList.Size();
	memcpy( pP, &num, sizeof(int));
	pP = pP + sizeof(int);
      
	CegoExpr** pExpr = _exprList.First();
	while ( pExpr )
	{
	    (*pExpr)->encode(pP);
	    pP = pP + (*pExpr)->getEncodingLength();	    
	    pExpr = _exprList.Next();
	}   

	break;
    }
    case CegoPredDesc::ISLIKE:
    case CegoPredDesc::ISNOTLIKE:
    {
	_pExpr1->encode(pP);
	pP = pP + _pExpr1->getEncodingLength();

	char len = _pattern.length();

	memcpy( pP, &len, sizeof(char));
	pP = pP + sizeof(char);

	memcpy( pP, (char*)_pattern, len);
	pP = pP + len;

	break;
    }
    }
}

void CegoPredDesc::decode(char *buf, CegoDistManager *pGTM, int tabSetId)
{
    char* pP = (char*)buf;

    memcpy( &_mode, pP, sizeof(CegoPredDesc::CompMode));
    pP = pP + sizeof(CegoPredDesc::CompMode);
    
    // init condition and notpred as zero
    _pC = 0;
    _pNotPred = 0;
    _pSelect = 0;
    _pExpr1=0;
    _pExpr2=0;
    _pExpr3=0;

    switch (_mode )
    {
    case CegoPredDesc::CONDITION:
    {
	_pC = new CegoCondDesc(pP, pGTM, tabSetId);
	pP = pP + _pC->getEncodingLength();
	break;
    }
    case CegoPredDesc::NOTPRED:
    {
	_pNotPred = new CegoPredDesc(pP, pGTM, tabSetId);
	pP = pP + _pNotPred->getEncodingLength();
	break;
    }
    case CegoPredDesc::EXPRCOMP:
    {	
	memcpy( &_comp, pP, sizeof(CegoComparison));
	pP = pP + sizeof(CegoComparison);

	_pExpr1 = new CegoExpr(pP, pGTM, tabSetId);
	pP = pP + _pExpr1->getEncodingLength();

	_pExpr2 = new CegoExpr(pP, pGTM, tabSetId);
	pP = pP + _pExpr2->getEncodingLength();

	break;
    }
    case CegoPredDesc::BETWEEN:
    {	
	_pExpr1 = new CegoExpr(pP, pGTM, tabSetId);
	pP = pP + _pExpr1->getEncodingLength();

	_pExpr2 = new CegoExpr(pP, pGTM, tabSetId);
	pP = pP + _pExpr2->getEncodingLength();

	_pExpr3 = new CegoExpr(pP, pGTM, tabSetId);
	pP = pP + _pExpr3->getEncodingLength();

	break;
    }
    case CegoPredDesc::NULLCOMP:
    case CegoPredDesc::NOTNULLCOMP:
    {
	_pExpr1 = new CegoExpr(pP, pGTM, tabSetId);
	pP = pP + _pExpr1->getEncodingLength();
	break;
    }
    case CegoPredDesc::EXISTSCOMP:
    {
	_pSelect = new CegoSelect(pP, pGTM, tabSetId);
	pP = pP + _pSelect->getEncodingLength();
	break;
    }
    case CegoPredDesc::INSUB:
    case CegoPredDesc::NOTINSUB:
    {
	_pExpr1 = new CegoExpr(pP, pGTM, tabSetId);
	pP = pP + _pExpr1->getEncodingLength();
	_pSelect = new CegoSelect(pP, pGTM, tabSetId);
	pP = pP + _pSelect->getEncodingLength();	
	break;
    }
    case CegoPredDesc::IN:
    case CegoPredDesc::NOTIN:
    {	
	_pExpr1 = new CegoExpr(pP, pGTM, tabSetId);
	pP = pP + _pExpr1->getEncodingLength();

	int num;
	memcpy( &num, pP, sizeof(int));
	pP = pP + sizeof(int);

	for ( int i=0; i<num; i++ )
	{
	    CegoExpr *pExpr = new CegoExpr(pP, pGTM, tabSetId);
	    _exprList.Insert(pExpr);
	    pP = pP + pExpr->getEncodingLength();
	}

	break;
    }
    case CegoPredDesc::ISLIKE:
    case CegoPredDesc::ISNOTLIKE:
    {
	_pExpr1 = new CegoExpr(pP, pGTM, tabSetId);
	pP = pP + _pExpr1->getEncodingLength();
	
	char len;
	memcpy( &len, pP, sizeof(char));
	pP = pP + sizeof(char);

	_pattern = Chain(pP, len);
	pP = pP + len;

	break;
    }
    }
}

int CegoPredDesc::getEncodingLength() const
{
    int len = 0;
    
    len += sizeof(CegoPredDesc::CompMode);
    
    switch (_mode )
    {
    case CegoPredDesc::CONDITION:
    {
	len += _pC->getEncodingLength();
	break;
    }
    case CegoPredDesc::NOTPRED:
    {
	len += _pNotPred->getEncodingLength();
	break;
    }
    case CegoPredDesc::EXPRCOMP:
    {
	len += sizeof(CegoComparison);
	len += _pExpr1->getEncodingLength();
	len += _pExpr2->getEncodingLength();
	break;
    }
    case CegoPredDesc::BETWEEN:
    {
	len += _pExpr1->getEncodingLength();
	len += _pExpr2->getEncodingLength();
	len += _pExpr3->getEncodingLength();
	break;
    }
    case CegoPredDesc::ISLIKE:
    case CegoPredDesc::ISNOTLIKE:
    {
	len += _pExpr1->getEncodingLength();
	len += 1; // sizeof pattern
	len += _pattern.length();
	break;
    }
    case CegoPredDesc::NULLCOMP:
    case CegoPredDesc::NOTNULLCOMP:
    {
	len += _pExpr1->getEncodingLength();
	break;
    }
    case CegoPredDesc::EXISTSCOMP:
    {
	len += _pSelect->getEncodingLength();
	break;
    }
    case CegoPredDesc::INSUB:
    case CegoPredDesc::NOTINSUB:
    {
	len += _pExpr1->getEncodingLength();
	len += _pSelect->getEncodingLength();
	break;
    }
    case CegoPredDesc::IN:
    case CegoPredDesc::NOTIN:
    {
	len += _pExpr1->getEncodingLength();

	len += sizeof(int);

	CegoExpr** pExpr = _exprList.First();
	while ( pExpr )
	{
	    len += (*pExpr)->getEncodingLength();	    
	    pExpr = _exprList.Next();
	}
	break;
    }
    }    
    return len;
}

SetT<Chain> CegoPredDesc::getTableRefSet() const
{
    SetT<Chain> tableRefSet;
    ListT<CegoAttrDesc*> adList = getAttrRefList();
    CegoAttrDesc **pAD = adList.First();
    while ( pAD )
    {
	tableRefSet.Insert((*pAD)->getTableName());
	pAD = adList.Next();
    }
    return tableRefSet;
}

ListT<CegoAttrDesc*> CegoPredDesc::getAttrRefList() const
{
    ListT<CegoAttrDesc*> al;

    switch (_mode )
    {
    case CegoPredDesc::CONDITION:
    {
	al = _pC->getAttrRefList();
	break;
    }
    case CegoPredDesc::NOTPRED:
    {
	al = _pNotPred->getAttrRefList();
	break;
    }
    case CegoPredDesc::EXPRCOMP:
    {
	al = _pExpr1->getAttrRefList();
	al += _pExpr2->getAttrRefList();
	break;
    }
    case CegoPredDesc::BETWEEN:
    {
	al = _pExpr1->getAttrRefList();
	al += _pExpr2->getAttrRefList();
	al += _pExpr3->getAttrRefList();
	break;
    }
    case CegoPredDesc::ISLIKE:
    case CegoPredDesc::ISNOTLIKE:
    {
	al += _pExpr1->getAttrRefList();
	break;
    }
    case CegoPredDesc::NULLCOMP:
    case CegoPredDesc::NOTNULLCOMP:
    {
	al += _pExpr1->getAttrRefList();
	break;
    }
    case CegoPredDesc::INSUB:
    case CegoPredDesc::NOTINSUB:
    {
	al = _pExpr1->getAttrRefList();
	al += _pSelect->getAttrRefList();
	break;
    }
    case CegoPredDesc::IN:
    case CegoPredDesc::NOTIN:
    {
	al = _pExpr1->getAttrRefList();

	CegoExpr** pExpr = _exprList.First();
	while ( pExpr )
	{
	    al += (*pExpr)->getAttrRefList();	    
	    pExpr = _exprList.Next();
	}
	break;
    }
    case CegoPredDesc::EXISTSCOMP:
    {
	al = _pSelect->getAttrRefList();
	break;
    }    
    }
    return al; 
}

int CegoPredDesc::evalReferences(CegoContentObject *pCO, const ListT<CegoField>& fl)
{
    int refCount = 0;
    
    switch (_mode )
    {
    case CegoPredDesc::CONDITION:
    {
	refCount += _pC->evalReferences(pCO, fl);
	break;
    }
    case CegoPredDesc::NOTPRED:
    {
	refCount += _pNotPred->evalReferences(pCO, fl);
	break;
    }
    case CegoPredDesc::EXPRCOMP:
    {
 	refCount += _pExpr1->evalReferences(pCO, fl);
	refCount += _pExpr2->evalReferences(pCO, fl);
	break;
    }
    case CegoPredDesc::BETWEEN:
    {
	refCount += _pExpr1->evalReferences(pCO, fl);
	refCount += _pExpr2->evalReferences(pCO, fl);
	refCount += _pExpr3->evalReferences(pCO, fl);
	break;
    }
    case CegoPredDesc::ISLIKE:
    case CegoPredDesc::ISNOTLIKE:
    {
	refCount += _pExpr1->evalReferences(pCO, fl);
	break;
    }
    case CegoPredDesc::NULLCOMP:
    case CegoPredDesc::NOTNULLCOMP:
    {
	refCount += _pExpr1->evalReferences(pCO, fl);
	break;
    }
    case CegoPredDesc::INSUB:
    case CegoPredDesc::NOTINSUB:
    {
	refCount += _pExpr1->evalReferences(pCO, fl);
	_pSelect->prepare();
	refCount += _pSelect->evalExtTableReferences(pCO, fl);
	break;
    }
    case CegoPredDesc::IN:
    case CegoPredDesc::NOTIN:
    {
	refCount += _pExpr1->evalReferences(pCO, fl);

	CegoExpr** pExpr = _exprList.First();
	while ( pExpr )
	{
	    refCount += (*pExpr)->evalReferences(pCO, fl);
	    pExpr = _exprList.Next();
	}
	break;
    }
    case CegoPredDesc::EXISTSCOMP:
    {
	_pSelect->prepare();
	refCount += _pSelect->evalExtTableReferences(pCO, fl);
	break;
    }    
    }

    return refCount;
}

ListT<CegoAggregation*> CegoPredDesc::getAggregationList()
{
    ListT<CegoAggregation*> aggList;	

    if ( _pExpr1 )
    {
	aggList = aggList + _pExpr1->getAggregationList();
    }

    if ( _pExpr2 )
    {
	aggList = aggList + _pExpr2->getAggregationList();
    }

    if ( _pExpr3 )
    {
	aggList = aggList + _pExpr3->getAggregationList();
    }

    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	aggList = aggList + (*pExpr)->getAggregationList();
	pExpr = _exprList.Next();
    }

    if (_pC)
    {	
	aggList = aggList + _pC->getAggregationList();
    }
    
    if (_pNotPred)
    {
	aggList = aggList + _pNotPred->getAggregationList();
    }
    
    return aggList;

}

void CegoPredDesc::setFieldListArray(ListT<CegoField> **pFLA)
{
    
    if ( _pExpr1 )
    {
	_pExpr1->setFieldListArray(pFLA);
    }

    if ( _pExpr2 )
    {
	_pExpr2->setFieldListArray(pFLA);
    }

    if ( _pExpr3 )
    {
	_pExpr3->setFieldListArray(pFLA);
    }
    CegoExpr** pExpr = _exprList.First();
    while ( pExpr )
    {
	(*pExpr)->setFieldListArray(pFLA);
	pExpr = _exprList.Next();
    }

    if ( _pC )
    {
	_pC->setFieldListArray(pFLA);
    }
    
    if (_pNotPred)
    {
	_pNotPred->setFieldListArray(pFLA);
    }

}


void CegoPredDesc::getFieldList(ListT<CegoField>& fl) const
{    
    switch (_mode )
    {
    case CegoPredDesc::CONDITION:
    {
	_pC->getFieldList(fl);
	break;
    }
    case CegoPredDesc::NOTPRED:
    {
	_pNotPred->getFieldList(fl);
	break;
    }
    case CegoPredDesc::EXPRCOMP:
    {
	_pExpr1->getFieldList(fl);
	_pExpr2->getFieldList(fl);
	break;
    }
    case CegoPredDesc::BETWEEN:
    {
	_pExpr1->getFieldList(fl);
	_pExpr2->getFieldList(fl);
	_pExpr3->getFieldList(fl);
	break;
    }
    case CegoPredDesc::ISLIKE:
    case CegoPredDesc::ISNOTLIKE:
    {
	_pExpr1->getFieldList(fl);
	break;
    }
    case CegoPredDesc::INSUB:
    case CegoPredDesc::NOTINSUB:
    {
	_pExpr1->getFieldList(fl);
	_pSelect->getFieldList(fl);
	break;
    }
    case CegoPredDesc::IN:
    case CegoPredDesc::NOTIN:
    {
	_pExpr1->getFieldList(fl);

	CegoExpr** pExpr = _exprList.First();
	while ( pExpr )
	{
	    (*pExpr)->getFieldList(fl);
	    pExpr = _exprList.Next();
	}
	break;
    }
    case CegoPredDesc::EXISTSCOMP:
    {
	_pSelect->getFieldList(fl);
	break;
    }
    case CegoPredDesc::NULLCOMP:
    case CegoPredDesc::NOTNULLCOMP:
    {
	_pExpr1->getFieldList(fl);
	break;
    }
    }
}

CegoPredDesc* CegoPredDesc::clone(bool isAttrRef)
{    
    if ( _mode == CegoPredDesc::EXPRCOMP )
    {
	return new CegoPredDesc(_pExpr1->clone(isAttrRef), _pExpr2->clone(isAttrRef), _comp);
    }
    else if ( _mode == CegoPredDesc::BETWEEN )
    {
	return new CegoPredDesc(_pExpr1->clone(isAttrRef), _pExpr2->clone(isAttrRef), _pExpr3->clone(isAttrRef));
    }
    else if ( _mode == CegoPredDesc::ISLIKE )
    {
	return new CegoPredDesc(_pExpr1->clone(isAttrRef), _pattern, false);
    }
    else if ( _mode == CegoPredDesc::ISNOTLIKE )
    {
	return new CegoPredDesc(_pExpr1->clone(isAttrRef), _pattern, true);
    }
    else if ( _mode == CegoPredDesc::CONDITION )
    {
	return new CegoPredDesc(_pC->clone(isAttrRef));
    }
    else if ( _mode == CegoPredDesc::NOTPRED )
    {
	return new CegoPredDesc(_pNotPred->clone(isAttrRef));
    }
    else if ( _mode == CegoPredDesc::EXISTSCOMP )
    {
	return new CegoPredDesc(_pSelect->clone(isAttrRef));
    }
    else if ( _mode == CegoPredDesc::INSUB )
    {
	return new CegoPredDesc(_pExpr1->clone(isAttrRef), _pSelect->clone(isAttrRef), false);
    }
    else if ( _mode == CegoPredDesc::NOTINSUB )
    {
	return new CegoPredDesc(_pExpr1->clone(isAttrRef), _pSelect->clone(isAttrRef), true);
    }
    else if ( _mode == CegoPredDesc::IN )
    {
	ListT<CegoExpr*> exprList;

	CegoExpr** pExpr = _exprList.First();
	while ( pExpr )
	{
	    exprList.Insert((*pExpr)->clone());
	    pExpr = _exprList.Next();
	}
	return new CegoPredDesc(_pExpr1->clone(isAttrRef), exprList, false);
    }
    else if ( _mode == CegoPredDesc::NOTIN )
    {
	ListT<CegoExpr*> exprList;

	CegoExpr** pExpr = _exprList.First();
	while ( pExpr )
	{
	    exprList.Insert((*pExpr)->clone());
	    pExpr = _exprList.Next();
	}
	return new CegoPredDesc(_pExpr1->clone(isAttrRef), exprList, true);
    }
    else if ( _mode == CegoPredDesc::NULLCOMP )
    {
	return new CegoPredDesc(_pExpr1->clone(isAttrRef), true);
    }
    else if ( _mode == CegoPredDesc::NOTNULLCOMP )
    {
	return new CegoPredDesc(_pExpr1->clone(isAttrRef), false);
    }
    throw Exception(EXLOC, "Clone not implemented");
}

Chain CegoPredDesc::getId() const
{
    Chain s;

    switch (_mode )
    {
    case CegoPredDesc::CONDITION:
    {
	if (_pC->Left())
	{
	    s = Chain("( ");
	    s += _pC->Left()->getId();

	    switch(_pC->getCondType())
	    {
	    case CegoCondDesc::AND:
		s += Chain("a");
		break;
	    case CegoCondDesc::OR:
		s += Chain("o");
		break;
	    case CegoCondDesc::PRED:
		break;
	    }
	    s += _pC->Right()->getId();
	}
	break;
    }
    case CegoPredDesc::NOTPRED:
    {
	s = Chain("n") + _pNotPred->getId();
	break;
    }
    case CegoPredDesc::EXPRCOMP:
    {	
	s = _pExpr1->getId();
	
	switch (_comp)
	{
	case EQUAL:
	    s += "=";
	    break;
	case NOT_EQUAL:
	    s += "!=";
	    break;
	case LESS_THAN:
	    s += "<";
	    break;
	case MORE_THAN:
	    s += ">";
	    break;
	case LESS_EQUAL_THAN:
	    s += "<=";
	    break;
	case MORE_EQUAL_THAN:
	    s += ">=";
	    break;
	}
	
	s +=  _pExpr2->getId();
	break;
    }
    case CegoPredDesc::BETWEEN:
    {	
	s = _pExpr2->getId();
	s += Chain("bw");
	s +=  _pExpr1->getId();
	s += Chain("a");
	s +=  _pExpr3->getId();

	break;
    }
    case CegoPredDesc::ISLIKE:
    {
	s = _pExpr1->getId();
	s += Chain("l");
	s += Chain("'") + _pattern + Chain("'");
	break;
    }
    case CegoPredDesc::ISNOTLIKE:
    {
	s = _pExpr1->getId();
	s += Chain("nl");
	s += Chain("'") + _pattern + Chain("'");
	break;
    }
    case CegoPredDesc::NULLCOMP:
    {
	s =  _pExpr1->getId();
	s += Chain("n");
	break;
    }
    case CegoPredDesc::NOTNULLCOMP:
    {
	s = _pExpr1->getId();
	s += Chain("nn");
	break;
    }
    case CegoPredDesc::EXISTSCOMP:
    {
	s = Chain("e");
	s += _pSelect->getQueryId();
	break;
    }
    case CegoPredDesc::INSUB:
    {
	s = _pExpr1->getId();
	s += Chain("i");
	s += Chain("(") + _pSelect->getQueryId() + Chain(")");
	break;
    }
    case CegoPredDesc::NOTINSUB:
    {
	s = _pExpr1->getId();
	s += Chain("ni");
	s += Chain("(") + _pSelect->getQueryId() + Chain(")");   
	break;
    }
    case CegoPredDesc::IN:
    {
	s = _pExpr1->getId();
	s += Chain("i");
    
	s += Chain("(");	
	CegoExpr **pExpr = _exprList.First();
	while ( pExpr )
	{
	    s += (*pExpr)->getId();
	    pExpr = _exprList.Next();
	    if ( pExpr )
		s += Chain(",");
	}
	s += Chain(")");
	break;
    }
    case CegoPredDesc::NOTIN:
    {
	s = _pExpr1->getId();
	s += Chain("ni");

	s += Chain("(");	
	CegoExpr **pExpr = _exprList.First();
	while ( pExpr )
	{
	    s += (*pExpr)->getId();
	    pExpr = _exprList.Next();
	    if ( pExpr )
		s += Chain(",");
	}
	s += Chain(")");
	
	break;
    }
    }
    return s;
}

Chain CegoPredDesc::toChain(const Chain& indent) const
{
    Chain s;

    switch (_mode )
    {
    case CegoPredDesc::CONDITION:
    {
	if (_pC->Left())
	{
	    s = Chain("( ");
	    s += _pC->Left()->toChain(indent);

	    s += Chain("\n") + indent;
	    Chain addIndent;
	    switch(_pC->getCondType())
	    {
	    case CegoCondDesc::AND:
		s += Chain("and ");
		addIndent = Chain(DEFAULTINDENT);
		break;
	    case CegoCondDesc::OR:
		s += Chain("or ");
		addIndent = Chain(DEFAULTINDENT);
		break;
	    case CegoCondDesc::PRED:
		break;
	    }

	    s += _pC->Right()->toChain(indent + addIndent);
	    s += Chain(" )");
	}
	break;
    }
    case CegoPredDesc::NOTPRED:
    {
	s = Chain("not ") + _pNotPred->toChain(indent + Chain(DEFAULTINDENT));
	break;
    }
    case CegoPredDesc::EXPRCOMP:
    {	
	s = _pExpr1->toChain(indent);
	
	switch (_comp)
	{
	case EQUAL:
	    s += " = ";
	    break;
	case NOT_EQUAL:
	    s += " != ";
	    break;
	case LESS_THAN:
	    s += " < ";
	    break;
	case MORE_THAN:
	    s += " > ";
	    break;
	case LESS_EQUAL_THAN:
	    s += " <= ";
	    break;
	case MORE_EQUAL_THAN:
	    s += " >= ";
	    break;
	}
	
	Chain addIndent;
	for ( int i=1; i<s.length(); i++ )
	    addIndent += Chain(" ");
	
	s +=  _pExpr2->toChain(indent + addIndent);
	break;
    }
    case CegoPredDesc::BETWEEN:
    {	
	s = _pExpr1->toChain(indent);
	s += Chain("\n") + indent;
	s += Chain("between ");

	Chain addIndent;

	addIndent = Chain(DEFAULTINDENT);
	s += _pExpr2->toChain(indent + addIndent);
	s += Chain("\n") + indent;
	s += Chain("and ");	

	addIndent=Chain(DEFAULTINDENT);
	s += _pExpr3->toChain(indent + addIndent);

	break;
    }
    case CegoPredDesc::ISLIKE:
    {
	s = _pExpr1->toChain(indent);
	s += Chain(" like ");
	s += Chain("'") + _pattern + Chain("'");
	break;
    }
    case CegoPredDesc::ISNOTLIKE:
    {
	s = _pExpr1->toChain(indent);
	s += Chain(" not like ");
	s += Chain("'") + _pattern + Chain("'");
	break;
    }
    case CegoPredDesc::NULLCOMP:
    {
	s = _pExpr1->toChain(indent);
	s += Chain(" is null ");
	break;
    }
    case CegoPredDesc::NOTNULLCOMP:
    {
	s = _pExpr1->toChain(indent);
	s += Chain(" is not null ");
	break;
    }
    case CegoPredDesc::EXISTSCOMP:
    {
	s = Chain("exists ( ");
	s += _pSelect->toChain(indent + Chain(DEFAULTINDENT)) + Chain(" )");
	break;
    }
    case CegoPredDesc::INSUB:
    {
	s = _pExpr1->toChain(indent);
	s += Chain(" in ( ");

	Chain addIndent;
	for ( int i=1; i<s.length(); i++ )
	    addIndent += Chain(" ");

	s += _pSelect->toChain(indent + addIndent) + Chain(" )");   
	break;
    }
    case CegoPredDesc::NOTINSUB:
    {
	s = _pExpr1->toChain(indent);
	s += Chain(" not in ( ");

	Chain addIndent;
	for ( int i=1; i<s.length(); i++ )
	    addIndent += Chain(" ");

	s += _pSelect->toChain(indent + addIndent) + Chain(" )");   
	break;
    }
    case CegoPredDesc::IN:
    {
	s = _pExpr1->toChain(indent);
	s += Chain(" in ( ");

	CegoExpr **pExpr = _exprList.First();
	while ( pExpr )
	{
	    s += (*pExpr)->toChain();
	    pExpr = _exprList.Next();
	    if ( pExpr )
		s += Chain(",");
	}
	s += Chain(")");

	break;
    }
    case CegoPredDesc::NOTIN:
    {
	s = _pExpr1->toChain(indent);
	s += Chain(" not in ( ");

	CegoExpr **pExpr = _exprList.First();
	while ( pExpr )
	{
	    s += (*pExpr)->toChain();
	    pExpr = _exprList.Next();
	    if ( pExpr )
		s += Chain(",");
	}
	s += Chain(")");

	break;
    }
    }
    return s;
}

Chain CegoPredDesc::dbFormat(CegoDatabaseFormater *pForm)
{
    return pForm->formatPred(_mode, _pExpr1, _pExpr2, _pExpr3, _exprList, _comp, _pattern, _pC, _pNotPred, _pSelect);
}
    
Element* CegoPredDesc::toElement() const
{
    Element *pPredElement = new Element(XML_PRED_ELEMENT);

    switch (_mode )
    {
    case CegoPredDesc::CONDITION:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_COND_VALUE );
	pPredElement->addContent(  _pC->toElement() );	
	break;
    }
    case CegoPredDesc::NOTPRED:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_NOTPRED_VALUE );
	pPredElement->addContent(  _pNotPred->toElement() );	
	break;
    }
    case CegoPredDesc::EXPRCOMP:
    {
	pPredElement->addContent(  _pExpr1->toElement() );
	pPredElement->addContent(  _pExpr2->toElement() );
	pPredElement->setAttribute( XML_PRED_ATTR, XML_EXPRCOMP_VALUE );
    
	switch (_comp)
	{
	case EQUAL:
	    pPredElement->setAttribute( XML_COMP_ATTR, XML_EQUAL_VALUE );    
	    break;
	case NOT_EQUAL:
	    pPredElement->setAttribute( XML_COMP_ATTR, XML_NOTEQUAL_VALUE ); 
	    break;
	case LESS_THAN:
	    pPredElement->setAttribute( XML_COMP_ATTR, XML_LESSTHAN_VALUE ); 
	    break;
	case MORE_THAN:
	    pPredElement->setAttribute( XML_COMP_ATTR, XML_MORETHAN_VALUE ); 
	    break;
	case LESS_EQUAL_THAN:
	    pPredElement->setAttribute( XML_COMP_ATTR, XML_LESSEQUALTHAN_VALUE ); 
	    break;
	case MORE_EQUAL_THAN:
	    pPredElement->setAttribute( XML_COMP_ATTR, XML_MOREEQUALTHAN_VALUE );
	    break;
	}	
	break;
    }
    case CegoPredDesc::BETWEEN:
    {
	pPredElement->addContent(  _pExpr1->toElement() );
	pPredElement->addContent(  _pExpr2->toElement() );
	pPredElement->addContent(  _pExpr3->toElement() );
	pPredElement->setAttribute( XML_PRED_ATTR, XML_BETWEEN_VALUE );
	break;
    }
    case CegoPredDesc::ISLIKE:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_ISLIKE_VALUE ); 
	pPredElement->addContent(  _pExpr1->toElement() );
	pPredElement->setAttribute( XML_PATTERN_ATTR, _pattern );
	break;
    }
    case CegoPredDesc::ISNOTLIKE:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_ISLIKE_VALUE ); 
	pPredElement->addContent(  _pExpr1->toElement() );
	pPredElement->setAttribute( XML_PATTERN_ATTR, _pattern ); 
	break;
    }
    case CegoPredDesc::NULLCOMP:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_NULLCOMP_VALUE ); 
	pPredElement->addContent ( _pExpr1->toElement() );
	break;
    }
    case CegoPredDesc::NOTNULLCOMP:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_NOTNULLCOMP_VALUE ); 
	pPredElement->addContent ( _pExpr1->toElement() );
	break;
    }
    case CegoPredDesc::EXISTSCOMP:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_EXISTS_VALUE ); 
	pPredElement->addContent( _pSelect->toElement() );
	break;
    }
    case CegoPredDesc::INSUB:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_INQUERY_VALUE ); 
	pPredElement->addContent ( _pExpr1->toElement() );
	pPredElement->addContent( _pSelect->toElement() );
	break;
    }
    case CegoPredDesc::NOTINSUB:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_NOTINQUERY_VALUE ); 
	pPredElement->addContent ( _pExpr1->toElement() );
	pPredElement->addContent( _pSelect->toElement() );
	break;
    }       
    case CegoPredDesc::IN:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_IN_VALUE ); 
	pPredElement->addContent ( _pExpr1->toElement() );

	CegoExpr **pExpr = _exprList.First();
	while ( pExpr )
	{
	    pPredElement->addContent ( (*pExpr)->toElement() );
	    pExpr = _exprList.Next();
	}
	break;
    }       
    case CegoPredDesc::NOTIN:
    {
	pPredElement->setAttribute( XML_PRED_ATTR, XML_NOTIN_VALUE ); 
	pPredElement->addContent ( _pExpr1->toElement() );
	
	CegoExpr **pExpr = _exprList.First();
	while ( pExpr )
	{
	    pPredElement->addContent ( (*pExpr)->toElement() );
	    pExpr = _exprList.Next();
	}
	break;
    }
    }
    return pPredElement;
}

void CegoPredDesc::fromElement(Element* pPredElement, CegoDistManager *pGTM)
{
    if ( _pExpr1 )
	delete _pExpr1;
    if ( _pExpr2 )
	delete _pExpr2;
    if ( _pExpr3 )
	delete _pExpr3;
    if ( _pC )
	delete _pC;
    if ( _pNotPred )
	delete _pNotPred;
    if ( _pSelect )
	delete _pSelect;

    _pExpr1 = 0;
    _pExpr2 = 0;
    _pExpr3 = 0;
    _pC = 0;
    _pNotPred = 0;
    _isChecked = false;
    _pSelect = 0;

    Chain mode = pPredElement->getAttributeValue( XML_PRED_ATTR );

    if ( mode == Chain(XML_COND_VALUE) )
    {
	_mode = CegoPredDesc::CONDITION;

	ListT<Element*> cel = pPredElement->getChildren(XML_COND_ELEMENT);
	Element **pCE = cel.First();
	if ( pCE ) 
	{
	    _pC = new CegoCondDesc(*pCE, pGTM);
	}
	else
	{
	    throw Exception(EXLOC, "Invalid element");
	} 
    }
    else if ( mode == Chain(XML_NOTPRED_VALUE) )
    {
	_mode = CegoPredDesc::NOTPRED;
	
	ListT<Element*> npl = pPredElement->getChildren(XML_PRED_ELEMENT);
	Element **pNPE = npl.First();
	if ( pNPE ) 
	{
	    _pNotPred = new CegoPredDesc(*pNPE, pGTM);
	}
	else
	{
	    throw Exception(EXLOC, "Invalid element");
	} 
    }
    else if ( mode == Chain(XML_EXPRCOMP_VALUE) )
    {
	_mode = CegoPredDesc::EXPRCOMP;

	ListT<Element*> el = pPredElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr1 = new CegoExpr(*pEE, pGTM);
	}
	pEE = el.Next();
	if ( pEE )
	{
	    _pExpr2 = new CegoExpr(*pEE, pGTM);
	}

	Chain compString = pPredElement->getAttributeValue( XML_COMP_ATTR );    

	if ( compString == Chain(XML_EQUAL_VALUE) )
	{
	    _comp = EQUAL;
	}
	else if ( compString == Chain(XML_NOTEQUAL_VALUE) )
	{
	    _comp = NOT_EQUAL;
	}
	else if ( compString == Chain(XML_LESSTHAN_VALUE) )
	{
	    _comp = LESS_THAN;
	}
	else if ( compString == Chain(XML_MORETHAN_VALUE) )
	{
	    _comp = MORE_THAN;
	}
	else if ( compString == Chain(XML_LESSEQUALTHAN_VALUE) )
	{
	    _comp = LESS_EQUAL_THAN;
	}
	else if ( compString == Chain(XML_MOREEQUALTHAN_VALUE) )
	{
	    _comp = MORE_EQUAL_THAN;
	}	
    }
    else if ( mode == Chain(XML_BETWEEN_VALUE) )
    {
	_mode = CegoPredDesc::BETWEEN;

	ListT<Element*> el = pPredElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr1 = new CegoExpr(*pEE, pGTM);
	}
	pEE = el.Next();
	if ( pEE )
	{
	    _pExpr2 = new CegoExpr(*pEE, pGTM);
	}
	pEE = el.Next();
	if ( pEE )
	{
	    _pExpr3 = new CegoExpr(*pEE, pGTM);
	}
    }
    else if ( mode == Chain(XML_ISLIKE_VALUE) )
    {
	_mode = CegoPredDesc::ISLIKE;
	ListT<Element*> el = pPredElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr1 = new CegoExpr(*pEE, pGTM);
	}

	_pattern = pPredElement->getAttributeValue( XML_PATTERN_ATTR );
    }
    else if ( mode == Chain(XML_ISNOTLIKE_VALUE) )
    {
	_mode = CegoPredDesc::ISNOTLIKE;
	ListT<Element*> el = pPredElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr1 = new CegoExpr(*pEE, pGTM);
	}

	_pattern = pPredElement->getAttributeValue( XML_PATTERN_ATTR );
    }
    else if ( mode == Chain(XML_NULLCOMP_VALUE) )
    {
	_mode = CegoPredDesc::NULLCOMP;
	ListT<Element*> el = pPredElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr1 = new CegoExpr(*pEE, pGTM);
	}
    }
    else if ( mode == Chain(XML_NOTNULLCOMP_VALUE) )
    {
	_mode = CegoPredDesc::NOTNULLCOMP;
	ListT<Element*> el = pPredElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr1 = new CegoExpr(*pEE, pGTM);
	}
    }
    else if ( mode == Chain(XML_EXISTS_VALUE) )
    {
	_mode = CegoPredDesc::EXISTSCOMP;
	ListT<Element*> sl = pPredElement->getChildren(XML_SELECT_ELEMENT);
	Element **pSE = sl.First();
	if ( pSE )
	{
	    _pSelect = new CegoSelect(*pSE, pGTM);
	}
    }
    else if ( mode == Chain(XML_IN_VALUE) )
    {
	_mode = CegoPredDesc::IN;

	ListT<Element*> el = pPredElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr1 = new CegoExpr(*pEE, pGTM);
	}

	pEE = el.Next();
	while ( pEE )
	{
	    _exprList.Insert( new CegoExpr(*pEE, pGTM) );
	    pEE = el.Next();
	}
    }
    else if ( mode == Chain(XML_NOTIN_VALUE) )
    {
	_mode = CegoPredDesc::NOTIN;

	ListT<Element*> el = pPredElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr1 = new CegoExpr(*pEE, pGTM);
	}
		
	pEE = el.Next();
	while ( pEE )
	{
	    _exprList.Insert( new CegoExpr(*pEE, pGTM) );
	    pEE = el.Next();
	}	
    }
    else if ( mode == Chain(XML_INQUERY_VALUE) )
    {
	_mode = CegoPredDesc::INSUB;

	ListT<Element*> el = pPredElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr1 = new CegoExpr(*pEE, pGTM);
	}

	ListT<Element*> sl = pPredElement->getChildren(XML_SELECT_ELEMENT);
	Element **pSE = sl.First();
	if ( pSE )
	{
	    _pSelect = new CegoSelect(*pSE, pGTM);
	}
    }
    else if ( mode == Chain(XML_NOTINQUERY_VALUE) )
    {
	_mode = CegoPredDesc::NOTINSUB;

	ListT<Element*> el = pPredElement->getChildren(XML_EXPR_ELEMENT);
	Element **pEE = el.First();
	if ( pEE )
	{
	    _pExpr1 = new CegoExpr(*pEE, pGTM);
	}

	ListT<Element*> sl = pPredElement->getChildren(XML_SELECT_ELEMENT);
	Element **pSE = sl.First();
	if ( pSE )
	{
	    _pSelect = new CegoSelect(*pSE, pGTM);
	}
    }
}

ostream& operator << (ostream& s, const CegoPredDesc& p)
{
    s << p.toChain();
    return s;
}
