﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/datazone/DataZone_EXPORTS.h>
#include <aws/datazone/model/FormEntryOutput.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace DataZone {
namespace Model {

/**
 * <p>The details of the asset type.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/datazone-2018-05-10/AssetTypeItem">AWS
 * API Reference</a></p>
 */
class AssetTypeItem {
 public:
  AWS_DATAZONE_API AssetTypeItem() = default;
  AWS_DATAZONE_API AssetTypeItem(Aws::Utils::Json::JsonView jsonValue);
  AWS_DATAZONE_API AssetTypeItem& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_DATAZONE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The identifier of the Amazon DataZone domain where the asset type exists.</p>
   */
  inline const Aws::String& GetDomainId() const { return m_domainId; }
  inline bool DomainIdHasBeenSet() const { return m_domainIdHasBeenSet; }
  template <typename DomainIdT = Aws::String>
  void SetDomainId(DomainIdT&& value) {
    m_domainIdHasBeenSet = true;
    m_domainId = std::forward<DomainIdT>(value);
  }
  template <typename DomainIdT = Aws::String>
  AssetTypeItem& WithDomainId(DomainIdT&& value) {
    SetDomainId(std::forward<DomainIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the asset type.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  AssetTypeItem& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The revision of the asset type.</p>
   */
  inline const Aws::String& GetRevision() const { return m_revision; }
  inline bool RevisionHasBeenSet() const { return m_revisionHasBeenSet; }
  template <typename RevisionT = Aws::String>
  void SetRevision(RevisionT&& value) {
    m_revisionHasBeenSet = true;
    m_revision = std::forward<RevisionT>(value);
  }
  template <typename RevisionT = Aws::String>
  AssetTypeItem& WithRevision(RevisionT&& value) {
    SetRevision(std::forward<RevisionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the asset type.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  AssetTypeItem& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The forms included in the details of the asset type.</p>
   */
  inline const Aws::Map<Aws::String, FormEntryOutput>& GetFormsOutput() const { return m_formsOutput; }
  inline bool FormsOutputHasBeenSet() const { return m_formsOutputHasBeenSet; }
  template <typename FormsOutputT = Aws::Map<Aws::String, FormEntryOutput>>
  void SetFormsOutput(FormsOutputT&& value) {
    m_formsOutputHasBeenSet = true;
    m_formsOutput = std::forward<FormsOutputT>(value);
  }
  template <typename FormsOutputT = Aws::Map<Aws::String, FormEntryOutput>>
  AssetTypeItem& WithFormsOutput(FormsOutputT&& value) {
    SetFormsOutput(std::forward<FormsOutputT>(value));
    return *this;
  }
  template <typename FormsOutputKeyT = Aws::String, typename FormsOutputValueT = FormEntryOutput>
  AssetTypeItem& AddFormsOutput(FormsOutputKeyT&& key, FormsOutputValueT&& value) {
    m_formsOutputHasBeenSet = true;
    m_formsOutput.emplace(std::forward<FormsOutputKeyT>(key), std::forward<FormsOutputValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon DataZone project that owns the asset type.</p>
   */
  inline const Aws::String& GetOwningProjectId() const { return m_owningProjectId; }
  inline bool OwningProjectIdHasBeenSet() const { return m_owningProjectIdHasBeenSet; }
  template <typename OwningProjectIdT = Aws::String>
  void SetOwningProjectId(OwningProjectIdT&& value) {
    m_owningProjectIdHasBeenSet = true;
    m_owningProjectId = std::forward<OwningProjectIdT>(value);
  }
  template <typename OwningProjectIdT = Aws::String>
  AssetTypeItem& WithOwningProjectId(OwningProjectIdT&& value) {
    SetOwningProjectId(std::forward<OwningProjectIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon DataZone domain where the asset type was
   * originally created.</p>
   */
  inline const Aws::String& GetOriginDomainId() const { return m_originDomainId; }
  inline bool OriginDomainIdHasBeenSet() const { return m_originDomainIdHasBeenSet; }
  template <typename OriginDomainIdT = Aws::String>
  void SetOriginDomainId(OriginDomainIdT&& value) {
    m_originDomainIdHasBeenSet = true;
    m_originDomainId = std::forward<OriginDomainIdT>(value);
  }
  template <typename OriginDomainIdT = Aws::String>
  AssetTypeItem& WithOriginDomainId(OriginDomainIdT&& value) {
    SetOriginDomainId(std::forward<OriginDomainIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the Amazon DataZone project where the asset type
   * exists.</p>
   */
  inline const Aws::String& GetOriginProjectId() const { return m_originProjectId; }
  inline bool OriginProjectIdHasBeenSet() const { return m_originProjectIdHasBeenSet; }
  template <typename OriginProjectIdT = Aws::String>
  void SetOriginProjectId(OriginProjectIdT&& value) {
    m_originProjectIdHasBeenSet = true;
    m_originProjectId = std::forward<OriginProjectIdT>(value);
  }
  template <typename OriginProjectIdT = Aws::String>
  AssetTypeItem& WithOriginProjectId(OriginProjectIdT&& value) {
    SetOriginProjectId(std::forward<OriginProjectIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp of when the asset type was created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedAt() const { return m_createdAt; }
  inline bool CreatedAtHasBeenSet() const { return m_createdAtHasBeenSet; }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  void SetCreatedAt(CreatedAtT&& value) {
    m_createdAtHasBeenSet = true;
    m_createdAt = std::forward<CreatedAtT>(value);
  }
  template <typename CreatedAtT = Aws::Utils::DateTime>
  AssetTypeItem& WithCreatedAt(CreatedAtT&& value) {
    SetCreatedAt(std::forward<CreatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon DataZone user who created the asset type.</p>
   */
  inline const Aws::String& GetCreatedBy() const { return m_createdBy; }
  inline bool CreatedByHasBeenSet() const { return m_createdByHasBeenSet; }
  template <typename CreatedByT = Aws::String>
  void SetCreatedBy(CreatedByT&& value) {
    m_createdByHasBeenSet = true;
    m_createdBy = std::forward<CreatedByT>(value);
  }
  template <typename CreatedByT = Aws::String>
  AssetTypeItem& WithCreatedBy(CreatedByT&& value) {
    SetCreatedBy(std::forward<CreatedByT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp of when the asset type was updated.</p>
   */
  inline const Aws::Utils::DateTime& GetUpdatedAt() const { return m_updatedAt; }
  inline bool UpdatedAtHasBeenSet() const { return m_updatedAtHasBeenSet; }
  template <typename UpdatedAtT = Aws::Utils::DateTime>
  void SetUpdatedAt(UpdatedAtT&& value) {
    m_updatedAtHasBeenSet = true;
    m_updatedAt = std::forward<UpdatedAtT>(value);
  }
  template <typename UpdatedAtT = Aws::Utils::DateTime>
  AssetTypeItem& WithUpdatedAt(UpdatedAtT&& value) {
    SetUpdatedAt(std::forward<UpdatedAtT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon DataZone user who updated the asset type.</p>
   */
  inline const Aws::String& GetUpdatedBy() const { return m_updatedBy; }
  inline bool UpdatedByHasBeenSet() const { return m_updatedByHasBeenSet; }
  template <typename UpdatedByT = Aws::String>
  void SetUpdatedBy(UpdatedByT&& value) {
    m_updatedByHasBeenSet = true;
    m_updatedBy = std::forward<UpdatedByT>(value);
  }
  template <typename UpdatedByT = Aws::String>
  AssetTypeItem& WithUpdatedBy(UpdatedByT&& value) {
    SetUpdatedBy(std::forward<UpdatedByT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainId;

  Aws::String m_name;

  Aws::String m_revision;

  Aws::String m_description;

  Aws::Map<Aws::String, FormEntryOutput> m_formsOutput;

  Aws::String m_owningProjectId;

  Aws::String m_originDomainId;

  Aws::String m_originProjectId;

  Aws::Utils::DateTime m_createdAt{};

  Aws::String m_createdBy;

  Aws::Utils::DateTime m_updatedAt{};

  Aws::String m_updatedBy;
  bool m_domainIdHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_revisionHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_formsOutputHasBeenSet = false;
  bool m_owningProjectIdHasBeenSet = false;
  bool m_originDomainIdHasBeenSet = false;
  bool m_originProjectIdHasBeenSet = false;
  bool m_createdAtHasBeenSet = false;
  bool m_createdByHasBeenSet = false;
  bool m_updatedAtHasBeenSet = false;
  bool m_updatedByHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataZone
}  // namespace Aws
