"""Logic for ignoring paths."""

import fnmatch
import os
import re


class Ignore:
    def __init__(self, regular, regex_based):
        """Prepare the function that determines whether a path should be ignored."""
        self.regular_patterns = regular
        self.regex_based_patterns = list(map(re.compile, regex_based))

    def __repr__(self):
        return (
            f"Ignore(regular={self.regular_patterns!r}, "
            f"regex_based={self.regex_based_patterns!r})"
        )

    def __call__(self, path):
        """Determine if 'path' should be ignored."""
        # Any regular pattern matches.
        for pattern in self.regular_patterns:
            if path.startswith((pattern + os.sep, pattern + "/")):
                return True
            if fnmatch.fnmatch(path, pattern):
                return True

        # Any regular expression matches.
        for regex in self.regex_based_patterns:
            if regex.search(path):
                return True

        return False
