﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/devops-guru/DevOpsGuruRequest.h>
#include <aws/devops-guru/DevOpsGuru_EXPORTS.h>
#include <aws/devops-guru/model/CostEstimationResourceCollectionFilter.h>

#include <utility>

namespace Aws {
namespace DevOpsGuru {
namespace Model {

/**
 */
class StartCostEstimationRequest : public DevOpsGuruRequest {
 public:
  AWS_DEVOPSGURU_API StartCostEstimationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartCostEstimation"; }

  AWS_DEVOPSGURU_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The collection of Amazon Web Services resources used to create a monthly
   * DevOps Guru cost estimate.</p>
   */
  inline const CostEstimationResourceCollectionFilter& GetResourceCollection() const { return m_resourceCollection; }
  inline bool ResourceCollectionHasBeenSet() const { return m_resourceCollectionHasBeenSet; }
  template <typename ResourceCollectionT = CostEstimationResourceCollectionFilter>
  void SetResourceCollection(ResourceCollectionT&& value) {
    m_resourceCollectionHasBeenSet = true;
    m_resourceCollection = std::forward<ResourceCollectionT>(value);
  }
  template <typename ResourceCollectionT = CostEstimationResourceCollectionFilter>
  StartCostEstimationRequest& WithResourceCollection(ResourceCollectionT&& value) {
    SetResourceCollection(std::forward<ResourceCollectionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The idempotency token used to identify each cost estimate request.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartCostEstimationRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  CostEstimationResourceCollectionFilter m_resourceCollection;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_resourceCollectionHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace DevOpsGuru
}  // namespace Aws
