/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*-
 * 
 * make-iso.c: code to generate iso files
 *
 * Copyright (C) 2002-2004 Red Hat, Inc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authors: Alexander Larsson <alexl@redhat.com>
 */

#include "config.h"

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#include <sys/param.h>
#include <sys/mount.h>
#else
#include <sys/vfs.h>
#endif /* __FreeBSD__ || __NetBSD__ || __OpenBSD__ */
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gtk/gtkmessagedialog.h>

#include "nautilus-cd-burner.h"
#include "make-iso.h"

#ifndef HAVE_MKDTEMP
#include "mkdtemp.h"
#endif

GQuark
nautilus_burn_iso_error_quark (void)
{
	static GQuark quark = 0;
	if (!quark)
		quark = g_quark_from_static_string ("nautilus_burn_iso_error");

	return quark;
}

static gboolean
make_iso_get_free_space (const char       *filename,
			 GnomeVFSFileSize *size)
{
	GnomeVFSURI *uri;

	*size = 0;

	uri = gnome_vfs_uri_new (filename);
	if (uri == NULL)
		return FALSE;

	if (gnome_vfs_get_volume_free_space (uri, size) != GNOME_VFS_OK) {
		gnome_vfs_uri_unref (uri);
		return FALSE;
	}

	gnome_vfs_uri_unref (uri);
	return TRUE;
}

struct mkisofs_output {
	GMainLoop  *loop;
	int         result;
	int         pid;
	const char *filename;
	GError     *error;
	gboolean    debug;
};

static struct mkisofs_output *mkisofs_output_ptr;

/**
 * nautilus_burn_make_iso_cancel:
 *
 * Cancel current creation process
 *
 **/
void
nautilus_burn_make_iso_cancel (void)
{
	if (mkisofs_output_ptr) {
		kill (mkisofs_output_ptr->pid, SIGINT);
		unlink (mkisofs_output_ptr->filename);
		mkisofs_output_ptr->error = g_error_new_literal (NAUTILUS_BURN_ISO_ERROR, 
								 NAUTILUS_BURN_ISO_ERROR_GENERAL,
								 _("The operation was cancelled by the user."));
		g_main_loop_quit (mkisofs_output_ptr->loop);
	}
}

static void
write_all (int   fd,
	   char *buf,
	   int   len)
{
	int bytes;
	int res;
	
	bytes = 0;
	while (bytes < len) {
		res = write (fd, buf + bytes, len - bytes);
		if (res <= 0) {
			return;
		}
		bytes += res;
	}
	return;
}

static void
copy_file (const char *source,
	   const char *dest)
{
	int         sfd, dfd;
	struct stat stat_buf;
	char        buffer [1024*8];
	ssize_t     len;

	if (link (source, dest) == 0) {
		return;
	}
	
	if (stat (source, &stat_buf) != 0) {
		g_warning ("Trying to copy nonexisting file\n");
		return;
	}

	sfd = open (source, O_RDONLY);
	if (sfd == -1) {
		g_warning ("Can't copy file (open source failed)\n");
		return;
	}
	
	dfd = open (dest, O_WRONLY | O_CREAT, stat_buf.st_mode);
	if (dfd == -1) {
		close (sfd);
		g_warning ("Can't copy file (open dest '%s' failed)\n", dest);
		perror ("error:");
		return;
	}

	while ( (len = read (sfd, &buffer, sizeof (buffer))) > 0) {
		write_all (dfd, buffer, len);
	}
	close (dfd);
	close (sfd);
}

static char *
escape_path (const char *str)
{
	char       *escaped, *d;
	const char *s;
	int         len;
	
	s = str;
	len = 1;
	while (*s != 0) {
		if (*s == '\\' ||
		    *s == '=') {
			len++;
		}
		
		len++;
		s++;
	}
	
	escaped = g_malloc (len);
	
	s = str;
	d = escaped;
	while (*s != 0) {
		if (*s == '\\' ||
		    *s == '=') {
			*d++ = '\\';
		}
		
		*d++ = *s++;
	}
	*d = 0;
	
	return escaped;
}

static gboolean
dir_is_empty (const char *virtual_path)
{
	GnomeVFSFileInfo        *info;
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSResult           result;
	char                    *escaped_path, *uri;
	gboolean                 found_file;
	
	escaped_path = gnome_vfs_escape_path_string (virtual_path);
	uri = g_strconcat ("burn:///", escaped_path, NULL);
	g_free (escaped_path);
	
	result = gnome_vfs_directory_open (&handle, uri, GNOME_VFS_FILE_INFO_DEFAULT);
	if (result != GNOME_VFS_OK) {
		g_free (uri);
		return TRUE;
	}
	
	info = gnome_vfs_file_info_new ();

	found_file = FALSE;
	
	while (TRUE) {
		result = gnome_vfs_directory_read_next (handle, info);
		if (result != GNOME_VFS_OK)
			break;

		/* Skip "." and "..".  */
		if (info->name [0] == '.'
		    && (info->name [1] == 0
			|| (info->name [1] == '.' && info->name [2] == 0))) {
			gnome_vfs_file_info_clear (info);
			continue;
		}
		
		found_file = TRUE;
		break;
	}

	gnome_vfs_directory_close (handle);
	gnome_vfs_file_info_unref (info);

	return !found_file;
}

static char *
get_backing_file (const char *virtual_path)
{
	GnomeVFSHandle *handle;
	GnomeVFSResult  res;
	char           *escaped_path, *uri;
	char           *mapping;

	escaped_path = gnome_vfs_escape_path_string (virtual_path);
	uri = g_strconcat ("burn:///", escaped_path, NULL);
	g_free (escaped_path);
	res = gnome_vfs_open (&handle,
			      uri,
			      GNOME_VFS_OPEN_READ);
	g_free (uri);
	if (res == GNOME_VFS_OK) {
		res =  gnome_vfs_file_control (handle,
					       "mapping:get_mapping",
					       &mapping);
		gnome_vfs_close	(handle);
		if (res == GNOME_VFS_OK) {
			return mapping;
		}
	}
	return NULL;
}

static gboolean
ask_disable_joliet (GtkWindow *parent)
{
	GtkWidget *dialog;
	int        res;

	dialog = gtk_message_dialog_new (parent,
					 GTK_DIALOG_DESTROY_WITH_PARENT,
					 GTK_MESSAGE_QUESTION,
					 GTK_BUTTONS_OK_CANCEL,
					 _("Disable Microsoft Windows compatibility?"));
	gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						  _("Some files don't have a suitable name for a Windows-compatible CD.\nDo you want to continue with Windows compatibility disabled?"));
	gtk_window_set_title (GTK_WINDOW (dialog), _("Windows compatibility"));
	res = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
	return (res == GTK_RESPONSE_OK);
}

struct mkisofs_state {
	FILE    *graft_file;
	char    *emptydir;
	int      depth;
	char    *tmpdir;
	char    *copy_to_dir;
	int      copy_depth;
	GList   *remove_files;
	gboolean found_file;
};

static void
graft_file_end_dir_visitor (struct mkisofs_state *state)
{
	char *last_slash;
	
	if (state->copy_depth > 0) {
		state->copy_depth--;
		if (state->copy_depth == 0) {
			g_free (state->copy_to_dir);
			state->copy_to_dir = NULL;
		} else {
			last_slash = strrchr (state->copy_to_dir, '/');
			if (last_slash != NULL) {
				*last_slash = 0;
			}
		}
	}
}

static gboolean
graft_file_visitor (const gchar          *rel_path,
		    GnomeVFSFileInfo     *info,
		    struct mkisofs_state *state,
		    gboolean             *recurse)
{
	char *mapping, *path1, *path2;
	char *new_copy_dir;
	char *copy_path;

	*recurse = TRUE;
	
	if (state->copy_to_dir != NULL) {
		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
			new_copy_dir = g_build_filename (state->copy_to_dir,
							 info->name,
							 NULL);
			g_free (state->copy_to_dir);
			state->copy_to_dir = new_copy_dir;
			mkdir (state->copy_to_dir, 0777);
			state->remove_files = g_list_prepend (state->remove_files, g_strdup (state->copy_to_dir));
			state->copy_depth++;
		} else {
			copy_path = g_build_filename (state->copy_to_dir, info->name, NULL);
			mapping = get_backing_file (rel_path);
			if (mapping != NULL) {
				copy_file (mapping, copy_path);
				state->remove_files = g_list_prepend (state->remove_files, g_strdup (copy_path));
			}
		}
		return TRUE;
	}
	
	if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY) {
		mapping = get_backing_file (rel_path);
		if (mapping != NULL) {
			path1 = escape_path (rel_path);
			path2 = escape_path (mapping);
			state->found_file = TRUE;
			fprintf (state->graft_file, "%s=%s\n", path1, path2);
			g_free (path1);
			g_free (path2);
			g_free (mapping);
		}
	} else {
		if (dir_is_empty (rel_path)) {
			path1 = escape_path (rel_path);
			path2 = escape_path (state->emptydir);
			state->found_file = TRUE;
			fprintf (state->graft_file, "%s/=%s\n", path1, path2);
			g_free (path1);
			g_free (path2);
		} else if (state->depth >= 6) {
			new_copy_dir = g_build_filename (state->tmpdir, "subdir.XXXXXX", NULL);
			copy_path = mkdtemp (new_copy_dir);
			if (copy_path != NULL) {
				state->remove_files = g_list_prepend (state->remove_files, g_strdup (copy_path));
				state->copy_depth = 1;
				state->copy_to_dir = copy_path;
				path1 = escape_path (rel_path);
				path2 = escape_path (copy_path);
				state->found_file = TRUE;
				fprintf (state->graft_file, "%s/=%s\n", path1, path2);
				g_free (path1);
				g_free (path2);
			} else {
				g_free (new_copy_dir);
				g_warning ("Couldn't create temp subdir\n");
				*recurse = FALSE;
			}
		} 
	}

	return TRUE;
}
	
static void
create_graft_file (GnomeVFSURI          *uri,
		   const gchar          *prefix,
		   struct mkisofs_state *state)
{
	GnomeVFSFileInfo        *info;
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSResult           result;
	gboolean                 stop;

	result = gnome_vfs_directory_open_from_uri (&handle, uri, GNOME_VFS_FILE_INFO_DEFAULT);
	if (result != GNOME_VFS_OK)
		return;

	info = gnome_vfs_file_info_new ();

	stop = FALSE;
	while (! stop) {
		gchar *rel_path;
		gboolean recurse;

		result = gnome_vfs_directory_read_next (handle, info);
		if (result != GNOME_VFS_OK)
			break;

		/* Skip "." and "..".  */
		if (info->name [0] == '.'
		    && (info->name [1] == 0
			|| (info->name [1] == '.' && info->name [2] == 0))) {
			gnome_vfs_file_info_clear (info);
			continue;
		}

		if (prefix == NULL)
			rel_path = g_strdup (info->name);
		else
			rel_path = g_strconcat (prefix, info->name, NULL);

		recurse = FALSE;
		stop = ! graft_file_visitor (rel_path, info, state, &recurse);
		
		if (! stop
		    && recurse
		    && info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
			GnomeVFSURI *new_uri;
			gchar *new_prefix;

			if (prefix == NULL)
				new_prefix = g_strconcat (info->name, "/",
							  NULL);
			else
				new_prefix = g_strconcat (prefix, info->name,
							  "/", NULL);

			new_uri = gnome_vfs_uri_append_file_name (uri, info->name);

			state->depth++;
			create_graft_file (new_uri,
					   new_prefix,
					   state);
			state->depth--;
			graft_file_end_dir_visitor (state);

			gnome_vfs_uri_unref (new_uri);
			g_free (new_prefix);
		}

		g_free (rel_path);

		gnome_vfs_file_info_clear (info);

		if (stop)
			break;
	}

	gnome_vfs_directory_close (handle);
	gnome_vfs_file_info_unref (info);
}

static gboolean  
mkisofs_stdout_read (GIOChannel   *source,
		     GIOCondition  condition,
		     gpointer      data)
{
	struct mkisofs_output *mkisofs_output = data;
	char *line;
	GIOStatus status;

	if (condition & G_IO_IN) {
		status = g_io_channel_read_line (source,
						 &line, NULL, NULL, NULL);

		if (line && mkisofs_output->debug) {
			g_print ("make_iso stdout: %s", line);
		}

		if (status == G_IO_STATUS_NORMAL) {
			g_free (line);
		}
	} else if (condition & G_IO_HUP) {
		if (mkisofs_output->debug)
			g_print ("make_iso stdout: HUP\n");
		return FALSE;
	}

	return TRUE;
}

static gboolean  
mkisofs_stderr_read (GIOChannel   *source,
		     GIOCondition  condition,
		     gpointer      data)
{
	struct mkisofs_output *mkisofs_output = data;
	char                  *line;
	char                   fraction_str [7];
	double                 fraction;
	GIOStatus              status;

	if (condition & G_IO_IN) {
		status = g_io_channel_read_line (source, &line, NULL, NULL, NULL);

		if (line && mkisofs_output->debug) {
			g_print ("make_iso stderr: %s", line);
		}

		if (status == G_IO_STATUS_NORMAL) {
			if (strncmp (line, "Total translation table size", 28) == 0) {
				nautilus_burn_progress_set_fraction (1.0);
				mkisofs_output->result = NAUTILUS_BURN_RECORDER_RESULT_FINISHED;
			}

			if (strstr (line, "estimate finish")) {
				if (sscanf (line, "%6c%% done, estimate finish",
					    fraction_str) == 1) {
					fraction_str [6] = 0;
					fraction = g_strtod (fraction_str, NULL);
					nautilus_burn_progress_set_fraction (fraction/100.0);
				}
			}

			if (strstr (line, "Incorrectly encoded string")) {
				nautilus_burn_progress_set_fraction (1.0);
				if (!mkisofs_output->error)
					mkisofs_output->error = g_error_new_literal (NAUTILUS_BURN_ISO_ERROR,
										     NAUTILUS_BURN_ISO_ERROR_GENERAL,
										     _("Incorrectly encoded string."));
				mkisofs_output->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
			}

			if (strstr (line, "Unknown charset")) {
				nautilus_burn_progress_set_fraction (1.0);
				/* FIXME: use better error message after string freeze */
				if (!mkisofs_output->error)
					mkisofs_output->error = g_error_new_literal (NAUTILUS_BURN_ISO_ERROR,
										     NAUTILUS_BURN_ISO_ERROR_GENERAL,
										     _("Incorrectly encoded string."));
				mkisofs_output->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
			}

			if (strstr (line, "No space left on device")) {
				nautilus_burn_progress_set_fraction (1.0);
				if (!mkisofs_output->error)
					mkisofs_output->error = g_error_new_literal (NAUTILUS_BURN_ISO_ERROR,
										     NAUTILUS_BURN_ISO_ERROR_GENERAL,
										     _("There is no space left on the device."));
				mkisofs_output->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
			}

			if (strstr (line, "Value too large for defined data type")) {
				/* TODO: get filename from error message */
				nautilus_burn_progress_set_fraction (1.0);
				if (!mkisofs_output->error)
					mkisofs_output->error = g_error_new_literal (NAUTILUS_BURN_ISO_ERROR,
										     NAUTILUS_BURN_ISO_ERROR_GENERAL,
										     _("File too large for filesystem."));
				mkisofs_output->result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
			}

			g_free (line);
		}
	} else if (condition & G_IO_HUP) {
		/* only handle the HUP when we have read all available lines of output */
		if (mkisofs_output->debug)
			g_print ("make_iso stderr: HUP\n");
		nautilus_burn_progress_set_fraction (1.0);
		g_main_loop_quit (mkisofs_output->loop);
		return FALSE;
	}

	return TRUE;
}

/* this is an ugly hack until utf8/iconv support is added into upstream mkisofs */
static gboolean
ncb_mkisofs_supports_utf8 (void)
{
	char    *standard_error;
	gboolean supported;
	gboolean res;

	res = g_spawn_command_line_sync ("mkisofs -input-charset utf8", NULL, &standard_error, NULL, NULL);
	if (res && !g_strrstr (standard_error, "Unknown charset"))
		supported = TRUE;
	else
		supported = FALSE;

	g_free (standard_error);

	return supported;
}

/**
 * nautilus_burn_make_iso:
 * @filename:
 * @label:
 * @warn_low_space:
 * @use_joliet:
 * @debug:
 *
 * Create an ISO image in filename from the data files in burn:///
 *
 * Return value: 
 **/
int
nautilus_burn_make_iso (const char *filename,
			const char *label,
			gboolean    warn_low_space,
			gboolean    use_joliet,
			gboolean    debug,
			GError    **error)
{
	GnomeVFSURI          *uri;
	char                 *filelist = NULL;
	struct mkisofs_state  state = {NULL};
	char                 *tempdir;
	GList                *l;
	int                   stdout_pipe, stderr_pipe;
	int                   i;
	GError               *sub_error;
	const char           *argv [20]; /* Shouldn't need more than 20 arguments */
	struct mkisofs_output mkisofs_output;
	GIOChannel           *channel;
	guint                 stdout_tag, stderr_tag;
	char                 *stdout_data, *stderr_data;
	char                 *dirname;
	int                   exit_status, res;
	guint64               iso_size;
	GnomeVFSFileSize      size;
	gboolean              has_utf8_support;

	if (label && (strlen (label) > 32)) {
		g_set_error (error,
			     G_FILE_ERROR,
			     0,
			     _("The label for the image is too long."));
		return NAUTILUS_BURN_RECORDER_RESULT_ERROR;
	}

	mkisofs_output.error = NULL;
	mkisofs_output.debug = debug;

	dirname = g_strdup_printf ("iso-%s.XXXXXX", g_get_user_name ());
	tempdir = g_build_filename (g_get_tmp_dir (), dirname, NULL);
	g_free (dirname);
	state.tmpdir = mkdtemp (tempdir);

	mkisofs_output.result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;

	if (state.tmpdir == 0) {
		mkisofs_output.error = g_error_new (NAUTILUS_BURN_ISO_ERROR,
						    NAUTILUS_BURN_ISO_ERROR_GENERAL,
						    _("Unable to create temporary directory: %s."),
						    g_strerror (errno));
		goto cleanup;
	}
	
	state.emptydir = g_build_filename (state.tmpdir, "emptydir", NULL);
	mkdir (state.emptydir, 0777);
	state.remove_files = g_list_prepend (state.remove_files, g_strdup (state.emptydir));

	filelist = g_build_filename (state.tmpdir, "filelist", NULL);

	state.graft_file = fopen (filelist, "w");
	if (state.graft_file == NULL) {
		mkisofs_output.error = g_error_new (NAUTILUS_BURN_ISO_ERROR, 
						    NAUTILUS_BURN_ISO_ERROR_GENERAL,
						    _("Unable to create temporary file: %s."),
						    g_strerror (errno));
		goto cleanup;
	}
	
	uri = gnome_vfs_uri_new ("burn:///");
	state.found_file = FALSE;
	create_graft_file (uri, NULL, &state);
	gnome_vfs_uri_unref (uri);

	fclose (state.graft_file);
	state.remove_files = g_list_prepend (state.remove_files, g_strdup (filelist));

	if (!state.found_file) {
		mkisofs_output.error = g_error_new_literal (NAUTILUS_BURN_ISO_ERROR, 
							    NAUTILUS_BURN_ISO_ERROR_GENERAL,
							    _("There are no files to write to disc."));
		mkisofs_output.result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		goto cleanup;
	}

	has_utf8_support = ncb_mkisofs_supports_utf8 ();

 retry:
	i = 0;
	argv [i++] = "mkisofs";
	argv [i++] = "-r";
	if (use_joliet) {
		argv [i++] = "-J";
	}
	if (has_utf8_support) {
		argv [i++] = "-input-charset";
		argv [i++] = "utf8";
	}
	argv [i++] = "-q";
	argv [i++] = "-graft-points";
	argv [i++] = "-path-list";
	argv [i++] = filelist;
	argv [i++] = "-print-size";
	argv [i++] = NULL;

	sub_error = NULL;

	if (debug) {
		g_print ("launching command: ");
		for (i = 0; argv [i] != NULL; i++) {
			g_print ("%s ", argv [i]);
		}
		g_print ("\n");
	}

	if (!g_spawn_sync (NULL,
			   (char **)argv,
			   NULL,
			   G_SPAWN_SEARCH_PATH,
			   NULL, NULL,
			   &stdout_data,
			   &stderr_data,
			   &exit_status,
			   &sub_error)) {
		mkisofs_output.error = g_error_new (NAUTILUS_BURN_ISO_ERROR, 
						    NAUTILUS_BURN_ISO_ERROR_GENERAL,
						    _("Could not run sub process: %s."),
						    sub_error->message);
		g_error_free (sub_error);
		mkisofs_output.result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		goto cleanup;
	}

	if (exit_status != 0 && use_joliet) {
		if (strstr (stderr_data, "Joliet tree sort failed.") != NULL) {
			g_free (stdout_data);
			g_free (stderr_data);
			if (ask_disable_joliet (nautilus_burn_progress_get_window ())) {
				use_joliet = FALSE;
				goto retry;
			} else {
				mkisofs_output.result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
				mkisofs_output.error = g_error_new_literal (NAUTILUS_BURN_ISO_ERROR, 
									    NAUTILUS_BURN_ISO_ERROR_GENERAL,
									    _("The operation was cancelled by the user."));
				goto cleanup;
			}
		}
	}

	g_free (stderr_data);
	iso_size = (guint64)atol (stdout_data) * 2048 ; /* mkisofs reports blocks of 2048 bytes */
	g_free (stdout_data);

	dirname = g_path_get_dirname (filename);
	res = make_iso_get_free_space (dirname, &size);

	if (res == -1) {
		g_warning ("Cannot get free space at %s\n", dirname);
		g_free (dirname);
	} else if (iso_size > size) {
		g_free (dirname);
		if (warn_low_space) {
			mkisofs_output.error = g_error_new (NAUTILUS_BURN_ISO_ERROR, 
							    NAUTILUS_BURN_ISO_ERROR_GENERAL,
							    _("The selected location does not have enough space to store the disc image (%ld MiB needed)."),
							    (unsigned long)iso_size / 1048576);
			mkisofs_output.result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		} else {
			mkisofs_output.result = NAUTILUS_BURN_RECORDER_RESULT_RETRY;
		}

		goto cleanup;
	}

	i = 0;
	argv [i++] = "mkisofs";
	argv [i++] = "-r";
	if (use_joliet) {
		argv [i++] = "-J";
	}
	if (has_utf8_support) {
		argv [i++] = "-input-charset";
		argv [i++] = "utf8";
	}
	argv [i++] = "-graft-points";
	argv [i++] = "-path-list";
	argv [i++] = filelist;
	if (label) {
		argv [i++] = "-V";
		argv [i++] = label;
	}
	argv [i++] = "-o";
	argv [i++] = filename;
	argv [i++] = NULL;

	if (debug) {
		g_print ("launching command: ");
		for (i = 0; argv [i] != NULL; i++) {
			g_print ("%s ", argv [i]);
		}
		g_print ("\n");
	}

	nautilus_burn_progress_set_text (_("Creating CD image"));
	nautilus_burn_progress_set_fraction (0.0);
	nautilus_burn_progress_set_image_spinning (TRUE);
	sub_error = NULL;
	if (!g_spawn_async_with_pipes  (NULL,
					(char **)argv,
					NULL,
					G_SPAWN_SEARCH_PATH,
					NULL, NULL,
					&mkisofs_output.pid,
					/*stdin*/NULL,
					&stdout_pipe,
					&stderr_pipe,
					&sub_error)) {
		mkisofs_output.error = g_error_new (NAUTILUS_BURN_ISO_ERROR, 
						    NAUTILUS_BURN_ISO_ERROR_GENERAL,
						    _("Command failed: %s"),
						    sub_error->message);
		g_error_free (sub_error);
		mkisofs_output.result = NAUTILUS_BURN_RECORDER_RESULT_ERROR;
		goto cleanup;
	} else {
		mkisofs_output.loop = g_main_loop_new (NULL, FALSE);
	
		channel = g_io_channel_unix_new (stdout_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stdout_tag = g_io_add_watch (channel, 
					     (G_IO_IN | G_IO_HUP | G_IO_ERR), 
					     mkisofs_stdout_read,
					     &mkisofs_output);
		g_io_channel_unref (channel);
		channel = g_io_channel_unix_new (stderr_pipe);
		g_io_channel_set_encoding (channel, NULL, NULL);
		stderr_tag = g_io_add_watch (channel, 
					     (G_IO_IN | G_IO_HUP | G_IO_ERR), 
					     mkisofs_stderr_read,
					     &mkisofs_output);
		g_io_channel_unref (channel);

		mkisofs_output_ptr = &mkisofs_output;
		mkisofs_output.filename = filename;
		
		g_main_loop_run (mkisofs_output.loop);
		g_main_loop_unref (mkisofs_output.loop);
		
		g_source_remove (stdout_tag);
		g_source_remove (stderr_tag);
	}
	mkisofs_output_ptr = NULL;

 cleanup:
	for (l = state.remove_files; l != NULL; l = l->next) {
		remove ((char *)l->data);
		g_free (l->data);
	}
	g_list_free (state.remove_files);
	
	g_free (filelist);
	g_free (state.emptydir);
	rmdir (tempdir);
	g_free (tempdir);
	nautilus_burn_progress_set_image_spinning (FALSE);

	if (mkisofs_output.result == NAUTILUS_BURN_RECORDER_RESULT_ERROR) {
		if (!mkisofs_output.error)
			mkisofs_output.error = g_error_new_literal (NAUTILUS_BURN_ISO_ERROR,
								    NAUTILUS_BURN_ISO_ERROR_GENERAL,
								    _("Unknown error"));

		g_propagate_error (error, mkisofs_output.error);
	}

	return mkisofs_output.result;
}

/* Originally eel_make_valid_utf8 */
static char *
ncb_make_valid_utf8 (const char *name)
{
	GString    *string;
	const char *remainder, *invalid;
	int         remaining_bytes, valid_bytes;

	string = NULL;
	remainder = name;
	remaining_bytes = strlen (name);

	while (remaining_bytes != 0) {
		if (g_utf8_validate (remainder, remaining_bytes, &invalid)) {
			break;
		}
		valid_bytes = invalid - remainder;

		if (string == NULL) {
			string = g_string_sized_new (remaining_bytes);
		}
		g_string_append_len (string, remainder, valid_bytes);
		g_string_append_c (string, '?');

		remaining_bytes -= valid_bytes + 1;
		remainder = invalid + 1;
	}

	if (string == NULL) {
		return g_strdup (name);
	}

	g_string_append (string, remainder);
	g_string_append (string, _(" (invalid Unicode)"));
	g_assert (g_utf8_validate (string->str, -1, NULL));

	return g_string_free (string, FALSE);
}

/**
 * nautilus_burn_verify_iso:
 * @filename:
 * @iso_label:
 * @error:
 *
 * Verify that filename is a valid ISO image
 *
 * Return value: %TRUE if filename is a valid ISO image, otherwise %FALSE
 **/
gboolean
nautilus_burn_verify_iso (const char *filename,
			  char      **iso_label,
			  GError    **error)
{
	FILE  *file;
#define BUFFER_SIZE 128
	char  buf [BUFFER_SIZE+1];
	int   res;
	char *str, *str2;

	file = fopen (filename, "rb");
	if (file == NULL) {
		int err = errno;
		*error = g_error_new_literal (g_file_error_quark (),
					      g_file_error_from_errno (err),
					      strerror (err));
		return FALSE;
	}
	/* Verify we have an ISO image */
	/* This check is for the raw sector images */
	res = fseek (file, 37633L, SEEK_SET);
	if (res) {
		goto bail;
	}
	res = fread (buf, sizeof (char), 5, file);
	if (res != 5 || strncmp (buf, "CD001", 5) != 0) {
		/* Standard ISO images */
		res = fseek (file, 32769L, SEEK_SET);
		if (res) {
			goto bail;
		}
		res = fread (buf, sizeof (char), 5, file);
		if (res != 5 || strncmp (buf, "CD001", 5) != 0) {
			/* High Sierra images */
			res = fseek (file, 32776L, SEEK_SET);
			if (res) {
				goto bail;
			}
			res = fread (buf, sizeof (char), 5, file);
			if (res != 5 || strncmp (buf, "CDROM", 5) != 0) {
				goto bail;
			}
		}
	}
	/* Extract the volume label from the image */
	res = fseek (file, 32808L, SEEK_SET);
	if (res) {
		goto bail;
	}
	res = fread (buf, sizeof(char), BUFFER_SIZE, file);
	if (res != BUFFER_SIZE) {
		goto bail;
	}
	buf [BUFFER_SIZE] = '\0';
	str = g_strdup (g_strstrip (buf));
	if (!g_utf8_validate (str, -1, NULL)) {
		/* Hmm, not UTF-8. Try the current locale. */
		str2 = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
		if (str2 == NULL) {
			str2 = ncb_make_valid_utf8 (str);
		}
		g_free (str);
		str = str2;
	}
	fclose (file);
	*iso_label = str;
	return TRUE;

 bail:
	*error = g_error_new_literal (NAUTILUS_BURN_ISO_ERROR,
				      NAUTILUS_BURN_ISO_ERROR_GENERAL,
				      _("Not a valid CD image."));

	return FALSE;
}

