/*
 * Copyright (C) 2003 Sun Microsystems, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * Authors:
 *      Mark McLoughlin <mark@skynet.ie>
 */

#include <config.h>

#include "netstatus-dialog.h"

#include <string.h>
#include <glade/glade-xml.h>
#include <libgnome/gnome-i18n.h>
#include <gconf/gconf-client.h>

#include "netstatus-icon.h"
#include "netstatus-util.h"

#define NETWORK_CONFIG_TOOL_DIR "/apps/netstatus_applet"
#define NETWORK_CONFIG_TOOL_KEY NETWORK_CONFIG_TOOL_DIR "/config_tool"

static const char *network_config_tools[] = {
  "network-admin --configure %i",
  "redhat-config-network",
  "system-control-network"
};

#define UNKNOWN_STR(t,s) G_STMT_START {       \
	if (!((t) = (s))) (t) = _("Unknown"); \
} G_STMT_END

typedef struct
{
  GtkWidget      *dialog;

  NetstatusIface *iface;
  NetstatusIcon  *icon;

  GConfClient    *client;
  char           *config_tool;
  guint           listener;

  GtkWidget      *name;
  GtkWidget      *status;
  GtkWidget      *received;
  GtkWidget      *sent;
  GtkWidget      *configure_button;

  GtkWidget      *inet4_frame;
  GtkWidget      *inet4_table;
  GtkWidget      *inet4_addr;
  GtkWidget      *inet4_addr_title;
  GtkWidget      *inet4_dest;
  GtkWidget      *inet4_dest_title;
  GtkWidget      *inet4_bcast;
  GtkWidget      *inet4_bcast_title;
  GtkWidget      *inet4_mask;
  GtkWidget      *inet4_mask_title;

  GtkWidget      *dev_frame;
  GtkWidget      *dev_type;
  GtkWidget      *dev_addr;

  guint           iface_list_monitor;
  int             n_ifaces;
} NetstatusDialogData;


static void
netstatus_dialog_update_state (NetstatusDialogData *data)
{
  NetstatusState state;
  
  state = netstatus_iface_get_state (data->iface);
  gtk_label_set_text (GTK_LABEL (data->status),
		      netstatus_get_state_string (state));
}

static void
netstatus_dialog_update_name (NetstatusDialogData *data)
{
  const char *iface_name;
  const char *text;

  iface_name = netstatus_iface_get_name (data->iface);
  if (!iface_name)
    {
      gtk_window_set_title (GTK_WINDOW (data->dialog),
			    _("Connection Properties"));
    }
  else
    {
      char *p;

      p = g_strdup_printf (_("Connection Properties: %s"), iface_name);
      gtk_window_set_title (GTK_WINDOW (data->dialog), p);
      g_free (p);
    }
  
  UNKNOWN_STR (text, iface_name);
  gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (data->name)->entry), text);
}

static inline void
print_packets_string (GString *str,
		      gulong   packets)
{
  g_string_printf (str, _("%lu packets"), packets);
}

static inline void
print_bytes_string (GString *str,
		    gulong   bytes)
{
  bytes *= 10;

  if (bytes > 1048576)
    {
      bytes = bytes / 1045576;

      g_string_append_printf (str, " (%ld.%ld Mb)", bytes / 10, bytes % 10);
    }
  else if (bytes > 1024)
    {
      bytes = bytes / 1024;

      g_string_append_printf (str, " (%ld.%ld Kb)", bytes / 10, bytes % 10);
    }
  else if (bytes >= 0)
    {
      g_string_append_printf (str, " (%ld.%ld b)", bytes / 10, bytes % 10);
    }
}

static void
netstatus_dialog_update_activity (NetstatusDialogData *data)
{
  GString *str;
  long     in_packets = 0;
  long     out_packets = 0;
  long     in_bytes = 0;
  long     out_bytes = 0;

  netstatus_iface_get_packet_statistics (data->iface, &in_packets, &out_packets);
  netstatus_iface_get_byte_statistics (data->iface, &in_bytes, &out_bytes);

  str = g_string_new (NULL);

  print_packets_string (str, out_packets);
  print_bytes_string (str, out_bytes);
  gtk_label_set_text (GTK_LABEL (data->sent), str->str);
	
  print_packets_string (str, in_packets);
  print_bytes_string (str, in_bytes);
  gtk_label_set_text (GTK_LABEL (data->received), str->str);
  
  g_string_free (str, TRUE);
}

static void
netstatus_dialog_update_inet4_support (NetstatusDialogData *data)
{
  char *addr = NULL;
  char *dest = NULL;
  char *bcast = NULL;
  char *mask = NULL;

  if (netstatus_iface_get_inet4_details (data->iface, &addr, &dest, &bcast, &mask))
    {
      gtk_widget_show (data->inet4_frame);

      /* Address */
      if (addr)
	{
	  gtk_table_set_row_spacing (GTK_TABLE (data->inet4_table), 0, 6);
	  gtk_label_set_text (GTK_LABEL (data->inet4_addr), addr);
	  gtk_widget_show (data->inet4_addr);
	  gtk_widget_show (data->inet4_addr_title);
	}
      else
	{
	  gtk_table_set_row_spacing (GTK_TABLE (data->inet4_table), 0, 0);
	  gtk_widget_hide (data->inet4_addr);
	  gtk_widget_hide (data->inet4_addr_title);
	}

      /* Destination */
      if (dest)
	{
	  gtk_table_set_row_spacing (GTK_TABLE (data->inet4_table), 1, 6);
	  gtk_label_set_text (GTK_LABEL (data->inet4_dest), dest);
	  gtk_widget_show (data->inet4_dest);
	  gtk_widget_show (data->inet4_dest_title);
	}
      else
	{
	  gtk_table_set_row_spacing (GTK_TABLE (data->inet4_table), 1, 0);
	  gtk_widget_hide (data->inet4_dest);
	  gtk_widget_hide (data->inet4_dest_title);
	}

      /* Broadcast */
      if (bcast)
	{
	  gtk_table_set_row_spacing (GTK_TABLE (data->inet4_table), 2, 6);
	  gtk_label_set_text (GTK_LABEL (data->inet4_bcast), bcast);
	  gtk_widget_show (data->inet4_bcast);
	  gtk_widget_show (data->inet4_bcast_title);
	}
      else
	{
	  gtk_table_set_row_spacing (GTK_TABLE (data->inet4_table), 2, 0);
	  gtk_widget_hide (data->inet4_bcast);
	  gtk_widget_hide (data->inet4_bcast_title);
	}

      /* Subnet Mask */
      if (mask)
	{
	  gtk_label_set_text (GTK_LABEL (data->inet4_mask), mask);
	  gtk_widget_show (data->inet4_mask);
	  gtk_widget_show (data->inet4_mask_title);
	}
      else
	{
	  gtk_widget_hide (data->inet4_mask);
	  gtk_widget_hide (data->inet4_mask_title);
	}

      g_free (addr);
      g_free (dest);
      g_free (bcast);
      g_free (mask);
    }
  else
    {
      gtk_widget_hide (data->inet4_frame);
    }
}

static void
netstatus_dialog_update_device_support (NetstatusDialogData *data)
{
  const char *type = NULL;
  char        *addr = NULL;

  if (netstatus_iface_get_device_details (data->iface, &type, &addr))
    {
      g_assert (type != NULL);

      gtk_widget_show (data->dev_frame);

      /* Device type */
      gtk_label_set_text (GTK_LABEL (data->dev_type), type);
      
      /* Address */
      if (addr)
	gtk_label_set_text (GTK_LABEL (data->dev_addr), addr);
      else
	gtk_label_set_text (GTK_LABEL (data->dev_addr), _("Unknown"));

      g_free (addr);
    }
  else
    {
      gtk_widget_hide (data->dev_frame);
    }
}

static void
netstatus_dialog_iface_state_changed (NetstatusIface      *iface,
				      GParamSpec          *pspec,
				      NetstatusDialogData *data)
{
  netstatus_dialog_update_state (data);
  netstatus_dialog_update_activity (data);
  
  netstatus_dialog_update_inet4_support (data);
  netstatus_dialog_update_device_support (data);
}

static void
netstatus_dialog_iface_name_changed (NetstatusIface      *iface,
				     GParamSpec          *pspec,
				     NetstatusDialogData *data)
{
  netstatus_dialog_update_name (data);
}

static void
netstatus_dialog_set_iface_name (NetstatusDialogData *data,
				 GtkEntry            *entry)
{
  const char *iface_name;

  iface_name = gtk_entry_get_text (entry);
  if (iface_name && iface_name [0])
    {
      netstatus_iface_set_name (data->iface, iface_name);

      gtk_widget_set_sensitive (data->configure_button,
				!netstatus_iface_get_is_loopback (data->iface));
    }
}

static void
netstatus_dialog_response (GtkWidget *dialog)
{
  gtk_widget_hide (dialog);
}

static void
netstatus_dialog_destroy (GtkWidget *dialog)
{
  NetstatusDialogData *data;

  data = g_object_get_data (G_OBJECT (dialog), "netstatus-dialog-data");
  if (!data)
    return;

  if (data->config_tool)
    g_free (data->config_tool);
  data->config_tool = NULL;

  if (data->listener)
    {
      g_assert (data->client != NULL);

      gconf_client_notify_remove (data->client, data->listener);
      data->listener = 0;

      gconf_client_remove_dir (data->client, NETWORK_CONFIG_TOOL_DIR, NULL);
    }

  if (data->client)
    g_object_unref (data->client);
  data->client = NULL;

  if (data->iface_list_monitor)
    g_source_remove (data->iface_list_monitor);
  data->iface_list_monitor = 0;

  if (data->iface)
    g_object_unref (data->iface);
  data->iface = NULL;

  g_free (data);
}

static gboolean
netstatus_dialog_check_config_tool (NetstatusDialogData *dialog_data,
				    const char          *config_tool)
{
  char     **argv = NULL;
  gboolean   found = FALSE;

  if (config_tool && g_shell_parse_argv (config_tool, NULL, &argv, NULL))
    {
      char *path;

      g_assert (argv != NULL);

      if ((path = g_find_program_in_path (argv [0])))
	{
	  if (dialog_data->config_tool)
	    g_free (dialog_data->config_tool);
	  dialog_data->config_tool = g_strdup (config_tool);
	  found = TRUE;
	}

      g_free (path);
      g_strfreev (argv);
    }

  return found;
}

static void
netstatus_dialog_config_tool_notify (GConfClient         *client,
				     guint                cnx_id,
				     GConfEntry          *entry,
				     NetstatusDialogData *dialog_data)
{
  if (!entry->value || !entry->value->type == GCONF_VALUE_STRING)
    return;

  netstatus_dialog_check_config_tool (dialog_data,
				      gconf_value_get_string (entry->value));
}

static void
netstatus_dialog_detect_configuration_tool (NetstatusDialogData *dialog_data)
{
  char *config_tool;
  int   i;

  dialog_data->client = gconf_client_get_default ();

  gconf_client_add_dir (dialog_data->client,
			NETWORK_CONFIG_TOOL_DIR,
			GCONF_CLIENT_PRELOAD_NONE,
			NULL);

  dialog_data->listener =
    gconf_client_notify_add (dialog_data->client,
			     NETWORK_CONFIG_TOOL_KEY,
			     (GConfClientNotifyFunc) netstatus_dialog_config_tool_notify,
			     dialog_data, NULL, NULL);

  config_tool = gconf_client_get_string (dialog_data->client,
					 NETWORK_CONFIG_TOOL_KEY,
					 NULL);
  if (netstatus_dialog_check_config_tool (dialog_data, config_tool))
    {
      g_free (config_tool);
      return;
    }

  for (i = 0; i < G_N_ELEMENTS (network_config_tools); i++)
    {
      if (netstatus_dialog_check_config_tool (dialog_data, network_config_tools [i]))
	break;
    }
}

static void
netstatus_iface_configure (GtkWidget           *configure_button,
			   NetstatusDialogData *dialog_data)
{
  GError     *error;
  GdkScreen  *screen;
  GString    *command;
  char      **argv = NULL;
  int         i;

  g_return_if_fail (dialog_data != NULL);
  g_return_if_fail (dialog_data->config_tool != NULL);

  g_shell_parse_argv (dialog_data->config_tool, NULL, &argv, NULL);

  /* We've already verified config_tool as valid */
  g_assert (argv != NULL);

  command = g_string_new (argv [0]);

  for (i = 1; argv [i]; i++)
    {
      command = g_string_append_c (command, ' ');

      if (!strcmp (argv [i], "%i"))
	command = g_string_append (command,
				   netstatus_iface_get_name (dialog_data->iface));
      else
	command = g_string_append (command, argv [i]);
    }

  screen = gtk_window_get_screen (GTK_WINDOW (dialog_data->dialog));

  error = NULL;
  if (!gdk_spawn_command_line_on_screen (screen, command->str, &error))
    {
      GtkWidget *error_dialog;

      error_dialog = gtk_message_dialog_new (NULL,
					     GTK_DIALOG_DESTROY_WITH_PARENT,
					     GTK_MESSAGE_ERROR,
					     GTK_BUTTONS_OK,
					     _("Failed to launch time configuration tool: %s"),
					     error->message);
      g_signal_connect (error_dialog, "response",
			G_CALLBACK (gtk_widget_destroy), NULL);

      gtk_window_set_resizable (GTK_WINDOW (error_dialog), FALSE);
      gtk_window_set_screen (GTK_WINDOW (error_dialog), screen);
      
      gtk_widget_show_all (error_dialog);

      g_error_free (error);
    }

  g_string_free (command, TRUE);
  g_strfreev (argv);
}

static void
netstatus_dialog_setup_configure_button (NetstatusDialogData *data,
					 GladeXML            *xml)
{
  data->configure_button = glade_xml_get_widget (xml, "configure_button");

  g_signal_connect (data->configure_button, "clicked",
		    G_CALLBACK (netstatus_iface_configure), data);

  netstatus_dialog_detect_configuration_tool (data);
  
  if (!data->config_tool)
    gtk_widget_hide (data->configure_button);

  gtk_widget_set_sensitive (data->configure_button,
			    !netstatus_iface_get_is_loopback (data->iface));
}

static void
netstatus_dialog_setup_connection (NetstatusDialogData *data,
				   GladeXML            *xml)
{
  GtkWidget *hbox;
  GtkWidget *icon;
  
  hbox = glade_xml_get_widget (xml, "connection_hbox");

  icon = netstatus_icon_new (data->iface);
  netstatus_icon_set_tooltips_enabled (NETSTATUS_ICON (icon), FALSE);
  gtk_box_pack_end (GTK_BOX (hbox), icon, FALSE, TRUE, 4);
  gtk_widget_show (icon);

  data->icon = NETSTATUS_ICON (icon);

  data->name   = glade_xml_get_widget (xml, "name_combo");
  data->status = glade_xml_get_widget (xml, "status_label");

  netstatus_dialog_update_name (data);
  netstatus_dialog_update_state (data);
}

static void
netstatus_dialog_setup_activity (NetstatusDialogData *data,
				 GladeXML            *xml)
{
  data->sent     = glade_xml_get_widget (xml, "sent_label");
  data->received = glade_xml_get_widget (xml, "received_label");

  netstatus_dialog_update_activity (data);
}

static void
netstatus_dialog_setup_inet4_support (NetstatusDialogData *data,
				      GladeXML            *xml)
{
  data->inet4_frame       = glade_xml_get_widget (xml, "inet4_frame");
  data->inet4_table       = glade_xml_get_widget (xml, "inet4_table");
  data->inet4_addr        = glade_xml_get_widget (xml, "inet4_addr_label");
  data->inet4_addr_title  = glade_xml_get_widget (xml, "inet4_addr_title");
  data->inet4_dest        = glade_xml_get_widget (xml, "inet4_dest_label");
  data->inet4_dest_title  = glade_xml_get_widget (xml, "inet4_dest_title");
  data->inet4_bcast       = glade_xml_get_widget (xml, "inet4_bcast_label");
  data->inet4_bcast_title = glade_xml_get_widget (xml, "inet4_bcast_title");
  data->inet4_mask        = glade_xml_get_widget (xml, "inet4_mask_label");
  data->inet4_mask_title  = glade_xml_get_widget (xml, "inet4_mask_title");

  netstatus_dialog_update_inet4_support (data);
}
static void
netstatus_dialog_setup_device_support (NetstatusDialogData *data,
				       GladeXML            *xml)
{
  data->dev_frame = glade_xml_get_widget (xml, "dev_frame");
  data->dev_type  = glade_xml_get_widget (xml, "dev_type_label");
  data->dev_addr  = glade_xml_get_widget (xml, "dev_addr_label");
  
  netstatus_dialog_update_device_support (data);
}

static void
netstatus_dialog_set_icon (GtkWidget *dialog)
{
  gtk_window_set_icon_from_file (GTK_WINDOW (dialog),
				 NETSTATUS_ICONDIR "/gnome-netstatus-tx.png",
				 NULL);
}

static gboolean
netstatus_dialog_iface_list_monitor (NetstatusDialogData *data)
{
  GList *iface_names, *l;
  int    n_ifaces;

  iface_names = netstatus_list_interface_names (NULL);

  n_ifaces = g_list_length (iface_names);

  if (data->n_ifaces != n_ifaces)
    {
      g_signal_handlers_block_by_func (GTK_COMBO (data->name)->entry,
				       G_CALLBACK (netstatus_dialog_set_iface_name), data);
				       
      gtk_combo_set_popdown_strings (GTK_COMBO (data->name), iface_names);
      netstatus_dialog_update_name (data);

      g_signal_handlers_unblock_by_func (GTK_COMBO (data->name)->entry,
					 G_CALLBACK (netstatus_dialog_set_iface_name), data);
    }

  data->n_ifaces = n_ifaces;

  for (l = iface_names; l; l = l->next)
    g_free (l->data);
  g_list_free (iface_names);

  return TRUE;
}

GtkWidget *
netstatus_dialog_new (NetstatusIface *iface)
{
  NetstatusDialogData *data;
  GladeXML            *xml;

  xml = glade_xml_new (NETSTATUS_GLADEDIR "/gnome-netstatus.glade", NULL, NULL);
  if (!xml)
    return NULL;

  data = g_new0 (NetstatusDialogData, 1);

  data->dialog = glade_xml_get_widget (xml, "network_status_dialog");
  g_object_set_data (G_OBJECT (data->dialog), "netstatus-dialog-data", data);

  netstatus_dialog_set_icon (data->dialog);

  data->iface = g_object_ref (iface);
  netstatus_connect_signal_while_alive (data->iface,
					"notify::state",
					G_CALLBACK (netstatus_dialog_iface_state_changed),
					data,
					data->dialog);

  netstatus_connect_signal_while_alive (data->iface,
					"notify::name",
					G_CALLBACK (netstatus_dialog_iface_name_changed),
					data,
					data->dialog);

  g_signal_connect (data->dialog, "response",
		    G_CALLBACK (netstatus_dialog_response), NULL);

  g_signal_connect (data->dialog, "destroy",
		    G_CALLBACK (netstatus_dialog_destroy), NULL);

  netstatus_dialog_setup_connection (data, xml);
  netstatus_dialog_setup_activity (data, xml);
  netstatus_dialog_setup_inet4_support (data, xml);
  netstatus_dialog_setup_device_support (data, xml);
  netstatus_dialog_setup_configure_button (data, xml);

  data->iface_list_monitor = g_timeout_add (2 * 1000,
					    (GSourceFunc) netstatus_dialog_iface_list_monitor,
					    data);
  netstatus_dialog_iface_list_monitor (data);

  g_signal_connect_swapped (GTK_COMBO (data->name)->entry, "changed",
			    G_CALLBACK (netstatus_dialog_set_iface_name),
			    data);
  g_object_unref (xml);

  return data->dialog;
}
