/*
 * flist-entry.c: an autocompleting entry control
 *
 * Authors:
 *    Jacob Berkman  <jacob@ximian.com>
 *
 * Copyright 2001 Ximian, Inc.
 */

#include <config.h>

#include "flist-entry.h"

#include <bonobo/bonobo-control.h>
#include <bonobo/bonobo-exception.h>
#include <bonobo/bonobo-property-bag.h>

#include <gal/e-text/e-entry.h>

#include <gtk/gtksignal.h>

#include <libgnome/gnome-i18n.h>

enum {
	PROP_TEXT
};

struct _FListEntryPrivate {
	FList             *flist;
	EEntry            *entry;
	BonoboPropertyBag *pb;
};

static void
on_entry_activate (EEntry *eentry, FListEntry *fentry)
{
	flist_request_uri (fentry->priv->flist, 
			   e_entry_get_text (eentry),
			   FLIST_REQUEST_TRY_RELATIVE);
}

static void
on_entry_changed (EEntry *eentry, FListEntry *fentry)
{
	BonoboArg *arg;

	arg = bonobo_arg_new (BONOBO_ARG_STRING);
	BONOBO_ARG_SET_STRING (arg, e_entry_get_text (eentry));
	
	bonobo_property_bag_notify_listeners (fentry->priv->pb,
					      "Text", arg,
					      NULL);

	bonobo_arg_release (arg);
}

void
flist_entry_free (FListEntry *fentry)
{
	g_message ("freeing FListEntry...");
	
	flist_completion_free (fentry->completion);
	bonobo_object_unref (BONOBO_OBJECT (fentry->priv->pb));

	bonobo_object_unref (fentry->control);

	memset (fentry->priv, 0, sizeof (FListEntryPrivate));
	g_free (fentry->priv);

	memset (fentry, 0, sizeof (FListEntry));
	g_free (fentry);
}

static void
entry_get_prop (BonoboPropertyBag *pb,
		BonoboArg         *arg,
		guint              arg_id,
		CORBA_Environment *ev,
		FListEntry        *entry)
{
	if (arg_id == PROP_TEXT) {
		const char *s;
		s = e_entry_get_text (E_ENTRY (entry->completion->entry));
		BONOBO_ARG_SET_STRING (arg, s);
	} else {
		bonobo_exception_set (ev, ex_Bonobo_PropertyBag_NotFound);
	}
}

static void
entry_set_prop (BonoboPropertyBag *pb,
		BonoboArg         *arg,
		guint              arg_id,
		CORBA_Environment *ev,
		FListEntry        *entry)
{
	if (arg_id == PROP_TEXT) {
		char *s;
		s = BONOBO_ARG_GET_STRING (arg);
		e_entry_set_text (E_ENTRY (entry->completion->entry), s);
		/* activate flist? */
	} else {
		bonobo_exception_set (ev, ex_Bonobo_PropertyBag_NotFound);
	}
}

FListEntry *
flist_entry_new (FList *flist)
{
	FListEntry *entry;
	GtkWidget *w;

	g_return_val_if_fail (flist != NULL, NULL);

	entry       = g_new0 (FListEntry, 1);
	entry->priv = g_new0 (FListEntryPrivate, 1);

	entry->priv->flist = flist;

	entry->completion = flist_completion_new ();
	w = entry->completion->entry;
	entry->control = BONOBO_OBJECT (bonobo_control_new (w));
	gtk_widget_show (w);

	gtk_signal_connect (GTK_OBJECT (w), "activate",
			    GTK_SIGNAL_FUNC (on_entry_activate),
			    entry);

	gtk_signal_connect (GTK_OBJECT (w), "changed",
			    GTK_SIGNAL_FUNC (on_entry_changed),
			    entry);

	entry->priv->pb = bonobo_property_bag_new ((BonoboPropertyGetFn)entry_get_prop, 
						   (BonoboPropertySetFn)entry_set_prop, 
						   entry);

	bonobo_property_bag_add (entry->priv->pb, "Text", PROP_TEXT,
				 BONOBO_ARG_STRING, NULL,
				 _("Text in the entry"),
				 BONOBO_PROPERTY_READABLE | BONOBO_PROPERTY_WRITEABLE);

	bonobo_control_set_properties (BONOBO_CONTROL (entry->control), 
				       entry->priv->pb);

	return entry;
}
