/**
 * <copyright>
 *
 * Copyright (c) 2005, 2006, 2007, 2008 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal
 * </copyright>
 *
 * $Id: FeatureMapWrapper.java,v 1.8 2008/02/28 07:09:02 mtaal Exp $
 */

package org.eclipse.emf.teneo.jpox.elist;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.jdo.JDOHelper;
import javax.jdo.spi.PersistenceCapable;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.emf.teneo.jpox.JpoxStoreException;
import org.eclipse.emf.teneo.jpox.JpoxUtil;
import org.eclipse.emf.teneo.jpox.resource.JPOXResource;
import org.eclipse.emf.teneo.mapping.elist.PersistableEList;
import org.eclipse.emf.teneo.mapping.elist.PersistableFeatureMap;
import org.eclipse.emf.teneo.resource.StoreResource;
import org.eclipse.emf.teneo.type.FeatureMapEntry;
import org.eclipse.emf.teneo.util.AssertUtil;
import org.eclipse.emf.teneo.util.StoreUtil;
import org.jpox.StateManager;
import org.jpox.sco.SCO;
import org.jpox.sco.SCOList;
import org.jpox.sco.exceptions.QueryUnownedSCOException;
import org.jpox.state.FetchPlanState;
import org.jpox.store.expression.QueryExpression;
import org.jpox.store.query.Queryable;
import org.jpox.store.query.ResultObjectFactory;
import org.jpox.util.ClassUtils;

/**
 * This class works as a wrapper around the EMF FeatureMap. This is required because the EMF - JPOX implementation can
 * not work with the standard FeatureMap.Entry it requires specific implementations which are generated by the EMF OR
 * Mapper.
 * 
 * A special featuremap entry has been developed: AnyFeatureMapEntry. This special feature map entry is used to handle
 * the xsd:any schema construction.
 * 
 * For the backing store a jpox arraylist is used. The delegate list is actually a normal arraylist, it was not possible
 * to use the backingstore as the delegate because the list can be detached.
 * 
 * @author <a href="mailto:mtaal@elver.org">Martin Taal</a>
 * @version $Revision: 1.8 $ $Date: 2008/02/28 07:09:02 $
 */

public class FeatureMapWrapper extends PersistableFeatureMap implements SCO, Queryable, SCOList {
	/**
	 * Generated Serial ID
	 */
	private static final long serialVersionUID = 7582443003647643367L;

	/** The logger */
	private static Log log = LogFactory.getLog(FeatureMapWrapper.class);

	/** The statemanager used to get to all the jdo information */
	private StateManager stateManager = null;

	/** The fieldname for which this feature map works */
	private String fieldName = null;

	/** The backing store of this elist */
	private JPOXArrayList jdoDelegate;

	/**
	 * An ArrayList of deletedobjects, is set by the didRemove method and used by the remove methods at the end
	 */
	private final ArrayList<Entry> deletedObjects = new ArrayList<Entry>();

	private final ArrayList<Integer> deletedObjectsIndex = new ArrayList<Integer>();

	/**
	 * Constructor, using the StateManager of the "owner" and the field name.
	 * 
	 * @param ownerSM
	 *            The owner StateManager
	 * @param fieldName
	 *            The name of the field of the SCO.
	 */
	public FeatureMapWrapper(StateManager ownerSM, String fieldName) {
		this(ownerSM, fieldName, new ArrayList<Entry>());
	}

	/**
	 * Constructor, using the StateManager of the "owner" and the field name.
	 * 
	 * @param ownerSM
	 *            The owner StateManager
	 * @param fieldName
	 *            The name of the field of the SCO.
	 */
	public FeatureMapWrapper(StateManager ownerSM, String featureFieldName, List<Entry> list) {
		super((InternalEObject) ownerSM.getObject(), StoreUtil.getEStructuralFeature((InternalEObject) ownerSM
			.getObject(), featureFieldName), list);

		stateManager = ownerSM;
		fieldName = featureFieldName;
		jdoDelegate = new JPOXArrayList(ownerSM, featureFieldName);

		log.debug("Creating feature map wrapper for " + getElementType().getName());

		// if already loaded then load the content of the jdo delegate to here
		if (jdoDelegate.isLoaded()) {
			load();
		}
	}

	/** Returns the element type to be used */
	@Override
	protected Class<? extends FeatureMap.Entry> determineElementType() {
		// determines which class should be used as the element type
		if (StoreUtil.isWildCard(getEStructuralFeature())) {
			return AnyFeatureMapEntry.class;
		} else {
			return GenericFeatureMapEntry.class; // FeatureMapEntry.getEntryClass(owner.getClass(),
			// getEStructuralFeature().getName());
		}
	}

	/** Nullify the delegate and stateManager before serializing */
	private void writeObject(java.io.ObjectOutputStream out) throws IOException {
		jdoDelegate = null;
		stateManager = null;
		out.defaultWriteObject();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.teneo.mapping.elist.PersistableFeatureMap#isPersistencyWrapped()
	 */
	@Override
	public boolean isPersistencyWrapped() {
		return true;
	}

	/** Is the owner detached */
	private boolean isOwnerDetached() {
		final PersistenceCapable pc = (PersistenceCapable) getEObject();
		return pc.jdoIsDetached();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.teneo.mapping.elist.PersistableFeatureMap#isPersistencyWrapped()
	 */
	@Override
	public Object clone() {
		throw new UnsupportedOperationException("Not yet supported for featuremap");
	}

	// ----------------------- Extra methods
	// ----------------------------------------

	/**
	 * Checks if a certain feature is containment and if so removes the object from the store
	 */
	protected void deleteDependent(EStructuralFeature feature, Object value) {
		if (isOwnerDetached()) {
			return; // do not do this when detached, this done at attach
		}

		if (feature instanceof EReference) {
			final EReference eref = (EReference) feature;
			if (eref.isContainment() && value instanceof PersistenceCapable) {
				PersistenceCapable pc = (PersistenceCapable) owner;
				pc.jdoGetPersistenceManager().deletePersistent(value);
			}
		}
	}

	// ----------------------- Implementation of JPOXEList methods
	// -------------------

	/** Detach self, means nullify all references to jdo */
	public void detachSelf() {
		// and detach ourselves also
		jdoDelegate = null;
		stateManager = null;
	}

	// ----------------------- Implementation of SCO methods -------------------

	/**
	 * Utility to mark the object as dirty
	 */
	public void makeDirty() {
		if (jdoDelegate != null) {
			jdoDelegate.makeDirty();
		}
	}

	/** Does the actual loading from the store in a synchronized manner */
	@Override
	protected synchronized void doLoad() {
		AssertUtil.assertTrue("EList " + getLogString() + " is already loaded", !isLoaded());

		final Resource res = getEObject().eResource();
		final boolean setLoading = res != null && res instanceof JPOXResource && !((StoreResource) res).isLoading();
		if (setLoading) {
			((StoreResource) res).setIsLoading(true);
		}

		// note add directly to the delegate to prevent infinite looping and
		// notifications
		final List<Entry> list = getDelegate();
		Iterator<?> iter = jdoDelegate.iterator();
		while (iter.hasNext()) {
			final Entry child = (Entry) iter.next();
			assert (getElementType().isInstance(child));
			list.add(child);

			// also set the container
			// TODO check if the content of the featuremap entry should also
			// not be
			// added to the resource explicitly, is maybe already done
			// correctly
			// in the setContainer method
			((FeatureMapEntry) child).setContainer(owner);
		}

		log.debug("Loaded " + list.size() + " objects from the backing store for elist " + getLogString());
		if (setLoading) {
			((JPOXResource) res).setIsLoading(false);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jpox.sco.SCO#attachCopy(java.lang.Object, boolean)
	 */
	public void attachCopy(Object value) {
		if (!((PersistableEList<?>) value).isLoaded()) {
			return;
		}

		load();

		java.util.Collection<?> c = (java.util.Collection<?>) value;

		// Attach all of the elements in the new list
		ArrayList<Entry> attachedElements = new ArrayList<Entry>(c.size());
		Iterator<?> iter = c.iterator();
		while (iter.hasNext()) {
			Object detachedElement = iter.next();
			if (ClassUtils.isPersistenceCapable(detachedElement)) {
				attachedElements.add((Entry) stateManager.getPersistenceManager().attachCopy(detachedElement, true));
			} else {
				// should this ever happen?
				throw new JpoxStoreException("A FeatureMap may only contain persistable objects, this object " +
						detachedElement.getClass().getName() + " is not persistable.");
				// attachedElements.add(detachedElement);
			}
		}

		log.debug("Attaching " + attachedElements.size() + " objects to " + getLogString());
		// NOTE: needed to pass the jdodelegate because the update actions
		// should go directly to the
		// underlying datastore and not through this list. Tests showed that if
		// the updates where
		// done directly in this list that in case of bidirectional reatlions
		// elements which had
		// been moved from one list to another ended up with empty relations
		// because the remove
		// action in the updateListWithListElements cleared the one side of the
		// bidirectional case.
		JpoxUtil.updateListWithListElements(jdoDelegate, attachedElements);
	}

	/**
	 * Method to detach the elements of this list.
	 */
	public void detach(FetchPlanState state) {
		load();
		jdoDelegate.detach(state);

		final Object[] values = toArray();
		for (Object element : values) {
			if (element != null && element instanceof PersistenceCapable) {
				stateManager.getPersistenceManager().detachInternal(element, state);
			}
		}
		stateManager = null;
		jdoDelegate = null;
	}

	/**
	 * Method to make transient all elements of this list
	 */
	public void makeTransient(FetchPlanState state) {
		final Object[] values = toArray();
		for (Object element : values) {
			if (element != null && element instanceof PersistenceCapable) {
				stateManager.getPersistenceManager().findStateManager((PersistenceCapable) element)
					.makeTransient(state);
			}
		}
		stateManager = null;
		jdoDelegate = null;
	}

	/**
	 * Method to return a detached copy of the list
	 */
	public Object detachCopy(FetchPlanState state) {
		load();
		jdoDelegate.detachCopy(state);

		final ArrayList<Entry> detached = new ArrayList<Entry>();
		final Object[] values = toArray();
		for (Object object : values) {
			if (object == null) {
				detached.add(null);
			} else {
				if (object instanceof PersistenceCapable) {
					detached.add((Entry) stateManager.getPersistenceManager().detachCopyInternal(object, state));
				} else {
					detached.add((Entry) object);
				}
			}
		}
		return detached;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jpox.sco.SCO#getFieldName()
	 */
	public String getFieldName() {
		return fieldName;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jpox.sco.SCO#getOwner()
	 */
	public Object getOwner() {
		return getEObject();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jpox.sco.SCO#setValueFrom(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	public void setValueFrom(Object obj) throws ClassCastException {
		Collection c = (Collection) obj;
		clear();
		addAll(c);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jpox.sco.SCO#setValueFrom(java.lang.Object, boolean)
	 */
	@SuppressWarnings("unchecked")
	public void setValueFrom(Object arg0, boolean arg1) throws ClassCastException {
		final Collection c = (Collection) arg0;
		if (arg1) {
			clear();
			addAll(c);
		} else {
			getDelegate().clear();
			getDelegate().addAll(c);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jpox.sco.SCO#unsetOwner()
	 */
	public void unsetOwner() {
		log.debug("Unsetting owner of " + getLogString());

		if (jdoDelegate != null) {
			jdoDelegate = null;
			stateManager = null;
			fieldName = null;
		}
	}

	/**
	 * Method to run reachability on this SCO.
	 * 
	 * @param reachables
	 *            List of StateManagers reachable so far
	 */
	@SuppressWarnings("unchecked")
	public void runReachability(java.util.Set reachables) {

		for (Object element : this) {
			final FeatureMapEntry gfm = (FeatureMapEntry) element;

			// first add the gfm to the reachables
			final StateManager sm = stateManager.getPersistenceManager().findStateManager((PersistenceCapable) gfm);
			if (!reachables.contains(sm.getInternalObjectId())) {
				sm.flush();
				reachables.add(sm.getInternalObjectId());
			}

			// now check if the gfm contains a pc
			final Object value = gfm.getValue();
			if (value instanceof PersistenceCapable) {
				final StateManager valueSM =
						stateManager.getPersistenceManager().findStateManager((PersistenceCapable) value);
				valueSM.runReachability(reachables);
			}
		}
	}

	/** Set method for SCO.set */
	public Object set(int arg0, Object arg1, boolean arg2) {
		return set(arg0, (Entry) arg1);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jpox.sco.SCOCollection#updateEmbeddedElement(java.lang.Object, int,
	 *      java.lang.Object)
	 */
	public void updateEmbeddedElement(Object element, int fieldNumber, Object value) {
		if (jdoDelegate == null) {
			throw new JpoxStoreException("List is detached can not update embedded element, " + getLogString());
		}

		jdoDelegate.updateEmbeddedElement(element, fieldNumber, value);
	}

	// ------------------------ Query Statement methods ------------------------

	/**
	 * Method to generate a QueryStatement for the SCO.
	 * 
	 * @return The QueryStatement
	 */
	public synchronized QueryExpression newQueryStatement() {
		if (jdoDelegate == null) {
			throw new QueryUnownedSCOException();
		}

		return jdoDelegate.newQueryStatement();
	}

	/**
	 * Method to return a QueryStatement, using the specified candidate class.
	 * 
	 * @param candidate_class
	 *            The candidate class
	 * @return The QueryStatement
	 */
	@SuppressWarnings("unchecked")
	public synchronized QueryExpression newQueryStatement(Class candidate_class) {
		if (jdoDelegate == null) {
			throw new QueryUnownedSCOException();
		}

		return jdoDelegate.newQueryStatement(candidate_class);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jpox.store.query.Queryable#newResultObjectFactory(org.jpox.store.expression.QueryExpression,
	 *      boolean, java.lang.Class, boolean)
	 */
	@SuppressWarnings("unchecked")
	public synchronized ResultObjectFactory newResultObjectFactory(QueryExpression stmt, boolean ignoreCache,
			Class resultClass, boolean useFetchPlan) {
		if (jdoDelegate == null) {
			throw new QueryUnownedSCOException();
		}

		return jdoDelegate.newResultObjectFactory(stmt, ignoreCache, resultClass, useFetchPlan);
	}

	// -------------------------- did Methods which are called after add,
	// remove, move, set and change ----------
	// The did... method check some invariants and update the backing store

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.common.util.BasicEList#didChange()
	 */
	@Override
	protected void didChange() {
		if (!isLoaded()) {
			return; // in load state so no change
		}
		makeDirty();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.common.util.BasicEList#didAdd(int, java.lang.Object)
	 */
	@Override
	protected void didAdd(int index, Entry newObject) {
		assert (newObject instanceof FeatureMapEntry);

		if (!isLoaded()) {
			return; // this can only happen when we are loading so do not add to
			// the store
		}

		if (jdoDelegate != null) {
			// now there is an id!
			if (newObject instanceof AnyFeatureMapEntry) {
				final Object value = ((FeatureMapEntry) newObject).getValue();
				if (value instanceof PersistenceCapable) {
					if (((PersistenceCapable) value).jdoGetObjectId() == null) // to
					// persist,
					// do
					// it
					// now
					{
						stateManager.getPersistenceManager().makePersistent(value);
						// set field again!
						((AnyFeatureMapEntry) newObject).initializeSpecificImplementation();
					}
				}
			}

			jdoDelegate.add(index, newObject);
		}

		super.didAdd(index, newObject);
	}

	/**
	 * Updates the backing store and deletes dependent objects if required
	 * 
	 * @see org.eclipse.emf.common.util.BasicEList#didClear(int, java.lang.Object[])
	 */
	@Override
	protected void didClear(int size, Object[] oldObjects) {
		if (oldObjects == null) {
			return;
		}

		if (jdoDelegate != null) {
			jdoDelegate.clear();

			/*
			 * for (int i = 0; i < oldObjects.length; i++) { if (oldObjects[i] == null) continue;
			 * assert(oldObjects[i] instanceof FeatureMapEntry); // get the values before the actual
			 * remove final FeatureMapEntry entry = (FeatureMapEntry)oldObjects[i]; final
			 * EStructuralFeature feature = entry.getEStructuralFeature(); final Object value =
			 * entry.getValue();
			 * 
			 * if (ret && value instanceof PersistenceCapable) { deleteDependent(feature, value); } }
			 */
		}
		super.didClear(size, oldObjects);
	}

	/**
	 * Updates the backing store.
	 * 
	 * @see org.eclipse.emf.common.util.BasicEList#didMove(int, java.lang.Object, int)
	 */
	@Override
	protected void didMove(int index, Entry movedObject, int oldIndex) {
		assert (movedObject instanceof FeatureMapEntry);
		if (jdoDelegate != null) {
			final Object newObject =
					createEntry(((FeatureMapEntry) movedObject).getEStructuralFeature(),
						((FeatureMapEntry) movedObject).getValue());

			jdoDelegate.remove(oldIndex);

			jdoDelegate.add(index, newObject);
			assert (JDOHelper.isPersistent(newObject));
		}
		super.didMove(index, movedObject, oldIndex);
	}

	/**
	 * Adds the oldObject to the deletedObjects arraylist. The objects are actually deleted from the backing store at
	 * the end of the remove, removeAll methods. The delete from the store can not be done here because after the
	 * didRemove method the objects are still accessed to do inverseremove.
	 * 
	 * @see org.eclipse.emf.common.util.BasicEList#didRemove(int, java.lang.Object)
	 */
	@Override
	protected void didRemove(int index, Entry oldObject) {
		assert (oldObject instanceof FeatureMapEntry);
		assert (!deletedObjects.contains(oldObject));
		deletedObjects.add(oldObject);
		deletedObjectsIndex.add(new Integer(index));
		super.didRemove(index, oldObject);
	}

	/**
	 * Performs the actual remove is called at the end of each remove method below
	 */
	private synchronized void doRemove() {
		if (deletedObjects.size() == 0) {
			return;
		}

		if (jdoDelegate == null) {
			deletedObjects.clear();
			deletedObjectsIndex.clear();
			return;
		}

		try {
			for (int i = 0; i < deletedObjects.size(); i++) {
				final int index = (deletedObjectsIndex.get(i)).intValue();
				final Object oldObject = deletedObjects.get(i);

				// get the values before the actual remove
				final FeatureMapEntry entry = (FeatureMapEntry) oldObject;
				final EStructuralFeature feature = entry.getEStructuralFeature();
				final Object value = entry.getValue();

				jdoDelegate.remove(index);

				deleteDependent(feature, value);
			}
		} finally {
			deletedObjects.clear();
			deletedObjectsIndex.clear();
		}
	}

	/*
	 * Updates the backing store
	 * 
	 * @see org.eclipse.emf.common.util.BasicEList#didSet(int, java.lang.Object, java.lang.Object)
	 */
	@Override
	protected void didSet(int index, Entry newObject, Entry oldObject) {
		assert (newObject instanceof FeatureMapEntry);
		assert (oldObject instanceof FeatureMapEntry);

		if (jdoDelegate != null) {
			final Object obj = jdoDelegate.set(index, newObject);
			assert (obj == oldObject);
			if (oldObject != newObject) {
				final FeatureMapEntry entry = (FeatureMapEntry) oldObject;
				final EStructuralFeature feature = entry.getEStructuralFeature();
				final Object value = entry.getValue();

				// first delete the entry itself
				// PersistenceCapable pc = (PersistenceCapable) owner;
				// pc.jdoGetPersistenceManager().deletePersistent(entry);
				jdoDelegate.remove(entry);

				// and the dependent entry
				deleteDependent(feature, value);
			}
		}

		super.didSet(index, newObject, oldObject);
	}

	/** Creates an entry with the correct type */
	@Override
	public FeatureMap.Entry createEntry(EStructuralFeature feature, Object value) {
		try {
			FeatureMapEntry entry = (FeatureMapEntry) getElementType().newInstance();
			entry.setFields(feature, value);
			return entry;
		} catch (Exception e) {
			throw createException("Exception while instantiating for elementClass " + getElementType().getName(), e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.ecore.util.BasicFeatureMap#clear(org.eclipse.emf.ecore.EStructuralFeature)
	 */
	@Override
	public void clear(EStructuralFeature feature) {
		load();
		try {
			assert (deletedObjects.size() == 0);
			super.clear(feature);
		} finally {
			doRemove();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.ecore.util.BasicFeatureMap#remove(org.eclipse.emf.ecore.EStructuralFeature,
	 *      int)
	 */
	@Override
	public synchronized Object remove(EStructuralFeature feature, int index) {
		load();
		try {
			assert (deletedObjects.size() == 0);
			return super.remove(feature, index);
		} finally {
			doRemove();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.ecore.util.BasicFeatureMap#remove(org.eclipse.emf.ecore.EStructuralFeature,
	 *      java.lang.Object)
	 */
	@Override
	public synchronized boolean remove(EStructuralFeature feature, Object object) {
		load();
		try {
			assert (deletedObjects.size() == 0);
			return super.remove(feature, object);
		} finally {
			doRemove();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.ecore.util.BasicFeatureMap#removeAll(org.eclipse.emf.ecore.EStructuralFeature,
	 *      java.util.Collection)
	 */
	@Override
	public synchronized boolean removeAll(EStructuralFeature feature, Collection<?> collection) {
		load();
		try {
			assert (deletedObjects.size() == 0);
			return super.removeAll(feature, collection);
		} finally {
			doRemove();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.common.notify.impl.NotifyingListImpl#clear()
	 */
	@Override
	public void clear() {
		load();
		// doRemove does not need to be called because the jdoDelegate is cleared in didClear
		super.clear();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.common.notify.impl.NotifyingListImpl#remove(int)
	 */
	@Override
	public Entry remove(int index) {
		load();
		try {
			assert (deletedObjects.size() == 0);
			return super.remove(index);
		} finally {
			doRemove();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.common.notify.impl.NotifyingListImpl#removeAll(java.util.Collection)
	 */
	@Override
	public boolean removeAll(Collection<?> collection) {
		load();
		try {
			assert (deletedObjects.size() == 0);
			return super.removeAll(collection);
		} finally {
			doRemove();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.emf.common.util.BasicEList#remove(java.lang.Object)
	 */
	@Override
	public synchronized boolean remove(Object object) {
		load();
		try {
			assert (deletedObjects.size() == 0);
			return super.remove(object);
		} finally {
			doRemove();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.util.AbstractList#removeRange(int, int)
	 */
	@Override
	protected synchronized void removeRange(int arg0, int arg1) {
		load();
		try {
			assert (deletedObjects.size() == 0);
			super.removeRange(arg0, arg1);
		} finally {
			doRemove();
		}
	}

	/** For now returns the same as isLoaded */
	public boolean isInitialized() {
		return isLoaded();
	}

	/**
	 * Overridden to make public
	 */
	@Override
	public void load() {
		// TODO Auto-generated method stub
		super.load();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jpox.sco.SCOContainer#loadFieldsInFetchPlan(org.jpox.state.FetchPlanState)
	 */
	public void loadFieldsInFetchPlan(FetchPlanState arg0) {
		Object[] values = toArray();
		for (Object element : values) {
			if (element != null && element instanceof PersistenceCapable) {
				stateManager.getPersistenceManager().findStateManager((PersistenceCapable) element)
					.loadFieldsInFetchPlan(arg0);
			}
		}
	}
}