/**
 * Copyright (c) 2016-2018 TypeFox and others.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 * 
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */
package org.eclipse.lsp4j;

import java.util.List;
import org.eclipse.lsp4j.jsonrpc.ProtocolSince;
import org.eclipse.lsp4j.jsonrpc.util.Preconditions;
import org.eclipse.lsp4j.jsonrpc.util.ToStringBuilder;
import org.eclipse.lsp4j.jsonrpc.validation.NonNull;

/**
 * Contains additional diagnostic information about the context in which a code action is run.
 */
@SuppressWarnings("all")
public class CodeActionContext {
  /**
   * An array of diagnostics known on the client side overlapping the range
   * provided to the {@code textDocument/codeAction} request. They are provided
   * so that the server knows which errors are currently presented to the user
   * for the given range. There is no guarantee that these accurately reflect the
   * error state of the resource. The primary parameter to compute code actions
   * is the provided range.
   * <p>
   * Note that the client should check the {@link DiagnosticServerCapabilities#markupMessageSupport}
   * server capability before sending diagnostics with markup messages to a server.
   * Diagnostics with markup messages should be excluded for servers that do not
   * support them.
   */
  @NonNull
  private List<Diagnostic> diagnostics;

  /**
   * Requested kind of actions to return.
   * <p>
   * Actions not of this kind are filtered out by the client before being shown. So servers
   * can omit computing them.
   * <p>
   * See {@link CodeActionKind} for some predefined code action kinds.
   */
  private List<String> only;

  /**
   * The reason why code actions were requested.
   */
  @ProtocolSince("3.17.0")
  private CodeActionTriggerKind triggerKind;

  public CodeActionContext() {
  }

  public CodeActionContext(@NonNull final List<Diagnostic> diagnostics) {
    this.diagnostics = Preconditions.<List<Diagnostic>>checkNotNull(diagnostics, "diagnostics");
  }

  public CodeActionContext(@NonNull final List<Diagnostic> diagnostics, final List<String> only) {
    this(diagnostics);
    this.only = only;
  }

  /**
   * An array of diagnostics known on the client side overlapping the range
   * provided to the {@code textDocument/codeAction} request. They are provided
   * so that the server knows which errors are currently presented to the user
   * for the given range. There is no guarantee that these accurately reflect the
   * error state of the resource. The primary parameter to compute code actions
   * is the provided range.
   * <p>
   * Note that the client should check the {@link DiagnosticServerCapabilities#markupMessageSupport}
   * server capability before sending diagnostics with markup messages to a server.
   * Diagnostics with markup messages should be excluded for servers that do not
   * support them.
   */
  @NonNull
  public List<Diagnostic> getDiagnostics() {
    return this.diagnostics;
  }

  /**
   * An array of diagnostics known on the client side overlapping the range
   * provided to the {@code textDocument/codeAction} request. They are provided
   * so that the server knows which errors are currently presented to the user
   * for the given range. There is no guarantee that these accurately reflect the
   * error state of the resource. The primary parameter to compute code actions
   * is the provided range.
   * <p>
   * Note that the client should check the {@link DiagnosticServerCapabilities#markupMessageSupport}
   * server capability before sending diagnostics with markup messages to a server.
   * Diagnostics with markup messages should be excluded for servers that do not
   * support them.
   */
  public void setDiagnostics(@NonNull final List<Diagnostic> diagnostics) {
    this.diagnostics = Preconditions.checkNotNull(diagnostics, "diagnostics");
  }

  /**
   * Requested kind of actions to return.
   * <p>
   * Actions not of this kind are filtered out by the client before being shown. So servers
   * can omit computing them.
   * <p>
   * See {@link CodeActionKind} for some predefined code action kinds.
   */
  public List<String> getOnly() {
    return this.only;
  }

  /**
   * Requested kind of actions to return.
   * <p>
   * Actions not of this kind are filtered out by the client before being shown. So servers
   * can omit computing them.
   * <p>
   * See {@link CodeActionKind} for some predefined code action kinds.
   */
  public void setOnly(final List<String> only) {
    this.only = only;
  }

  /**
   * The reason why code actions were requested.
   */
  @ProtocolSince("3.17.0")
  public CodeActionTriggerKind getTriggerKind() {
    return this.triggerKind;
  }

  /**
   * The reason why code actions were requested.
   */
  @ProtocolSince("3.17.0")
  public void setTriggerKind(final CodeActionTriggerKind triggerKind) {
    this.triggerKind = triggerKind;
  }

  @Override
  public String toString() {
    ToStringBuilder b = new ToStringBuilder(this);
    b.add("diagnostics", this.diagnostics);
    b.add("only", this.only);
    b.add("triggerKind", this.triggerKind);
    return b.toString();
  }

  @Override
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    CodeActionContext other = (CodeActionContext) obj;
    if (this.diagnostics == null) {
      if (other.diagnostics != null)
        return false;
    } else if (!this.diagnostics.equals(other.diagnostics))
      return false;
    if (this.only == null) {
      if (other.only != null)
        return false;
    } else if (!this.only.equals(other.only))
      return false;
    if (this.triggerKind == null) {
      if (other.triggerKind != null)
        return false;
    } else if (!this.triggerKind.equals(other.triggerKind))
      return false;
    return true;
  }

  @Override
  public int hashCode() {
    final int prime = 31;
    int result = 1;
    result = prime * result + ((this.diagnostics== null) ? 0 : this.diagnostics.hashCode());
    result = prime * result + ((this.only== null) ? 0 : this.only.hashCode());
    return prime * result + ((this.triggerKind== null) ? 0 : this.triggerKind.hashCode());
  }
}
