
package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link IpamOptions}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableIpamOptions.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableIpamOptions implements IpamOptions {
  private final Driver driver;
  private final @Nullable List<IpamConfig> configs;

  private ImmutableIpamOptions(
      Driver driver,
      @Nullable List<IpamConfig> configs) {
    this.driver = driver;
    this.configs = configs;
  }

  /**
   * @return The value of the {@code driver} attribute
   */
  @JsonProperty("Driver")
  @Override
  public Driver driver() {
    return driver;
  }

  /**
   * @return The value of the {@code configs} attribute
   */
  @JsonProperty("Configs")
  @Override
  public @Nullable List<IpamConfig> configs() {
    return configs;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link IpamOptions#driver() driver} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for driver
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableIpamOptions withDriver(Driver value) {
    if (this.driver == value) return this;
    Driver newValue = Objects.requireNonNull(value, "driver");
    return new ImmutableIpamOptions(newValue, this.configs);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link IpamOptions#configs() configs}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableIpamOptions withConfigs(@Nullable IpamConfig... elements) {
    if (elements == null) {
      if (this.configs == null) return this;
      return new ImmutableIpamOptions(this.driver, null);
    }
    List<IpamConfig> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableIpamOptions(this.driver, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link IpamOptions#configs() configs}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of configs elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableIpamOptions withConfigs(@Nullable Iterable<? extends IpamConfig> elements) {
    if (this.configs == elements) return this;
    @Nullable List<IpamConfig> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableIpamOptions(this.driver, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableIpamOptions} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableIpamOptions
        && equalsByValue((ImmutableIpamOptions) another);
  }

  private boolean equalsByValue(ImmutableIpamOptions another) {
    return driver.equals(another.driver)
        && Objects.equals(configs, another.configs);
  }

  /**
   * Computes a hash code from attributes: {@code driver}, {@code configs}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + driver.hashCode();
    h += (h << 5) + Objects.hashCode(configs);
    return h;
  }

  /**
   * Prints the immutable value {@code IpamOptions} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "IpamOptions{"
        + "driver=" + driver
        + ", configs=" + configs
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link IpamOptions} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable IpamOptions instance
   */
  public static ImmutableIpamOptions copyOf(IpamOptions instance) {
    if (instance instanceof ImmutableIpamOptions) {
      return (ImmutableIpamOptions) instance;
    }
    return ImmutableIpamOptions.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableIpamOptions ImmutableIpamOptions}.
   * <pre>
   * ImmutableIpamOptions.builder()
   *    .driver(org.mandas.docker.client.messages.swarm.Driver) // required {@link IpamOptions#driver() driver}
   *    .configs(List&lt;org.mandas.docker.client.messages.swarm.IpamConfig&gt; | null) // nullable {@link IpamOptions#configs() configs}
   *    .build();
   * </pre>
   * @return A new ImmutableIpamOptions builder
   */
  public static ImmutableIpamOptions.Builder builder() {
    return new ImmutableIpamOptions.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableIpamOptions ImmutableIpamOptions}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_DRIVER = 0x1L;
    private long initBits = 0x1L;

    private Driver driver;
    private List<IpamConfig> configs = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code IpamOptions} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(IpamOptions instance) {
      Objects.requireNonNull(instance, "instance");
      this.driver(instance.driver());
      @Nullable List<IpamConfig> configsValue = instance.configs();
      if (configsValue != null) {
        addAllConfigs(configsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link IpamOptions#driver() driver} attribute.
     * @param driver The value for driver 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Driver")
    public final Builder driver(Driver driver) {
      this.driver = Objects.requireNonNull(driver, "driver");
      initBits &= ~INIT_BIT_DRIVER;
      return this;
    }

    /**
     * Adds one element to {@link IpamOptions#configs() configs} list.
     * @param element A configs element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder config(IpamConfig element) {
      if (this.configs == null) {
        this.configs = new ArrayList<IpamConfig>();
      }
      this.configs.add(Objects.requireNonNull(element, "configs element"));
      return this;
    }

    /**
     * Adds elements to {@link IpamOptions#configs() configs} list.
     * @param elements An array of configs elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder configs(IpamConfig... elements) {
      if (this.configs == null) {
        this.configs = new ArrayList<IpamConfig>();
      }
      for (IpamConfig element : elements) {
        this.configs.add(Objects.requireNonNull(element, "configs element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link IpamOptions#configs() configs} list.
     * @param elements An iterable of configs elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Configs")
    public final Builder configs(@Nullable Iterable<? extends IpamConfig> elements) {
      if (elements == null) {
        this.configs = null;
        return this;
      }
      this.configs = new ArrayList<IpamConfig>();
      return addAllConfigs(elements);
    }

    /**
     * Adds elements to {@link IpamOptions#configs() configs} list.
     * @param elements An iterable of configs elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllConfigs(Iterable<? extends IpamConfig> elements) {
      Objects.requireNonNull(elements, "configs element");
      if (this.configs == null) {
        this.configs = new ArrayList<IpamConfig>();
      }
      for (IpamConfig element : elements) {
        this.configs.add(Objects.requireNonNull(element, "configs element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableIpamOptions ImmutableIpamOptions}.
     * @return An immutable instance of IpamOptions
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableIpamOptions build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableIpamOptions(driver, configs == null ? null : createUnmodifiableList(true, configs));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DRIVER) != 0) attributes.add("driver");
      return "Cannot build IpamOptions, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
