"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configRoutes = configRoutes;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _helper = require("../utils/helper");
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../common/constants");
var _MDSEnabledClientService = require("../../common/MDSEnabledClientService");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function configRoutes(router, dataSourceEnabled) {
  const getConfigsQuerySchema = {
    from_index: _configSchema.schema.number(),
    max_items: _configSchema.schema.number(),
    query: _configSchema.schema.maybe(_configSchema.schema.string()),
    config_type: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()]),
    is_enabled: _configSchema.schema.maybe(_configSchema.schema.boolean()),
    sort_field: _configSchema.schema.string(),
    sort_order: _configSchema.schema.string(),
    config_id_list: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])),
    'smtp_account.method': _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()]))
  };
  if (dataSourceEnabled) {
    getConfigsQuerySchema.dataSourceId = _configSchema.schema.string();
  }
  const genericBodyAndDataSourceIdQuery = {
    body: _configSchema.schema.any()
  };
  if (dataSourceEnabled) {
    genericBodyAndDataSourceIdQuery.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const genericParamsAndDataSourceIdQuery = {
    params: _configSchema.schema.any()
  };
  if (dataSourceEnabled) {
    genericParamsAndDataSourceIdQuery.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const updateQuerySchema = {
    body: _configSchema.schema.any(),
    params: _configSchema.schema.object({
      configId: _configSchema.schema.string()
    })
  };
  if (dataSourceEnabled) {
    updateQuerySchema.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const deleteQuerySchema = {
    config_id_list: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])
  };
  if (dataSourceEnabled) {
    deleteQuerySchema.dataSourceId = _configSchema.schema.string();
  }
  router.get({
    path: _common.NODE_API.GET_CONFIGS,
    validate: {
      query: _configSchema.schema.object(getConfigsQuerySchema)
    }
  }, async (context, request, response) => {
    const config_type = (0, _helper.joinRequestParams)(request.query.config_type);
    const config_id_list = (0, _helper.joinRequestParams)(request.query.config_id_list);
    const encryption_method = (0, _helper.joinRequestParams)(request.query['smtp_account.method']);
    const query = request.query.query;
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getConfigs', {
        from_index: request.query.from_index,
        max_items: request.query.max_items,
        is_enabled: request.query.is_enabled,
        sort_field: request.query.sort_field,
        sort_order: request.query.sort_order,
        config_type,
        ...(query && {
          text_query: query
        }),
        // text_query will exclude keyword fields
        ...(config_id_list && {
          config_id_list
        }),
        ...(encryption_method && {
          'smtp_account.method': encryption_method
        })
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 404,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_common.NODE_API.GET_CONFIG}/{configId}`,
    validate: genericParamsAndDataSourceIdQuery
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getConfigById', {
        configId: request.params.configId
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _common.NODE_API.CREATE_CONFIG,
    validate: genericBodyAndDataSourceIdQuery
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.createConfig', {
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_common.NODE_API.UPDATE_CONFIG}/{configId}`,
    validate: updateQuerySchema
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.updateConfigById', {
        configId: request.params.configId,
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: _common.NODE_API.DELETE_CONFIGS,
    validate: {
      query: _configSchema.schema.object(deleteQuerySchema)
    }
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    const config_id_list = (0, _helper.joinRequestParams)(request.query.config_id_list);
    try {
      const resp = await client('notifications.deleteConfigs', {
        config_id_list
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: _common.NODE_API.GET_AVAILABLE_FEATURES,
    validate: dataSourceEnabled ? {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.string()
      })
    } : false
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getServerFeatures');
      const config_type_list = resp.allowed_config_type_list;
      const channelTypes = {};
      for (let channel of config_type_list) {
        if (_constants.CHANNEL_TYPE[channel]) {
          channelTypes[channel] = _constants.CHANNEL_TYPE[channel];
        }
      }
      const availableFeature = {
        availableChannels: channelTypes,
        availableConfigTypes: config_type_list,
        tooltipSupport: _lodash.default.get(response, ['plugin_features', 'tooltip_support']) === 'true'
      };
      return response.ok({
        body: availableFeature
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 404,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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