"""Mappings from Adobe glyph names to Unicode characters.

In some CMap tables, Adobe glyph names are used for specifying
Unicode characters instead of using decimal/hex character code.

The following data was taken by

  $ wget https://partners.adobe.com/public/developer/en/opentype/glyphlist.txt

```python
from pdfminer.glyphlist import convert_glyphlist

convert_glyphlist("glyphlist.txt")
"""

# ###################################################################################
# Copyright (c) 1997,1998,2002,2007 Adobe Systems Incorporated
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this documentation file to use, copy, publish, distribute,
# sublicense, and/or sell copies of the documentation, and to permit
# others to do the same, provided that:
# - No modification, editing or other alteration of this document is
# allowed; and
# - The above copyright notice and this permission notice shall be
# included in all copies of the documentation.
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this documentation file, to create their own derivative works
# from the content of this document to use, copy, publish, distribute,
# sublicense, and/or sell the derivative works, and to permit others to do
# the same, provided that the derived work is not represented as being a
# copy or version of this document.
#
# Adobe shall not be liable to any party for any loss of revenue or profit
# or for indirect, incidental, special, consequential, or other similar
# damages, whether based on tort (including without limitation negligence
# or strict liability), contract or other legal or equitable grounds even
# if Adobe has been advised or had reason to know of the possibility of
# such damages. The Adobe materials are provided on an "AS IS" basis.
# Adobe specifically disclaims all express, statutory, or implied
# warranties relating to the Adobe materials, including but not limited to
# those concerning merchantability or fitness for a particular purpose or
# non-infringement of any third party rights regarding the Adobe
# materials.
# ###################################################################################
# Name:          Adobe Glyph List
# Table version: 2.0
# Date:          September 20, 2002
#
# See http://partners.adobe.com/asn/developer/typeforum/unicodegn.html
#
# Format: Semicolon-delimited fields:
#            (1) glyph name
#            (2) Unicode scalar value


def convert_glyphlist(path: str) -> None:
    """Convert a glyph list into a python representation.

    See output below.
    """
    state = 0
    with open(path) as fileinput:
        for line in fileinput.readlines():
            line = line.strip()
            if not line or line.startswith("#"):
                if state == 1:
                    state = 2
                    print("}\n")
                print(line)
                continue
            if state == 0:
                print("\nglyphname2unicode = {")
                state = 1
            (name, x) = line.split(";")
            codes = x.split(" ")
            print(
                " {!r}: u'{}',".format(name, "".join(f"\\u{code}" for code in codes)),
            )


glyphname2unicode = {
    "A": "\u0041",
    "AE": "\u00c6",
    "AEacute": "\u01fc",
    "AEmacron": "\u01e2",
    "AEsmall": "\uf7e6",
    "Aacute": "\u00c1",
    "Aacutesmall": "\uf7e1",
    "Abreve": "\u0102",
    "Abreveacute": "\u1eae",
    "Abrevecyrillic": "\u04d0",
    "Abrevedotbelow": "\u1eb6",
    "Abrevegrave": "\u1eb0",
    "Abrevehookabove": "\u1eb2",
    "Abrevetilde": "\u1eb4",
    "Acaron": "\u01cd",
    "Acircle": "\u24b6",
    "Acircumflex": "\u00c2",
    "Acircumflexacute": "\u1ea4",
    "Acircumflexdotbelow": "\u1eac",
    "Acircumflexgrave": "\u1ea6",
    "Acircumflexhookabove": "\u1ea8",
    "Acircumflexsmall": "\uf7e2",
    "Acircumflextilde": "\u1eaa",
    "Acute": "\uf6c9",
    "Acutesmall": "\uf7b4",
    "Acyrillic": "\u0410",
    "Adblgrave": "\u0200",
    "Adieresis": "\u00c4",
    "Adieresiscyrillic": "\u04d2",
    "Adieresismacron": "\u01de",
    "Adieresissmall": "\uf7e4",
    "Adotbelow": "\u1ea0",
    "Adotmacron": "\u01e0",
    "Agrave": "\u00c0",
    "Agravesmall": "\uf7e0",
    "Ahookabove": "\u1ea2",
    "Aiecyrillic": "\u04d4",
    "Ainvertedbreve": "\u0202",
    "Alpha": "\u0391",
    "Alphatonos": "\u0386",
    "Amacron": "\u0100",
    "Amonospace": "\uff21",
    "Aogonek": "\u0104",
    "Aring": "\u00c5",
    "Aringacute": "\u01fa",
    "Aringbelow": "\u1e00",
    "Aringsmall": "\uf7e5",
    "Asmall": "\uf761",
    "Atilde": "\u00c3",
    "Atildesmall": "\uf7e3",
    "Aybarmenian": "\u0531",
    "B": "\u0042",
    "Bcircle": "\u24b7",
    "Bdotaccent": "\u1e02",
    "Bdotbelow": "\u1e04",
    "Becyrillic": "\u0411",
    "Benarmenian": "\u0532",
    "Beta": "\u0392",
    "Bhook": "\u0181",
    "Blinebelow": "\u1e06",
    "Bmonospace": "\uff22",
    "Brevesmall": "\uf6f4",
    "Bsmall": "\uf762",
    "Btopbar": "\u0182",
    "C": "\u0043",
    "Caarmenian": "\u053e",
    "Cacute": "\u0106",
    "Caron": "\uf6ca",
    "Caronsmall": "\uf6f5",
    "Ccaron": "\u010c",
    "Ccedilla": "\u00c7",
    "Ccedillaacute": "\u1e08",
    "Ccedillasmall": "\uf7e7",
    "Ccircle": "\u24b8",
    "Ccircumflex": "\u0108",
    "Cdot": "\u010a",
    "Cdotaccent": "\u010a",
    "Cedillasmall": "\uf7b8",
    "Chaarmenian": "\u0549",
    "Cheabkhasiancyrillic": "\u04bc",
    "Checyrillic": "\u0427",
    "Chedescenderabkhasiancyrillic": "\u04be",
    "Chedescendercyrillic": "\u04b6",
    "Chedieresiscyrillic": "\u04f4",
    "Cheharmenian": "\u0543",
    "Chekhakassiancyrillic": "\u04cb",
    "Cheverticalstrokecyrillic": "\u04b8",
    "Chi": "\u03a7",
    "Chook": "\u0187",
    "Circumflexsmall": "\uf6f6",
    "Cmonospace": "\uff23",
    "Coarmenian": "\u0551",
    "Csmall": "\uf763",
    "D": "\u0044",
    "DZ": "\u01f1",
    "DZcaron": "\u01c4",
    "Daarmenian": "\u0534",
    "Dafrican": "\u0189",
    "Dcaron": "\u010e",
    "Dcedilla": "\u1e10",
    "Dcircle": "\u24b9",
    "Dcircumflexbelow": "\u1e12",
    "Dcroat": "\u0110",
    "Ddotaccent": "\u1e0a",
    "Ddotbelow": "\u1e0c",
    "Decyrillic": "\u0414",
    "Deicoptic": "\u03ee",
    "Delta": "\u2206",
    "Deltagreek": "\u0394",
    "Dhook": "\u018a",
    "Dieresis": "\uf6cb",
    "DieresisAcute": "\uf6cc",
    "DieresisGrave": "\uf6cd",
    "Dieresissmall": "\uf7a8",
    "Digammagreek": "\u03dc",
    "Djecyrillic": "\u0402",
    "Dlinebelow": "\u1e0e",
    "Dmonospace": "\uff24",
    "Dotaccentsmall": "\uf6f7",
    "Dslash": "\u0110",
    "Dsmall": "\uf764",
    "Dtopbar": "\u018b",
    "Dz": "\u01f2",
    "Dzcaron": "\u01c5",
    "Dzeabkhasiancyrillic": "\u04e0",
    "Dzecyrillic": "\u0405",
    "Dzhecyrillic": "\u040f",
    "E": "\u0045",
    "Eacute": "\u00c9",
    "Eacutesmall": "\uf7e9",
    "Ebreve": "\u0114",
    "Ecaron": "\u011a",
    "Ecedillabreve": "\u1e1c",
    "Echarmenian": "\u0535",
    "Ecircle": "\u24ba",
    "Ecircumflex": "\u00ca",
    "Ecircumflexacute": "\u1ebe",
    "Ecircumflexbelow": "\u1e18",
    "Ecircumflexdotbelow": "\u1ec6",
    "Ecircumflexgrave": "\u1ec0",
    "Ecircumflexhookabove": "\u1ec2",
    "Ecircumflexsmall": "\uf7ea",
    "Ecircumflextilde": "\u1ec4",
    "Ecyrillic": "\u0404",
    "Edblgrave": "\u0204",
    "Edieresis": "\u00cb",
    "Edieresissmall": "\uf7eb",
    "Edot": "\u0116",
    "Edotaccent": "\u0116",
    "Edotbelow": "\u1eb8",
    "Efcyrillic": "\u0424",
    "Egrave": "\u00c8",
    "Egravesmall": "\uf7e8",
    "Eharmenian": "\u0537",
    "Ehookabove": "\u1eba",
    "Eightroman": "\u2167",
    "Einvertedbreve": "\u0206",
    "Eiotifiedcyrillic": "\u0464",
    "Elcyrillic": "\u041b",
    "Elevenroman": "\u216a",
    "Emacron": "\u0112",
    "Emacronacute": "\u1e16",
    "Emacrongrave": "\u1e14",
    "Emcyrillic": "\u041c",
    "Emonospace": "\uff25",
    "Encyrillic": "\u041d",
    "Endescendercyrillic": "\u04a2",
    "Eng": "\u014a",
    "Enghecyrillic": "\u04a4",
    "Enhookcyrillic": "\u04c7",
    "Eogonek": "\u0118",
    "Eopen": "\u0190",
    "Epsilon": "\u0395",
    "Epsilontonos": "\u0388",
    "Ercyrillic": "\u0420",
    "Ereversed": "\u018e",
    "Ereversedcyrillic": "\u042d",
    "Escyrillic": "\u0421",
    "Esdescendercyrillic": "\u04aa",
    "Esh": "\u01a9",
    "Esmall": "\uf765",
    "Eta": "\u0397",
    "Etarmenian": "\u0538",
    "Etatonos": "\u0389",
    "Eth": "\u00d0",
    "Ethsmall": "\uf7f0",
    "Etilde": "\u1ebc",
    "Etildebelow": "\u1e1a",
    "Euro": "\u20ac",
    "Ezh": "\u01b7",
    "Ezhcaron": "\u01ee",
    "Ezhreversed": "\u01b8",
    "F": "\u0046",
    "Fcircle": "\u24bb",
    "Fdotaccent": "\u1e1e",
    "Feharmenian": "\u0556",
    "Feicoptic": "\u03e4",
    "Fhook": "\u0191",
    "Fitacyrillic": "\u0472",
    "Fiveroman": "\u2164",
    "Fmonospace": "\uff26",
    "Fourroman": "\u2163",
    "Fsmall": "\uf766",
    "G": "\u0047",
    "GBsquare": "\u3387",
    "Gacute": "\u01f4",
    "Gamma": "\u0393",
    "Gammaafrican": "\u0194",
    "Gangiacoptic": "\u03ea",
    "Gbreve": "\u011e",
    "Gcaron": "\u01e6",
    "Gcedilla": "\u0122",
    "Gcircle": "\u24bc",
    "Gcircumflex": "\u011c",
    "Gcommaaccent": "\u0122",
    "Gdot": "\u0120",
    "Gdotaccent": "\u0120",
    "Gecyrillic": "\u0413",
    "Ghadarmenian": "\u0542",
    "Ghemiddlehookcyrillic": "\u0494",
    "Ghestrokecyrillic": "\u0492",
    "Gheupturncyrillic": "\u0490",
    "Ghook": "\u0193",
    "Gimarmenian": "\u0533",
    "Gjecyrillic": "\u0403",
    "Gmacron": "\u1e20",
    "Gmonospace": "\uff27",
    "Grave": "\uf6ce",
    "Gravesmall": "\uf760",
    "Gsmall": "\uf767",
    "Gsmallhook": "\u029b",
    "Gstroke": "\u01e4",
    "H": "\u0048",
    "H18533": "\u25cf",
    "H18543": "\u25aa",
    "H18551": "\u25ab",
    "H22073": "\u25a1",
    "HPsquare": "\u33cb",
    "Haabkhasiancyrillic": "\u04a8",
    "Hadescendercyrillic": "\u04b2",
    "Hardsigncyrillic": "\u042a",
    "Hbar": "\u0126",
    "Hbrevebelow": "\u1e2a",
    "Hcedilla": "\u1e28",
    "Hcircle": "\u24bd",
    "Hcircumflex": "\u0124",
    "Hdieresis": "\u1e26",
    "Hdotaccent": "\u1e22",
    "Hdotbelow": "\u1e24",
    "Hmonospace": "\uff28",
    "Hoarmenian": "\u0540",
    "Horicoptic": "\u03e8",
    "Hsmall": "\uf768",
    "Hungarumlaut": "\uf6cf",
    "Hungarumlautsmall": "\uf6f8",
    "Hzsquare": "\u3390",
    "I": "\u0049",
    "IAcyrillic": "\u042f",
    "IJ": "\u0132",
    "IUcyrillic": "\u042e",
    "Iacute": "\u00cd",
    "Iacutesmall": "\uf7ed",
    "Ibreve": "\u012c",
    "Icaron": "\u01cf",
    "Icircle": "\u24be",
    "Icircumflex": "\u00ce",
    "Icircumflexsmall": "\uf7ee",
    "Icyrillic": "\u0406",
    "Idblgrave": "\u0208",
    "Idieresis": "\u00cf",
    "Idieresisacute": "\u1e2e",
    "Idieresiscyrillic": "\u04e4",
    "Idieresissmall": "\uf7ef",
    "Idot": "\u0130",
    "Idotaccent": "\u0130",
    "Idotbelow": "\u1eca",
    "Iebrevecyrillic": "\u04d6",
    "Iecyrillic": "\u0415",
    "Ifraktur": "\u2111",
    "Igrave": "\u00cc",
    "Igravesmall": "\uf7ec",
    "Ihookabove": "\u1ec8",
    "Iicyrillic": "\u0418",
    "Iinvertedbreve": "\u020a",
    "Iishortcyrillic": "\u0419",
    "Imacron": "\u012a",
    "Imacroncyrillic": "\u04e2",
    "Imonospace": "\uff29",
    "Iniarmenian": "\u053b",
    "Iocyrillic": "\u0401",
    "Iogonek": "\u012e",
    "Iota": "\u0399",
    "Iotaafrican": "\u0196",
    "Iotadieresis": "\u03aa",
    "Iotatonos": "\u038a",
    "Ismall": "\uf769",
    "Istroke": "\u0197",
    "Itilde": "\u0128",
    "Itildebelow": "\u1e2c",
    "Izhitsacyrillic": "\u0474",
    "Izhitsadblgravecyrillic": "\u0476",
    "J": "\u004a",
    "Jaarmenian": "\u0541",
    "Jcircle": "\u24bf",
    "Jcircumflex": "\u0134",
    "Jecyrillic": "\u0408",
    "Jheharmenian": "\u054b",
    "Jmonospace": "\uff2a",
    "Jsmall": "\uf76a",
    "K": "\u004b",
    "KBsquare": "\u3385",
    "KKsquare": "\u33cd",
    "Kabashkircyrillic": "\u04a0",
    "Kacute": "\u1e30",
    "Kacyrillic": "\u041a",
    "Kadescendercyrillic": "\u049a",
    "Kahookcyrillic": "\u04c3",
    "Kappa": "\u039a",
    "Kastrokecyrillic": "\u049e",
    "Kaverticalstrokecyrillic": "\u049c",
    "Kcaron": "\u01e8",
    "Kcedilla": "\u0136",
    "Kcircle": "\u24c0",
    "Kcommaaccent": "\u0136",
    "Kdotbelow": "\u1e32",
    "Keharmenian": "\u0554",
    "Kenarmenian": "\u053f",
    "Khacyrillic": "\u0425",
    "Kheicoptic": "\u03e6",
    "Khook": "\u0198",
    "Kjecyrillic": "\u040c",
    "Klinebelow": "\u1e34",
    "Kmonospace": "\uff2b",
    "Koppacyrillic": "\u0480",
    "Koppagreek": "\u03de",
    "Ksicyrillic": "\u046e",
    "Ksmall": "\uf76b",
    "L": "\u004c",
    "LJ": "\u01c7",
    "LL": "\uf6bf",
    "Lacute": "\u0139",
    "Lambda": "\u039b",
    "Lcaron": "\u013d",
    "Lcedilla": "\u013b",
    "Lcircle": "\u24c1",
    "Lcircumflexbelow": "\u1e3c",
    "Lcommaaccent": "\u013b",
    "Ldot": "\u013f",
    "Ldotaccent": "\u013f",
    "Ldotbelow": "\u1e36",
    "Ldotbelowmacron": "\u1e38",
    "Liwnarmenian": "\u053c",
    "Lj": "\u01c8",
    "Ljecyrillic": "\u0409",
    "Llinebelow": "\u1e3a",
    "Lmonospace": "\uff2c",
    "Lslash": "\u0141",
    "Lslashsmall": "\uf6f9",
    "Lsmall": "\uf76c",
    "M": "\u004d",
    "MBsquare": "\u3386",
    "Macron": "\uf6d0",
    "Macronsmall": "\uf7af",
    "Macute": "\u1e3e",
    "Mcircle": "\u24c2",
    "Mdotaccent": "\u1e40",
    "Mdotbelow": "\u1e42",
    "Menarmenian": "\u0544",
    "Mmonospace": "\uff2d",
    "Msmall": "\uf76d",
    "Mturned": "\u019c",
    "Mu": "\u039c",
    "N": "\u004e",
    "NJ": "\u01ca",
    "Nacute": "\u0143",
    "Ncaron": "\u0147",
    "Ncedilla": "\u0145",
    "Ncircle": "\u24c3",
    "Ncircumflexbelow": "\u1e4a",
    "Ncommaaccent": "\u0145",
    "Ndotaccent": "\u1e44",
    "Ndotbelow": "\u1e46",
    "Nhookleft": "\u019d",
    "Nineroman": "\u2168",
    "Nj": "\u01cb",
    "Njecyrillic": "\u040a",
    "Nlinebelow": "\u1e48",
    "Nmonospace": "\uff2e",
    "Nowarmenian": "\u0546",
    "Nsmall": "\uf76e",
    "Ntilde": "\u00d1",
    "Ntildesmall": "\uf7f1",
    "Nu": "\u039d",
    "O": "\u004f",
    "OE": "\u0152",
    "OEsmall": "\uf6fa",
    "Oacute": "\u00d3",
    "Oacutesmall": "\uf7f3",
    "Obarredcyrillic": "\u04e8",
    "Obarreddieresiscyrillic": "\u04ea",
    "Obreve": "\u014e",
    "Ocaron": "\u01d1",
    "Ocenteredtilde": "\u019f",
    "Ocircle": "\u24c4",
    "Ocircumflex": "\u00d4",
    "Ocircumflexacute": "\u1ed0",
    "Ocircumflexdotbelow": "\u1ed8",
    "Ocircumflexgrave": "\u1ed2",
    "Ocircumflexhookabove": "\u1ed4",
    "Ocircumflexsmall": "\uf7f4",
    "Ocircumflextilde": "\u1ed6",
    "Ocyrillic": "\u041e",
    "Odblacute": "\u0150",
    "Odblgrave": "\u020c",
    "Odieresis": "\u00d6",
    "Odieresiscyrillic": "\u04e6",
    "Odieresissmall": "\uf7f6",
    "Odotbelow": "\u1ecc",
    "Ogoneksmall": "\uf6fb",
    "Ograve": "\u00d2",
    "Ogravesmall": "\uf7f2",
    "Oharmenian": "\u0555",
    "Ohm": "\u2126",
    "Ohookabove": "\u1ece",
    "Ohorn": "\u01a0",
    "Ohornacute": "\u1eda",
    "Ohorndotbelow": "\u1ee2",
    "Ohorngrave": "\u1edc",
    "Ohornhookabove": "\u1ede",
    "Ohorntilde": "\u1ee0",
    "Ohungarumlaut": "\u0150",
    "Oi": "\u01a2",
    "Oinvertedbreve": "\u020e",
    "Omacron": "\u014c",
    "Omacronacute": "\u1e52",
    "Omacrongrave": "\u1e50",
    "Omega": "\u2126",
    "Omegacyrillic": "\u0460",
    "Omegagreek": "\u03a9",
    "Omegaroundcyrillic": "\u047a",
    "Omegatitlocyrillic": "\u047c",
    "Omegatonos": "\u038f",
    "Omicron": "\u039f",
    "Omicrontonos": "\u038c",
    "Omonospace": "\uff2f",
    "Oneroman": "\u2160",
    "Oogonek": "\u01ea",
    "Oogonekmacron": "\u01ec",
    "Oopen": "\u0186",
    "Oslash": "\u00d8",
    "Oslashacute": "\u01fe",
    "Oslashsmall": "\uf7f8",
    "Osmall": "\uf76f",
    "Ostrokeacute": "\u01fe",
    "Otcyrillic": "\u047e",
    "Otilde": "\u00d5",
    "Otildeacute": "\u1e4c",
    "Otildedieresis": "\u1e4e",
    "Otildesmall": "\uf7f5",
    "P": "\u0050",
    "Pacute": "\u1e54",
    "Pcircle": "\u24c5",
    "Pdotaccent": "\u1e56",
    "Pecyrillic": "\u041f",
    "Peharmenian": "\u054a",
    "Pemiddlehookcyrillic": "\u04a6",
    "Phi": "\u03a6",
    "Phook": "\u01a4",
    "Pi": "\u03a0",
    "Piwrarmenian": "\u0553",
    "Pmonospace": "\uff30",
    "Psi": "\u03a8",
    "Psicyrillic": "\u0470",
    "Psmall": "\uf770",
    "Q": "\u0051",
    "Qcircle": "\u24c6",
    "Qmonospace": "\uff31",
    "Qsmall": "\uf771",
    "R": "\u0052",
    "Raarmenian": "\u054c",
    "Racute": "\u0154",
    "Rcaron": "\u0158",
    "Rcedilla": "\u0156",
    "Rcircle": "\u24c7",
    "Rcommaaccent": "\u0156",
    "Rdblgrave": "\u0210",
    "Rdotaccent": "\u1e58",
    "Rdotbelow": "\u1e5a",
    "Rdotbelowmacron": "\u1e5c",
    "Reharmenian": "\u0550",
    "Rfraktur": "\u211c",
    "Rho": "\u03a1",
    "Ringsmall": "\uf6fc",
    "Rinvertedbreve": "\u0212",
    "Rlinebelow": "\u1e5e",
    "Rmonospace": "\uff32",
    "Rsmall": "\uf772",
    "Rsmallinverted": "\u0281",
    "Rsmallinvertedsuperior": "\u02b6",
    "S": "\u0053",
    "SF010000": "\u250c",
    "SF020000": "\u2514",
    "SF030000": "\u2510",
    "SF040000": "\u2518",
    "SF050000": "\u253c",
    "SF060000": "\u252c",
    "SF070000": "\u2534",
    "SF080000": "\u251c",
    "SF090000": "\u2524",
    "SF100000": "\u2500",
    "SF110000": "\u2502",
    "SF190000": "\u2561",
    "SF200000": "\u2562",
    "SF210000": "\u2556",
    "SF220000": "\u2555",
    "SF230000": "\u2563",
    "SF240000": "\u2551",
    "SF250000": "\u2557",
    "SF260000": "\u255d",
    "SF270000": "\u255c",
    "SF280000": "\u255b",
    "SF360000": "\u255e",
    "SF370000": "\u255f",
    "SF380000": "\u255a",
    "SF390000": "\u2554",
    "SF400000": "\u2569",
    "SF410000": "\u2566",
    "SF420000": "\u2560",
    "SF430000": "\u2550",
    "SF440000": "\u256c",
    "SF450000": "\u2567",
    "SF460000": "\u2568",
    "SF470000": "\u2564",
    "SF480000": "\u2565",
    "SF490000": "\u2559",
    "SF500000": "\u2558",
    "SF510000": "\u2552",
    "SF520000": "\u2553",
    "SF530000": "\u256b",
    "SF540000": "\u256a",
    "Sacute": "\u015a",
    "Sacutedotaccent": "\u1e64",
    "Sampigreek": "\u03e0",
    "Scaron": "\u0160",
    "Scarondotaccent": "\u1e66",
    "Scaronsmall": "\uf6fd",
    "Scedilla": "\u015e",
    "Schwa": "\u018f",
    "Schwacyrillic": "\u04d8",
    "Schwadieresiscyrillic": "\u04da",
    "Scircle": "\u24c8",
    "Scircumflex": "\u015c",
    "Scommaaccent": "\u0218",
    "Sdotaccent": "\u1e60",
    "Sdotbelow": "\u1e62",
    "Sdotbelowdotaccent": "\u1e68",
    "Seharmenian": "\u054d",
    "Sevenroman": "\u2166",
    "Shaarmenian": "\u0547",
    "Shacyrillic": "\u0428",
    "Shchacyrillic": "\u0429",
    "Sheicoptic": "\u03e2",
    "Shhacyrillic": "\u04ba",
    "Shimacoptic": "\u03ec",
    "Sigma": "\u03a3",
    "Sixroman": "\u2165",
    "Smonospace": "\uff33",
    "Softsigncyrillic": "\u042c",
    "Ssmall": "\uf773",
    "Stigmagreek": "\u03da",
    "T": "\u0054",
    "Tau": "\u03a4",
    "Tbar": "\u0166",
    "Tcaron": "\u0164",
    "Tcedilla": "\u0162",
    "Tcircle": "\u24c9",
    "Tcircumflexbelow": "\u1e70",
    "Tcommaaccent": "\u0162",
    "Tdotaccent": "\u1e6a",
    "Tdotbelow": "\u1e6c",
    "Tecyrillic": "\u0422",
    "Tedescendercyrillic": "\u04ac",
    "Tenroman": "\u2169",
    "Tetsecyrillic": "\u04b4",
    "Theta": "\u0398",
    "Thook": "\u01ac",
    "Thorn": "\u00de",
    "Thornsmall": "\uf7fe",
    "Threeroman": "\u2162",
    "Tildesmall": "\uf6fe",
    "Tiwnarmenian": "\u054f",
    "Tlinebelow": "\u1e6e",
    "Tmonospace": "\uff34",
    "Toarmenian": "\u0539",
    "Tonefive": "\u01bc",
    "Tonesix": "\u0184",
    "Tonetwo": "\u01a7",
    "Tretroflexhook": "\u01ae",
    "Tsecyrillic": "\u0426",
    "Tshecyrillic": "\u040b",
    "Tsmall": "\uf774",
    "Twelveroman": "\u216b",
    "Tworoman": "\u2161",
    "U": "\u0055",
    "Uacute": "\u00da",
    "Uacutesmall": "\uf7fa",
    "Ubreve": "\u016c",
    "Ucaron": "\u01d3",
    "Ucircle": "\u24ca",
    "Ucircumflex": "\u00db",
    "Ucircumflexbelow": "\u1e76",
    "Ucircumflexsmall": "\uf7fb",
    "Ucyrillic": "\u0423",
    "Udblacute": "\u0170",
    "Udblgrave": "\u0214",
    "Udieresis": "\u00dc",
    "Udieresisacute": "\u01d7",
    "Udieresisbelow": "\u1e72",
    "Udieresiscaron": "\u01d9",
    "Udieresiscyrillic": "\u04f0",
    "Udieresisgrave": "\u01db",
    "Udieresismacron": "\u01d5",
    "Udieresissmall": "\uf7fc",
    "Udotbelow": "\u1ee4",
    "Ugrave": "\u00d9",
    "Ugravesmall": "\uf7f9",
    "Uhookabove": "\u1ee6",
    "Uhorn": "\u01af",
    "Uhornacute": "\u1ee8",
    "Uhorndotbelow": "\u1ef0",
    "Uhorngrave": "\u1eea",
    "Uhornhookabove": "\u1eec",
    "Uhorntilde": "\u1eee",
    "Uhungarumlaut": "\u0170",
    "Uhungarumlautcyrillic": "\u04f2",
    "Uinvertedbreve": "\u0216",
    "Ukcyrillic": "\u0478",
    "Umacron": "\u016a",
    "Umacroncyrillic": "\u04ee",
    "Umacrondieresis": "\u1e7a",
    "Umonospace": "\uff35",
    "Uogonek": "\u0172",
    "Upsilon": "\u03a5",
    "Upsilon1": "\u03d2",
    "Upsilonacutehooksymbolgreek": "\u03d3",
    "Upsilonafrican": "\u01b1",
    "Upsilondieresis": "\u03ab",
    "Upsilondieresishooksymbolgreek": "\u03d4",
    "Upsilonhooksymbol": "\u03d2",
    "Upsilontonos": "\u038e",
    "Uring": "\u016e",
    "Ushortcyrillic": "\u040e",
    "Usmall": "\uf775",
    "Ustraightcyrillic": "\u04ae",
    "Ustraightstrokecyrillic": "\u04b0",
    "Utilde": "\u0168",
    "Utildeacute": "\u1e78",
    "Utildebelow": "\u1e74",
    "V": "\u0056",
    "Vcircle": "\u24cb",
    "Vdotbelow": "\u1e7e",
    "Vecyrillic": "\u0412",
    "Vewarmenian": "\u054e",
    "Vhook": "\u01b2",
    "Vmonospace": "\uff36",
    "Voarmenian": "\u0548",
    "Vsmall": "\uf776",
    "Vtilde": "\u1e7c",
    "W": "\u0057",
    "Wacute": "\u1e82",
    "Wcircle": "\u24cc",
    "Wcircumflex": "\u0174",
    "Wdieresis": "\u1e84",
    "Wdotaccent": "\u1e86",
    "Wdotbelow": "\u1e88",
    "Wgrave": "\u1e80",
    "Wmonospace": "\uff37",
    "Wsmall": "\uf777",
    "X": "\u0058",
    "Xcircle": "\u24cd",
    "Xdieresis": "\u1e8c",
    "Xdotaccent": "\u1e8a",
    "Xeharmenian": "\u053d",
    "Xi": "\u039e",
    "Xmonospace": "\uff38",
    "Xsmall": "\uf778",
    "Y": "\u0059",
    "Yacute": "\u00dd",
    "Yacutesmall": "\uf7fd",
    "Yatcyrillic": "\u0462",
    "Ycircle": "\u24ce",
    "Ycircumflex": "\u0176",
    "Ydieresis": "\u0178",
    "Ydieresissmall": "\uf7ff",
    "Ydotaccent": "\u1e8e",
    "Ydotbelow": "\u1ef4",
    "Yericyrillic": "\u042b",
    "Yerudieresiscyrillic": "\u04f8",
    "Ygrave": "\u1ef2",
    "Yhook": "\u01b3",
    "Yhookabove": "\u1ef6",
    "Yiarmenian": "\u0545",
    "Yicyrillic": "\u0407",
    "Yiwnarmenian": "\u0552",
    "Ymonospace": "\uff39",
    "Ysmall": "\uf779",
    "Ytilde": "\u1ef8",
    "Yusbigcyrillic": "\u046a",
    "Yusbigiotifiedcyrillic": "\u046c",
    "Yuslittlecyrillic": "\u0466",
    "Yuslittleiotifiedcyrillic": "\u0468",
    "Z": "\u005a",
    "Zaarmenian": "\u0536",
    "Zacute": "\u0179",
    "Zcaron": "\u017d",
    "Zcaronsmall": "\uf6ff",
    "Zcircle": "\u24cf",
    "Zcircumflex": "\u1e90",
    "Zdot": "\u017b",
    "Zdotaccent": "\u017b",
    "Zdotbelow": "\u1e92",
    "Zecyrillic": "\u0417",
    "Zedescendercyrillic": "\u0498",
    "Zedieresiscyrillic": "\u04de",
    "Zeta": "\u0396",
    "Zhearmenian": "\u053a",
    "Zhebrevecyrillic": "\u04c1",
    "Zhecyrillic": "\u0416",
    "Zhedescendercyrillic": "\u0496",
    "Zhedieresiscyrillic": "\u04dc",
    "Zlinebelow": "\u1e94",
    "Zmonospace": "\uff3a",
    "Zsmall": "\uf77a",
    "Zstroke": "\u01b5",
    "a": "\u0061",
    "aabengali": "\u0986",
    "aacute": "\u00e1",
    "aadeva": "\u0906",
    "aagujarati": "\u0a86",
    "aagurmukhi": "\u0a06",
    "aamatragurmukhi": "\u0a3e",
    "aarusquare": "\u3303",
    "aavowelsignbengali": "\u09be",
    "aavowelsigndeva": "\u093e",
    "aavowelsigngujarati": "\u0abe",
    "abbreviationmarkarmenian": "\u055f",
    "abbreviationsigndeva": "\u0970",
    "abengali": "\u0985",
    "abopomofo": "\u311a",
    "abreve": "\u0103",
    "abreveacute": "\u1eaf",
    "abrevecyrillic": "\u04d1",
    "abrevedotbelow": "\u1eb7",
    "abrevegrave": "\u1eb1",
    "abrevehookabove": "\u1eb3",
    "abrevetilde": "\u1eb5",
    "acaron": "\u01ce",
    "acircle": "\u24d0",
    "acircumflex": "\u00e2",
    "acircumflexacute": "\u1ea5",
    "acircumflexdotbelow": "\u1ead",
    "acircumflexgrave": "\u1ea7",
    "acircumflexhookabove": "\u1ea9",
    "acircumflextilde": "\u1eab",
    "acute": "\u00b4",
    "acutebelowcmb": "\u0317",
    "acutecmb": "\u0301",
    "acutecomb": "\u0301",
    "acutedeva": "\u0954",
    "acutelowmod": "\u02cf",
    "acutetonecmb": "\u0341",
    "acyrillic": "\u0430",
    "adblgrave": "\u0201",
    "addakgurmukhi": "\u0a71",
    "adeva": "\u0905",
    "adieresis": "\u00e4",
    "adieresiscyrillic": "\u04d3",
    "adieresismacron": "\u01df",
    "adotbelow": "\u1ea1",
    "adotmacron": "\u01e1",
    "ae": "\u00e6",
    "aeacute": "\u01fd",
    "aekorean": "\u3150",
    "aemacron": "\u01e3",
    "afii00208": "\u2015",
    "afii08941": "\u20a4",
    "afii10017": "\u0410",
    "afii10018": "\u0411",
    "afii10019": "\u0412",
    "afii10020": "\u0413",
    "afii10021": "\u0414",
    "afii10022": "\u0415",
    "afii10023": "\u0401",
    "afii10024": "\u0416",
    "afii10025": "\u0417",
    "afii10026": "\u0418",
    "afii10027": "\u0419",
    "afii10028": "\u041a",
    "afii10029": "\u041b",
    "afii10030": "\u041c",
    "afii10031": "\u041d",
    "afii10032": "\u041e",
    "afii10033": "\u041f",
    "afii10034": "\u0420",
    "afii10035": "\u0421",
    "afii10036": "\u0422",
    "afii10037": "\u0423",
    "afii10038": "\u0424",
    "afii10039": "\u0425",
    "afii10040": "\u0426",
    "afii10041": "\u0427",
    "afii10042": "\u0428",
    "afii10043": "\u0429",
    "afii10044": "\u042a",
    "afii10045": "\u042b",
    "afii10046": "\u042c",
    "afii10047": "\u042d",
    "afii10048": "\u042e",
    "afii10049": "\u042f",
    "afii10050": "\u0490",
    "afii10051": "\u0402",
    "afii10052": "\u0403",
    "afii10053": "\u0404",
    "afii10054": "\u0405",
    "afii10055": "\u0406",
    "afii10056": "\u0407",
    "afii10057": "\u0408",
    "afii10058": "\u0409",
    "afii10059": "\u040a",
    "afii10060": "\u040b",
    "afii10061": "\u040c",
    "afii10062": "\u040e",
    "afii10063": "\uf6c4",
    "afii10064": "\uf6c5",
    "afii10065": "\u0430",
    "afii10066": "\u0431",
    "afii10067": "\u0432",
    "afii10068": "\u0433",
    "afii10069": "\u0434",
    "afii10070": "\u0435",
    "afii10071": "\u0451",
    "afii10072": "\u0436",
    "afii10073": "\u0437",
    "afii10074": "\u0438",
    "afii10075": "\u0439",
    "afii10076": "\u043a",
    "afii10077": "\u043b",
    "afii10078": "\u043c",
    "afii10079": "\u043d",
    "afii10080": "\u043e",
    "afii10081": "\u043f",
    "afii10082": "\u0440",
    "afii10083": "\u0441",
    "afii10084": "\u0442",
    "afii10085": "\u0443",
    "afii10086": "\u0444",
    "afii10087": "\u0445",
    "afii10088": "\u0446",
    "afii10089": "\u0447",
    "afii10090": "\u0448",
    "afii10091": "\u0449",
    "afii10092": "\u044a",
    "afii10093": "\u044b",
    "afii10094": "\u044c",
    "afii10095": "\u044d",
    "afii10096": "\u044e",
    "afii10097": "\u044f",
    "afii10098": "\u0491",
    "afii10099": "\u0452",
    "afii10100": "\u0453",
    "afii10101": "\u0454",
    "afii10102": "\u0455",
    "afii10103": "\u0456",
    "afii10104": "\u0457",
    "afii10105": "\u0458",
    "afii10106": "\u0459",
    "afii10107": "\u045a",
    "afii10108": "\u045b",
    "afii10109": "\u045c",
    "afii10110": "\u045e",
    "afii10145": "\u040f",
    "afii10146": "\u0462",
    "afii10147": "\u0472",
    "afii10148": "\u0474",
    "afii10192": "\uf6c6",
    "afii10193": "\u045f",
    "afii10194": "\u0463",
    "afii10195": "\u0473",
    "afii10196": "\u0475",
    "afii10831": "\uf6c7",
    "afii10832": "\uf6c8",
    "afii10846": "\u04d9",
    "afii299": "\u200e",
    "afii300": "\u200f",
    "afii301": "\u200d",
    "afii57381": "\u066a",
    "afii57388": "\u060c",
    "afii57392": "\u0660",
    "afii57393": "\u0661",
    "afii57394": "\u0662",
    "afii57395": "\u0663",
    "afii57396": "\u0664",
    "afii57397": "\u0665",
    "afii57398": "\u0666",
    "afii57399": "\u0667",
    "afii57400": "\u0668",
    "afii57401": "\u0669",
    "afii57403": "\u061b",
    "afii57407": "\u061f",
    "afii57409": "\u0621",
    "afii57410": "\u0622",
    "afii57411": "\u0623",
    "afii57412": "\u0624",
    "afii57413": "\u0625",
    "afii57414": "\u0626",
    "afii57415": "\u0627",
    "afii57416": "\u0628",
    "afii57417": "\u0629",
    "afii57418": "\u062a",
    "afii57419": "\u062b",
    "afii57420": "\u062c",
    "afii57421": "\u062d",
    "afii57422": "\u062e",
    "afii57423": "\u062f",
    "afii57424": "\u0630",
    "afii57425": "\u0631",
    "afii57426": "\u0632",
    "afii57427": "\u0633",
    "afii57428": "\u0634",
    "afii57429": "\u0635",
    "afii57430": "\u0636",
    "afii57431": "\u0637",
    "afii57432": "\u0638",
    "afii57433": "\u0639",
    "afii57434": "\u063a",
    "afii57440": "\u0640",
    "afii57441": "\u0641",
    "afii57442": "\u0642",
    "afii57443": "\u0643",
    "afii57444": "\u0644",
    "afii57445": "\u0645",
    "afii57446": "\u0646",
    "afii57448": "\u0648",
    "afii57449": "\u0649",
    "afii57450": "\u064a",
    "afii57451": "\u064b",
    "afii57452": "\u064c",
    "afii57453": "\u064d",
    "afii57454": "\u064e",
    "afii57455": "\u064f",
    "afii57456": "\u0650",
    "afii57457": "\u0651",
    "afii57458": "\u0652",
    "afii57470": "\u0647",
    "afii57505": "\u06a4",
    "afii57506": "\u067e",
    "afii57507": "\u0686",
    "afii57508": "\u0698",
    "afii57509": "\u06af",
    "afii57511": "\u0679",
    "afii57512": "\u0688",
    "afii57513": "\u0691",
    "afii57514": "\u06ba",
    "afii57519": "\u06d2",
    "afii57534": "\u06d5",
    "afii57636": "\u20aa",
    "afii57645": "\u05be",
    "afii57658": "\u05c3",
    "afii57664": "\u05d0",
    "afii57665": "\u05d1",
    "afii57666": "\u05d2",
    "afii57667": "\u05d3",
    "afii57668": "\u05d4",
    "afii57669": "\u05d5",
    "afii57670": "\u05d6",
    "afii57671": "\u05d7",
    "afii57672": "\u05d8",
    "afii57673": "\u05d9",
    "afii57674": "\u05da",
    "afii57675": "\u05db",
    "afii57676": "\u05dc",
    "afii57677": "\u05dd",
    "afii57678": "\u05de",
    "afii57679": "\u05df",
    "afii57680": "\u05e0",
    "afii57681": "\u05e1",
    "afii57682": "\u05e2",
    "afii57683": "\u05e3",
    "afii57684": "\u05e4",
    "afii57685": "\u05e5",
    "afii57686": "\u05e6",
    "afii57687": "\u05e7",
    "afii57688": "\u05e8",
    "afii57689": "\u05e9",
    "afii57690": "\u05ea",
    "afii57694": "\ufb2a",
    "afii57695": "\ufb2b",
    "afii57700": "\ufb4b",
    "afii57705": "\ufb1f",
    "afii57716": "\u05f0",
    "afii57717": "\u05f1",
    "afii57718": "\u05f2",
    "afii57723": "\ufb35",
    "afii57793": "\u05b4",
    "afii57794": "\u05b5",
    "afii57795": "\u05b6",
    "afii57796": "\u05bb",
    "afii57797": "\u05b8",
    "afii57798": "\u05b7",
    "afii57799": "\u05b0",
    "afii57800": "\u05b2",
    "afii57801": "\u05b1",
    "afii57802": "\u05b3",
    "afii57803": "\u05c2",
    "afii57804": "\u05c1",
    "afii57806": "\u05b9",
    "afii57807": "\u05bc",
    "afii57839": "\u05bd",
    "afii57841": "\u05bf",
    "afii57842": "\u05c0",
    "afii57929": "\u02bc",
    "afii61248": "\u2105",
    "afii61289": "\u2113",
    "afii61352": "\u2116",
    "afii61573": "\u202c",
    "afii61574": "\u202d",
    "afii61575": "\u202e",
    "afii61664": "\u200c",
    "afii63167": "\u066d",
    "afii64937": "\u02bd",
    "agrave": "\u00e0",
    "agujarati": "\u0a85",
    "agurmukhi": "\u0a05",
    "ahiragana": "\u3042",
    "ahookabove": "\u1ea3",
    "aibengali": "\u0990",
    "aibopomofo": "\u311e",
    "aideva": "\u0910",
    "aiecyrillic": "\u04d5",
    "aigujarati": "\u0a90",
    "aigurmukhi": "\u0a10",
    "aimatragurmukhi": "\u0a48",
    "ainarabic": "\u0639",
    "ainfinalarabic": "\ufeca",
    "aininitialarabic": "\ufecb",
    "ainmedialarabic": "\ufecc",
    "ainvertedbreve": "\u0203",
    "aivowelsignbengali": "\u09c8",
    "aivowelsigndeva": "\u0948",
    "aivowelsigngujarati": "\u0ac8",
    "akatakana": "\u30a2",
    "akatakanahalfwidth": "\uff71",
    "akorean": "\u314f",
    "alef": "\u05d0",
    "alefarabic": "\u0627",
    "alefdageshhebrew": "\ufb30",
    "aleffinalarabic": "\ufe8e",
    "alefhamzaabovearabic": "\u0623",
    "alefhamzaabovefinalarabic": "\ufe84",
    "alefhamzabelowarabic": "\u0625",
    "alefhamzabelowfinalarabic": "\ufe88",
    "alefhebrew": "\u05d0",
    "aleflamedhebrew": "\ufb4f",
    "alefmaddaabovearabic": "\u0622",
    "alefmaddaabovefinalarabic": "\ufe82",
    "alefmaksuraarabic": "\u0649",
    "alefmaksurafinalarabic": "\ufef0",
    "alefmaksurainitialarabic": "\ufef3",
    "alefmaksuramedialarabic": "\ufef4",
    "alefpatahhebrew": "\ufb2e",
    "alefqamatshebrew": "\ufb2f",
    "aleph": "\u2135",
    "allequal": "\u224c",
    "alpha": "\u03b1",
    "alphatonos": "\u03ac",
    "amacron": "\u0101",
    "amonospace": "\uff41",
    "ampersand": "\u0026",
    "ampersandmonospace": "\uff06",
    "ampersandsmall": "\uf726",
    "amsquare": "\u33c2",
    "anbopomofo": "\u3122",
    "angbopomofo": "\u3124",
    "angkhankhuthai": "\u0e5a",
    "angle": "\u2220",
    "anglebracketleft": "\u3008",
    "anglebracketleftvertical": "\ufe3f",
    "anglebracketright": "\u3009",
    "anglebracketrightvertical": "\ufe40",
    "angleleft": "\u2329",
    "angleright": "\u232a",
    "angstrom": "\u212b",
    "anoteleia": "\u0387",
    "anudattadeva": "\u0952",
    "anusvarabengali": "\u0982",
    "anusvaradeva": "\u0902",
    "anusvaragujarati": "\u0a82",
    "aogonek": "\u0105",
    "apaatosquare": "\u3300",
    "aparen": "\u249c",
    "apostrophearmenian": "\u055a",
    "apostrophemod": "\u02bc",
    "apple": "\uf8ff",
    "approaches": "\u2250",
    "approxequal": "\u2248",
    "approxequalorimage": "\u2252",
    "approximatelyequal": "\u2245",
    "araeaekorean": "\u318e",
    "araeakorean": "\u318d",
    "arc": "\u2312",
    "arighthalfring": "\u1e9a",
    "aring": "\u00e5",
    "aringacute": "\u01fb",
    "aringbelow": "\u1e01",
    "arrowboth": "\u2194",
    "arrowdashdown": "\u21e3",
    "arrowdashleft": "\u21e0",
    "arrowdashright": "\u21e2",
    "arrowdashup": "\u21e1",
    "arrowdblboth": "\u21d4",
    "arrowdbldown": "\u21d3",
    "arrowdblleft": "\u21d0",
    "arrowdblright": "\u21d2",
    "arrowdblup": "\u21d1",
    "arrowdown": "\u2193",
    "arrowdownleft": "\u2199",
    "arrowdownright": "\u2198",
    "arrowdownwhite": "\u21e9",
    "arrowheaddownmod": "\u02c5",
    "arrowheadleftmod": "\u02c2",
    "arrowheadrightmod": "\u02c3",
    "arrowheadupmod": "\u02c4",
    "arrowhorizex": "\uf8e7",
    "arrowleft": "\u2190",
    "arrowleftdbl": "\u21d0",
    "arrowleftdblstroke": "\u21cd",
    "arrowleftoverright": "\u21c6",
    "arrowleftwhite": "\u21e6",
    "arrowright": "\u2192",
    "arrowrightdblstroke": "\u21cf",
    "arrowrightheavy": "\u279e",
    "arrowrightoverleft": "\u21c4",
    "arrowrightwhite": "\u21e8",
    "arrowtableft": "\u21e4",
    "arrowtabright": "\u21e5",
    "arrowup": "\u2191",
    "arrowupdn": "\u2195",
    "arrowupdnbse": "\u21a8",
    "arrowupdownbase": "\u21a8",
    "arrowupleft": "\u2196",
    "arrowupleftofdown": "\u21c5",
    "arrowupright": "\u2197",
    "arrowupwhite": "\u21e7",
    "arrowvertex": "\uf8e6",
    "asciicircum": "\u005e",
    "asciicircummonospace": "\uff3e",
    "asciitilde": "\u007e",
    "asciitildemonospace": "\uff5e",
    "ascript": "\u0251",
    "ascriptturned": "\u0252",
    "asmallhiragana": "\u3041",
    "asmallkatakana": "\u30a1",
    "asmallkatakanahalfwidth": "\uff67",
    "asterisk": "\u002a",
    "asteriskaltonearabic": "\u066d",
    "asteriskarabic": "\u066d",
    "asteriskmath": "\u2217",
    "asteriskmonospace": "\uff0a",
    "asterisksmall": "\ufe61",
    "asterism": "\u2042",
    "asuperior": "\uf6e9",
    "asymptoticallyequal": "\u2243",
    "at": "\u0040",
    "atilde": "\u00e3",
    "atmonospace": "\uff20",
    "atsmall": "\ufe6b",
    "aturned": "\u0250",
    "aubengali": "\u0994",
    "aubopomofo": "\u3120",
    "audeva": "\u0914",
    "augujarati": "\u0a94",
    "augurmukhi": "\u0a14",
    "aulengthmarkbengali": "\u09d7",
    "aumatragurmukhi": "\u0a4c",
    "auvowelsignbengali": "\u09cc",
    "auvowelsigndeva": "\u094c",
    "auvowelsigngujarati": "\u0acc",
    "avagrahadeva": "\u093d",
    "aybarmenian": "\u0561",
    "ayin": "\u05e2",
    "ayinaltonehebrew": "\ufb20",
    "ayinhebrew": "\u05e2",
    "b": "\u0062",
    "babengali": "\u09ac",
    "backslash": "\u005c",
    "backslashmonospace": "\uff3c",
    "badeva": "\u092c",
    "bagujarati": "\u0aac",
    "bagurmukhi": "\u0a2c",
    "bahiragana": "\u3070",
    "bahtthai": "\u0e3f",
    "bakatakana": "\u30d0",
    "bar": "\u007c",
    "barmonospace": "\uff5c",
    "bbopomofo": "\u3105",
    "bcircle": "\u24d1",
    "bdotaccent": "\u1e03",
    "bdotbelow": "\u1e05",
    "beamedsixteenthnotes": "\u266c",
    "because": "\u2235",
    "becyrillic": "\u0431",
    "beharabic": "\u0628",
    "behfinalarabic": "\ufe90",
    "behinitialarabic": "\ufe91",
    "behiragana": "\u3079",
    "behmedialarabic": "\ufe92",
    "behmeeminitialarabic": "\ufc9f",
    "behmeemisolatedarabic": "\ufc08",
    "behnoonfinalarabic": "\ufc6d",
    "bekatakana": "\u30d9",
    "benarmenian": "\u0562",
    "bet": "\u05d1",
    "beta": "\u03b2",
    "betasymbolgreek": "\u03d0",
    "betdagesh": "\ufb31",
    "betdageshhebrew": "\ufb31",
    "bethebrew": "\u05d1",
    "betrafehebrew": "\ufb4c",
    "bhabengali": "\u09ad",
    "bhadeva": "\u092d",
    "bhagujarati": "\u0aad",
    "bhagurmukhi": "\u0a2d",
    "bhook": "\u0253",
    "bihiragana": "\u3073",
    "bikatakana": "\u30d3",
    "bilabialclick": "\u0298",
    "bindigurmukhi": "\u0a02",
    "birusquare": "\u3331",
    "blackcircle": "\u25cf",
    "blackdiamond": "\u25c6",
    "blackdownpointingtriangle": "\u25bc",
    "blackleftpointingpointer": "\u25c4",
    "blackleftpointingtriangle": "\u25c0",
    "blacklenticularbracketleft": "\u3010",
    "blacklenticularbracketleftvertical": "\ufe3b",
    "blacklenticularbracketright": "\u3011",
    "blacklenticularbracketrightvertical": "\ufe3c",
    "blacklowerlefttriangle": "\u25e3",
    "blacklowerrighttriangle": "\u25e2",
    "blackrectangle": "\u25ac",
    "blackrightpointingpointer": "\u25ba",
    "blackrightpointingtriangle": "\u25b6",
    "blacksmallsquare": "\u25aa",
    "blacksmilingface": "\u263b",
    "blacksquare": "\u25a0",
    "blackstar": "\u2605",
    "blackupperlefttriangle": "\u25e4",
    "blackupperrighttriangle": "\u25e5",
    "blackuppointingsmalltriangle": "\u25b4",
    "blackuppointingtriangle": "\u25b2",
    "blank": "\u2423",
    "blinebelow": "\u1e07",
    "block": "\u2588",
    "bmonospace": "\uff42",
    "bobaimaithai": "\u0e1a",
    "bohiragana": "\u307c",
    "bokatakana": "\u30dc",
    "bparen": "\u249d",
    "bqsquare": "\u33c3",
    "braceex": "\uf8f4",
    "braceleft": "\u007b",
    "braceleftbt": "\uf8f3",
    "braceleftmid": "\uf8f2",
    "braceleftmonospace": "\uff5b",
    "braceleftsmall": "\ufe5b",
    "bracelefttp": "\uf8f1",
    "braceleftvertical": "\ufe37",
    "braceright": "\u007d",
    "bracerightbt": "\uf8fe",
    "bracerightmid": "\uf8fd",
    "bracerightmonospace": "\uff5d",
    "bracerightsmall": "\ufe5c",
    "bracerighttp": "\uf8fc",
    "bracerightvertical": "\ufe38",
    "bracketleft": "\u005b",
    "bracketleftbt": "\uf8f0",
    "bracketleftex": "\uf8ef",
    "bracketleftmonospace": "\uff3b",
    "bracketlefttp": "\uf8ee",
    "bracketright": "\u005d",
    "bracketrightbt": "\uf8fb",
    "bracketrightex": "\uf8fa",
    "bracketrightmonospace": "\uff3d",
    "bracketrighttp": "\uf8f9",
    "breve": "\u02d8",
    "brevebelowcmb": "\u032e",
    "brevecmb": "\u0306",
    "breveinvertedbelowcmb": "\u032f",
    "breveinvertedcmb": "\u0311",
    "breveinverteddoublecmb": "\u0361",
    "bridgebelowcmb": "\u032a",
    "bridgeinvertedbelowcmb": "\u033a",
    "brokenbar": "\u00a6",
    "bstroke": "\u0180",
    "bsuperior": "\uf6ea",
    "btopbar": "\u0183",
    "buhiragana": "\u3076",
    "bukatakana": "\u30d6",
    "bullet": "\u2022",
    "bulletinverse": "\u25d8",
    "bulletoperator": "\u2219",
    "bullseye": "\u25ce",
    "c": "\u0063",
    "caarmenian": "\u056e",
    "cabengali": "\u099a",
    "cacute": "\u0107",
    "cadeva": "\u091a",
    "cagujarati": "\u0a9a",
    "cagurmukhi": "\u0a1a",
    "calsquare": "\u3388",
    "candrabindubengali": "\u0981",
    "candrabinducmb": "\u0310",
    "candrabindudeva": "\u0901",
    "candrabindugujarati": "\u0a81",
    "capslock": "\u21ea",
    "careof": "\u2105",
    "caron": "\u02c7",
    "caronbelowcmb": "\u032c",
    "caroncmb": "\u030c",
    "carriagereturn": "\u21b5",
    "cbopomofo": "\u3118",
    "ccaron": "\u010d",
    "ccedilla": "\u00e7",
    "ccedillaacute": "\u1e09",
    "ccircle": "\u24d2",
    "ccircumflex": "\u0109",
    "ccurl": "\u0255",
    "cdot": "\u010b",
    "cdotaccent": "\u010b",
    "cdsquare": "\u33c5",
    "cedilla": "\u00b8",
    "cedillacmb": "\u0327",
    "cent": "\u00a2",
    "centigrade": "\u2103",
    "centinferior": "\uf6df",
    "centmonospace": "\uffe0",
    "centoldstyle": "\uf7a2",
    "centsuperior": "\uf6e0",
    "chaarmenian": "\u0579",
    "chabengali": "\u099b",
    "chadeva": "\u091b",
    "chagujarati": "\u0a9b",
    "chagurmukhi": "\u0a1b",
    "chbopomofo": "\u3114",
    "cheabkhasiancyrillic": "\u04bd",
    "checkmark": "\u2713",
    "checyrillic": "\u0447",
    "chedescenderabkhasiancyrillic": "\u04bf",
    "chedescendercyrillic": "\u04b7",
    "chedieresiscyrillic": "\u04f5",
    "cheharmenian": "\u0573",
    "chekhakassiancyrillic": "\u04cc",
    "cheverticalstrokecyrillic": "\u04b9",
    "chi": "\u03c7",
    "chieuchacirclekorean": "\u3277",
    "chieuchaparenkorean": "\u3217",
    "chieuchcirclekorean": "\u3269",
    "chieuchkorean": "\u314a",
    "chieuchparenkorean": "\u3209",
    "chochangthai": "\u0e0a",
    "chochanthai": "\u0e08",
    "chochingthai": "\u0e09",
    "chochoethai": "\u0e0c",
    "chook": "\u0188",
    "cieucacirclekorean": "\u3276",
    "cieucaparenkorean": "\u3216",
    "cieuccirclekorean": "\u3268",
    "cieuckorean": "\u3148",
    "cieucparenkorean": "\u3208",
    "cieucuparenkorean": "\u321c",
    "circle": "\u25cb",
    "circlemultiply": "\u2297",
    "circleot": "\u2299",
    "circleplus": "\u2295",
    "circlepostalmark": "\u3036",
    "circlewithlefthalfblack": "\u25d0",
    "circlewithrighthalfblack": "\u25d1",
    "circumflex": "\u02c6",
    "circumflexbelowcmb": "\u032d",
    "circumflexcmb": "\u0302",
    "clear": "\u2327",
    "clickalveolar": "\u01c2",
    "clickdental": "\u01c0",
    "clicklateral": "\u01c1",
    "clickretroflex": "\u01c3",
    "club": "\u2663",
    "clubsuitblack": "\u2663",
    "clubsuitwhite": "\u2667",
    "cmcubedsquare": "\u33a4",
    "cmonospace": "\uff43",
    "cmsquaredsquare": "\u33a0",
    "coarmenian": "\u0581",
    "colon": "\u003a",
    "colonmonetary": "\u20a1",
    "colonmonospace": "\uff1a",
    "colonsign": "\u20a1",
    "colonsmall": "\ufe55",
    "colontriangularhalfmod": "\u02d1",
    "colontriangularmod": "\u02d0",
    "comma": "\u002c",
    "commaabovecmb": "\u0313",
    "commaaboverightcmb": "\u0315",
    "commaaccent": "\uf6c3",
    "commaarabic": "\u060c",
    "commaarmenian": "\u055d",
    "commainferior": "\uf6e1",
    "commamonospace": "\uff0c",
    "commareversedabovecmb": "\u0314",
    "commareversedmod": "\u02bd",
    "commasmall": "\ufe50",
    "commasuperior": "\uf6e2",
    "commaturnedabovecmb": "\u0312",
    "commaturnedmod": "\u02bb",
    "compass": "\u263c",
    "congruent": "\u2245",
    "contourintegral": "\u222e",
    "control": "\u2303",
    "controlACK": "\u0006",
    "controlBEL": "\u0007",
    "controlBS": "\u0008",
    "controlCAN": "\u0018",
    "controlCR": "\u000d",
    "controlDC1": "\u0011",
    "controlDC2": "\u0012",
    "controlDC3": "\u0013",
    "controlDC4": "\u0014",
    "controlDEL": "\u007f",
    "controlDLE": "\u0010",
    "controlEM": "\u0019",
    "controlENQ": "\u0005",
    "controlEOT": "\u0004",
    "controlESC": "\u001b",
    "controlETB": "\u0017",
    "controlETX": "\u0003",
    "controlFF": "\u000c",
    "controlFS": "\u001c",
    "controlGS": "\u001d",
    "controlHT": "\u0009",
    "controlLF": "\u000a",
    "controlNAK": "\u0015",
    "controlRS": "\u001e",
    "controlSI": "\u000f",
    "controlSO": "\u000e",
    "controlSOT": "\u0002",
    "controlSTX": "\u0001",
    "controlSUB": "\u001a",
    "controlSYN": "\u0016",
    "controlUS": "\u001f",
    "controlVT": "\u000b",
    "copyright": "\u00a9",
    "copyrightsans": "\uf8e9",
    "copyrightserif": "\uf6d9",
    "cornerbracketleft": "\u300c",
    "cornerbracketlefthalfwidth": "\uff62",
    "cornerbracketleftvertical": "\ufe41",
    "cornerbracketright": "\u300d",
    "cornerbracketrighthalfwidth": "\uff63",
    "cornerbracketrightvertical": "\ufe42",
    "corporationsquare": "\u337f",
    "cosquare": "\u33c7",
    "coverkgsquare": "\u33c6",
    "cparen": "\u249e",
    "cruzeiro": "\u20a2",
    "cstretched": "\u0297",
    "curlyand": "\u22cf",
    "curlyor": "\u22ce",
    "currency": "\u00a4",
    "cyrBreve": "\uf6d1",
    "cyrFlex": "\uf6d2",
    "cyrbreve": "\uf6d4",
    "cyrflex": "\uf6d5",
    "d": "\u0064",
    "daarmenian": "\u0564",
    "dabengali": "\u09a6",
    "dadarabic": "\u0636",
    "dadeva": "\u0926",
    "dadfinalarabic": "\ufebe",
    "dadinitialarabic": "\ufebf",
    "dadmedialarabic": "\ufec0",
    "dagesh": "\u05bc",
    "dageshhebrew": "\u05bc",
    "dagger": "\u2020",
    "daggerdbl": "\u2021",
    "dagujarati": "\u0aa6",
    "dagurmukhi": "\u0a26",
    "dahiragana": "\u3060",
    "dakatakana": "\u30c0",
    "dalarabic": "\u062f",
    "dalet": "\u05d3",
    "daletdagesh": "\ufb33",
    "daletdageshhebrew": "\ufb33",
    "dalethatafpatah": "\u05d3\u05b2",
    "dalethatafpatahhebrew": "\u05d3\u05b2",
    "dalethatafsegol": "\u05d3\u05b1",
    "dalethatafsegolhebrew": "\u05d3\u05b1",
    "dalethebrew": "\u05d3",
    "dalethiriq": "\u05d3\u05b4",
    "dalethiriqhebrew": "\u05d3\u05b4",
    "daletholam": "\u05d3\u05b9",
    "daletholamhebrew": "\u05d3\u05b9",
    "daletpatah": "\u05d3\u05b7",
    "daletpatahhebrew": "\u05d3\u05b7",
    "daletqamats": "\u05d3\u05b8",
    "daletqamatshebrew": "\u05d3\u05b8",
    "daletqubuts": "\u05d3\u05bb",
    "daletqubutshebrew": "\u05d3\u05bb",
    "daletsegol": "\u05d3\u05b6",
    "daletsegolhebrew": "\u05d3\u05b6",
    "daletsheva": "\u05d3\u05b0",
    "daletshevahebrew": "\u05d3\u05b0",
    "dalettsere": "\u05d3\u05b5",
    "dalettserehebrew": "\u05d3\u05b5",
    "dalfinalarabic": "\ufeaa",
    "dammaarabic": "\u064f",
    "dammalowarabic": "\u064f",
    "dammatanaltonearabic": "\u064c",
    "dammatanarabic": "\u064c",
    "danda": "\u0964",
    "dargahebrew": "\u05a7",
    "dargalefthebrew": "\u05a7",
    "dasiapneumatacyrilliccmb": "\u0485",
    "dblGrave": "\uf6d3",
    "dblanglebracketleft": "\u300a",
    "dblanglebracketleftvertical": "\ufe3d",
    "dblanglebracketright": "\u300b",
    "dblanglebracketrightvertical": "\ufe3e",
    "dblarchinvertedbelowcmb": "\u032b",
    "dblarrowleft": "\u21d4",
    "dblarrowright": "\u21d2",
    "dbldanda": "\u0965",
    "dblgrave": "\uf6d6",
    "dblgravecmb": "\u030f",
    "dblintegral": "\u222c",
    "dbllowline": "\u2017",
    "dbllowlinecmb": "\u0333",
    "dbloverlinecmb": "\u033f",
    "dblprimemod": "\u02ba",
    "dblverticalbar": "\u2016",
    "dblverticallineabovecmb": "\u030e",
    "dbopomofo": "\u3109",
    "dbsquare": "\u33c8",
    "dcaron": "\u010f",
    "dcedilla": "\u1e11",
    "dcircle": "\u24d3",
    "dcircumflexbelow": "\u1e13",
    "dcroat": "\u0111",
    "ddabengali": "\u09a1",
    "ddadeva": "\u0921",
    "ddagujarati": "\u0aa1",
    "ddagurmukhi": "\u0a21",
    "ddalarabic": "\u0688",
    "ddalfinalarabic": "\ufb89",
    "dddhadeva": "\u095c",
    "ddhabengali": "\u09a2",
    "ddhadeva": "\u0922",
    "ddhagujarati": "\u0aa2",
    "ddhagurmukhi": "\u0a22",
    "ddotaccent": "\u1e0b",
    "ddotbelow": "\u1e0d",
    "decimalseparatorarabic": "\u066b",
    "decimalseparatorpersian": "\u066b",
    "decyrillic": "\u0434",
    "degree": "\u00b0",
    "dehihebrew": "\u05ad",
    "dehiragana": "\u3067",
    "deicoptic": "\u03ef",
    "dekatakana": "\u30c7",
    "deleteleft": "\u232b",
    "deleteright": "\u2326",
    "delta": "\u03b4",
    "deltaturned": "\u018d",
    "denominatorminusonenumeratorbengali": "\u09f8",
    "dezh": "\u02a4",
    "dhabengali": "\u09a7",
    "dhadeva": "\u0927",
    "dhagujarati": "\u0aa7",
    "dhagurmukhi": "\u0a27",
    "dhook": "\u0257",
    "dialytikatonos": "\u0385",
    "dialytikatonoscmb": "\u0344",
    "diamond": "\u2666",
    "diamondsuitwhite": "\u2662",
    "dieresis": "\u00a8",
    "dieresisacute": "\uf6d7",
    "dieresisbelowcmb": "\u0324",
    "dieresiscmb": "\u0308",
    "dieresisgrave": "\uf6d8",
    "dieresistonos": "\u0385",
    "dihiragana": "\u3062",
    "dikatakana": "\u30c2",
    "dittomark": "\u3003",
    "divide": "\u00f7",
    "divides": "\u2223",
    "divisionslash": "\u2215",
    "djecyrillic": "\u0452",
    "dkshade": "\u2593",
    "dlinebelow": "\u1e0f",
    "dlsquare": "\u3397",
    "dmacron": "\u0111",
    "dmonospace": "\uff44",
    "dnblock": "\u2584",
    "dochadathai": "\u0e0e",
    "dodekthai": "\u0e14",
    "dohiragana": "\u3069",
    "dokatakana": "\u30c9",
    "dollar": "\u0024",
    "dollarinferior": "\uf6e3",
    "dollarmonospace": "\uff04",
    "dollaroldstyle": "\uf724",
    "dollarsmall": "\ufe69",
    "dollarsuperior": "\uf6e4",
    "dong": "\u20ab",
    "dorusquare": "\u3326",
    "dotaccent": "\u02d9",
    "dotaccentcmb": "\u0307",
    "dotbelowcmb": "\u0323",
    "dotbelowcomb": "\u0323",
    "dotkatakana": "\u30fb",
    "dotlessi": "\u0131",
    "dotlessj": "\uf6be",
    "dotlessjstrokehook": "\u0284",
    "dotmath": "\u22c5",
    "dottedcircle": "\u25cc",
    "doubleyodpatah": "\ufb1f",
    "doubleyodpatahhebrew": "\ufb1f",
    "downtackbelowcmb": "\u031e",
    "downtackmod": "\u02d5",
    "dparen": "\u249f",
    "dsuperior": "\uf6eb",
    "dtail": "\u0256",
    "dtopbar": "\u018c",
    "duhiragana": "\u3065",
    "dukatakana": "\u30c5",
    "dz": "\u01f3",
    "dzaltone": "\u02a3",
    "dzcaron": "\u01c6",
    "dzcurl": "\u02a5",
    "dzeabkhasiancyrillic": "\u04e1",
    "dzecyrillic": "\u0455",
    "dzhecyrillic": "\u045f",
    "e": "\u0065",
    "eacute": "\u00e9",
    "earth": "\u2641",
    "ebengali": "\u098f",
    "ebopomofo": "\u311c",
    "ebreve": "\u0115",
    "ecandradeva": "\u090d",
    "ecandragujarati": "\u0a8d",
    "ecandravowelsigndeva": "\u0945",
    "ecandravowelsigngujarati": "\u0ac5",
    "ecaron": "\u011b",
    "ecedillabreve": "\u1e1d",
    "echarmenian": "\u0565",
    "echyiwnarmenian": "\u0587",
    "ecircle": "\u24d4",
    "ecircumflex": "\u00ea",
    "ecircumflexacute": "\u1ebf",
    "ecircumflexbelow": "\u1e19",
    "ecircumflexdotbelow": "\u1ec7",
    "ecircumflexgrave": "\u1ec1",
    "ecircumflexhookabove": "\u1ec3",
    "ecircumflextilde": "\u1ec5",
    "ecyrillic": "\u0454",
    "edblgrave": "\u0205",
    "edeva": "\u090f",
    "edieresis": "\u00eb",
    "edot": "\u0117",
    "edotaccent": "\u0117",
    "edotbelow": "\u1eb9",
    "eegurmukhi": "\u0a0f",
    "eematragurmukhi": "\u0a47",
    "efcyrillic": "\u0444",
    "egrave": "\u00e8",
    "egujarati": "\u0a8f",
    "eharmenian": "\u0567",
    "ehbopomofo": "\u311d",
    "ehiragana": "\u3048",
    "ehookabove": "\u1ebb",
    "eibopomofo": "\u311f",
    "eight": "\u0038",
    "eightarabic": "\u0668",
    "eightbengali": "\u09ee",
    "eightcircle": "\u2467",
    "eightcircleinversesansserif": "\u2791",
    "eightdeva": "\u096e",
    "eighteencircle": "\u2471",
    "eighteenparen": "\u2485",
    "eighteenperiod": "\u2499",
    "eightgujarati": "\u0aee",
    "eightgurmukhi": "\u0a6e",
    "eighthackarabic": "\u0668",
    "eighthangzhou": "\u3028",
    "eighthnotebeamed": "\u266b",
    "eightideographicparen": "\u3227",
    "eightinferior": "\u2088",
    "eightmonospace": "\uff18",
    "eightoldstyle": "\uf738",
    "eightparen": "\u247b",
    "eightperiod": "\u248f",
    "eightpersian": "\u06f8",
    "eightroman": "\u2177",
    "eightsuperior": "\u2078",
    "eightthai": "\u0e58",
    "einvertedbreve": "\u0207",
    "eiotifiedcyrillic": "\u0465",
    "ekatakana": "\u30a8",
    "ekatakanahalfwidth": "\uff74",
    "ekonkargurmukhi": "\u0a74",
    "ekorean": "\u3154",
    "elcyrillic": "\u043b",
    "element": "\u2208",
    "elevencircle": "\u246a",
    "elevenparen": "\u247e",
    "elevenperiod": "\u2492",
    "elevenroman": "\u217a",
    "ellipsis": "\u2026",
    "ellipsisvertical": "\u22ee",
    "emacron": "\u0113",
    "emacronacute": "\u1e17",
    "emacrongrave": "\u1e15",
    "emcyrillic": "\u043c",
    "emdash": "\u2014",
    "emdashvertical": "\ufe31",
    "emonospace": "\uff45",
    "emphasismarkarmenian": "\u055b",
    "emptyset": "\u2205",
    "enbopomofo": "\u3123",
    "encyrillic": "\u043d",
    "endash": "\u2013",
    "endashvertical": "\ufe32",
    "endescendercyrillic": "\u04a3",
    "eng": "\u014b",
    "engbopomofo": "\u3125",
    "enghecyrillic": "\u04a5",
    "enhookcyrillic": "\u04c8",
    "enspace": "\u2002",
    "eogonek": "\u0119",
    "eokorean": "\u3153",
    "eopen": "\u025b",
    "eopenclosed": "\u029a",
    "eopenreversed": "\u025c",
    "eopenreversedclosed": "\u025e",
    "eopenreversedhook": "\u025d",
    "eparen": "\u24a0",
    "epsilon": "\u03b5",
    "epsilontonos": "\u03ad",
    "equal": "\u003d",
    "equalmonospace": "\uff1d",
    "equalsmall": "\ufe66",
    "equalsuperior": "\u207c",
    "equivalence": "\u2261",
    "erbopomofo": "\u3126",
    "ercyrillic": "\u0440",
    "ereversed": "\u0258",
    "ereversedcyrillic": "\u044d",
    "escyrillic": "\u0441",
    "esdescendercyrillic": "\u04ab",
    "esh": "\u0283",
    "eshcurl": "\u0286",
    "eshortdeva": "\u090e",
    "eshortvowelsigndeva": "\u0946",
    "eshreversedloop": "\u01aa",
    "eshsquatreversed": "\u0285",
    "esmallhiragana": "\u3047",
    "esmallkatakana": "\u30a7",
    "esmallkatakanahalfwidth": "\uff6a",
    "estimated": "\u212e",
    "esuperior": "\uf6ec",
    "eta": "\u03b7",
    "etarmenian": "\u0568",
    "etatonos": "\u03ae",
    "eth": "\u00f0",
    "etilde": "\u1ebd",
    "etildebelow": "\u1e1b",
    "etnahtafoukhhebrew": "\u0591",
    "etnahtafoukhlefthebrew": "\u0591",
    "etnahtahebrew": "\u0591",
    "etnahtalefthebrew": "\u0591",
    "eturned": "\u01dd",
    "eukorean": "\u3161",
    "euro": "\u20ac",
    "evowelsignbengali": "\u09c7",
    "evowelsigndeva": "\u0947",
    "evowelsigngujarati": "\u0ac7",
    "exclam": "\u0021",
    "exclamarmenian": "\u055c",
    "exclamdbl": "\u203c",
    "exclamdown": "\u00a1",
    "exclamdownsmall": "\uf7a1",
    "exclammonospace": "\uff01",
    "exclamsmall": "\uf721",
    "existential": "\u2203",
    "ezh": "\u0292",
    "ezhcaron": "\u01ef",
    "ezhcurl": "\u0293",
    "ezhreversed": "\u01b9",
    "ezhtail": "\u01ba",
    "f": "\u0066",
    "fadeva": "\u095e",
    "fagurmukhi": "\u0a5e",
    "fahrenheit": "\u2109",
    "fathaarabic": "\u064e",
    "fathalowarabic": "\u064e",
    "fathatanarabic": "\u064b",
    "fbopomofo": "\u3108",
    "fcircle": "\u24d5",
    "fdotaccent": "\u1e1f",
    "feharabic": "\u0641",
    "feharmenian": "\u0586",
    "fehfinalarabic": "\ufed2",
    "fehinitialarabic": "\ufed3",
    "fehmedialarabic": "\ufed4",
    "feicoptic": "\u03e5",
    "female": "\u2640",
    "ff": "\ufb00",
    "ffi": "\ufb03",
    "ffl": "\ufb04",
    "fi": "\ufb01",
    "fifteencircle": "\u246e",
    "fifteenparen": "\u2482",
    "fifteenperiod": "\u2496",
    "figuredash": "\u2012",
    "filledbox": "\u25a0",
    "filledrect": "\u25ac",
    "finalkaf": "\u05da",
    "finalkafdagesh": "\ufb3a",
    "finalkafdageshhebrew": "\ufb3a",
    "finalkafhebrew": "\u05da",
    "finalkafqamats": "\u05da\u05b8",
    "finalkafqamatshebrew": "\u05da\u05b8",
    "finalkafsheva": "\u05da\u05b0",
    "finalkafshevahebrew": "\u05da\u05b0",
    "finalmem": "\u05dd",
    "finalmemhebrew": "\u05dd",
    "finalnun": "\u05df",
    "finalnunhebrew": "\u05df",
    "finalpe": "\u05e3",
    "finalpehebrew": "\u05e3",
    "finaltsadi": "\u05e5",
    "finaltsadihebrew": "\u05e5",
    "firsttonechinese": "\u02c9",
    "fisheye": "\u25c9",
    "fitacyrillic": "\u0473",
    "five": "\u0035",
    "fivearabic": "\u0665",
    "fivebengali": "\u09eb",
    "fivecircle": "\u2464",
    "fivecircleinversesansserif": "\u278e",
    "fivedeva": "\u096b",
    "fiveeighths": "\u215d",
    "fivegujarati": "\u0aeb",
    "fivegurmukhi": "\u0a6b",
    "fivehackarabic": "\u0665",
    "fivehangzhou": "\u3025",
    "fiveideographicparen": "\u3224",
    "fiveinferior": "\u2085",
    "fivemonospace": "\uff15",
    "fiveoldstyle": "\uf735",
    "fiveparen": "\u2478",
    "fiveperiod": "\u248c",
    "fivepersian": "\u06f5",
    "fiveroman": "\u2174",
    "fivesuperior": "\u2075",
    "fivethai": "\u0e55",
    "fl": "\ufb02",
    "florin": "\u0192",
    "fmonospace": "\uff46",
    "fmsquare": "\u3399",
    "fofanthai": "\u0e1f",
    "fofathai": "\u0e1d",
    "fongmanthai": "\u0e4f",
    "forall": "\u2200",
    "four": "\u0034",
    "fourarabic": "\u0664",
    "fourbengali": "\u09ea",
    "fourcircle": "\u2463",
    "fourcircleinversesansserif": "\u278d",
    "fourdeva": "\u096a",
    "fourgujarati": "\u0aea",
    "fourgurmukhi": "\u0a6a",
    "fourhackarabic": "\u0664",
    "fourhangzhou": "\u3024",
    "fourideographicparen": "\u3223",
    "fourinferior": "\u2084",
    "fourmonospace": "\uff14",
    "fournumeratorbengali": "\u09f7",
    "fouroldstyle": "\uf734",
    "fourparen": "\u2477",
    "fourperiod": "\u248b",
    "fourpersian": "\u06f4",
    "fourroman": "\u2173",
    "foursuperior": "\u2074",
    "fourteencircle": "\u246d",
    "fourteenparen": "\u2481",
    "fourteenperiod": "\u2495",
    "fourthai": "\u0e54",
    "fourthtonechinese": "\u02cb",
    "fparen": "\u24a1",
    "fraction": "\u2044",
    "franc": "\u20a3",
    "g": "\u0067",
    "gabengali": "\u0997",
    "gacute": "\u01f5",
    "gadeva": "\u0917",
    "gafarabic": "\u06af",
    "gaffinalarabic": "\ufb93",
    "gafinitialarabic": "\ufb94",
    "gafmedialarabic": "\ufb95",
    "gagujarati": "\u0a97",
    "gagurmukhi": "\u0a17",
    "gahiragana": "\u304c",
    "gakatakana": "\u30ac",
    "gamma": "\u03b3",
    "gammalatinsmall": "\u0263",
    "gammasuperior": "\u02e0",
    "gangiacoptic": "\u03eb",
    "gbopomofo": "\u310d",
    "gbreve": "\u011f",
    "gcaron": "\u01e7",
    "gcedilla": "\u0123",
    "gcircle": "\u24d6",
    "gcircumflex": "\u011d",
    "gcommaaccent": "\u0123",
    "gdot": "\u0121",
    "gdotaccent": "\u0121",
    "gecyrillic": "\u0433",
    "gehiragana": "\u3052",
    "gekatakana": "\u30b2",
    "geometricallyequal": "\u2251",
    "gereshaccenthebrew": "\u059c",
    "gereshhebrew": "\u05f3",
    "gereshmuqdamhebrew": "\u059d",
    "germandbls": "\u00df",
    "gershayimaccenthebrew": "\u059e",
    "gershayimhebrew": "\u05f4",
    "getamark": "\u3013",
    "ghabengali": "\u0998",
    "ghadarmenian": "\u0572",
    "ghadeva": "\u0918",
    "ghagujarati": "\u0a98",
    "ghagurmukhi": "\u0a18",
    "ghainarabic": "\u063a",
    "ghainfinalarabic": "\ufece",
    "ghaininitialarabic": "\ufecf",
    "ghainmedialarabic": "\ufed0",
    "ghemiddlehookcyrillic": "\u0495",
    "ghestrokecyrillic": "\u0493",
    "gheupturncyrillic": "\u0491",
    "ghhadeva": "\u095a",
    "ghhagurmukhi": "\u0a5a",
    "ghook": "\u0260",
    "ghzsquare": "\u3393",
    "gihiragana": "\u304e",
    "gikatakana": "\u30ae",
    "gimarmenian": "\u0563",
    "gimel": "\u05d2",
    "gimeldagesh": "\ufb32",
    "gimeldageshhebrew": "\ufb32",
    "gimelhebrew": "\u05d2",
    "gjecyrillic": "\u0453",
    "glottalinvertedstroke": "\u01be",
    "glottalstop": "\u0294",
    "glottalstopinverted": "\u0296",
    "glottalstopmod": "\u02c0",
    "glottalstopreversed": "\u0295",
    "glottalstopreversedmod": "\u02c1",
    "glottalstopreversedsuperior": "\u02e4",
    "glottalstopstroke": "\u02a1",
    "glottalstopstrokereversed": "\u02a2",
    "gmacron": "\u1e21",
    "gmonospace": "\uff47",
    "gohiragana": "\u3054",
    "gokatakana": "\u30b4",
    "gparen": "\u24a2",
    "gpasquare": "\u33ac",
    "gradient": "\u2207",
    "grave": "\u0060",
    "gravebelowcmb": "\u0316",
    "gravecmb": "\u0300",
    "gravecomb": "\u0300",
    "gravedeva": "\u0953",
    "gravelowmod": "\u02ce",
    "gravemonospace": "\uff40",
    "gravetonecmb": "\u0340",
    "greater": "\u003e",
    "greaterequal": "\u2265",
    "greaterequalorless": "\u22db",
    "greatermonospace": "\uff1e",
    "greaterorequivalent": "\u2273",
    "greaterorless": "\u2277",
    "greateroverequal": "\u2267",
    "greatersmall": "\ufe65",
    "gscript": "\u0261",
    "gstroke": "\u01e5",
    "guhiragana": "\u3050",
    "guillemotleft": "\u00ab",
    "guillemotright": "\u00bb",
    "guilsinglleft": "\u2039",
    "guilsinglright": "\u203a",
    "gukatakana": "\u30b0",
    "guramusquare": "\u3318",
    "gysquare": "\u33c9",
    "h": "\u0068",
    "haabkhasiancyrillic": "\u04a9",
    "haaltonearabic": "\u06c1",
    "habengali": "\u09b9",
    "hadescendercyrillic": "\u04b3",
    "hadeva": "\u0939",
    "hagujarati": "\u0ab9",
    "hagurmukhi": "\u0a39",
    "haharabic": "\u062d",
    "hahfinalarabic": "\ufea2",
    "hahinitialarabic": "\ufea3",
    "hahiragana": "\u306f",
    "hahmedialarabic": "\ufea4",
    "haitusquare": "\u332a",
    "hakatakana": "\u30cf",
    "hakatakanahalfwidth": "\uff8a",
    "halantgurmukhi": "\u0a4d",
    "hamzaarabic": "\u0621",
    "hamzadammaarabic": "\u0621\u064f",
    "hamzadammatanarabic": "\u0621\u064c",
    "hamzafathaarabic": "\u0621\u064e",
    "hamzafathatanarabic": "\u0621\u064b",
    "hamzalowarabic": "\u0621",
    "hamzalowkasraarabic": "\u0621\u0650",
    "hamzalowkasratanarabic": "\u0621\u064d",
    "hamzasukunarabic": "\u0621\u0652",
    "hangulfiller": "\u3164",
    "hardsigncyrillic": "\u044a",
    "harpoonleftbarbup": "\u21bc",
    "harpoonrightbarbup": "\u21c0",
    "hasquare": "\u33ca",
    "hatafpatah": "\u05b2",
    "hatafpatah16": "\u05b2",
    "hatafpatah23": "\u05b2",
    "hatafpatah2f": "\u05b2",
    "hatafpatahhebrew": "\u05b2",
    "hatafpatahnarrowhebrew": "\u05b2",
    "hatafpatahquarterhebrew": "\u05b2",
    "hatafpatahwidehebrew": "\u05b2",
    "hatafqamats": "\u05b3",
    "hatafqamats1b": "\u05b3",
    "hatafqamats28": "\u05b3",
    "hatafqamats34": "\u05b3",
    "hatafqamatshebrew": "\u05b3",
    "hatafqamatsnarrowhebrew": "\u05b3",
    "hatafqamatsquarterhebrew": "\u05b3",
    "hatafqamatswidehebrew": "\u05b3",
    "hatafsegol": "\u05b1",
    "hatafsegol17": "\u05b1",
    "hatafsegol24": "\u05b1",
    "hatafsegol30": "\u05b1",
    "hatafsegolhebrew": "\u05b1",
    "hatafsegolnarrowhebrew": "\u05b1",
    "hatafsegolquarterhebrew": "\u05b1",
    "hatafsegolwidehebrew": "\u05b1",
    "hbar": "\u0127",
    "hbopomofo": "\u310f",
    "hbrevebelow": "\u1e2b",
    "hcedilla": "\u1e29",
    "hcircle": "\u24d7",
    "hcircumflex": "\u0125",
    "hdieresis": "\u1e27",
    "hdotaccent": "\u1e23",
    "hdotbelow": "\u1e25",
    "he": "\u05d4",
    "heart": "\u2665",
    "heartsuitblack": "\u2665",
    "heartsuitwhite": "\u2661",
    "hedagesh": "\ufb34",
    "hedageshhebrew": "\ufb34",
    "hehaltonearabic": "\u06c1",
    "heharabic": "\u0647",
    "hehebrew": "\u05d4",
    "hehfinalaltonearabic": "\ufba7",
    "hehfinalalttwoarabic": "\ufeea",
    "hehfinalarabic": "\ufeea",
    "hehhamzaabovefinalarabic": "\ufba5",
    "hehhamzaaboveisolatedarabic": "\ufba4",
    "hehinitialaltonearabic": "\ufba8",
    "hehinitialarabic": "\ufeeb",
    "hehiragana": "\u3078",
    "hehmedialaltonearabic": "\ufba9",
    "hehmedialarabic": "\ufeec",
    "heiseierasquare": "\u337b",
    "hekatakana": "\u30d8",
    "hekatakanahalfwidth": "\uff8d",
    "hekutaarusquare": "\u3336",
    "henghook": "\u0267",
    "herutusquare": "\u3339",
    "het": "\u05d7",
    "hethebrew": "\u05d7",
    "hhook": "\u0266",
    "hhooksuperior": "\u02b1",
    "hieuhacirclekorean": "\u327b",
    "hieuhaparenkorean": "\u321b",
    "hieuhcirclekorean": "\u326d",
    "hieuhkorean": "\u314e",
    "hieuhparenkorean": "\u320d",
    "hihiragana": "\u3072",
    "hikatakana": "\u30d2",
    "hikatakanahalfwidth": "\uff8b",
    "hiriq": "\u05b4",
    "hiriq14": "\u05b4",
    "hiriq21": "\u05b4",
    "hiriq2d": "\u05b4",
    "hiriqhebrew": "\u05b4",
    "hiriqnarrowhebrew": "\u05b4",
    "hiriqquarterhebrew": "\u05b4",
    "hiriqwidehebrew": "\u05b4",
    "hlinebelow": "\u1e96",
    "hmonospace": "\uff48",
    "hoarmenian": "\u0570",
    "hohipthai": "\u0e2b",
    "hohiragana": "\u307b",
    "hokatakana": "\u30db",
    "hokatakanahalfwidth": "\uff8e",
    "holam": "\u05b9",
    "holam19": "\u05b9",
    "holam26": "\u05b9",
    "holam32": "\u05b9",
    "holamhebrew": "\u05b9",
    "holamnarrowhebrew": "\u05b9",
    "holamquarterhebrew": "\u05b9",
    "holamwidehebrew": "\u05b9",
    "honokhukthai": "\u0e2e",
    "hookabovecomb": "\u0309",
    "hookcmb": "\u0309",
    "hookpalatalizedbelowcmb": "\u0321",
    "hookretroflexbelowcmb": "\u0322",
    "hoonsquare": "\u3342",
    "horicoptic": "\u03e9",
    "horizontalbar": "\u2015",
    "horncmb": "\u031b",
    "hotsprings": "\u2668",
    "house": "\u2302",
    "hparen": "\u24a3",
    "hsuperior": "\u02b0",
    "hturned": "\u0265",
    "huhiragana": "\u3075",
    "huiitosquare": "\u3333",
    "hukatakana": "\u30d5",
    "hukatakanahalfwidth": "\uff8c",
    "hungarumlaut": "\u02dd",
    "hungarumlautcmb": "\u030b",
    "hv": "\u0195",
    "hyphen": "\u002d",
    "hypheninferior": "\uf6e5",
    "hyphenmonospace": "\uff0d",
    "hyphensmall": "\ufe63",
    "hyphensuperior": "\uf6e6",
    "hyphentwo": "\u2010",
    "i": "\u0069",
    "iacute": "\u00ed",
    "iacyrillic": "\u044f",
    "ibengali": "\u0987",
    "ibopomofo": "\u3127",
    "ibreve": "\u012d",
    "icaron": "\u01d0",
    "icircle": "\u24d8",
    "icircumflex": "\u00ee",
    "icyrillic": "\u0456",
    "idblgrave": "\u0209",
    "ideographearthcircle": "\u328f",
    "ideographfirecircle": "\u328b",
    "ideographicallianceparen": "\u323f",
    "ideographiccallparen": "\u323a",
    "ideographiccentrecircle": "\u32a5",
    "ideographicclose": "\u3006",
    "ideographiccomma": "\u3001",
    "ideographiccommaleft": "\uff64",
    "ideographiccongratulationparen": "\u3237",
    "ideographiccorrectcircle": "\u32a3",
    "ideographicearthparen": "\u322f",
    "ideographicenterpriseparen": "\u323d",
    "ideographicexcellentcircle": "\u329d",
    "ideographicfestivalparen": "\u3240",
    "ideographicfinancialcircle": "\u3296",
    "ideographicfinancialparen": "\u3236",
    "ideographicfireparen": "\u322b",
    "ideographichaveparen": "\u3232",
    "ideographichighcircle": "\u32a4",
    "ideographiciterationmark": "\u3005",
    "ideographiclaborcircle": "\u3298",
    "ideographiclaborparen": "\u3238",
    "ideographicleftcircle": "\u32a7",
    "ideographiclowcircle": "\u32a6",
    "ideographicmedicinecircle": "\u32a9",
    "ideographicmetalparen": "\u322e",
    "ideographicmoonparen": "\u322a",
    "ideographicnameparen": "\u3234",
    "ideographicperiod": "\u3002",
    "ideographicprintcircle": "\u329e",
    "ideographicreachparen": "\u3243",
    "ideographicrepresentparen": "\u3239",
    "ideographicresourceparen": "\u323e",
    "ideographicrightcircle": "\u32a8",
    "ideographicsecretcircle": "\u3299",
    "ideographicselfparen": "\u3242",
    "ideographicsocietyparen": "\u3233",
    "ideographicspace": "\u3000",
    "ideographicspecialparen": "\u3235",
    "ideographicstockparen": "\u3231",
    "ideographicstudyparen": "\u323b",
    "ideographicsunparen": "\u3230",
    "ideographicsuperviseparen": "\u323c",
    "ideographicwaterparen": "\u322c",
    "ideographicwoodparen": "\u322d",
    "ideographiczero": "\u3007",
    "ideographmetalcircle": "\u328e",
    "ideographmooncircle": "\u328a",
    "ideographnamecircle": "\u3294",
    "ideographsuncircle": "\u3290",
    "ideographwatercircle": "\u328c",
    "ideographwoodcircle": "\u328d",
    "ideva": "\u0907",
    "idieresis": "\u00ef",
    "idieresisacute": "\u1e2f",
    "idieresiscyrillic": "\u04e5",
    "idotbelow": "\u1ecb",
    "iebrevecyrillic": "\u04d7",
    "iecyrillic": "\u0435",
    "ieungacirclekorean": "\u3275",
    "ieungaparenkorean": "\u3215",
    "ieungcirclekorean": "\u3267",
    "ieungkorean": "\u3147",
    "ieungparenkorean": "\u3207",
    "igrave": "\u00ec",
    "igujarati": "\u0a87",
    "igurmukhi": "\u0a07",
    "ihiragana": "\u3044",
    "ihookabove": "\u1ec9",
    "iibengali": "\u0988",
    "iicyrillic": "\u0438",
    "iideva": "\u0908",
    "iigujarati": "\u0a88",
    "iigurmukhi": "\u0a08",
    "iimatragurmukhi": "\u0a40",
    "iinvertedbreve": "\u020b",
    "iishortcyrillic": "\u0439",
    "iivowelsignbengali": "\u09c0",
    "iivowelsigndeva": "\u0940",
    "iivowelsigngujarati": "\u0ac0",
    "ij": "\u0133",
    "ikatakana": "\u30a4",
    "ikatakanahalfwidth": "\uff72",
    "ikorean": "\u3163",
    "ilde": "\u02dc",
    "iluyhebrew": "\u05ac",
    "imacron": "\u012b",
    "imacroncyrillic": "\u04e3",
    "imageorapproximatelyequal": "\u2253",
    "imatragurmukhi": "\u0a3f",
    "imonospace": "\uff49",
    "increment": "\u2206",
    "infinity": "\u221e",
    "iniarmenian": "\u056b",
    "integral": "\u222b",
    "integralbottom": "\u2321",
    "integralbt": "\u2321",
    "integralex": "\uf8f5",
    "integraltop": "\u2320",
    "integraltp": "\u2320",
    "intersection": "\u2229",
    "intisquare": "\u3305",
    "invbullet": "\u25d8",
    "invcircle": "\u25d9",
    "invsmileface": "\u263b",
    "iocyrillic": "\u0451",
    "iogonek": "\u012f",
    "iota": "\u03b9",
    "iotadieresis": "\u03ca",
    "iotadieresistonos": "\u0390",
    "iotalatin": "\u0269",
    "iotatonos": "\u03af",
    "iparen": "\u24a4",
    "irigurmukhi": "\u0a72",
    "ismallhiragana": "\u3043",
    "ismallkatakana": "\u30a3",
    "ismallkatakanahalfwidth": "\uff68",
    "issharbengali": "\u09fa",
    "istroke": "\u0268",
    "isuperior": "\uf6ed",
    "iterationhiragana": "\u309d",
    "iterationkatakana": "\u30fd",
    "itilde": "\u0129",
    "itildebelow": "\u1e2d",
    "iubopomofo": "\u3129",
    "iucyrillic": "\u044e",
    "ivowelsignbengali": "\u09bf",
    "ivowelsigndeva": "\u093f",
    "ivowelsigngujarati": "\u0abf",
    "izhitsacyrillic": "\u0475",
    "izhitsadblgravecyrillic": "\u0477",
    "j": "\u006a",
    "jaarmenian": "\u0571",
    "jabengali": "\u099c",
    "jadeva": "\u091c",
    "jagujarati": "\u0a9c",
    "jagurmukhi": "\u0a1c",
    "jbopomofo": "\u3110",
    "jcaron": "\u01f0",
    "jcircle": "\u24d9",
    "jcircumflex": "\u0135",
    "jcrossedtail": "\u029d",
    "jdotlessstroke": "\u025f",
    "jecyrillic": "\u0458",
    "jeemarabic": "\u062c",
    "jeemfinalarabic": "\ufe9e",
    "jeeminitialarabic": "\ufe9f",
    "jeemmedialarabic": "\ufea0",
    "jeharabic": "\u0698",
    "jehfinalarabic": "\ufb8b",
    "jhabengali": "\u099d",
    "jhadeva": "\u091d",
    "jhagujarati": "\u0a9d",
    "jhagurmukhi": "\u0a1d",
    "jheharmenian": "\u057b",
    "jis": "\u3004",
    "jmonospace": "\uff4a",
    "jparen": "\u24a5",
    "jsuperior": "\u02b2",
    "k": "\u006b",
    "kabashkircyrillic": "\u04a1",
    "kabengali": "\u0995",
    "kacute": "\u1e31",
    "kacyrillic": "\u043a",
    "kadescendercyrillic": "\u049b",
    "kadeva": "\u0915",
    "kaf": "\u05db",
    "kafarabic": "\u0643",
    "kafdagesh": "\ufb3b",
    "kafdageshhebrew": "\ufb3b",
    "kaffinalarabic": "\ufeda",
    "kafhebrew": "\u05db",
    "kafinitialarabic": "\ufedb",
    "kafmedialarabic": "\ufedc",
    "kafrafehebrew": "\ufb4d",
    "kagujarati": "\u0a95",
    "kagurmukhi": "\u0a15",
    "kahiragana": "\u304b",
    "kahookcyrillic": "\u04c4",
    "kakatakana": "\u30ab",
    "kakatakanahalfwidth": "\uff76",
    "kappa": "\u03ba",
    "kappasymbolgreek": "\u03f0",
    "kapyeounmieumkorean": "\u3171",
    "kapyeounphieuphkorean": "\u3184",
    "kapyeounpieupkorean": "\u3178",
    "kapyeounssangpieupkorean": "\u3179",
    "karoriisquare": "\u330d",
    "kashidaautoarabic": "\u0640",
    "kashidaautonosidebearingarabic": "\u0640",
    "kasmallkatakana": "\u30f5",
    "kasquare": "\u3384",
    "kasraarabic": "\u0650",
    "kasratanarabic": "\u064d",
    "kastrokecyrillic": "\u049f",
    "katahiraprolongmarkhalfwidth": "\uff70",
    "kaverticalstrokecyrillic": "\u049d",
    "kbopomofo": "\u310e",
    "kcalsquare": "\u3389",
    "kcaron": "\u01e9",
    "kcedilla": "\u0137",
    "kcircle": "\u24da",
    "kcommaaccent": "\u0137",
    "kdotbelow": "\u1e33",
    "keharmenian": "\u0584",
    "kehiragana": "\u3051",
    "kekatakana": "\u30b1",
    "kekatakanahalfwidth": "\uff79",
    "kenarmenian": "\u056f",
    "kesmallkatakana": "\u30f6",
    "kgreenlandic": "\u0138",
    "khabengali": "\u0996",
    "khacyrillic": "\u0445",
    "khadeva": "\u0916",
    "khagujarati": "\u0a96",
    "khagurmukhi": "\u0a16",
    "khaharabic": "\u062e",
    "khahfinalarabic": "\ufea6",
    "khahinitialarabic": "\ufea7",
    "khahmedialarabic": "\ufea8",
    "kheicoptic": "\u03e7",
    "khhadeva": "\u0959",
    "khhagurmukhi": "\u0a59",
    "khieukhacirclekorean": "\u3278",
    "khieukhaparenkorean": "\u3218",
    "khieukhcirclekorean": "\u326a",
    "khieukhkorean": "\u314b",
    "khieukhparenkorean": "\u320a",
    "khokhaithai": "\u0e02",
    "khokhonthai": "\u0e05",
    "khokhuatthai": "\u0e03",
    "khokhwaithai": "\u0e04",
    "khomutthai": "\u0e5b",
    "khook": "\u0199",
    "khorakhangthai": "\u0e06",
    "khzsquare": "\u3391",
    "kihiragana": "\u304d",
    "kikatakana": "\u30ad",
    "kikatakanahalfwidth": "\uff77",
    "kiroguramusquare": "\u3315",
    "kiromeetorusquare": "\u3316",
    "kirosquare": "\u3314",
    "kiyeokacirclekorean": "\u326e",
    "kiyeokaparenkorean": "\u320e",
    "kiyeokcirclekorean": "\u3260",
    "kiyeokkorean": "\u3131",
    "kiyeokparenkorean": "\u3200",
    "kiyeoksioskorean": "\u3133",
    "kjecyrillic": "\u045c",
    "klinebelow": "\u1e35",
    "klsquare": "\u3398",
    "kmcubedsquare": "\u33a6",
    "kmonospace": "\uff4b",
    "kmsquaredsquare": "\u33a2",
    "kohiragana": "\u3053",
    "kohmsquare": "\u33c0",
    "kokaithai": "\u0e01",
    "kokatakana": "\u30b3",
    "kokatakanahalfwidth": "\uff7a",
    "kooposquare": "\u331e",
    "koppacyrillic": "\u0481",
    "koreanstandardsymbol": "\u327f",
    "koroniscmb": "\u0343",
    "kparen": "\u24a6",
    "kpasquare": "\u33aa",
    "ksicyrillic": "\u046f",
    "ktsquare": "\u33cf",
    "kturned": "\u029e",
    "kuhiragana": "\u304f",
    "kukatakana": "\u30af",
    "kukatakanahalfwidth": "\uff78",
    "kvsquare": "\u33b8",
    "kwsquare": "\u33be",
    "l": "\u006c",
    "labengali": "\u09b2",
    "lacute": "\u013a",
    "ladeva": "\u0932",
    "lagujarati": "\u0ab2",
    "lagurmukhi": "\u0a32",
    "lakkhangyaothai": "\u0e45",
    "lamaleffinalarabic": "\ufefc",
    "lamalefhamzaabovefinalarabic": "\ufef8",
    "lamalefhamzaaboveisolatedarabic": "\ufef7",
    "lamalefhamzabelowfinalarabic": "\ufefa",
    "lamalefhamzabelowisolatedarabic": "\ufef9",
    "lamalefisolatedarabic": "\ufefb",
    "lamalefmaddaabovefinalarabic": "\ufef6",
    "lamalefmaddaaboveisolatedarabic": "\ufef5",
    "lamarabic": "\u0644",
    "lambda": "\u03bb",
    "lambdastroke": "\u019b",
    "lamed": "\u05dc",
    "lameddagesh": "\ufb3c",
    "lameddageshhebrew": "\ufb3c",
    "lamedhebrew": "\u05dc",
    "lamedholam": "\u05dc\u05b9",
    "lamedholamdagesh": "\u05dc\u05b9\u05bc",
    "lamedholamdageshhebrew": "\u05dc\u05b9\u05bc",
    "lamedholamhebrew": "\u05dc\u05b9",
    "lamfinalarabic": "\ufede",
    "lamhahinitialarabic": "\ufcca",
    "laminitialarabic": "\ufedf",
    "lamjeeminitialarabic": "\ufcc9",
    "lamkhahinitialarabic": "\ufccb",
    "lamlamhehisolatedarabic": "\ufdf2",
    "lammedialarabic": "\ufee0",
    "lammeemhahinitialarabic": "\ufd88",
    "lammeeminitialarabic": "\ufccc",
    "lammeemjeeminitialarabic": "\ufedf\ufee4\ufea0",
    "lammeemkhahinitialarabic": "\ufedf\ufee4\ufea8",
    "largecircle": "\u25ef",
    "lbar": "\u019a",
    "lbelt": "\u026c",
    "lbopomofo": "\u310c",
    "lcaron": "\u013e",
    "lcedilla": "\u013c",
    "lcircle": "\u24db",
    "lcircumflexbelow": "\u1e3d",
    "lcommaaccent": "\u013c",
    "ldot": "\u0140",
    "ldotaccent": "\u0140",
    "ldotbelow": "\u1e37",
    "ldotbelowmacron": "\u1e39",
    "leftangleabovecmb": "\u031a",
    "lefttackbelowcmb": "\u0318",
    "less": "\u003c",
    "lessequal": "\u2264",
    "lessequalorgreater": "\u22da",
    "lessmonospace": "\uff1c",
    "lessorequivalent": "\u2272",
    "lessorgreater": "\u2276",
    "lessoverequal": "\u2266",
    "lesssmall": "\ufe64",
    "lezh": "\u026e",
    "lfblock": "\u258c",
    "lhookretroflex": "\u026d",
    "lira": "\u20a4",
    "liwnarmenian": "\u056c",
    "lj": "\u01c9",
    "ljecyrillic": "\u0459",
    "ll": "\uf6c0",
    "lladeva": "\u0933",
    "llagujarati": "\u0ab3",
    "llinebelow": "\u1e3b",
    "llladeva": "\u0934",
    "llvocalicbengali": "\u09e1",
    "llvocalicdeva": "\u0961",
    "llvocalicvowelsignbengali": "\u09e3",
    "llvocalicvowelsigndeva": "\u0963",
    "lmiddletilde": "\u026b",
    "lmonospace": "\uff4c",
    "lmsquare": "\u33d0",
    "lochulathai": "\u0e2c",
    "logicaland": "\u2227",
    "logicalnot": "\u00ac",
    "logicalnotreversed": "\u2310",
    "logicalor": "\u2228",
    "lolingthai": "\u0e25",
    "longs": "\u017f",
    "lowlinecenterline": "\ufe4e",
    "lowlinecmb": "\u0332",
    "lowlinedashed": "\ufe4d",
    "lozenge": "\u25ca",
    "lparen": "\u24a7",
    "lslash": "\u0142",
    "lsquare": "\u2113",
    "lsuperior": "\uf6ee",
    "ltshade": "\u2591",
    "luthai": "\u0e26",
    "lvocalicbengali": "\u098c",
    "lvocalicdeva": "\u090c",
    "lvocalicvowelsignbengali": "\u09e2",
    "lvocalicvowelsigndeva": "\u0962",
    "lxsquare": "\u33d3",
    "m": "\u006d",
    "mabengali": "\u09ae",
    "macron": "\u00af",
    "macronbelowcmb": "\u0331",
    "macroncmb": "\u0304",
    "macronlowmod": "\u02cd",
    "macronmonospace": "\uffe3",
    "macute": "\u1e3f",
    "madeva": "\u092e",
    "magujarati": "\u0aae",
    "magurmukhi": "\u0a2e",
    "mahapakhhebrew": "\u05a4",
    "mahapakhlefthebrew": "\u05a4",
    "mahiragana": "\u307e",
    "maichattawalowleftthai": "\uf895",
    "maichattawalowrightthai": "\uf894",
    "maichattawathai": "\u0e4b",
    "maichattawaupperleftthai": "\uf893",
    "maieklowleftthai": "\uf88c",
    "maieklowrightthai": "\uf88b",
    "maiekthai": "\u0e48",
    "maiekupperleftthai": "\uf88a",
    "maihanakatleftthai": "\uf884",
    "maihanakatthai": "\u0e31",
    "maitaikhuleftthai": "\uf889",
    "maitaikhuthai": "\u0e47",
    "maitholowleftthai": "\uf88f",
    "maitholowrightthai": "\uf88e",
    "maithothai": "\u0e49",
    "maithoupperleftthai": "\uf88d",
    "maitrilowleftthai": "\uf892",
    "maitrilowrightthai": "\uf891",
    "maitrithai": "\u0e4a",
    "maitriupperleftthai": "\uf890",
    "maiyamokthai": "\u0e46",
    "makatakana": "\u30de",
    "makatakanahalfwidth": "\uff8f",
    "male": "\u2642",
    "mansyonsquare": "\u3347",
    "maqafhebrew": "\u05be",
    "mars": "\u2642",
    "masoracirclehebrew": "\u05af",
    "masquare": "\u3383",
    "mbopomofo": "\u3107",
    "mbsquare": "\u33d4",
    "mcircle": "\u24dc",
    "mcubedsquare": "\u33a5",
    "mdotaccent": "\u1e41",
    "mdotbelow": "\u1e43",
    "meemarabic": "\u0645",
    "meemfinalarabic": "\ufee2",
    "meeminitialarabic": "\ufee3",
    "meemmedialarabic": "\ufee4",
    "meemmeeminitialarabic": "\ufcd1",
    "meemmeemisolatedarabic": "\ufc48",
    "meetorusquare": "\u334d",
    "mehiragana": "\u3081",
    "meizierasquare": "\u337e",
    "mekatakana": "\u30e1",
    "mekatakanahalfwidth": "\uff92",
    "mem": "\u05de",
    "memdagesh": "\ufb3e",
    "memdageshhebrew": "\ufb3e",
    "memhebrew": "\u05de",
    "menarmenian": "\u0574",
    "merkhahebrew": "\u05a5",
    "merkhakefulahebrew": "\u05a6",
    "merkhakefulalefthebrew": "\u05a6",
    "merkhalefthebrew": "\u05a5",
    "mhook": "\u0271",
    "mhzsquare": "\u3392",
    "middledotkatakanahalfwidth": "\uff65",
    "middot": "\u00b7",
    "mieumacirclekorean": "\u3272",
    "mieumaparenkorean": "\u3212",
    "mieumcirclekorean": "\u3264",
    "mieumkorean": "\u3141",
    "mieumpansioskorean": "\u3170",
    "mieumparenkorean": "\u3204",
    "mieumpieupkorean": "\u316e",
    "mieumsioskorean": "\u316f",
    "mihiragana": "\u307f",
    "mikatakana": "\u30df",
    "mikatakanahalfwidth": "\uff90",
    "minus": "\u2212",
    "minusbelowcmb": "\u0320",
    "minuscircle": "\u2296",
    "minusmod": "\u02d7",
    "minusplus": "\u2213",
    "minute": "\u2032",
    "miribaarusquare": "\u334a",
    "mirisquare": "\u3349",
    "mlonglegturned": "\u0270",
    "mlsquare": "\u3396",
    "mmcubedsquare": "\u33a3",
    "mmonospace": "\uff4d",
    "mmsquaredsquare": "\u339f",
    "mohiragana": "\u3082",
    "mohmsquare": "\u33c1",
    "mokatakana": "\u30e2",
    "mokatakanahalfwidth": "\uff93",
    "molsquare": "\u33d6",
    "momathai": "\u0e21",
    "moverssquare": "\u33a7",
    "moverssquaredsquare": "\u33a8",
    "mparen": "\u24a8",
    "mpasquare": "\u33ab",
    "mssquare": "\u33b3",
    "msuperior": "\uf6ef",
    "mturned": "\u026f",
    "mu": "\u00b5",
    "mu1": "\u00b5",
    "muasquare": "\u3382",
    "muchgreater": "\u226b",
    "muchless": "\u226a",
    "mufsquare": "\u338c",
    "mugreek": "\u03bc",
    "mugsquare": "\u338d",
    "muhiragana": "\u3080",
    "mukatakana": "\u30e0",
    "mukatakanahalfwidth": "\uff91",
    "mulsquare": "\u3395",
    "multiply": "\u00d7",
    "mumsquare": "\u339b",
    "munahhebrew": "\u05a3",
    "munahlefthebrew": "\u05a3",
    "musicalnote": "\u266a",
    "musicalnotedbl": "\u266b",
    "musicflatsign": "\u266d",
    "musicsharpsign": "\u266f",
    "mussquare": "\u33b2",
    "muvsquare": "\u33b6",
    "muwsquare": "\u33bc",
    "mvmegasquare": "\u33b9",
    "mvsquare": "\u33b7",
    "mwmegasquare": "\u33bf",
    "mwsquare": "\u33bd",
    "n": "\u006e",
    "nabengali": "\u09a8",
    "nabla": "\u2207",
    "nacute": "\u0144",
    "nadeva": "\u0928",
    "nagujarati": "\u0aa8",
    "nagurmukhi": "\u0a28",
    "nahiragana": "\u306a",
    "nakatakana": "\u30ca",
    "nakatakanahalfwidth": "\uff85",
    "napostrophe": "\u0149",
    "nasquare": "\u3381",
    "nbopomofo": "\u310b",
    "nbspace": "\u00a0",
    "ncaron": "\u0148",
    "ncedilla": "\u0146",
    "ncircle": "\u24dd",
    "ncircumflexbelow": "\u1e4b",
    "ncommaaccent": "\u0146",
    "ndotaccent": "\u1e45",
    "ndotbelow": "\u1e47",
    "nehiragana": "\u306d",
    "nekatakana": "\u30cd",
    "nekatakanahalfwidth": "\uff88",
    "newsheqelsign": "\u20aa",
    "nfsquare": "\u338b",
    "ngabengali": "\u0999",
    "ngadeva": "\u0919",
    "ngagujarati": "\u0a99",
    "ngagurmukhi": "\u0a19",
    "ngonguthai": "\u0e07",
    "nhiragana": "\u3093",
    "nhookleft": "\u0272",
    "nhookretroflex": "\u0273",
    "nieunacirclekorean": "\u326f",
    "nieunaparenkorean": "\u320f",
    "nieuncieuckorean": "\u3135",
    "nieuncirclekorean": "\u3261",
    "nieunhieuhkorean": "\u3136",
    "nieunkorean": "\u3134",
    "nieunpansioskorean": "\u3168",
    "nieunparenkorean": "\u3201",
    "nieunsioskorean": "\u3167",
    "nieuntikeutkorean": "\u3166",
    "nihiragana": "\u306b",
    "nikatakana": "\u30cb",
    "nikatakanahalfwidth": "\uff86",
    "nikhahitleftthai": "\uf899",
    "nikhahitthai": "\u0e4d",
    "nine": "\u0039",
    "ninearabic": "\u0669",
    "ninebengali": "\u09ef",
    "ninecircle": "\u2468",
    "ninecircleinversesansserif": "\u2792",
    "ninedeva": "\u096f",
    "ninegujarati": "\u0aef",
    "ninegurmukhi": "\u0a6f",
    "ninehackarabic": "\u0669",
    "ninehangzhou": "\u3029",
    "nineideographicparen": "\u3228",
    "nineinferior": "\u2089",
    "ninemonospace": "\uff19",
    "nineoldstyle": "\uf739",
    "nineparen": "\u247c",
    "nineperiod": "\u2490",
    "ninepersian": "\u06f9",
    "nineroman": "\u2178",
    "ninesuperior": "\u2079",
    "nineteencircle": "\u2472",
    "nineteenparen": "\u2486",
    "nineteenperiod": "\u249a",
    "ninethai": "\u0e59",
    "nj": "\u01cc",
    "njecyrillic": "\u045a",
    "nkatakana": "\u30f3",
    "nkatakanahalfwidth": "\uff9d",
    "nlegrightlong": "\u019e",
    "nlinebelow": "\u1e49",
    "nmonospace": "\uff4e",
    "nmsquare": "\u339a",
    "nnabengali": "\u09a3",
    "nnadeva": "\u0923",
    "nnagujarati": "\u0aa3",
    "nnagurmukhi": "\u0a23",
    "nnnadeva": "\u0929",
    "nohiragana": "\u306e",
    "nokatakana": "\u30ce",
    "nokatakanahalfwidth": "\uff89",
    "nonbreakingspace": "\u00a0",
    "nonenthai": "\u0e13",
    "nonuthai": "\u0e19",
    "noonarabic": "\u0646",
    "noonfinalarabic": "\ufee6",
    "noonghunnaarabic": "\u06ba",
    "noonghunnafinalarabic": "\ufb9f",
    "noonhehinitialarabic": "\ufee7\ufeec",
    "nooninitialarabic": "\ufee7",
    "noonjeeminitialarabic": "\ufcd2",
    "noonjeemisolatedarabic": "\ufc4b",
    "noonmedialarabic": "\ufee8",
    "noonmeeminitialarabic": "\ufcd5",
    "noonmeemisolatedarabic": "\ufc4e",
    "noonnoonfinalarabic": "\ufc8d",
    "notcontains": "\u220c",
    "notelement": "\u2209",
    "notelementof": "\u2209",
    "notequal": "\u2260",
    "notgreater": "\u226f",
    "notgreaternorequal": "\u2271",
    "notgreaternorless": "\u2279",
    "notidentical": "\u2262",
    "notless": "\u226e",
    "notlessnorequal": "\u2270",
    "notparallel": "\u2226",
    "notprecedes": "\u2280",
    "notsubset": "\u2284",
    "notsucceeds": "\u2281",
    "notsuperset": "\u2285",
    "nowarmenian": "\u0576",
    "nparen": "\u24a9",
    "nssquare": "\u33b1",
    "nsuperior": "\u207f",
    "ntilde": "\u00f1",
    "nu": "\u03bd",
    "nuhiragana": "\u306c",
    "nukatakana": "\u30cc",
    "nukatakanahalfwidth": "\uff87",
    "nuktabengali": "\u09bc",
    "nuktadeva": "\u093c",
    "nuktagujarati": "\u0abc",
    "nuktagurmukhi": "\u0a3c",
    "numbersign": "\u0023",
    "numbersignmonospace": "\uff03",
    "numbersignsmall": "\ufe5f",
    "numeralsigngreek": "\u0374",
    "numeralsignlowergreek": "\u0375",
    "numero": "\u2116",
    "nun": "\u05e0",
    "nundagesh": "\ufb40",
    "nundageshhebrew": "\ufb40",
    "nunhebrew": "\u05e0",
    "nvsquare": "\u33b5",
    "nwsquare": "\u33bb",
    "nyabengali": "\u099e",
    "nyadeva": "\u091e",
    "nyagujarati": "\u0a9e",
    "nyagurmukhi": "\u0a1e",
    "o": "\u006f",
    "oacute": "\u00f3",
    "oangthai": "\u0e2d",
    "obarred": "\u0275",
    "obarredcyrillic": "\u04e9",
    "obarreddieresiscyrillic": "\u04eb",
    "obengali": "\u0993",
    "obopomofo": "\u311b",
    "obreve": "\u014f",
    "ocandradeva": "\u0911",
    "ocandragujarati": "\u0a91",
    "ocandravowelsigndeva": "\u0949",
    "ocandravowelsigngujarati": "\u0ac9",
    "ocaron": "\u01d2",
    "ocircle": "\u24de",
    "ocircumflex": "\u00f4",
    "ocircumflexacute": "\u1ed1",
    "ocircumflexdotbelow": "\u1ed9",
    "ocircumflexgrave": "\u1ed3",
    "ocircumflexhookabove": "\u1ed5",
    "ocircumflextilde": "\u1ed7",
    "ocyrillic": "\u043e",
    "odblacute": "\u0151",
    "odblgrave": "\u020d",
    "odeva": "\u0913",
    "odieresis": "\u00f6",
    "odieresiscyrillic": "\u04e7",
    "odotbelow": "\u1ecd",
    "oe": "\u0153",
    "oekorean": "\u315a",
    "ogonek": "\u02db",
    "ogonekcmb": "\u0328",
    "ograve": "\u00f2",
    "ogujarati": "\u0a93",
    "oharmenian": "\u0585",
    "ohiragana": "\u304a",
    "ohookabove": "\u1ecf",
    "ohorn": "\u01a1",
    "ohornacute": "\u1edb",
    "ohorndotbelow": "\u1ee3",
    "ohorngrave": "\u1edd",
    "ohornhookabove": "\u1edf",
    "ohorntilde": "\u1ee1",
    "ohungarumlaut": "\u0151",
    "oi": "\u01a3",
    "oinvertedbreve": "\u020f",
    "okatakana": "\u30aa",
    "okatakanahalfwidth": "\uff75",
    "okorean": "\u3157",
    "olehebrew": "\u05ab",
    "omacron": "\u014d",
    "omacronacute": "\u1e53",
    "omacrongrave": "\u1e51",
    "omdeva": "\u0950",
    "omega": "\u03c9",
    "omega1": "\u03d6",
    "omegacyrillic": "\u0461",
    "omegalatinclosed": "\u0277",
    "omegaroundcyrillic": "\u047b",
    "omegatitlocyrillic": "\u047d",
    "omegatonos": "\u03ce",
    "omgujarati": "\u0ad0",
    "omicron": "\u03bf",
    "omicrontonos": "\u03cc",
    "omonospace": "\uff4f",
    "one": "\u0031",
    "onearabic": "\u0661",
    "onebengali": "\u09e7",
    "onecircle": "\u2460",
    "onecircleinversesansserif": "\u278a",
    "onedeva": "\u0967",
    "onedotenleader": "\u2024",
    "oneeighth": "\u215b",
    "onefitted": "\uf6dc",
    "onegujarati": "\u0ae7",
    "onegurmukhi": "\u0a67",
    "onehackarabic": "\u0661",
    "onehalf": "\u00bd",
    "onehangzhou": "\u3021",
    "oneideographicparen": "\u3220",
    "oneinferior": "\u2081",
    "onemonospace": "\uff11",
    "onenumeratorbengali": "\u09f4",
    "oneoldstyle": "\uf731",
    "oneparen": "\u2474",
    "oneperiod": "\u2488",
    "onepersian": "\u06f1",
    "onequarter": "\u00bc",
    "oneroman": "\u2170",
    "onesuperior": "\u00b9",
    "onethai": "\u0e51",
    "onethird": "\u2153",
    "oogonek": "\u01eb",
    "oogonekmacron": "\u01ed",
    "oogurmukhi": "\u0a13",
    "oomatragurmukhi": "\u0a4b",
    "oopen": "\u0254",
    "oparen": "\u24aa",
    "openbullet": "\u25e6",
    "option": "\u2325",
    "ordfeminine": "\u00aa",
    "ordmasculine": "\u00ba",
    "orthogonal": "\u221f",
    "oshortdeva": "\u0912",
    "oshortvowelsigndeva": "\u094a",
    "oslash": "\u00f8",
    "oslashacute": "\u01ff",
    "osmallhiragana": "\u3049",
    "osmallkatakana": "\u30a9",
    "osmallkatakanahalfwidth": "\uff6b",
    "ostrokeacute": "\u01ff",
    "osuperior": "\uf6f0",
    "otcyrillic": "\u047f",
    "otilde": "\u00f5",
    "otildeacute": "\u1e4d",
    "otildedieresis": "\u1e4f",
    "oubopomofo": "\u3121",
    "overline": "\u203e",
    "overlinecenterline": "\ufe4a",
    "overlinecmb": "\u0305",
    "overlinedashed": "\ufe49",
    "overlinedblwavy": "\ufe4c",
    "overlinewavy": "\ufe4b",
    "overscore": "\u00af",
    "ovowelsignbengali": "\u09cb",
    "ovowelsigndeva": "\u094b",
    "ovowelsigngujarati": "\u0acb",
    "p": "\u0070",
    "paampssquare": "\u3380",
    "paasentosquare": "\u332b",
    "pabengali": "\u09aa",
    "pacute": "\u1e55",
    "padeva": "\u092a",
    "pagedown": "\u21df",
    "pageup": "\u21de",
    "pagujarati": "\u0aaa",
    "pagurmukhi": "\u0a2a",
    "pahiragana": "\u3071",
    "paiyannoithai": "\u0e2f",
    "pakatakana": "\u30d1",
    "palatalizationcyrilliccmb": "\u0484",
    "palochkacyrillic": "\u04c0",
    "pansioskorean": "\u317f",
    "paragraph": "\u00b6",
    "parallel": "\u2225",
    "parenleft": "\u0028",
    "parenleftaltonearabic": "\ufd3e",
    "parenleftbt": "\uf8ed",
    "parenleftex": "\uf8ec",
    "parenleftinferior": "\u208d",
    "parenleftmonospace": "\uff08",
    "parenleftsmall": "\ufe59",
    "parenleftsuperior": "\u207d",
    "parenlefttp": "\uf8eb",
    "parenleftvertical": "\ufe35",
    "parenright": "\u0029",
    "parenrightaltonearabic": "\ufd3f",
    "parenrightbt": "\uf8f8",
    "parenrightex": "\uf8f7",
    "parenrightinferior": "\u208e",
    "parenrightmonospace": "\uff09",
    "parenrightsmall": "\ufe5a",
    "parenrightsuperior": "\u207e",
    "parenrighttp": "\uf8f6",
    "parenrightvertical": "\ufe36",
    "partialdiff": "\u2202",
    "paseqhebrew": "\u05c0",
    "pashtahebrew": "\u0599",
    "pasquare": "\u33a9",
    "patah": "\u05b7",
    "patah11": "\u05b7",
    "patah1d": "\u05b7",
    "patah2a": "\u05b7",
    "patahhebrew": "\u05b7",
    "patahnarrowhebrew": "\u05b7",
    "patahquarterhebrew": "\u05b7",
    "patahwidehebrew": "\u05b7",
    "pazerhebrew": "\u05a1",
    "pbopomofo": "\u3106",
    "pcircle": "\u24df",
    "pdotaccent": "\u1e57",
    "pe": "\u05e4",
    "pecyrillic": "\u043f",
    "pedagesh": "\ufb44",
    "pedageshhebrew": "\ufb44",
    "peezisquare": "\u333b",
    "pefinaldageshhebrew": "\ufb43",
    "peharabic": "\u067e",
    "peharmenian": "\u057a",
    "pehebrew": "\u05e4",
    "pehfinalarabic": "\ufb57",
    "pehinitialarabic": "\ufb58",
    "pehiragana": "\u307a",
    "pehmedialarabic": "\ufb59",
    "pekatakana": "\u30da",
    "pemiddlehookcyrillic": "\u04a7",
    "perafehebrew": "\ufb4e",
    "percent": "\u0025",
    "percentarabic": "\u066a",
    "percentmonospace": "\uff05",
    "percentsmall": "\ufe6a",
    "period": "\u002e",
    "periodarmenian": "\u0589",
    "periodcentered": "\u00b7",
    "periodhalfwidth": "\uff61",
    "periodinferior": "\uf6e7",
    "periodmonospace": "\uff0e",
    "periodsmall": "\ufe52",
    "periodsuperior": "\uf6e8",
    "perispomenigreekcmb": "\u0342",
    "perpendicular": "\u22a5",
    "perthousand": "\u2030",
    "peseta": "\u20a7",
    "pfsquare": "\u338a",
    "phabengali": "\u09ab",
    "phadeva": "\u092b",
    "phagujarati": "\u0aab",
    "phagurmukhi": "\u0a2b",
    "phi": "\u03c6",
    "phi1": "\u03d5",
    "phieuphacirclekorean": "\u327a",
    "phieuphaparenkorean": "\u321a",
    "phieuphcirclekorean": "\u326c",
    "phieuphkorean": "\u314d",
    "phieuphparenkorean": "\u320c",
    "philatin": "\u0278",
    "phinthuthai": "\u0e3a",
    "phisymbolgreek": "\u03d5",
    "phook": "\u01a5",
    "phophanthai": "\u0e1e",
    "phophungthai": "\u0e1c",
    "phosamphaothai": "\u0e20",
    "pi": "\u03c0",
    "pieupacirclekorean": "\u3273",
    "pieupaparenkorean": "\u3213",
    "pieupcieuckorean": "\u3176",
    "pieupcirclekorean": "\u3265",
    "pieupkiyeokkorean": "\u3172",
    "pieupkorean": "\u3142",
    "pieupparenkorean": "\u3205",
    "pieupsioskiyeokkorean": "\u3174",
    "pieupsioskorean": "\u3144",
    "pieupsiostikeutkorean": "\u3175",
    "pieupthieuthkorean": "\u3177",
    "pieuptikeutkorean": "\u3173",
    "pihiragana": "\u3074",
    "pikatakana": "\u30d4",
    "pisymbolgreek": "\u03d6",
    "piwrarmenian": "\u0583",
    "plus": "\u002b",
    "plusbelowcmb": "\u031f",
    "pluscircle": "\u2295",
    "plusminus": "\u00b1",
    "plusmod": "\u02d6",
    "plusmonospace": "\uff0b",
    "plussmall": "\ufe62",
    "plussuperior": "\u207a",
    "pmonospace": "\uff50",
    "pmsquare": "\u33d8",
    "pohiragana": "\u307d",
    "pointingindexdownwhite": "\u261f",
    "pointingindexleftwhite": "\u261c",
    "pointingindexrightwhite": "\u261e",
    "pointingindexupwhite": "\u261d",
    "pokatakana": "\u30dd",
    "poplathai": "\u0e1b",
    "postalmark": "\u3012",
    "postalmarkface": "\u3020",
    "pparen": "\u24ab",
    "precedes": "\u227a",
    "prescription": "\u211e",
    "primemod": "\u02b9",
    "primereversed": "\u2035",
    "product": "\u220f",
    "projective": "\u2305",
    "prolongedkana": "\u30fc",
    "propellor": "\u2318",
    "propersubset": "\u2282",
    "propersuperset": "\u2283",
    "proportion": "\u2237",
    "proportional": "\u221d",
    "psi": "\u03c8",
    "psicyrillic": "\u0471",
    "psilipneumatacyrilliccmb": "\u0486",
    "pssquare": "\u33b0",
    "puhiragana": "\u3077",
    "pukatakana": "\u30d7",
    "pvsquare": "\u33b4",
    "pwsquare": "\u33ba",
    "q": "\u0071",
    "qadeva": "\u0958",
    "qadmahebrew": "\u05a8",
    "qafarabic": "\u0642",
    "qaffinalarabic": "\ufed6",
    "qafinitialarabic": "\ufed7",
    "qafmedialarabic": "\ufed8",
    "qamats": "\u05b8",
    "qamats10": "\u05b8",
    "qamats1a": "\u05b8",
    "qamats1c": "\u05b8",
    "qamats27": "\u05b8",
    "qamats29": "\u05b8",
    "qamats33": "\u05b8",
    "qamatsde": "\u05b8",
    "qamatshebrew": "\u05b8",
    "qamatsnarrowhebrew": "\u05b8",
    "qamatsqatanhebrew": "\u05b8",
    "qamatsqatannarrowhebrew": "\u05b8",
    "qamatsqatanquarterhebrew": "\u05b8",
    "qamatsqatanwidehebrew": "\u05b8",
    "qamatsquarterhebrew": "\u05b8",
    "qamatswidehebrew": "\u05b8",
    "qarneyparahebrew": "\u059f",
    "qbopomofo": "\u3111",
    "qcircle": "\u24e0",
    "qhook": "\u02a0",
    "qmonospace": "\uff51",
    "qof": "\u05e7",
    "qofdagesh": "\ufb47",
    "qofdageshhebrew": "\ufb47",
    "qofhatafpatah": "\u05e7\u05b2",
    "qofhatafpatahhebrew": "\u05e7\u05b2",
    "qofhatafsegol": "\u05e7\u05b1",
    "qofhatafsegolhebrew": "\u05e7\u05b1",
    "qofhebrew": "\u05e7",
    "qofhiriq": "\u05e7\u05b4",
    "qofhiriqhebrew": "\u05e7\u05b4",
    "qofholam": "\u05e7\u05b9",
    "qofholamhebrew": "\u05e7\u05b9",
    "qofpatah": "\u05e7\u05b7",
    "qofpatahhebrew": "\u05e7\u05b7",
    "qofqamats": "\u05e7\u05b8",
    "qofqamatshebrew": "\u05e7\u05b8",
    "qofqubuts": "\u05e7\u05bb",
    "qofqubutshebrew": "\u05e7\u05bb",
    "qofsegol": "\u05e7\u05b6",
    "qofsegolhebrew": "\u05e7\u05b6",
    "qofsheva": "\u05e7\u05b0",
    "qofshevahebrew": "\u05e7\u05b0",
    "qoftsere": "\u05e7\u05b5",
    "qoftserehebrew": "\u05e7\u05b5",
    "qparen": "\u24ac",
    "quarternote": "\u2669",
    "qubuts": "\u05bb",
    "qubuts18": "\u05bb",
    "qubuts25": "\u05bb",
    "qubuts31": "\u05bb",
    "qubutshebrew": "\u05bb",
    "qubutsnarrowhebrew": "\u05bb",
    "qubutsquarterhebrew": "\u05bb",
    "qubutswidehebrew": "\u05bb",
    "question": "\u003f",
    "questionarabic": "\u061f",
    "questionarmenian": "\u055e",
    "questiondown": "\u00bf",
    "questiondownsmall": "\uf7bf",
    "questiongreek": "\u037e",
    "questionmonospace": "\uff1f",
    "questionsmall": "\uf73f",
    "quotedbl": "\u0022",
    "quotedblbase": "\u201e",
    "quotedblleft": "\u201c",
    "quotedblmonospace": "\uff02",
    "quotedblprime": "\u301e",
    "quotedblprimereversed": "\u301d",
    "quotedblright": "\u201d",
    "quoteleft": "\u2018",
    "quoteleftreversed": "\u201b",
    "quotereversed": "\u201b",
    "quoteright": "\u2019",
    "quoterightn": "\u0149",
    "quotesinglbase": "\u201a",
    "quotesingle": "\u0027",
    "quotesinglemonospace": "\uff07",
    "r": "\u0072",
    "raarmenian": "\u057c",
    "rabengali": "\u09b0",
    "racute": "\u0155",
    "radeva": "\u0930",
    "radical": "\u221a",
    "radicalex": "\uf8e5",
    "radoverssquare": "\u33ae",
    "radoverssquaredsquare": "\u33af",
    "radsquare": "\u33ad",
    "rafe": "\u05bf",
    "rafehebrew": "\u05bf",
    "ragujarati": "\u0ab0",
    "ragurmukhi": "\u0a30",
    "rahiragana": "\u3089",
    "rakatakana": "\u30e9",
    "rakatakanahalfwidth": "\uff97",
    "ralowerdiagonalbengali": "\u09f1",
    "ramiddlediagonalbengali": "\u09f0",
    "ramshorn": "\u0264",
    "ratio": "\u2236",
    "rbopomofo": "\u3116",
    "rcaron": "\u0159",
    "rcedilla": "\u0157",
    "rcircle": "\u24e1",
    "rcommaaccent": "\u0157",
    "rdblgrave": "\u0211",
    "rdotaccent": "\u1e59",
    "rdotbelow": "\u1e5b",
    "rdotbelowmacron": "\u1e5d",
    "referencemark": "\u203b",
    "reflexsubset": "\u2286",
    "reflexsuperset": "\u2287",
    "registered": "\u00ae",
    "registersans": "\uf8e8",
    "registerserif": "\uf6da",
    "reharabic": "\u0631",
    "reharmenian": "\u0580",
    "rehfinalarabic": "\ufeae",
    "rehiragana": "\u308c",
    "rehyehaleflamarabic": "\u0631\ufef3\ufe8e\u0644",
    "rekatakana": "\u30ec",
    "rekatakanahalfwidth": "\uff9a",
    "resh": "\u05e8",
    "reshdageshhebrew": "\ufb48",
    "reshhatafpatah": "\u05e8\u05b2",
    "reshhatafpatahhebrew": "\u05e8\u05b2",
    "reshhatafsegol": "\u05e8\u05b1",
    "reshhatafsegolhebrew": "\u05e8\u05b1",
    "reshhebrew": "\u05e8",
    "reshhiriq": "\u05e8\u05b4",
    "reshhiriqhebrew": "\u05e8\u05b4",
    "reshholam": "\u05e8\u05b9",
    "reshholamhebrew": "\u05e8\u05b9",
    "reshpatah": "\u05e8\u05b7",
    "reshpatahhebrew": "\u05e8\u05b7",
    "reshqamats": "\u05e8\u05b8",
    "reshqamatshebrew": "\u05e8\u05b8",
    "reshqubuts": "\u05e8\u05bb",
    "reshqubutshebrew": "\u05e8\u05bb",
    "reshsegol": "\u05e8\u05b6",
    "reshsegolhebrew": "\u05e8\u05b6",
    "reshsheva": "\u05e8\u05b0",
    "reshshevahebrew": "\u05e8\u05b0",
    "reshtsere": "\u05e8\u05b5",
    "reshtserehebrew": "\u05e8\u05b5",
    "reversedtilde": "\u223d",
    "reviahebrew": "\u0597",
    "reviamugrashhebrew": "\u0597",
    "revlogicalnot": "\u2310",
    "rfishhook": "\u027e",
    "rfishhookreversed": "\u027f",
    "rhabengali": "\u09dd",
    "rhadeva": "\u095d",
    "rho": "\u03c1",
    "rhook": "\u027d",
    "rhookturned": "\u027b",
    "rhookturnedsuperior": "\u02b5",
    "rhosymbolgreek": "\u03f1",
    "rhotichookmod": "\u02de",
    "rieulacirclekorean": "\u3271",
    "rieulaparenkorean": "\u3211",
    "rieulcirclekorean": "\u3263",
    "rieulhieuhkorean": "\u3140",
    "rieulkiyeokkorean": "\u313a",
    "rieulkiyeoksioskorean": "\u3169",
    "rieulkorean": "\u3139",
    "rieulmieumkorean": "\u313b",
    "rieulpansioskorean": "\u316c",
    "rieulparenkorean": "\u3203",
    "rieulphieuphkorean": "\u313f",
    "rieulpieupkorean": "\u313c",
    "rieulpieupsioskorean": "\u316b",
    "rieulsioskorean": "\u313d",
    "rieulthieuthkorean": "\u313e",
    "rieultikeutkorean": "\u316a",
    "rieulyeorinhieuhkorean": "\u316d",
    "rightangle": "\u221f",
    "righttackbelowcmb": "\u0319",
    "righttriangle": "\u22bf",
    "rihiragana": "\u308a",
    "rikatakana": "\u30ea",
    "rikatakanahalfwidth": "\uff98",
    "ring": "\u02da",
    "ringbelowcmb": "\u0325",
    "ringcmb": "\u030a",
    "ringhalfleft": "\u02bf",
    "ringhalfleftarmenian": "\u0559",
    "ringhalfleftbelowcmb": "\u031c",
    "ringhalfleftcentered": "\u02d3",
    "ringhalfright": "\u02be",
    "ringhalfrightbelowcmb": "\u0339",
    "ringhalfrightcentered": "\u02d2",
    "rinvertedbreve": "\u0213",
    "rittorusquare": "\u3351",
    "rlinebelow": "\u1e5f",
    "rlongleg": "\u027c",
    "rlonglegturned": "\u027a",
    "rmonospace": "\uff52",
    "rohiragana": "\u308d",
    "rokatakana": "\u30ed",
    "rokatakanahalfwidth": "\uff9b",
    "roruathai": "\u0e23",
    "rparen": "\u24ad",
    "rrabengali": "\u09dc",
    "rradeva": "\u0931",
    "rragurmukhi": "\u0a5c",
    "rreharabic": "\u0691",
    "rrehfinalarabic": "\ufb8d",
    "rrvocalicbengali": "\u09e0",
    "rrvocalicdeva": "\u0960",
    "rrvocalicgujarati": "\u0ae0",
    "rrvocalicvowelsignbengali": "\u09c4",
    "rrvocalicvowelsigndeva": "\u0944",
    "rrvocalicvowelsigngujarati": "\u0ac4",
    "rsuperior": "\uf6f1",
    "rtblock": "\u2590",
    "rturned": "\u0279",
    "rturnedsuperior": "\u02b4",
    "ruhiragana": "\u308b",
    "rukatakana": "\u30eb",
    "rukatakanahalfwidth": "\uff99",
    "rupeemarkbengali": "\u09f2",
    "rupeesignbengali": "\u09f3",
    "rupiah": "\uf6dd",
    "ruthai": "\u0e24",
    "rvocalicbengali": "\u098b",
    "rvocalicdeva": "\u090b",
    "rvocalicgujarati": "\u0a8b",
    "rvocalicvowelsignbengali": "\u09c3",
    "rvocalicvowelsigndeva": "\u0943",
    "rvocalicvowelsigngujarati": "\u0ac3",
    "s": "\u0073",
    "sabengali": "\u09b8",
    "sacute": "\u015b",
    "sacutedotaccent": "\u1e65",
    "sadarabic": "\u0635",
    "sadeva": "\u0938",
    "sadfinalarabic": "\ufeba",
    "sadinitialarabic": "\ufebb",
    "sadmedialarabic": "\ufebc",
    "sagujarati": "\u0ab8",
    "sagurmukhi": "\u0a38",
    "sahiragana": "\u3055",
    "sakatakana": "\u30b5",
    "sakatakanahalfwidth": "\uff7b",
    "sallallahoualayhewasallamarabic": "\ufdfa",
    "samekh": "\u05e1",
    "samekhdagesh": "\ufb41",
    "samekhdageshhebrew": "\ufb41",
    "samekhhebrew": "\u05e1",
    "saraaathai": "\u0e32",
    "saraaethai": "\u0e41",
    "saraaimaimalaithai": "\u0e44",
    "saraaimaimuanthai": "\u0e43",
    "saraamthai": "\u0e33",
    "saraathai": "\u0e30",
    "saraethai": "\u0e40",
    "saraiileftthai": "\uf886",
    "saraiithai": "\u0e35",
    "saraileftthai": "\uf885",
    "saraithai": "\u0e34",
    "saraothai": "\u0e42",
    "saraueeleftthai": "\uf888",
    "saraueethai": "\u0e37",
    "saraueleftthai": "\uf887",
    "sarauethai": "\u0e36",
    "sarauthai": "\u0e38",
    "sarauuthai": "\u0e39",
    "sbopomofo": "\u3119",
    "scaron": "\u0161",
    "scarondotaccent": "\u1e67",
    "scedilla": "\u015f",
    "schwa": "\u0259",
    "schwacyrillic": "\u04d9",
    "schwadieresiscyrillic": "\u04db",
    "schwahook": "\u025a",
    "scircle": "\u24e2",
    "scircumflex": "\u015d",
    "scommaaccent": "\u0219",
    "sdotaccent": "\u1e61",
    "sdotbelow": "\u1e63",
    "sdotbelowdotaccent": "\u1e69",
    "seagullbelowcmb": "\u033c",
    "second": "\u2033",
    "secondtonechinese": "\u02ca",
    "section": "\u00a7",
    "seenarabic": "\u0633",
    "seenfinalarabic": "\ufeb2",
    "seeninitialarabic": "\ufeb3",
    "seenmedialarabic": "\ufeb4",
    "segol": "\u05b6",
    "segol13": "\u05b6",
    "segol1f": "\u05b6",
    "segol2c": "\u05b6",
    "segolhebrew": "\u05b6",
    "segolnarrowhebrew": "\u05b6",
    "segolquarterhebrew": "\u05b6",
    "segoltahebrew": "\u0592",
    "segolwidehebrew": "\u05b6",
    "seharmenian": "\u057d",
    "sehiragana": "\u305b",
    "sekatakana": "\u30bb",
    "sekatakanahalfwidth": "\uff7e",
    "semicolon": "\u003b",
    "semicolonarabic": "\u061b",
    "semicolonmonospace": "\uff1b",
    "semicolonsmall": "\ufe54",
    "semivoicedmarkkana": "\u309c",
    "semivoicedmarkkanahalfwidth": "\uff9f",
    "sentisquare": "\u3322",
    "sentosquare": "\u3323",
    "seven": "\u0037",
    "sevenarabic": "\u0667",
    "sevenbengali": "\u09ed",
    "sevencircle": "\u2466",
    "sevencircleinversesansserif": "\u2790",
    "sevendeva": "\u096d",
    "seveneighths": "\u215e",
    "sevengujarati": "\u0aed",
    "sevengurmukhi": "\u0a6d",
    "sevenhackarabic": "\u0667",
    "sevenhangzhou": "\u3027",
    "sevenideographicparen": "\u3226",
    "seveninferior": "\u2087",
    "sevenmonospace": "\uff17",
    "sevenoldstyle": "\uf737",
    "sevenparen": "\u247a",
    "sevenperiod": "\u248e",
    "sevenpersian": "\u06f7",
    "sevenroman": "\u2176",
    "sevensuperior": "\u2077",
    "seventeencircle": "\u2470",
    "seventeenparen": "\u2484",
    "seventeenperiod": "\u2498",
    "seventhai": "\u0e57",
    "sfthyphen": "\u00ad",
    "shaarmenian": "\u0577",
    "shabengali": "\u09b6",
    "shacyrillic": "\u0448",
    "shaddaarabic": "\u0651",
    "shaddadammaarabic": "\ufc61",
    "shaddadammatanarabic": "\ufc5e",
    "shaddafathaarabic": "\ufc60",
    "shaddafathatanarabic": "\u0651\u064b",
    "shaddakasraarabic": "\ufc62",
    "shaddakasratanarabic": "\ufc5f",
    "shade": "\u2592",
    "shadedark": "\u2593",
    "shadelight": "\u2591",
    "shademedium": "\u2592",
    "shadeva": "\u0936",
    "shagujarati": "\u0ab6",
    "shagurmukhi": "\u0a36",
    "shalshelethebrew": "\u0593",
    "shbopomofo": "\u3115",
    "shchacyrillic": "\u0449",
    "sheenarabic": "\u0634",
    "sheenfinalarabic": "\ufeb6",
    "sheeninitialarabic": "\ufeb7",
    "sheenmedialarabic": "\ufeb8",
    "sheicoptic": "\u03e3",
    "sheqel": "\u20aa",
    "sheqelhebrew": "\u20aa",
    "sheva": "\u05b0",
    "sheva115": "\u05b0",
    "sheva15": "\u05b0",
    "sheva22": "\u05b0",
    "sheva2e": "\u05b0",
    "shevahebrew": "\u05b0",
    "shevanarrowhebrew": "\u05b0",
    "shevaquarterhebrew": "\u05b0",
    "shevawidehebrew": "\u05b0",
    "shhacyrillic": "\u04bb",
    "shimacoptic": "\u03ed",
    "shin": "\u05e9",
    "shindagesh": "\ufb49",
    "shindageshhebrew": "\ufb49",
    "shindageshshindot": "\ufb2c",
    "shindageshshindothebrew": "\ufb2c",
    "shindageshsindot": "\ufb2d",
    "shindageshsindothebrew": "\ufb2d",
    "shindothebrew": "\u05c1",
    "shinhebrew": "\u05e9",
    "shinshindot": "\ufb2a",
    "shinshindothebrew": "\ufb2a",
    "shinsindot": "\ufb2b",
    "shinsindothebrew": "\ufb2b",
    "shook": "\u0282",
    "sigma": "\u03c3",
    "sigma1": "\u03c2",
    "sigmafinal": "\u03c2",
    "sigmalunatesymbolgreek": "\u03f2",
    "sihiragana": "\u3057",
    "sikatakana": "\u30b7",
    "sikatakanahalfwidth": "\uff7c",
    "siluqhebrew": "\u05bd",
    "siluqlefthebrew": "\u05bd",
    "similar": "\u223c",
    "sindothebrew": "\u05c2",
    "siosacirclekorean": "\u3274",
    "siosaparenkorean": "\u3214",
    "sioscieuckorean": "\u317e",
    "sioscirclekorean": "\u3266",
    "sioskiyeokkorean": "\u317a",
    "sioskorean": "\u3145",
    "siosnieunkorean": "\u317b",
    "siosparenkorean": "\u3206",
    "siospieupkorean": "\u317d",
    "siostikeutkorean": "\u317c",
    "six": "\u0036",
    "sixarabic": "\u0666",
    "sixbengali": "\u09ec",
    "sixcircle": "\u2465",
    "sixcircleinversesansserif": "\u278f",
    "sixdeva": "\u096c",
    "sixgujarati": "\u0aec",
    "sixgurmukhi": "\u0a6c",
    "sixhackarabic": "\u0666",
    "sixhangzhou": "\u3026",
    "sixideographicparen": "\u3225",
    "sixinferior": "\u2086",
    "sixmonospace": "\uff16",
    "sixoldstyle": "\uf736",
    "sixparen": "\u2479",
    "sixperiod": "\u248d",
    "sixpersian": "\u06f6",
    "sixroman": "\u2175",
    "sixsuperior": "\u2076",
    "sixteencircle": "\u246f",
    "sixteencurrencydenominatorbengali": "\u09f9",
    "sixteenparen": "\u2483",
    "sixteenperiod": "\u2497",
    "sixthai": "\u0e56",
    "slash": "\u002f",
    "slashmonospace": "\uff0f",
    "slong": "\u017f",
    "slongdotaccent": "\u1e9b",
    "smileface": "\u263a",
    "smonospace": "\uff53",
    "sofpasuqhebrew": "\u05c3",
    "softhyphen": "\u00ad",
    "softsigncyrillic": "\u044c",
    "sohiragana": "\u305d",
    "sokatakana": "\u30bd",
    "sokatakanahalfwidth": "\uff7f",
    "soliduslongoverlaycmb": "\u0338",
    "solidusshortoverlaycmb": "\u0337",
    "sorusithai": "\u0e29",
    "sosalathai": "\u0e28",
    "sosothai": "\u0e0b",
    "sosuathai": "\u0e2a",
    "space": "\u0020",
    "spacehackarabic": "\u0020",
    "spade": "\u2660",
    "spadesuitblack": "\u2660",
    "spadesuitwhite": "\u2664",
    "sparen": "\u24ae",
    "squarebelowcmb": "\u033b",
    "squarecc": "\u33c4",
    "squarecm": "\u339d",
    "squarediagonalcrosshatchfill": "\u25a9",
    "squarehorizontalfill": "\u25a4",
    "squarekg": "\u338f",
    "squarekm": "\u339e",
    "squarekmcapital": "\u33ce",
    "squareln": "\u33d1",
    "squarelog": "\u33d2",
    "squaremg": "\u338e",
    "squaremil": "\u33d5",
    "squaremm": "\u339c",
    "squaremsquared": "\u33a1",
    "squareorthogonalcrosshatchfill": "\u25a6",
    "squareupperlefttolowerrightfill": "\u25a7",
    "squareupperrighttolowerleftfill": "\u25a8",
    "squareverticalfill": "\u25a5",
    "squarewhitewithsmallblack": "\u25a3",
    "srsquare": "\u33db",
    "ssabengali": "\u09b7",
    "ssadeva": "\u0937",
    "ssagujarati": "\u0ab7",
    "ssangcieuckorean": "\u3149",
    "ssanghieuhkorean": "\u3185",
    "ssangieungkorean": "\u3180",
    "ssangkiyeokkorean": "\u3132",
    "ssangnieunkorean": "\u3165",
    "ssangpieupkorean": "\u3143",
    "ssangsioskorean": "\u3146",
    "ssangtikeutkorean": "\u3138",
    "ssuperior": "\uf6f2",
    "sterling": "\u00a3",
    "sterlingmonospace": "\uffe1",
    "strokelongoverlaycmb": "\u0336",
    "strokeshortoverlaycmb": "\u0335",
    "subset": "\u2282",
    "subsetnotequal": "\u228a",
    "subsetorequal": "\u2286",
    "succeeds": "\u227b",
    "suchthat": "\u220b",
    "suhiragana": "\u3059",
    "sukatakana": "\u30b9",
    "sukatakanahalfwidth": "\uff7d",
    "sukunarabic": "\u0652",
    "summation": "\u2211",
    "sun": "\u263c",
    "superset": "\u2283",
    "supersetnotequal": "\u228b",
    "supersetorequal": "\u2287",
    "svsquare": "\u33dc",
    "syouwaerasquare": "\u337c",
    "t": "\u0074",
    "tabengali": "\u09a4",
    "tackdown": "\u22a4",
    "tackleft": "\u22a3",
    "tadeva": "\u0924",
    "tagujarati": "\u0aa4",
    "tagurmukhi": "\u0a24",
    "taharabic": "\u0637",
    "tahfinalarabic": "\ufec2",
    "tahinitialarabic": "\ufec3",
    "tahiragana": "\u305f",
    "tahmedialarabic": "\ufec4",
    "taisyouerasquare": "\u337d",
    "takatakana": "\u30bf",
    "takatakanahalfwidth": "\uff80",
    "tatweelarabic": "\u0640",
    "tau": "\u03c4",
    "tav": "\u05ea",
    "tavdages": "\ufb4a",
    "tavdagesh": "\ufb4a",
    "tavdageshhebrew": "\ufb4a",
    "tavhebrew": "\u05ea",
    "tbar": "\u0167",
    "tbopomofo": "\u310a",
    "tcaron": "\u0165",
    "tccurl": "\u02a8",
    "tcedilla": "\u0163",
    "tcheharabic": "\u0686",
    "tchehfinalarabic": "\ufb7b",
    "tchehinitialarabic": "\ufb7c",
    "tchehmedialarabic": "\ufb7d",
    "tchehmeeminitialarabic": "\ufb7c\ufee4",
    "tcircle": "\u24e3",
    "tcircumflexbelow": "\u1e71",
    "tcommaaccent": "\u0163",
    "tdieresis": "\u1e97",
    "tdotaccent": "\u1e6b",
    "tdotbelow": "\u1e6d",
    "tecyrillic": "\u0442",
    "tedescendercyrillic": "\u04ad",
    "teharabic": "\u062a",
    "tehfinalarabic": "\ufe96",
    "tehhahinitialarabic": "\ufca2",
    "tehhahisolatedarabic": "\ufc0c",
    "tehinitialarabic": "\ufe97",
    "tehiragana": "\u3066",
    "tehjeeminitialarabic": "\ufca1",
    "tehjeemisolatedarabic": "\ufc0b",
    "tehmarbutaarabic": "\u0629",
    "tehmarbutafinalarabic": "\ufe94",
    "tehmedialarabic": "\ufe98",
    "tehmeeminitialarabic": "\ufca4",
    "tehmeemisolatedarabic": "\ufc0e",
    "tehnoonfinalarabic": "\ufc73",
    "tekatakana": "\u30c6",
    "tekatakanahalfwidth": "\uff83",
    "telephone": "\u2121",
    "telephoneblack": "\u260e",
    "telishagedolahebrew": "\u05a0",
    "telishaqetanahebrew": "\u05a9",
    "tencircle": "\u2469",
    "tenideographicparen": "\u3229",
    "tenparen": "\u247d",
    "tenperiod": "\u2491",
    "tenroman": "\u2179",
    "tesh": "\u02a7",
    "tet": "\u05d8",
    "tetdagesh": "\ufb38",
    "tetdageshhebrew": "\ufb38",
    "tethebrew": "\u05d8",
    "tetsecyrillic": "\u04b5",
    "tevirhebrew": "\u059b",
    "tevirlefthebrew": "\u059b",
    "thabengali": "\u09a5",
    "thadeva": "\u0925",
    "thagujarati": "\u0aa5",
    "thagurmukhi": "\u0a25",
    "thalarabic": "\u0630",
    "thalfinalarabic": "\ufeac",
    "thanthakhatlowleftthai": "\uf898",
    "thanthakhatlowrightthai": "\uf897",
    "thanthakhatthai": "\u0e4c",
    "thanthakhatupperleftthai": "\uf896",
    "theharabic": "\u062b",
    "thehfinalarabic": "\ufe9a",
    "thehinitialarabic": "\ufe9b",
    "thehmedialarabic": "\ufe9c",
    "thereexists": "\u2203",
    "therefore": "\u2234",
    "theta": "\u03b8",
    "theta1": "\u03d1",
    "thetasymbolgreek": "\u03d1",
    "thieuthacirclekorean": "\u3279",
    "thieuthaparenkorean": "\u3219",
    "thieuthcirclekorean": "\u326b",
    "thieuthkorean": "\u314c",
    "thieuthparenkorean": "\u320b",
    "thirteencircle": "\u246c",
    "thirteenparen": "\u2480",
    "thirteenperiod": "\u2494",
    "thonangmonthothai": "\u0e11",
    "thook": "\u01ad",
    "thophuthaothai": "\u0e12",
    "thorn": "\u00fe",
    "thothahanthai": "\u0e17",
    "thothanthai": "\u0e10",
    "thothongthai": "\u0e18",
    "thothungthai": "\u0e16",
    "thousandcyrillic": "\u0482",
    "thousandsseparatorarabic": "\u066c",
    "thousandsseparatorpersian": "\u066c",
    "three": "\u0033",
    "threearabic": "\u0663",
    "threebengali": "\u09e9",
    "threecircle": "\u2462",
    "threecircleinversesansserif": "\u278c",
    "threedeva": "\u0969",
    "threeeighths": "\u215c",
    "threegujarati": "\u0ae9",
    "threegurmukhi": "\u0a69",
    "threehackarabic": "\u0663",
    "threehangzhou": "\u3023",
    "threeideographicparen": "\u3222",
    "threeinferior": "\u2083",
    "threemonospace": "\uff13",
    "threenumeratorbengali": "\u09f6",
    "threeoldstyle": "\uf733",
    "threeparen": "\u2476",
    "threeperiod": "\u248a",
    "threepersian": "\u06f3",
    "threequarters": "\u00be",
    "threequartersemdash": "\uf6de",
    "threeroman": "\u2172",
    "threesuperior": "\u00b3",
    "threethai": "\u0e53",
    "thzsquare": "\u3394",
    "tihiragana": "\u3061",
    "tikatakana": "\u30c1",
    "tikatakanahalfwidth": "\uff81",
    "tikeutacirclekorean": "\u3270",
    "tikeutaparenkorean": "\u3210",
    "tikeutcirclekorean": "\u3262",
    "tikeutkorean": "\u3137",
    "tikeutparenkorean": "\u3202",
    "tilde": "\u02dc",
    "tildebelowcmb": "\u0330",
    "tildecmb": "\u0303",
    "tildecomb": "\u0303",
    "tildedoublecmb": "\u0360",
    "tildeoperator": "\u223c",
    "tildeoverlaycmb": "\u0334",
    "tildeverticalcmb": "\u033e",
    "timescircle": "\u2297",
    "tipehahebrew": "\u0596",
    "tipehalefthebrew": "\u0596",
    "tippigurmukhi": "\u0a70",
    "titlocyrilliccmb": "\u0483",
    "tiwnarmenian": "\u057f",
    "tlinebelow": "\u1e6f",
    "tmonospace": "\uff54",
    "toarmenian": "\u0569",
    "tohiragana": "\u3068",
    "tokatakana": "\u30c8",
    "tokatakanahalfwidth": "\uff84",
    "tonebarextrahighmod": "\u02e5",
    "tonebarextralowmod": "\u02e9",
    "tonebarhighmod": "\u02e6",
    "tonebarlowmod": "\u02e8",
    "tonebarmidmod": "\u02e7",
    "tonefive": "\u01bd",
    "tonesix": "\u0185",
    "tonetwo": "\u01a8",
    "tonos": "\u0384",
    "tonsquare": "\u3327",
    "topatakthai": "\u0e0f",
    "tortoiseshellbracketleft": "\u3014",
    "tortoiseshellbracketleftsmall": "\ufe5d",
    "tortoiseshellbracketleftvertical": "\ufe39",
    "tortoiseshellbracketright": "\u3015",
    "tortoiseshellbracketrightsmall": "\ufe5e",
    "tortoiseshellbracketrightvertical": "\ufe3a",
    "totaothai": "\u0e15",
    "tpalatalhook": "\u01ab",
    "tparen": "\u24af",
    "trademark": "\u2122",
    "trademarksans": "\uf8ea",
    "trademarkserif": "\uf6db",
    "tretroflexhook": "\u0288",
    "triagdn": "\u25bc",
    "triaglf": "\u25c4",
    "triagrt": "\u25ba",
    "triagup": "\u25b2",
    "ts": "\u02a6",
    "tsadi": "\u05e6",
    "tsadidagesh": "\ufb46",
    "tsadidageshhebrew": "\ufb46",
    "tsadihebrew": "\u05e6",
    "tsecyrillic": "\u0446",
    "tsere": "\u05b5",
    "tsere12": "\u05b5",
    "tsere1e": "\u05b5",
    "tsere2b": "\u05b5",
    "tserehebrew": "\u05b5",
    "tserenarrowhebrew": "\u05b5",
    "tserequarterhebrew": "\u05b5",
    "tserewidehebrew": "\u05b5",
    "tshecyrillic": "\u045b",
    "tsuperior": "\uf6f3",
    "ttabengali": "\u099f",
    "ttadeva": "\u091f",
    "ttagujarati": "\u0a9f",
    "ttagurmukhi": "\u0a1f",
    "tteharabic": "\u0679",
    "ttehfinalarabic": "\ufb67",
    "ttehinitialarabic": "\ufb68",
    "ttehmedialarabic": "\ufb69",
    "tthabengali": "\u09a0",
    "tthadeva": "\u0920",
    "tthagujarati": "\u0aa0",
    "tthagurmukhi": "\u0a20",
    "tturned": "\u0287",
    "tuhiragana": "\u3064",
    "tukatakana": "\u30c4",
    "tukatakanahalfwidth": "\uff82",
    "tusmallhiragana": "\u3063",
    "tusmallkatakana": "\u30c3",
    "tusmallkatakanahalfwidth": "\uff6f",
    "twelvecircle": "\u246b",
    "twelveparen": "\u247f",
    "twelveperiod": "\u2493",
    "twelveroman": "\u217b",
    "twentycircle": "\u2473",
    "twentyhangzhou": "\u5344",
    "twentyparen": "\u2487",
    "twentyperiod": "\u249b",
    "two": "\u0032",
    "twoarabic": "\u0662",
    "twobengali": "\u09e8",
    "twocircle": "\u2461",
    "twocircleinversesansserif": "\u278b",
    "twodeva": "\u0968",
    "twodotenleader": "\u2025",
    "twodotleader": "\u2025",
    "twodotleadervertical": "\ufe30",
    "twogujarati": "\u0ae8",
    "twogurmukhi": "\u0a68",
    "twohackarabic": "\u0662",
    "twohangzhou": "\u3022",
    "twoideographicparen": "\u3221",
    "twoinferior": "\u2082",
    "twomonospace": "\uff12",
    "twonumeratorbengali": "\u09f5",
    "twooldstyle": "\uf732",
    "twoparen": "\u2475",
    "twoperiod": "\u2489",
    "twopersian": "\u06f2",
    "tworoman": "\u2171",
    "twostroke": "\u01bb",
    "twosuperior": "\u00b2",
    "twothai": "\u0e52",
    "twothirds": "\u2154",
    "u": "\u0075",
    "uacute": "\u00fa",
    "ubar": "\u0289",
    "ubengali": "\u0989",
    "ubopomofo": "\u3128",
    "ubreve": "\u016d",
    "ucaron": "\u01d4",
    "ucircle": "\u24e4",
    "ucircumflex": "\u00fb",
    "ucircumflexbelow": "\u1e77",
    "ucyrillic": "\u0443",
    "udattadeva": "\u0951",
    "udblacute": "\u0171",
    "udblgrave": "\u0215",
    "udeva": "\u0909",
    "udieresis": "\u00fc",
    "udieresisacute": "\u01d8",
    "udieresisbelow": "\u1e73",
    "udieresiscaron": "\u01da",
    "udieresiscyrillic": "\u04f1",
    "udieresisgrave": "\u01dc",
    "udieresismacron": "\u01d6",
    "udotbelow": "\u1ee5",
    "ugrave": "\u00f9",
    "ugujarati": "\u0a89",
    "ugurmukhi": "\u0a09",
    "uhiragana": "\u3046",
    "uhookabove": "\u1ee7",
    "uhorn": "\u01b0",
    "uhornacute": "\u1ee9",
    "uhorndotbelow": "\u1ef1",
    "uhorngrave": "\u1eeb",
    "uhornhookabove": "\u1eed",
    "uhorntilde": "\u1eef",
    "uhungarumlaut": "\u0171",
    "uhungarumlautcyrillic": "\u04f3",
    "uinvertedbreve": "\u0217",
    "ukatakana": "\u30a6",
    "ukatakanahalfwidth": "\uff73",
    "ukcyrillic": "\u0479",
    "ukorean": "\u315c",
    "umacron": "\u016b",
    "umacroncyrillic": "\u04ef",
    "umacrondieresis": "\u1e7b",
    "umatragurmukhi": "\u0a41",
    "umonospace": "\uff55",
    "underscore": "\u005f",
    "underscoredbl": "\u2017",
    "underscoremonospace": "\uff3f",
    "underscorevertical": "\ufe33",
    "underscorewavy": "\ufe4f",
    "union": "\u222a",
    "universal": "\u2200",
    "uogonek": "\u0173",
    "uparen": "\u24b0",
    "upblock": "\u2580",
    "upperdothebrew": "\u05c4",
    "upsilon": "\u03c5",
    "upsilondieresis": "\u03cb",
    "upsilondieresistonos": "\u03b0",
    "upsilonlatin": "\u028a",
    "upsilontonos": "\u03cd",
    "uptackbelowcmb": "\u031d",
    "uptackmod": "\u02d4",
    "uragurmukhi": "\u0a73",
    "uring": "\u016f",
    "ushortcyrillic": "\u045e",
    "usmallhiragana": "\u3045",
    "usmallkatakana": "\u30a5",
    "usmallkatakanahalfwidth": "\uff69",
    "ustraightcyrillic": "\u04af",
    "ustraightstrokecyrillic": "\u04b1",
    "utilde": "\u0169",
    "utildeacute": "\u1e79",
    "utildebelow": "\u1e75",
    "uubengali": "\u098a",
    "uudeva": "\u090a",
    "uugujarati": "\u0a8a",
    "uugurmukhi": "\u0a0a",
    "uumatragurmukhi": "\u0a42",
    "uuvowelsignbengali": "\u09c2",
    "uuvowelsigndeva": "\u0942",
    "uuvowelsigngujarati": "\u0ac2",
    "uvowelsignbengali": "\u09c1",
    "uvowelsigndeva": "\u0941",
    "uvowelsigngujarati": "\u0ac1",
    "v": "\u0076",
    "vadeva": "\u0935",
    "vagujarati": "\u0ab5",
    "vagurmukhi": "\u0a35",
    "vakatakana": "\u30f7",
    "vav": "\u05d5",
    "vavdagesh": "\ufb35",
    "vavdagesh65": "\ufb35",
    "vavdageshhebrew": "\ufb35",
    "vavhebrew": "\u05d5",
    "vavholam": "\ufb4b",
    "vavholamhebrew": "\ufb4b",
    "vavvavhebrew": "\u05f0",
    "vavyodhebrew": "\u05f1",
    "vcircle": "\u24e5",
    "vdotbelow": "\u1e7f",
    "vecyrillic": "\u0432",
    "veharabic": "\u06a4",
    "vehfinalarabic": "\ufb6b",
    "vehinitialarabic": "\ufb6c",
    "vehmedialarabic": "\ufb6d",
    "vekatakana": "\u30f9",
    "venus": "\u2640",
    "verticalbar": "\u007c",
    "verticallineabovecmb": "\u030d",
    "verticallinebelowcmb": "\u0329",
    "verticallinelowmod": "\u02cc",
    "verticallinemod": "\u02c8",
    "vewarmenian": "\u057e",
    "vhook": "\u028b",
    "vikatakana": "\u30f8",
    "viramabengali": "\u09cd",
    "viramadeva": "\u094d",
    "viramagujarati": "\u0acd",
    "visargabengali": "\u0983",
    "visargadeva": "\u0903",
    "visargagujarati": "\u0a83",
    "vmonospace": "\uff56",
    "voarmenian": "\u0578",
    "voicediterationhiragana": "\u309e",
    "voicediterationkatakana": "\u30fe",
    "voicedmarkkana": "\u309b",
    "voicedmarkkanahalfwidth": "\uff9e",
    "vokatakana": "\u30fa",
    "vparen": "\u24b1",
    "vtilde": "\u1e7d",
    "vturned": "\u028c",
    "vuhiragana": "\u3094",
    "vukatakana": "\u30f4",
    "w": "\u0077",
    "wacute": "\u1e83",
    "waekorean": "\u3159",
    "wahiragana": "\u308f",
    "wakatakana": "\u30ef",
    "wakatakanahalfwidth": "\uff9c",
    "wakorean": "\u3158",
    "wasmallhiragana": "\u308e",
    "wasmallkatakana": "\u30ee",
    "wattosquare": "\u3357",
    "wavedash": "\u301c",
    "wavyunderscorevertical": "\ufe34",
    "wawarabic": "\u0648",
    "wawfinalarabic": "\ufeee",
    "wawhamzaabovearabic": "\u0624",
    "wawhamzaabovefinalarabic": "\ufe86",
    "wbsquare": "\u33dd",
    "wcircle": "\u24e6",
    "wcircumflex": "\u0175",
    "wdieresis": "\u1e85",
    "wdotaccent": "\u1e87",
    "wdotbelow": "\u1e89",
    "wehiragana": "\u3091",
    "weierstrass": "\u2118",
    "wekatakana": "\u30f1",
    "wekorean": "\u315e",
    "weokorean": "\u315d",
    "wgrave": "\u1e81",
    "whitebullet": "\u25e6",
    "whitecircle": "\u25cb",
    "whitecircleinverse": "\u25d9",
    "whitecornerbracketleft": "\u300e",
    "whitecornerbracketleftvertical": "\ufe43",
    "whitecornerbracketright": "\u300f",
    "whitecornerbracketrightvertical": "\ufe44",
    "whitediamond": "\u25c7",
    "whitediamondcontainingblacksmalldiamond": "\u25c8",
    "whitedownpointingsmalltriangle": "\u25bf",
    "whitedownpointingtriangle": "\u25bd",
    "whiteleftpointingsmalltriangle": "\u25c3",
    "whiteleftpointingtriangle": "\u25c1",
    "whitelenticularbracketleft": "\u3016",
    "whitelenticularbracketright": "\u3017",
    "whiterightpointingsmalltriangle": "\u25b9",
    "whiterightpointingtriangle": "\u25b7",
    "whitesmallsquare": "\u25ab",
    "whitesmilingface": "\u263a",
    "whitesquare": "\u25a1",
    "whitestar": "\u2606",
    "whitetelephone": "\u260f",
    "whitetortoiseshellbracketleft": "\u3018",
    "whitetortoiseshellbracketright": "\u3019",
    "whiteuppointingsmalltriangle": "\u25b5",
    "whiteuppointingtriangle": "\u25b3",
    "wihiragana": "\u3090",
    "wikatakana": "\u30f0",
    "wikorean": "\u315f",
    "wmonospace": "\uff57",
    "wohiragana": "\u3092",
    "wokatakana": "\u30f2",
    "wokatakanahalfwidth": "\uff66",
    "won": "\u20a9",
    "wonmonospace": "\uffe6",
    "wowaenthai": "\u0e27",
    "wparen": "\u24b2",
    "wring": "\u1e98",
    "wsuperior": "\u02b7",
    "wturned": "\u028d",
    "wynn": "\u01bf",
    "x": "\u0078",
    "xabovecmb": "\u033d",
    "xbopomofo": "\u3112",
    "xcircle": "\u24e7",
    "xdieresis": "\u1e8d",
    "xdotaccent": "\u1e8b",
    "xeharmenian": "\u056d",
    "xi": "\u03be",
    "xmonospace": "\uff58",
    "xparen": "\u24b3",
    "xsuperior": "\u02e3",
    "y": "\u0079",
    "yaadosquare": "\u334e",
    "yabengali": "\u09af",
    "yacute": "\u00fd",
    "yadeva": "\u092f",
    "yaekorean": "\u3152",
    "yagujarati": "\u0aaf",
    "yagurmukhi": "\u0a2f",
    "yahiragana": "\u3084",
    "yakatakana": "\u30e4",
    "yakatakanahalfwidth": "\uff94",
    "yakorean": "\u3151",
    "yamakkanthai": "\u0e4e",
    "yasmallhiragana": "\u3083",
    "yasmallkatakana": "\u30e3",
    "yasmallkatakanahalfwidth": "\uff6c",
    "yatcyrillic": "\u0463",
    "ycircle": "\u24e8",
    "ycircumflex": "\u0177",
    "ydieresis": "\u00ff",
    "ydotaccent": "\u1e8f",
    "ydotbelow": "\u1ef5",
    "yeharabic": "\u064a",
    "yehbarreearabic": "\u06d2",
    "yehbarreefinalarabic": "\ufbaf",
    "yehfinalarabic": "\ufef2",
    "yehhamzaabovearabic": "\u0626",
    "yehhamzaabovefinalarabic": "\ufe8a",
    "yehhamzaaboveinitialarabic": "\ufe8b",
    "yehhamzaabovemedialarabic": "\ufe8c",
    "yehinitialarabic": "\ufef3",
    "yehmedialarabic": "\ufef4",
    "yehmeeminitialarabic": "\ufcdd",
    "yehmeemisolatedarabic": "\ufc58",
    "yehnoonfinalarabic": "\ufc94",
    "yehthreedotsbelowarabic": "\u06d1",
    "yekorean": "\u3156",
    "yen": "\u00a5",
    "yenmonospace": "\uffe5",
    "yeokorean": "\u3155",
    "yeorinhieuhkorean": "\u3186",
    "yerahbenyomohebrew": "\u05aa",
    "yerahbenyomolefthebrew": "\u05aa",
    "yericyrillic": "\u044b",
    "yerudieresiscyrillic": "\u04f9",
    "yesieungkorean": "\u3181",
    "yesieungpansioskorean": "\u3183",
    "yesieungsioskorean": "\u3182",
    "yetivhebrew": "\u059a",
    "ygrave": "\u1ef3",
    "yhook": "\u01b4",
    "yhookabove": "\u1ef7",
    "yiarmenian": "\u0575",
    "yicyrillic": "\u0457",
    "yikorean": "\u3162",
    "yinyang": "\u262f",
    "yiwnarmenian": "\u0582",
    "ymonospace": "\uff59",
    "yod": "\u05d9",
    "yoddagesh": "\ufb39",
    "yoddageshhebrew": "\ufb39",
    "yodhebrew": "\u05d9",
    "yodyodhebrew": "\u05f2",
    "yodyodpatahhebrew": "\ufb1f",
    "yohiragana": "\u3088",
    "yoikorean": "\u3189",
    "yokatakana": "\u30e8",
    "yokatakanahalfwidth": "\uff96",
    "yokorean": "\u315b",
    "yosmallhiragana": "\u3087",
    "yosmallkatakana": "\u30e7",
    "yosmallkatakanahalfwidth": "\uff6e",
    "yotgreek": "\u03f3",
    "yoyaekorean": "\u3188",
    "yoyakorean": "\u3187",
    "yoyakthai": "\u0e22",
    "yoyingthai": "\u0e0d",
    "yparen": "\u24b4",
    "ypogegrammeni": "\u037a",
    "ypogegrammenigreekcmb": "\u0345",
    "yr": "\u01a6",
    "yring": "\u1e99",
    "ysuperior": "\u02b8",
    "ytilde": "\u1ef9",
    "yturned": "\u028e",
    "yuhiragana": "\u3086",
    "yuikorean": "\u318c",
    "yukatakana": "\u30e6",
    "yukatakanahalfwidth": "\uff95",
    "yukorean": "\u3160",
    "yusbigcyrillic": "\u046b",
    "yusbigiotifiedcyrillic": "\u046d",
    "yuslittlecyrillic": "\u0467",
    "yuslittleiotifiedcyrillic": "\u0469",
    "yusmallhiragana": "\u3085",
    "yusmallkatakana": "\u30e5",
    "yusmallkatakanahalfwidth": "\uff6d",
    "yuyekorean": "\u318b",
    "yuyeokorean": "\u318a",
    "yyabengali": "\u09df",
    "yyadeva": "\u095f",
    "z": "\u007a",
    "zaarmenian": "\u0566",
    "zacute": "\u017a",
    "zadeva": "\u095b",
    "zagurmukhi": "\u0a5b",
    "zaharabic": "\u0638",
    "zahfinalarabic": "\ufec6",
    "zahinitialarabic": "\ufec7",
    "zahiragana": "\u3056",
    "zahmedialarabic": "\ufec8",
    "zainarabic": "\u0632",
    "zainfinalarabic": "\ufeb0",
    "zakatakana": "\u30b6",
    "zaqefgadolhebrew": "\u0595",
    "zaqefqatanhebrew": "\u0594",
    "zarqahebrew": "\u0598",
    "zayin": "\u05d6",
    "zayindagesh": "\ufb36",
    "zayindageshhebrew": "\ufb36",
    "zayinhebrew": "\u05d6",
    "zbopomofo": "\u3117",
    "zcaron": "\u017e",
    "zcircle": "\u24e9",
    "zcircumflex": "\u1e91",
    "zcurl": "\u0291",
    "zdot": "\u017c",
    "zdotaccent": "\u017c",
    "zdotbelow": "\u1e93",
    "zecyrillic": "\u0437",
    "zedescendercyrillic": "\u0499",
    "zedieresiscyrillic": "\u04df",
    "zehiragana": "\u305c",
    "zekatakana": "\u30bc",
    "zero": "\u0030",
    "zeroarabic": "\u0660",
    "zerobengali": "\u09e6",
    "zerodeva": "\u0966",
    "zerogujarati": "\u0ae6",
    "zerogurmukhi": "\u0a66",
    "zerohackarabic": "\u0660",
    "zeroinferior": "\u2080",
    "zeromonospace": "\uff10",
    "zerooldstyle": "\uf730",
    "zeropersian": "\u06f0",
    "zerosuperior": "\u2070",
    "zerothai": "\u0e50",
    "zerowidthjoiner": "\ufeff",
    "zerowidthnonjoiner": "\u200c",
    "zerowidthspace": "\u200b",
    "zeta": "\u03b6",
    "zhbopomofo": "\u3113",
    "zhearmenian": "\u056a",
    "zhebrevecyrillic": "\u04c2",
    "zhecyrillic": "\u0436",
    "zhedescendercyrillic": "\u0497",
    "zhedieresiscyrillic": "\u04dd",
    "zihiragana": "\u3058",
    "zikatakana": "\u30b8",
    "zinorhebrew": "\u05ae",
    "zlinebelow": "\u1e95",
    "zmonospace": "\uff5a",
    "zohiragana": "\u305e",
    "zokatakana": "\u30be",
    "zparen": "\u24b5",
    "zretroflexhook": "\u0290",
    "zstroke": "\u01b6",
    "zuhiragana": "\u305a",
    "zukatakana": "\u30ba",
}
# --end
