! This file is part of mstore.
! SPDX-Identifier: Apache-2.0
!
! Licensed under the Apache License, Version 2.0 (the "License");
! you may not use this file except in compliance with the License.
! You may obtain a copy of the License at
!
!     http://www.apache.org/licenses/LICENSE-2.0
!
! Unless required by applicable law or agreed to in writing, software
! distributed under the License is distributed on an "AS IS" BASIS,
! WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
! See the License for the specific language governing permissions and
! limitations under the License.

module mstore_ice10
   use mctc_env_accuracy, only : wp
   use mctc_io_structure, only : structure_type, new
   use mstore_data_record, only : record_type, new_record, select_record
   implicit none
   private

   public :: get_ice10_records

contains

subroutine get_ice10_records(records)
   type(record_type), allocatable, intent(out) :: records(:)

   records = [ &
      new_record("gas", gas), &
      new_record("ih", ih), &
      new_record("ii", ii), &
      new_record("iii", iii), &
      new_record("ix", ix), &
      new_record("vi", vi), &
      new_record("vii", vii), &
      new_record("viii", viii), &
      new_record("xii", xii), &
      new_record("xiv", xiv), &
      new_record("xv", xv) &
      ]

end subroutine get_ice10_records

subroutine gas(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 3
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -0.02298820517725_wp,  0.00000000000000_wp, -1.76188954246096_wp, &
      &  1.65369502723146_wp,  0.00000000000000_wp,  0.60848805100320_wp, &
      & -0.10273226709885_wp,  0.00000000000000_wp,  0.07266269355725_wp],&
      & shape(xyz))
   call new(self, sym, xyz)
end subroutine gas

subroutine ih(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 48
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", &
      & "O", "O", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & 16.42503981766265_wp,  2.38527128643930_wp, 13.31135046761087_wp, &
      &  8.19463151601804_wp,  2.34900064134087_wp, 13.31512016724140_wp, &
      &  4.07599319232570_wp,  9.44521501610396_wp, 13.31788142774561_wp, &
      & 12.30621323786204_wp,  9.48157001219778_wp, 13.31404037767425_wp, &
      & 16.42702758764218_wp,  7.12622543214977_wp,  1.57166189539912_wp, &
      &  8.19668801530851_wp,  7.08997278965705_wp,  1.57548090402631_wp, &
      &  4.08778071658890_wp, 14.18661652260924_wp,  1.57759700702706_wp, &
      & 12.31813452137214_wp, 14.22289392805834_wp,  1.57366150348415_wp, &
      &  0.01357847610767_wp,  7.11573106872871_wp,  6.61201236605236_wp, &
      &  8.24372760412218_wp,  7.15198846146367_wp,  6.60828927008199_wp, &
      &  4.13718129888922_wp, 14.24655156605102_wp,  6.61109575017329_wp, &
      & 12.36756536879136_wp, 14.28238867219011_wp,  6.60709224174748_wp, &
      &  0.01221477701658_wp,  2.36918454119591_wp,  8.28501454283597_wp, &
      &  8.24284794873104_wp,  2.40555291814863_wp,  8.28129830615214_wp, &
      &  4.11813243623393_wp,  9.50051216997717_wp,  8.28314279379894_wp, &
      & 12.34842096758405_wp,  9.53663437248632_wp,  8.27922469210884_wp, &
      &  0.00053891634583_wp,  4.09557428934763_wp,  0.56166471938922_wp, &
      &  1.53420392673361_wp,  1.45348788584379_wp,  0.55371680388980_wp, &
      & 13.88466723294377_wp,  0.90318276449335_wp,  0.98382665484178_wp, &
      & 16.44131825796324_wp,  2.41357441169632_wp, 10.18198698486136_wp, &
      &  8.23087728053609_wp,  4.13177538788867_wp,  0.55796030451312_wp, &
      &  5.65427883013520_wp,  0.86689604913579_wp,  0.98769084454382_wp, &
      &  9.76459303199522_wp,  1.48973378075367_wp,  0.54995628460120_wp, &
      &  8.21115039613426_wp,  2.37739578575422_wp, 10.18575145797905_wp, &
      &  1.53291578292495_wp,  7.96289708208228_wp,  0.98984771339105_wp, &
      &  5.64580838977845_wp,  8.58490540772820_wp,  0.55204857765194_wp, &
      &  4.11517826178286_wp, 11.22826853288969_wp,  0.55975760143061_wp, &
      &  4.08796579621938_wp,  9.47214303812714_wp, 10.18756204853901_wp, &
      & 13.87602984873237_wp,  8.62129864302814_wp,  0.54816410292051_wp, &
      &  9.76329152598113_wp,  7.99917960819162_wp,  0.98621354452219_wp, &
      & 12.34545710915991_wp, 11.26467878422636_wp,  0.55583639425996_wp, &
      & 12.31826836160734_wp,  9.50827657866902_wp, 10.18363962985962_wp, &
      & 16.45874992579497_wp,  7.15654398159620_wp,  3.47611739467390_wp, &
      &  8.22837285315125_wp,  7.12030099129342_wp,  3.47995727499166_wp, &
      &  4.12010113935550_wp, 14.21640371195067_wp,  3.48194974506931_wp, &
      & 12.35046552570059_wp, 14.25255541893259_wp,  3.47803158853913_wp, &
      & 16.43555426178161_wp,  5.40029670409040_wp,  7.26113014945246_wp, &
      &  2.55038744398632_wp,  8.59213059486598_wp,  7.69519513140145_wp, &
      & 14.90005062785963_wp,  8.04167883353592_wp,  7.25429389192429_wp, &
      &  8.20508457017650_wp,  5.36408454018843_wp,  7.26490132308965_wp, &
      &  6.66953983392570_wp,  8.00551885385648_wp,  7.25815147251989_wp, &
      & 10.78069196895946_wp,  8.62820877151099_wp,  7.69131326114004_wp, &
      &  2.56409647369436_wp,  0.87419351398715_wp,  7.26052296945525_wp, &
      &  6.67594885582298_wp,  1.49564488066037_wp,  7.69361350602711_wp, &
      &  4.09262720592578_wp, 12.45852812608314_wp,  7.26695997633321_wp, &
      & 14.90613425719830_wp,  1.53166581252119_wp,  7.68981187024135_wp, &
      & 10.79460211498338_wp,  0.91014523018028_wp,  7.25665904387940_wp, &
      & 12.32299734995302_wp, 12.49436676529903_wp,  7.26302158037976_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 16.46073869794094_wp,  0.07247999712643_wp, -0.00744728782614_wp, &
      & -0.00002208935775_wp, 14.22852383291001_wp, -0.00024536152829_wp, &
      &  0.00379016165476_wp,  0.00469557595474_wp, 13.41439204290543_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine ih

subroutine ii(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & -6.36259896051429_wp,  8.72727378497866_wp,  2.70478130343492_wp, &
      & -2.14363388692479_wp,  9.65864925423242_wp, -0.00733935292597_wp, &
      & -3.17284440313189_wp,  4.44335494925429_wp,  5.05841632628302_wp, &
      & -1.03229310567974_wp, -7.01999005464759_wp,  8.52151363212054_wp, &
      & -2.23776880521153_wp,  0.20890363081734_wp,  7.65930684679623_wp, &
      &  0.80832932564347_wp,  7.22201937880047_wp,  3.49023590889647_wp, &
      &  2.07867340630709_wp, -2.30016436678228_wp,  6.67805043955573_wp, &
      &  5.03234965092924_wp,  4.55078261745637_wp,  2.46938024694689_wp, &
      &  9.73238638913123_wp, -1.25658880920138_wp,  1.68677454632174_wp, &
      &  6.12015360270871_wp,  0.31865153265006_wp,  5.10420865666712_wp, &
      & 10.56305238832328_wp,  3.47962735253886_wp,  0.12122159150878_wp, &
      & -5.08139833017193_wp, -4.14386342829269_wp,  7.51846488340064_wp, &
      & -4.97875089857594_wp,  9.00621900834578_wp,  1.44756661984399_wp, &
      &  1.19737043430266_wp, -3.21089119882809_wp,  5.27306144504198_wp, &
      &  6.27471538469595_wp,  5.97456577863801_wp,  2.43077256837360_wp, &
      &  6.62004436041869_wp, 10.10934248762266_wp,  2.40684388942644_wp, &
      & -7.14405871832253_wp,  4.32497977158064_wp,  9.24286671430892_wp, &
      & -0.88864560110298_wp,  8.74353190845093_wp,  1.06948176835250_wp, &
      & -7.68001971411425_wp,  0.59753669052898_wp,  8.26129935700016_wp, &
      & -5.54343372235825_wp,  5.66197389348676_wp,  7.10370358152120_wp, &
      & -2.81919094885052_wp,  2.75169313268280_wp,  5.83613388458039_wp, &
      & -0.53034309760756_wp,  6.12040126582794_wp,  4.25529628022367_wp, &
      & -3.66546224505788_wp,  4.11533837055119_wp,  3.26979082312197_wp, &
      &  5.11306015547582_wp,  3.61647044184943_wp,  0.83336133927840_wp, &
      & -3.65587625145297_wp, -5.37117925262418_wp,  7.75482288083001_wp, &
      & -0.52200045652707_wp, -0.54644412539985_wp,  7.38280309771730_wp, &
      &  3.48658631876403_wp, 10.61293187170784_wp,  0.23468267656923_wp, &
      & -3.42264907976766_wp, -1.25417534784219_wp,  7.52865267528511_wp, &
      &  2.19609208215251_wp,  6.07703848358843_wp,  2.90930860150869_wp, &
      &  8.35912500627539_wp, -4.56027615533327_wp,  5.99022324164996_wp, &
      &  3.50477425530508_wp, -1.33757897185502_wp,  5.88131726110261_wp, &
      &  3.73840408980528_wp, -4.10612372832848_wp,  8.93566903911312_wp, &
      &  5.74211716475252_wp,  1.99318418531489_wp,  4.30712780312761_wp, &
      &  7.37143563208835_wp, -0.48761570737582_wp,  3.94461424873559_wp, &
      & 10.05195586825285_wp,  0.43387501259758_wp,  0.89164951127295_wp, &
      &  6.64429168413262_wp, -4.62829193201241_wp,  9.39292463572718_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 14.32432148996177_wp,  0.09400695600551_wp,  0.06502555158319_wp, &
      & -5.76485447974523_wp, 13.03717806738958_wp, -0.05345069540062_wp, &
      & -5.63063934400149_wp, -8.39659192120953_wp, 10.03037623248034_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine ii

subroutine iii(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  1.13074350636051_wp, 11.14078501074342_wp,  3.10538972938921_wp, &
      &  1.40614614327253_wp,  3.84620807613723_wp,  3.33976407643169_wp, &
      &  2.34282582674967_wp,  7.59363293959745_wp,  6.56288371718799_wp, &
      &  3.93221789630481_wp,  1.58830637241286_wp,  9.22744826176765_wp, &
      &  4.59955573873274_wp, 10.11949734498213_wp, 12.44990752912687_wp, &
      &  4.95809877682228_wp,  5.05525823608184_wp, 12.77854170083423_wp, &
      &  7.41927555570466_wp,  7.31533212397221_wp,  6.33236276968785_wp, &
      &  7.77792204252969_wp,  2.25120891733126_wp,  6.00395633452655_wp, &
      &  8.44492794039951_wp, 10.78229987068811_wp,  2.78115590477119_wp, &
      & 10.03384398946361_wp,  4.77662103462205_wp,  0.11664348784886_wp, &
      & 10.97085610038584_wp,  8.52419419146926_wp,  9.78522631297125_wp, &
      & 11.24636239508761_wp,  1.22962353118794_wp,  9.55104846927791_wp, &
      &  1.59375588001625_wp,  9.92454127328342_wp,  4.48606084639241_wp, &
      &  3.15688751189140_wp, 10.47532134535653_wp,  0.74683737658230_wp, &
      & 12.23152770048412_wp,  4.23039489693533_wp,  2.31894179598950_wp, &
      &  1.40469661323181_wp,  1.96602130353446_wp,  3.54991055729204_wp, &
      &  1.95532278863120_wp,  6.04126251942291_wp,  5.54279133356166_wp, &
      &  4.22284616020382_wp,  7.59001430650664_wp,  6.77408345658150_wp, &
      &  0.72775521303610_wp,  1.47070689786070_wp,  9.25302848038510_wp, &
      &  4.28853229490680_wp,  0.05073557538667_wp, 10.29238978007020_wp, &
      &  4.71829506141115_wp,  6.91381570622472_wp, 12.47737041408787_wp, &
      &  6.13660925578614_wp, 10.47348282457067_wp,  0.62284622942323_wp, &
      &  3.73923137654350_wp,  4.59273700502027_wp,  1.26582361416065_wp, &
      &  4.29041824914147_wp,  3.02969107196882_wp, 10.41595068705418_wp, &
      &  8.63754753045447_wp,  7.77764197817464_wp,  7.71134849952975_wp, &
      &  8.08680335923183_wp,  9.34091210251729_wp,  3.96969856718851_wp, &
      &  6.24026413092443_wp,  1.89699186102504_wp,  7.06894282196374_wp, &
      &  7.65919107406958_wp,  5.45677212038867_wp,  6.03120449865717_wp, &
      & 11.64932978848782_wp, 10.89973396820597_wp,  2.80631289786803_wp, &
      &  8.08873423168094_wp, 12.31987847618171_wp,  3.84611409034617_wp, &
      & 10.42197174834332_wp,  6.32925587490371_wp, 11.98828224280367_wp, &
      &  8.15385932712455_wp,  4.78027102322909_wp,  0.32811308109502_wp, &
      & 10.97239549902133_wp, 10.40438740785897_wp,  9.99533294889628_wp, &
      &  0.14537539996357_wp,  8.13995999416505_wp,  8.76541814311185_wp, &
      & 10.78346542230009_wp,  2.44586294527444_wp, 10.93175629514105_wp, &
      &  9.21999639019618_wp,  1.89506634539881_wp,  7.19257104646166_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 12.37789008654214_wp, -0.00189282664150_wp, -0.00060070703040_wp, &
      & -0.00160590405415_wp, 12.37217253663016_wp, -0.00011623344133_wp, &
      &  0.00072646458515_wp,  0.00029354659568_wp, 12.89184015385954_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine iii

subroutine vi(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 30
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.21334354365044_wp,  2.80355217573427_wp,  4.01644940043440_wp, &
      &  2.78582911270444_wp,  0.22741086135532_wp,  9.32257211790264_wp, &
      &  2.91863860840941_wp,  5.34693466835458_wp,  9.20940865879242_wp, &
      &  2.74116416417077_wp,  8.52609299374414_wp,  2.65923630161886_wp, &
      &  5.33192529155566_wp,  2.96669024374123_wp,  3.90526237629484_wp, &
      &  5.97890932241212_wp,  8.74885363176614_wp,  6.61022356549085_wp, &
      &  8.49564029553922_wp,  2.78984541958738_wp,  7.96658930905883_wp, &
      &  8.71681618105723_wp,  6.04597167339754_wp,  1.30608341874107_wp, &
      &  8.49017892057323_wp, 11.17426695194603_wp,  1.41400444994767_wp, &
      & 11.10806700535631_wp,  8.55273402917048_wp,  6.71973292658878_wp, &
      &  2.10324747327548_wp,  2.86125486424038_wp,  3.98951200327589_wp, &
      &  9.95800131427187_wp,  2.84613205493289_wp,  6.77496984862766_wp, &
      & 11.10930839837739_wp,  4.26369268812414_wp,  3.03565531078583_wp, &
      &  9.97237706050123_wp,  0.45338507573733_wp,  2.35507569818172_wp, &
      &  0.37762255505523_wp,  9.97030251553710_wp,  7.65282234242515_wp, &
      &  2.83226286853966_wp,  2.11764192402391_wp,  9.29483097955289_wp, &
      &  2.78929607721117_wp,  9.98746478775367_wp,  1.46634602751987_wp, &
      &  4.18005488494396_wp, 11.13183123527983_wp,  8.33677405266008_wp, &
      &  1.50604857153895_wp,  6.17974617072995_wp,  8.28817929536084_wp, &
      &  2.95374142907705_wp,  6.23527830659310_wp,  0.27332219307531_wp, &
      &  5.32845969225440_wp,  7.28558381530286_wp,  7.58284990386175_wp, &
      &  0.44592870691315_wp,  8.57512809860286_wp,  5.03778868679784_wp, &
      &  4.19259581760742_wp,  8.69792003839573_wp,  3.85341335388771_wp, &
      &  6.20571434373128_wp,  2.98508865675001_wp,  5.57907035727821_wp, &
      &  7.25660913851501_wp,  5.38687449566849_wp,  2.27755820042579_wp, &
      &  6.17413982373106_wp,  1.55854141957568_wp,  2.98578818308809_wp, &
      &  7.86743643348928_wp,  8.67446518566461_wp,  6.63318968971602_wp, &
      &  8.65696718113680_wp,  4.24336755854004_wp,  9.15940063870071_wp, &
      &  8.55597925528463_wp,  0.49695009080028_wp, 10.34801118409168_wp, &
      &  8.63123134281422_wp,  7.93391047072912_wp,  1.32838134951373_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 11.53685852212402_wp,  0.00051656305940_wp,  0.00786811148897_wp, &
      & -0.07032894361989_wp, 11.53685471247226_wp,  0.00086254294560_wp, &
      & -0.02404161950271_wp, -0.01405784974577_wp, 10.60956191141812_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine vi

subroutine vii(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 48
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", &
      & "O", "O", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & 12.65569713388768_wp, 12.59994558020358_wp, 12.60558267311478_wp, &
      & 12.65674743978386_wp,  6.29957192972608_wp,  6.30388451859802_wp, &
      &  6.23282935445088_wp, -0.00029583774020_wp,  6.30242056952651_wp, &
      &  6.23650725904263_wp,  6.30013696062607_wp,  0.00164593322217_wp, &
      &  3.02136936008884_wp,  3.14942538912051_wp,  3.15148439628299_wp, &
      &  3.02259637010595_wp,  9.44981104965204_wp,  9.45202522900747_wp, &
      &  9.44310409833886_wp,  3.14963365151232_wp,  9.45653106415158_wp, &
      &  9.44547080552246_wp,  9.45150772016954_wp,  3.15572670497980_wp, &
      &  6.60894405283853_wp,  6.29948974075517_wp,  6.30244534382114_wp, &
      &  6.61201126962729_wp, 12.60135996927205_wp,  0.00227096391380_wp, &
      &  0.18477636817302_wp,  6.30127852940220_wp, 12.60319790374508_wp, &
      &  0.18561711069391_wp, -0.00011848350243_wp,  6.30154776037643_wp, &
      &  9.82062058714311_wp,  9.45064995658905_wp,  9.45335404386883_wp, &
      &  9.82205326874477_wp,  3.14912874077110_wp,  3.15246242058227_wp, &
      &  3.40036209174323_wp,  9.45120983055747_wp,  3.15021014996598_wp, &
      &  3.39698518194185_wp,  3.14839961495000_wp,  9.45316745228815_wp, &
      &  0.94299385357203_wp,  1.05448772704816_wp,  1.05516523523706_wp, &
      &  0.94160189447612_wp, 11.54632955269710_wp, 11.54886473849769_wp, &
      & 10.57416311203685_wp,  2.09374845055362_wp, 10.51012830399478_wp, &
      & 10.57619298016365_wp, 10.50522122013792_wp,  2.09980644676760_wp, &
      &  0.94208298540546_wp,  5.24568918979350_wp,  5.24664953204772_wp, &
      &  0.94289700317890_wp,  7.35552730559214_wp,  7.35654420688945_wp, &
      & 10.57391347877593_wp,  4.20381420862548_wp,  8.40120129946463_wp, &
      & 10.57677196938271_wp,  8.39605137785361_wp,  4.20952649319224_wp, &
      &  4.15217282875889_wp,  2.09482847678819_wp,  4.20668839898257_wp, &
      &  4.15354735195694_wp, 10.50532740532217_wp,  8.39816192214919_wp, &
      &  7.36275967083906_wp,  1.05421689875443_wp,  7.35835574262929_wp, &
      &  7.36637955637736_wp, 11.54707581209815_wp,  5.24916077220419_wp, &
      &  4.15273934777761_wp,  4.20408653083686_wp,  2.09717871489465_wp, &
      &  4.15333515109918_wp,  8.39529186954990_wp, 10.50725283125697_wp, &
      &  7.36367748369952_wp,  5.24368478926703_wp, 11.55042252162174_wp, &
      &  7.36656628605571_wp,  7.35454601571912_wp,  1.05749659745365_wp, &
      &  8.68981564084864_wp,  8.39544900410229_wp,  8.39892101365555_wp, &
      &  8.69089415406336_wp,  4.20371597893098_wp,  4.20717298987513_wp, &
      &  5.47898362739696_wp,  7.35479367742100_wp,  5.24719903609926_wp, &
      &  5.47775083835579_wp,  5.24496279134300_wp,  7.35703122367701_wp, &
      &  8.68955958562415_wp, 10.50494751040632_wp, 10.50847149371447_wp, &
      &  8.69114368173691_wp,  2.09451076097983_wp,  2.09751088193081_wp, &
      &  5.48114058619018_wp, 11.54614117389821_wp,  1.05660478151854_wp, &
      &  5.47552741079634_wp,  1.05305794367474_wp, 11.54896372430410_wp, &
      & 11.89974749571799_wp,  7.35489982434961_wp, 11.54978620694594_wp, &
      & 11.90220119056407_wp,  5.24598557844329_wp,  1.05687409344132_wp, &
      &  2.26914646601125_wp,  8.39670899714728_wp,  2.09543031248630_wp, &
      &  2.26632055954870_wp,  4.20403145647547_wp, 10.50724975560606_wp, &
      & 11.90133726508203_wp, 11.54581336545804_wp,  7.35873924476250_wp, &
      & 11.90047652626517_wp,  1.05362121646013_wp,  5.24833229392193_wp, &
      &  2.26965136327129_wp, 10.50574767393739_wp,  4.20538992999468_wp, &
      &  2.26580959050671_wp,  2.09432422047371_wp,  8.39793090422475_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 12.84486159527349_wp, -0.00141793718150_wp,  0.00221826730178_wp, &
      &  0.00217957628638_wp, 12.60233325220201_wp,  0.00092986554048_wp, &
      & -0.00383320658344_wp, -0.00092465566447_wp, 12.60277560585319_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine vii

subroutine viii(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 24
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "O", "O", "O", "O", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  0.00000000000000_wp,  3.72529102253171_wp,  2.53314601824047_wp, &
      &  4.46789931641409_wp,  8.19319033894580_wp,  8.97341762318620_wp, &
      &  4.46789931641409_wp,  5.21050761029647_wp,  8.97341762318620_wp, &
      &  0.00000000000000_wp,  0.74260829388238_wp,  2.53314601824047_wp, &
      &  7.44445726850351_wp,  6.70184897462114_wp,  5.75329470125660_wp, &
      &  5.95924068073876_wp,  2.23394965820705_wp, 12.19356630620233_wp, &
      &  4.46789931641409_wp,  3.72529102253171_wp,  3.90712558670525_wp, &
      &  0.00000000000000_wp,  8.19319033894580_wp, 10.34739719165098_wp, &
      &  0.00000000000000_wp,  5.21050761029647_wp, 10.34739719165098_wp, &
      &  4.46789931641409_wp,  0.74260829388238_wp,  3.90712558670525_wp, &
      &  5.95924068073876_wp,  6.70184897462114_wp,  0.68697690368912_wp, &
      &  7.44445726850351_wp,  2.23394965820705_wp,  7.12724850863485_wp, &
      &  1.49134136432467_wp,  2.23394965820705_wp,  7.12724850863485_wp, &
      &  2.97655795208942_wp,  6.70184897462114_wp,  0.68697690368912_wp, &
      &  2.97655795208942_wp,  2.23394965820705_wp, 12.19356630620233_wp, &
      &  1.49134136432467_wp,  6.70184897462114_wp,  5.75329470125660_wp, &
      &  0.00000000000000_wp,  2.23394965820705_wp,  1.40312943671443_wp, &
      &  4.46789931641409_wp,  6.70184897462114_wp,  7.84340104166016_wp, &
      &  0.00000000000000_wp,  6.70184897462114_wp,  4.62327811973047_wp, &
      &  4.46789931641409_wp,  2.23394965820705_wp, 11.06354972467620_wp, &
      &  4.46789931641409_wp,  2.23394965820705_wp,  5.03714216823129_wp, &
      &  0.00000000000000_wp,  6.70184897462114_wp, 11.47741377317702_wp, &
      &  4.46789931641409_wp,  6.70184897462114_wp,  1.81699348521525_wp, &
      &  0.00000000000000_wp,  2.23394965820705_wp,  8.25726509016098_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      &  8.93579863282818_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  8.93579863282818_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp, 12.88054320989145_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine viii

subroutine ix(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "O", "O", "O", "O", &
      & "O", "O", "O", "O", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      & 12.10235436244213_wp,  4.17373056307153_wp,  2.56202294191533_wp, &
      &  1.41505227515306_wp,  1.88760378511567_wp,  3.71734642024044_wp, &
      &  3.65807664386316_wp,  4.43334880077397_wp,  1.40570835746308_wp, &
      &  0.17927789832368_wp,  8.10781447965789_wp,  9.01743557004962_wp, &
      & 10.86658141056660_wp, 10.39394227722036_wp, 10.17275774130696_wp, &
      &  8.62355748611956_wp,  7.84819666854666_wp,  7.86112019181823_wp, &
      &  1.97089683834796_wp,  5.96537053587592_wp,  5.78594965354859_wp, &
      &  4.25568288092345_wp,  7.56115045157534_wp,  6.94115556485351_wp, &
      &  1.70212474516072_wp,  9.80442671732481_wp,  4.63373783444528_wp, &
      & 10.31073675204201_wp,  6.31617597758217_wp, 12.24136138738269_wp, &
      &  8.02595098449717_wp,  4.72039593674038_wp,  0.48574414140402_wp, &
      & 10.57950751692031_wp,  2.47711967227290_wp, 11.08914915411828_wp, &
      &  6.31445246008449_wp, 10.31057063884187_wp,  0.66941902068458_wp, &
      &  4.71910098594881_wp,  8.02573053562816_wp, 12.42494902542394_wp, &
      &  2.47561827491527_wp, 10.57828466235100_wp,  1.82219140101161_wp, &
      &  5.96717914714748_wp,  1.97097490674851_wp,  7.12483008649830_wp, &
      &  7.56253085171566_wp,  4.25581480254433_wp,  5.96953715426782_wp, &
      &  9.80601335049924_wp,  1.70326134433038_wp,  8.27760269497543_wp, &
      &  4.17417059559864_wp, 12.10221086955340_wp, 10.34834645025547_wp, &
      &  1.88872702942236_wp,  1.41577190532533_wp,  9.19329998685223_wp, &
      &  4.43411794609738_wp,  3.65769676592873_wp, 11.50487301786570_wp, &
      &  8.10746137031879_wp,  0.17933401451594_wp,  3.89293480556913_wp, &
      & 10.39290514657293_wp, 10.86577323237795_wp,  2.73788826025043_wp, &
      &  7.84751539022183_wp,  8.62384833842946_wp,  5.04946179078904_wp, &
      &  1.37845825429425_wp,  3.77532308960784_wp,  3.56595784596796_wp, &
      & 10.90317570043490_wp,  8.50622318468921_wp, 10.02137032516745_wp, &
      &  2.36819924006859_wp,  7.52318019042183_wp,  6.78897763321455_wp, &
      &  9.91343473421957_wp,  4.75836674798639_wp,  0.33356671786958_wp, &
      &  4.75616634811686_wp,  9.91310789096311_wp, 12.57801986871476_wp, &
      &  7.52546446616405_wp,  2.36843740364574_wp,  6.12260696820313_wp, &
      &  3.77648601843899_wp,  1.37847890574484_wp,  9.34432454154792_wp, &
      &  8.50514607918709_wp, 10.90306560175690_wp,  2.88891316402906_wp, &
      &  4.85956839599725_wp,  4.85906079266074_wp, 12.91079994852814_wp, &
      &  7.42206565554161_wp,  7.42248412818389_wp,  6.45538887716438_wp, &
      &  1.27443744203116_wp, 11.00534192427742_wp,  3.22801698820942_wp, &
      & 11.00719408779706_wp,  1.27620426018142_wp,  9.68342831352640_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 12.28088912361703_wp,  0.00074427500000_wp, -0.00000025847176_wp, &
      &  0.00074384965234_wp, 12.28080154974145_wp, -0.00000016520702_wp, &
      & -0.00000025695436_wp, -0.00000015367010_wp, 12.91082309691871_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine ix

subroutine xii(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 84
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", &
      & "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  5.56353889238787_wp,  8.95661063500987_wp,  3.72003453797662_wp, &
      &  3.99109073217384_wp,  6.78985219685064_wp,  4.98431706857828_wp, &
      &  1.92372578010702_wp,  9.87717361874703_wp,  2.39390095138968_wp, &
      &  7.57572863340933_wp, 12.67143642141157_wp,  3.92689331725310_wp, &
      &  9.22468006833280_wp, 10.35581427156036_wp,  5.14012758095162_wp, &
      & 15.26090767924439_wp, 10.37840849451730_wp, -4.51334785382691_wp, &
      &  3.76279152164069_wp,  6.27768331519031_wp,  0.54751147964221_wp, &
      &  5.05180110557072_wp,  6.81525663400754_wp,  9.20465484633034_wp, &
      &  2.78859691071408_wp,  4.93455864082094_wp,  8.80969532196928_wp, &
      &  4.90692673158483_wp,  5.04144924019610_wp, 16.94357691046783_wp, &
      &  1.23625008122123_wp,  0.16198067830362_wp,  3.27774819272414_wp, &
      &  5.03507625233573_wp,  1.92987043041748_wp,  2.95308841652497_wp, &
      &  6.07347151136249_wp,  0.66345608592373_wp,  0.49963707489225_wp, &
      &  0.68383831870801_wp,  2.90692186534485_wp,  4.51520634178614_wp, &
      &  2.48424655740747_wp,  2.04211719487237_wp, 12.38758004133232_wp, &
      &  4.05669471762150_wp, 13.70434563698814_wp, 11.12329751073066_wp, &
      &  6.12405966968843_wp,  2.96268017860953_wp, 13.71371362791922_wp, &
      &  0.47205681638601_wp,  5.75694298127407_wp, 12.18072126205584_wp, &
      & 14.83882235348488_wp,  3.44132083142286_wp,  5.26949401466752_wp, &
      &  8.80259474257329_wp,  3.46391505437980_wp, 14.92296944944606_wp, &
      &  4.28499392815465_wp, 13.19217675532781_wp, 15.56010309966673_wp, &
      &  2.99598434422462_wp, 13.72975007414504_wp,  6.90295973297860_wp, &
      &  5.25918853908125_wp, 11.84905208095844_wp,  7.29791925733966_wp, &
      &  3.14085871821051_wp, 11.95594268033360_wp, -0.83596233115889_wp, &
      &  6.81153536857411_wp,  7.07647411844122_wp, 12.82986638658480_wp, &
      &  3.01270919745961_wp,  8.84436387055498_wp, 13.15452616278397_wp, &
      &  1.97431393843285_wp,  7.57794952606123_wp, 15.60797750441669_wp, &
      &  7.36394713108732_wp,  9.82141530548235_wp, 11.59240823752280_wp, &
      & 10.49210504341864_wp,  4.87237624526513_wp,  9.53858354948742_wp, &
      & 12.06455320363267_wp,  7.03913468342436_wp,  8.27430101888576_wp, &
      & 14.13191815569960_wp,  3.95181326152797_wp, 10.86471713607433_wp, &
      &  8.47991530239718_wp,  1.15755045886343_wp,  9.33172477021094_wp, &
      &  6.83096386747371_wp,  3.47317260871464_wp,  8.11849050651242_wp, &
      &  0.79473625656212_wp,  3.45057838575770_wp, 17.77196594129095_wp, &
      & 12.29285241416582_wp,  7.55130356508469_wp, 12.71110660782183_wp, &
      & 11.00384283023579_wp,  7.01373024626746_wp,  4.05396324113371_wp, &
      & 13.26704702509243_wp,  8.89442823945406_wp,  4.44892276549476_wp, &
      & 11.14871720422168_wp,  8.78753764007890_wp, -3.68495882300378_wp, &
      & 14.81939385458528_wp, 13.66700620197128_wp,  9.98086989473991_wp, &
      & 11.02056768347078_wp, 11.89911644985752_wp, 10.30552967093907_wp, &
      &  9.98217242444402_wp, 13.16553079435127_wp, 12.75898101257179_wp, &
      & 15.37180561709850_wp, 10.92206501493015_wp,  8.74341174567791_wp, &
      & 13.57139737839904_wp, 11.78686968540263_wp,  0.87103804613172_wp, &
      & 11.99894921818501_wp,  0.12464124328686_wp,  2.13532057673338_wp, &
      &  9.93158426611808_wp, 10.86630670166547_wp, -0.45509554045518_wp, &
      & 15.58358711942050_wp,  8.07204389900093_wp,  1.07789682540820_wp, &
      &  1.21682158232163_wp, 10.38766604885214_wp,  7.98912407279652_wp, &
      &  7.25304919323322_wp, 10.36507182589520_wp, -1.66435136198201_wp, &
      & 11.77065000765186_wp,  0.63681012494719_wp, -2.30148501220269_wp, &
      & 13.05965959158189_wp,  0.09923680612996_wp,  6.35565835448544_wp, &
      & 10.79645539672526_wp,  1.97993479931656_wp,  5.96069883012438_wp, &
      & 12.91478521759600_wp,  1.87304419994140_wp, 14.09458041862293_wp, &
      &  9.24410856723240_wp,  6.75251276183378_wp,  0.42875170087924_wp, &
      & 13.04293473834690_wp,  4.98462300972002_wp,  0.10409192468007_wp, &
      & 14.08132999737366_wp,  6.25103735421377_wp, -2.34935941695265_wp, &
      &  8.69169680471919_wp,  4.00757157479265_wp,  1.66620984994124_wp, &
      &  4.14321295037954_wp,  7.74914975480804_wp,  3.36350822878680_wp, &
      &  7.50146252989267_wp, 11.11760986276612_wp,  4.98751092066806_wp, &
      &  0.90824662042001_wp, 11.20553496591533_wp,  1.50772991716379_wp, &
      &  4.35824061317413_wp,  5.55022972222264_wp,  7.96993154125457_wp, &
      &  3.28424987339997_wp,  5.56631869402894_wp, -1.13662037568452_wp, &
      &  6.65525118269577_wp,  1.50356204274663_wp,  2.09238527776915_wp, &
      &  1.97253300288302_wp,  1.53632125132345_wp,  4.33735309708567_wp, &
      &  3.90457249941580_wp,  0.83465631467054_wp, 12.74410635052214_wp, &
      &  0.54632291990267_wp,  4.20311642262862_wp, 11.12010365864088_wp, &
      &  7.13953882937532_wp,  4.29104152577783_wp, 14.59988466214515_wp, &
      &  3.68954483662121_wp, 12.46472316236014_wp,  8.13768303805437_wp, &
      &  4.76353557639537_wp, 12.48081213416644_wp, 17.24423495499346_wp, &
      &  1.39253426709957_wp,  8.41805548288413_wp, 14.01522930153979_wp, &
      &  6.07525244691232_wp,  8.45081469146105_wp, 11.77026148222327_wp, &
      & 11.91243098542697_wp,  6.07983712546696_wp,  9.89510985867725_wp, &
      &  8.55418140591384_wp,  2.71137701750888_wp,  8.27110716679598_wp, &
      & 15.14739731538650_wp,  2.62345191435958_wp, 11.75088817030025_wp, &
      & 11.69740332263238_wp,  8.27875715805236_wp,  5.28868654620947_wp, &
      & 12.77139406240654_wp,  8.26266818624606_wp, 14.39523846314856_wp, &
      &  9.40039275311074_wp, 12.32542483752837_wp, 11.16623280969489_wp, &
      & 14.08311093292349_wp, 12.29266562895145_wp,  8.92126499037837_wp, &
      & 12.15107143639071_wp, 12.99433056560446_wp,  0.51451173694190_wp, &
      & 15.50932101590384_wp,  9.62587045764638_wp,  2.13851442882317_wp, &
      &  8.91610510643119_wp,  9.53794535449717_wp, -1.34126657468110_wp, &
      & 12.36609909918531_wp,  1.36426371791486_wp,  5.12093504940967_wp, &
      & 11.29210835941114_wp,  1.34817474610856_wp, -3.98561686752942_wp, &
      & 14.66310966870694_wp,  5.41093139739087_wp, -0.75661121407574_wp, &
      &  9.98039148889419_wp,  5.37817218881395_wp,  1.48835660524077_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 16.01571697202234_wp,  0.00000000000000_wp, -5.69799298368980_wp, &
      &  0.00000000000000_wp, 13.82898688027500_wp,  0.00000000000000_wp, &
      &  0.03992696378417_wp,  0.00000000000000_wp, 18.95661107115384_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine xii

subroutine xiv(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 36
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "O", "O", "O", "O", &
      & "O", "O", "O", "O", "O", "O", "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  8.26641986097392_wp, 12.64054318635888_wp,  3.55943245160194_wp, &
      &  1.46746777548921_wp,  3.08471912015524_wp,  1.99189835897575_wp, &
      & 12.14172255404041_wp, 14.66377156503157_wp,  6.81827043142532_wp, &
      & 11.21551980433432_wp,  7.07622589110048_wp,  2.38190982893809_wp, &
      &  6.42576439841572_wp,  8.86901097952444_wp,  2.76421291893793_wp, &
      & 13.10425321522234_wp,  4.88152318898254_wp,  2.95675924158083_wp, &
      &  6.26436894550781_wp, 11.98271579459704_wp,  5.72801418109976_wp, &
      &  1.30607232258131_wp,  6.19842393522784_wp,  6.50032874106194_wp, &
      & 14.92909030201716_wp,  2.42687666095852_wp,  7.29554827372595_wp, &
      & 11.97999035865676_wp,  7.99119395621692_wp,  6.11802565106214_wp, &
      & 10.09125694776875_wp, 10.18591172576985_wp,  6.69287506370489_wp, &
      &  0.53458313997690_wp,  9.96059411833466_wp,  3.91279919264607_wp, &
      & 14.15760111941275_wp, 13.73214139260398_wp,  4.70801872531008_wp, &
      &  9.03790904357834_wp,  1.33529352214830_wp,  0.97190290318607_wp, &
      & 13.99620566650473_wp, 10.61843657753138_wp,  1.74421746314825_wp, &
      &  4.24815363765985_wp, 14.60994334847662_wp,  1.35420599318592_wp, &
      &  2.35942022677172_wp, 12.41524064635857_wp,  0.77935658054307_wp, &
      &  4.40988583304338_wp,  7.93738080709686_wp,  0.65396121282269_wp, &
      & 11.05378760895067_wp,  0.40366334972082_wp,  3.08215460930132_wp, &
      &  3.48368308333718_wp,  0.45747649884078_wp,  5.09032181530987_wp, &
      &  5.37241649422520_wp,  2.65219426839371_wp,  4.51547240266713_wp, &
      &  7.19725358102024_wp,  5.10682572898275_wp,  0.17668337052207_wp, &
      &  3.32195088795365_wp,  7.13005410765543_wp,  4.39007703494670_wp, &
      &  9.19930449648635_wp,  4.44899833722101_wp,  5.48033328527226_wp, &
      &  0.09326317168909_wp,  3.81853422038150_wp,  0.93259051234252_wp, &
      &  9.86724586952734_wp, 14.88570091885805_wp,  1.73893144534924_wp, &
      &  3.80786696076658_wp, 13.35200346552711_wp,  0.01638949501423_wp, &
      &  7.63857354930794_wp, 11.24890069437089_wp,  4.66870633446657_wp, &
      &  3.92396976023045_wp,  1.71543144922507_wp,  3.75250531713829_wp, &
      & 13.32826429346374_wp,  0.18173399589434_wp,  5.47504726747325_wp, &
      & 11.53970368176350_wp,  9.24914890660131_wp,  7.45584214923373_wp, &
      & 11.65580648122759_wp,  5.81828600815118_wp,  3.71972632710977_wp, &
      & 15.37041027030497_wp, 11.35225167775753_wp,  2.80352530978149_wp, &
      &  5.59642757246671_wp,  7.35198346148180_wp,  1.99718437677477_wp, &
      &  2.13540914853031_wp,  7.71545145327038_wp,  5.73330019889877_wp, &
      &  7.82509989268612_wp,  3.71518323699464_wp,  6.53964113190544_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 15.46367344199406_wp,  0.00000000000000_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp, 15.06743491475228_wp,  0.00000000000000_wp, &
      &  0.00000000000000_wp,  0.00000000000000_wp,  7.47223164424801_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine xiv

subroutine xv(self)
   type(structure_type), intent(out) :: self
   integer, parameter :: nat = 30
   character(len=*), parameter :: sym(nat) = [character(len=4)::&
      & "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", "H", &
      & "H", "H", "H", "H", "H", "H", "O", "O", "O", "O", "O", "O", "O", "O", &
      & "O", "O"]
   real(wp), parameter :: xyz(3, nat) = reshape([&
      &  2.82132582158723_wp,  4.23871426637512_wp,  3.80403834880621_wp, &
      &  2.87614482608943_wp,  1.31636709846106_wp,  3.74432157818635_wp, &
      &  2.90503356979895_wp,  7.82510256574375_wp,  6.70461819190271_wp, &
      &  7.89749054533177_wp,  2.85843448205158_wp,  9.37121655201425_wp, &
      &  4.27364611251997_wp,  5.21623801515079_wp,  7.53401948658210_wp, &
      &  4.37006469832010_wp,  0.47520098837517_wp,  7.55884780386910_wp, &
      &  1.45958997933034_wp,  0.37582480791409_wp,  7.58342939115770_wp, &
      &  0.48468235242007_wp,  4.19733867267633_wp,  8.29178273029482_wp, &
      &  5.21765773368706_wp,  2.84217866146634_wp,  0.17329064108438_wp, &
      &  0.55286474404154_wp,  2.81295461593246_wp,  0.17656089494690_wp, &
      &  6.27181445161658_wp,  8.60430038640814_wp, 10.55970195800288_wp, &
      & 10.93660744126193_wp,  8.63352443194201_wp, 10.55643170414036_wp, &
      &  8.66814636371632_wp,  7.20776478149936_wp,  6.92895425028105_wp, &
      &  8.61332741835298_wp, 10.13010042905663_wp,  6.98868177478003_wp, &
      &  8.58443861550460_wp,  3.62137648213073_wp,  4.02837440718456_wp, &
      &  3.59198167780010_wp,  8.58803305549914_wp,  1.36177606999445_wp, &
      &  7.21582607278358_wp,  6.23024103272369_wp,  3.19897311250516_wp, &
      &  7.11940748698345_wp, 10.97127805949930_wp,  3.17414479521816_wp, &
      & 10.02988220597321_wp, 11.07065423996038_wp,  3.14956320792949_wp, &
      & 11.00478985419401_wp,  7.24914036516512_wp,  2.44122059975013_wp, &
      &  2.86573742314222_wp,  2.79539362561694_wp,  2.58745013365098_wp, &
      &  2.79788569382970_wp,  5.94267677778210_wp,  6.62401649160071_wp, &
      &  6.01857322023104_wp,  2.85725029287676_wp,  9.19629461507930_wp, &
      &  2.89372142085032_wp, 11.08471528788928_wp,  6.70734370187699_wp, &
      & 11.15881685454565_wp,  2.75708866801304_wp,  9.28787646553196_wp, &
      &  8.62373482130019_wp,  8.65107390190066_wp,  8.14555321931540_wp, &
      &  8.69158651278439_wp,  5.50380226005935_wp,  4.10898683844416_wp, &
      &  5.47089896507243_wp,  8.58922875499772_wp,  1.53669798400797_wp, &
      &  0.33065535206844_wp,  8.68939036982849_wp,  1.44512686451299_wp, &
      &  8.59575076445323_wp,  0.36176375998519_wp,  4.02564889721035_wp],&
      & shape(xyz))
   real(wp), parameter :: lattice(3, 3) = reshape([&
      & 11.50598997441095_wp, -0.05381171732138_wp,  0.02495639897156_wp, &
      & -0.03782832335191_wp, 11.51032378931266_wp, -0.02292143596095_wp, &
      &  0.02131053424452_wp, -0.01003302411680_wp, 10.73095763607666_wp],&
      & shape(lattice))
   call new(self, sym, xyz, lattice=lattice)
end subroutine xv

end module mstore_ice10
