/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { describe, expect, it, vi } from 'vitest';
import { ApprovalMode, BaseDeclarativeTool, BaseToolInvocation, Kind, ToolConfirmationOutcome, } from '../index.js';
import { MockModifiableTool, MockTool } from '../test-utils/tools.js';
import { CoreToolScheduler, convertToFunctionResponse, } from './coreToolScheduler.js';
import { getPlanModeSystemReminder } from './prompts.js';
class TestApprovalTool extends BaseDeclarativeTool {
    config;
    static Name = 'testApprovalTool';
    constructor(config) {
        super(TestApprovalTool.Name, 'TestApprovalTool', 'A tool for testing approval logic', Kind.Edit, {
            properties: { id: { type: 'string' } },
            required: ['id'],
            type: 'object',
        });
        this.config = config;
    }
    createInvocation(params) {
        return new TestApprovalInvocation(this.config, params);
    }
}
class TestApprovalInvocation extends BaseToolInvocation {
    config;
    constructor(config, params) {
        super(params);
        this.config = config;
    }
    getDescription() {
        return `Test tool ${this.params.id}`;
    }
    async shouldConfirmExecute() {
        // Need confirmation unless approval mode is AUTO_EDIT
        if (this.config.getApprovalMode() === ApprovalMode.AUTO_EDIT) {
            return false;
        }
        return {
            type: 'edit',
            title: `Confirm Test Tool ${this.params.id}`,
            fileName: `test-${this.params.id}.txt`,
            filePath: `/test-${this.params.id}.txt`,
            fileDiff: 'Test diff content',
            originalContent: '',
            newContent: 'Test content',
            onConfirm: async (outcome) => {
                if (outcome === ToolConfirmationOutcome.ProceedAlways) {
                    this.config.setApprovalMode(ApprovalMode.AUTO_EDIT);
                }
            },
        };
    }
    async execute() {
        return {
            llmContent: `Executed test tool ${this.params.id}`,
            returnDisplay: `Executed test tool ${this.params.id}`,
        };
    }
}
class SimpleToolInvocation extends BaseToolInvocation {
    executeImpl;
    constructor(params, executeImpl) {
        super(params);
        this.executeImpl = executeImpl;
    }
    getDescription() {
        return 'simple tool invocation';
    }
    async execute() {
        return await Promise.resolve(this.executeImpl());
    }
}
class SimpleTool extends BaseDeclarativeTool {
    executeImpl;
    constructor(name, kind, executeImpl) {
        super(name, name, 'Simple test tool', kind, {
            type: 'object',
            properties: {},
            additionalProperties: true,
        });
        this.executeImpl = executeImpl;
    }
    createInvocation(params) {
        return new SimpleToolInvocation(params, this.executeImpl);
    }
}
async function waitForStatus(onToolCallsUpdate, status, timeout = 5000) {
    return new Promise((resolve, reject) => {
        const startTime = Date.now();
        const check = () => {
            if (Date.now() - startTime > timeout) {
                const seenStatuses = onToolCallsUpdate.mock.calls
                    .flatMap((call) => call[0])
                    .map((toolCall) => toolCall.status);
                reject(new Error(`Timed out waiting for status "${status}". Seen statuses: ${seenStatuses.join(', ')}`));
                return;
            }
            const foundCall = onToolCallsUpdate.mock.calls
                .flatMap((call) => call[0])
                .find((toolCall) => toolCall.status === status);
            if (foundCall) {
                resolve(foundCall);
            }
            else {
                setTimeout(check, 10); // Check again in 10ms
            }
        };
        check();
    });
}
describe('CoreToolScheduler', () => {
    it('should cancel a tool call if the signal is aborted before confirmation', async () => {
        const mockTool = new MockTool();
        mockTool.shouldConfirm = true;
        const declarativeTool = mockTool;
        const mockToolRegistry = {
            getTool: () => declarativeTool,
            getFunctionDeclarations: () => [],
            tools: new Map(),
            discovery: {},
            registerTool: () => { },
            getToolByName: () => declarativeTool,
            getToolByDisplayName: () => declarativeTool,
            getTools: () => [],
            discoverTools: async () => { },
            getAllTools: () => [],
            getToolsByServer: () => [],
        };
        const onAllToolCallsComplete = vi.fn();
        const onToolCallsUpdate = vi.fn();
        const mockConfig = {
            getSessionId: () => 'test-session-id',
            getUsageStatisticsEnabled: () => true,
            getDebugMode: () => false,
            getApprovalMode: () => ApprovalMode.DEFAULT,
            getAllowedTools: () => [],
            getContentGeneratorConfig: () => ({
                model: 'test-model',
                authType: 'oauth-personal',
            }),
            getToolRegistry: () => mockToolRegistry,
        };
        const scheduler = new CoreToolScheduler({
            config: mockConfig,
            onAllToolCallsComplete,
            onToolCallsUpdate,
            getPreferredEditor: () => 'vscode',
            onEditorClose: vi.fn(),
        });
        const abortController = new AbortController();
        const request = {
            callId: '1',
            name: 'mockTool',
            args: {},
            isClientInitiated: false,
            prompt_id: 'prompt-id-1',
        };
        abortController.abort();
        await scheduler.schedule([request], abortController.signal);
        expect(onAllToolCallsComplete).toHaveBeenCalled();
        const completedCalls = onAllToolCallsComplete.mock
            .calls[0][0];
        expect(completedCalls[0].status).toBe('cancelled');
    });
    describe('plan mode enforcement', () => {
        it('returns plan reminder and skips execution for edit tools', async () => {
            const executeSpy = vi.fn().mockResolvedValue({
                llmContent: 'should not execute',
                returnDisplay: 'should not execute',
            });
            // Use MockTool with shouldConfirm=true to simulate a tool that requires confirmation
            const tool = new MockTool('write_file');
            tool.shouldConfirm = true;
            tool.executeFn = executeSpy;
            const mockToolRegistry = {
                getTool: vi.fn().mockReturnValue(tool),
                getAllToolNames: vi.fn().mockReturnValue([tool.name]),
            };
            const onAllToolCallsComplete = vi.fn();
            const onToolCallsUpdate = vi.fn();
            const mockConfig = {
                getSessionId: () => 'plan-session',
                getUsageStatisticsEnabled: () => true,
                getDebugMode: () => false,
                getApprovalMode: vi.fn().mockReturnValue(ApprovalMode.PLAN),
                getAllowedTools: () => [],
                getContentGeneratorConfig: () => ({
                    model: 'test-model',
                    authType: 'oauth-personal',
                }),
                getToolRegistry: () => mockToolRegistry,
            };
            const scheduler = new CoreToolScheduler({
                config: mockConfig,
                onAllToolCallsComplete,
                onToolCallsUpdate,
                getPreferredEditor: () => 'vscode',
                onEditorClose: vi.fn(),
            });
            const request = {
                callId: 'plan-1',
                name: 'write_file',
                args: {},
                isClientInitiated: false,
                prompt_id: 'prompt-plan',
            };
            await scheduler.schedule([request], new AbortController().signal);
            const errorCall = (await waitForStatus(onToolCallsUpdate, 'error'));
            expect(executeSpy).not.toHaveBeenCalled();
            expect(errorCall.response.responseParts[0]?.functionResponse?.response?.['output']).toBe(getPlanModeSystemReminder());
            expect(errorCall.response.resultDisplay).toContain('Plan mode');
        });
        it('allows read tools to execute in plan mode', async () => {
            const executeSpy = vi.fn().mockResolvedValue({
                llmContent: 'read ok',
                returnDisplay: 'read ok',
            });
            const tool = new SimpleTool('read_file', Kind.Read, executeSpy);
            const mockToolRegistry = {
                getTool: vi.fn().mockReturnValue(tool),
                getAllToolNames: vi.fn().mockReturnValue([tool.name]),
            };
            const onAllToolCallsComplete = vi.fn();
            const onToolCallsUpdate = vi.fn();
            const mockConfig = {
                getSessionId: () => 'plan-session',
                getUsageStatisticsEnabled: () => true,
                getDebugMode: () => false,
                getApprovalMode: vi.fn().mockReturnValue(ApprovalMode.PLAN),
                getAllowedTools: () => [],
                getContentGeneratorConfig: () => ({
                    model: 'test-model',
                    authType: 'oauth-personal',
                }),
                getToolRegistry: () => mockToolRegistry,
            };
            const scheduler = new CoreToolScheduler({
                config: mockConfig,
                onAllToolCallsComplete,
                onToolCallsUpdate,
                getPreferredEditor: () => 'vscode',
                onEditorClose: vi.fn(),
            });
            const request = {
                callId: 'plan-2',
                name: tool.name,
                args: {},
                isClientInitiated: false,
                prompt_id: 'prompt-plan',
            };
            await scheduler.schedule([request], new AbortController().signal);
            const successCall = (await waitForStatus(onToolCallsUpdate, 'success'));
            expect(executeSpy).toHaveBeenCalledTimes(1);
            expect(successCall.response.responseParts[0]?.functionResponse?.response?.['output']).toBe('read ok');
        });
        it('enforces shell command restrictions in plan mode', async () => {
            const executeSpyAllowed = vi.fn().mockResolvedValue({
                llmContent: 'shell ok',
                returnDisplay: 'shell ok',
            });
            const allowedTool = new SimpleTool('run_shell_command', Kind.Execute, executeSpyAllowed);
            const allowedToolRegistry = {
                getTool: vi.fn().mockReturnValue(allowedTool),
                getAllToolNames: vi.fn().mockReturnValue([allowedTool.name]),
            };
            const allowedConfig = {
                getSessionId: () => 'plan-session',
                getUsageStatisticsEnabled: () => true,
                getDebugMode: () => false,
                getApprovalMode: vi.fn().mockReturnValue(ApprovalMode.PLAN),
                getAllowedTools: () => [],
                getContentGeneratorConfig: () => ({
                    model: 'test-model',
                    authType: 'oauth-personal',
                }),
                getToolRegistry: () => allowedToolRegistry,
            };
            const allowedUpdates = vi.fn();
            const allowedScheduler = new CoreToolScheduler({
                config: allowedConfig,
                onAllToolCallsComplete: vi.fn(),
                onToolCallsUpdate: allowedUpdates,
                getPreferredEditor: () => 'vscode',
                onEditorClose: vi.fn(),
            });
            const allowedRequest = {
                callId: 'plan-shell-allowed',
                name: allowedTool.name,
                args: { command: 'ls -la' },
                isClientInitiated: false,
                prompt_id: 'prompt-plan',
            };
            await allowedScheduler.schedule([allowedRequest], new AbortController().signal);
            await waitForStatus(allowedUpdates, 'success');
            expect(executeSpyAllowed).toHaveBeenCalledTimes(1);
            const executeSpyBlocked = vi.fn().mockResolvedValue({
                llmContent: 'blocked',
                returnDisplay: 'blocked',
            });
            // Use MockTool with shouldConfirm=true to simulate a shell tool that requires confirmation
            const blockedTool = new MockTool('run_shell_command');
            blockedTool.shouldConfirm = true;
            blockedTool.executeFn = executeSpyBlocked;
            const blockedToolRegistry = {
                getTool: vi.fn().mockReturnValue(blockedTool),
                getAllToolNames: vi.fn().mockReturnValue([blockedTool.name]),
            };
            const blockedConfig = {
                getSessionId: () => 'plan-session',
                getUsageStatisticsEnabled: () => true,
                getDebugMode: () => false,
                getApprovalMode: vi.fn().mockReturnValue(ApprovalMode.PLAN),
                getAllowedTools: () => [],
                getContentGeneratorConfig: () => ({
                    model: 'test-model',
                    authType: 'oauth-personal',
                }),
                getToolRegistry: () => blockedToolRegistry,
            };
            const blockedUpdates = vi.fn();
            const blockedScheduler = new CoreToolScheduler({
                config: blockedConfig,
                onAllToolCallsComplete: vi.fn(),
                onToolCallsUpdate: blockedUpdates,
                getPreferredEditor: () => 'vscode',
                onEditorClose: vi.fn(),
            });
            const blockedRequest = {
                callId: 'plan-shell-blocked',
                name: 'run_shell_command',
                args: { command: 'rm -rf tmp' },
                isClientInitiated: false,
                prompt_id: 'prompt-plan',
            };
            await blockedScheduler.schedule([blockedRequest], new AbortController().signal);
            const blockedCall = (await waitForStatus(blockedUpdates, 'error'));
            expect(executeSpyBlocked).not.toHaveBeenCalled();
            expect(blockedCall.response.responseParts[0]?.functionResponse?.response?.['output']).toBe(getPlanModeSystemReminder());
            const observedStatuses = blockedUpdates.mock.calls
                .flatMap((call) => call[0])
                .map((tc) => tc.status);
            expect(observedStatuses).not.toContain('awaiting_approval');
        });
    });
    describe('getToolSuggestion', () => {
        it('should suggest the top N closest tool names for a typo', () => {
            // Create mocked tool registry
            const mockConfig = {
                getToolRegistry: () => mockToolRegistry,
            };
            const mockToolRegistry = {
                getAllToolNames: () => ['list_files', 'read_file', 'write_file'],
            };
            // Create scheduler
            const scheduler = new CoreToolScheduler({
                config: mockConfig,
                getPreferredEditor: () => 'vscode',
                onEditorClose: vi.fn(),
            });
            // Test that the right tool is selected, with only 1 result, for typos
            // @ts-expect-error accessing private method
            const misspelledTool = scheduler.getToolSuggestion('list_fils', 1);
            expect(misspelledTool).toBe(' Did you mean "list_files"?');
            // Test that the right tool is selected, with only 1 result, for prefixes
            // @ts-expect-error accessing private method
            const prefixedTool = scheduler.getToolSuggestion('github.list_files', 1);
            expect(prefixedTool).toBe(' Did you mean "list_files"?');
            // Test that the right tool is first
            // @ts-expect-error accessing private method
            const suggestionMultiple = scheduler.getToolSuggestion('list_fils');
            expect(suggestionMultiple).toBe(' Did you mean one of: "list_files", "read_file", "write_file"?');
        });
    });
});
describe('CoreToolScheduler with payload', () => {
    it('should update args and diff and execute tool when payload is provided', async () => {
        const mockTool = new MockModifiableTool();
        const declarativeTool = mockTool;
        const mockToolRegistry = {
            getTool: () => declarativeTool,
            getFunctionDeclarations: () => [],
            tools: new Map(),
            discovery: {},
            registerTool: () => { },
            getToolByName: () => declarativeTool,
            getToolByDisplayName: () => declarativeTool,
            getTools: () => [],
            discoverTools: async () => { },
            getAllTools: () => [],
            getToolsByServer: () => [],
        };
        const onAllToolCallsComplete = vi.fn();
        const onToolCallsUpdate = vi.fn();
        const mockConfig = {
            getSessionId: () => 'test-session-id',
            getUsageStatisticsEnabled: () => true,
            getDebugMode: () => false,
            getApprovalMode: () => ApprovalMode.DEFAULT,
            getAllowedTools: () => [],
            getContentGeneratorConfig: () => ({
                model: 'test-model',
                authType: 'oauth-personal',
            }),
            getToolRegistry: () => mockToolRegistry,
        };
        const scheduler = new CoreToolScheduler({
            config: mockConfig,
            onAllToolCallsComplete,
            onToolCallsUpdate,
            getPreferredEditor: () => 'vscode',
            onEditorClose: vi.fn(),
        });
        const abortController = new AbortController();
        const request = {
            callId: '1',
            name: 'mockModifiableTool',
            args: {},
            isClientInitiated: false,
            prompt_id: 'prompt-id-2',
        };
        await scheduler.schedule([request], abortController.signal);
        const awaitingCall = (await waitForStatus(onToolCallsUpdate, 'awaiting_approval'));
        const confirmationDetails = awaitingCall.confirmationDetails;
        if (confirmationDetails) {
            const payload = { newContent: 'final version' };
            await confirmationDetails.onConfirm(ToolConfirmationOutcome.ProceedOnce, payload);
        }
        expect(onAllToolCallsComplete).toHaveBeenCalled();
        const completedCalls = onAllToolCallsComplete.mock
            .calls[0][0];
        expect(completedCalls[0].status).toBe('success');
        expect(mockTool.executeFn).toHaveBeenCalledWith({
            newContent: 'final version',
        });
    });
});
describe('convertToFunctionResponse', () => {
    const toolName = 'testTool';
    const callId = 'call1';
    it('should handle simple string llmContent', () => {
        const llmContent = 'Simple text output';
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: { output: 'Simple text output' },
                },
            },
        ]);
    });
    it('should handle llmContent as a single Part with text', () => {
        const llmContent = { text: 'Text from Part object' };
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: { output: 'Text from Part object' },
                },
            },
        ]);
    });
    it('should handle llmContent as a PartListUnion array with a single text Part', () => {
        const llmContent = [{ text: 'Text from array' }];
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: { output: 'Text from array' },
                },
            },
        ]);
    });
    it('should handle llmContent with inlineData', () => {
        const llmContent = {
            inlineData: { mimeType: 'image/png', data: 'base64...' },
        };
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: {
                        output: 'Binary content of type image/png was processed.',
                    },
                },
            },
            llmContent,
        ]);
    });
    it('should handle llmContent with fileData', () => {
        const llmContent = {
            fileData: { mimeType: 'application/pdf', fileUri: 'gs://...' },
        };
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: {
                        output: 'Binary content of type application/pdf was processed.',
                    },
                },
            },
            llmContent,
        ]);
    });
    it('should handle llmContent as an array of multiple Parts (text and inlineData)', () => {
        const llmContent = [
            { text: 'Some textual description' },
            { inlineData: { mimeType: 'image/jpeg', data: 'base64data...' } },
            { text: 'Another text part' },
        ];
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: { output: 'Tool execution succeeded.' },
                },
            },
            ...llmContent,
        ]);
    });
    it('should handle llmContent as an array with a single inlineData Part', () => {
        const llmContent = [
            { inlineData: { mimeType: 'image/gif', data: 'gifdata...' } },
        ];
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: {
                        output: 'Binary content of type image/gif was processed.',
                    },
                },
            },
            ...llmContent,
        ]);
    });
    it('should handle llmContent as a generic Part (not text, inlineData, or fileData)', () => {
        const llmContent = { functionCall: { name: 'test', args: {} } };
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: { output: 'Tool execution succeeded.' },
                },
            },
        ]);
    });
    it('should handle empty string llmContent', () => {
        const llmContent = '';
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: { output: '' },
                },
            },
        ]);
    });
    it('should handle llmContent as an empty array', () => {
        const llmContent = [];
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: { output: 'Tool execution succeeded.' },
                },
            },
        ]);
    });
    it('should handle llmContent as a Part with undefined inlineData/fileData/text', () => {
        const llmContent = {}; // An empty part object
        const result = convertToFunctionResponse(toolName, callId, llmContent);
        expect(result).toEqual([
            {
                functionResponse: {
                    name: toolName,
                    id: callId,
                    response: { output: 'Tool execution succeeded.' },
                },
            },
        ]);
    });
});
class MockEditToolInvocation extends BaseToolInvocation {
    constructor(params) {
        super(params);
    }
    getDescription() {
        return 'A mock edit tool invocation';
    }
    async shouldConfirmExecute(_abortSignal) {
        return {
            type: 'edit',
            title: 'Confirm Edit',
            fileName: 'test.txt',
            filePath: 'test.txt',
            fileDiff: '--- test.txt\n+++ test.txt\n@@ -1,1 +1,1 @@\n-old content\n+new content',
            originalContent: 'old content',
            newContent: 'new content',
            onConfirm: async () => { },
        };
    }
    async execute(_abortSignal) {
        return {
            llmContent: 'Edited successfully',
            returnDisplay: 'Edited successfully',
        };
    }
}
class MockEditTool extends BaseDeclarativeTool {
    constructor() {
        super('mockEditTool', 'mockEditTool', 'A mock edit tool', Kind.Edit, {});
    }
    createInvocation(params) {
        return new MockEditToolInvocation(params);
    }
}
describe('CoreToolScheduler edit cancellation', () => {
    it('should preserve diff when an edit is cancelled', async () => {
        const mockEditTool = new MockEditTool();
        const mockToolRegistry = {
            getTool: () => mockEditTool,
            getFunctionDeclarations: () => [],
            tools: new Map(),
            discovery: {},
            registerTool: () => { },
            getToolByName: () => mockEditTool,
            getToolByDisplayName: () => mockEditTool,
            getTools: () => [],
            discoverTools: async () => { },
            getAllTools: () => [],
            getToolsByServer: () => [],
        };
        const onAllToolCallsComplete = vi.fn();
        const onToolCallsUpdate = vi.fn();
        const mockConfig = {
            getSessionId: () => 'test-session-id',
            getUsageStatisticsEnabled: () => true,
            getDebugMode: () => false,
            getApprovalMode: () => ApprovalMode.DEFAULT,
            getAllowedTools: () => [],
            getContentGeneratorConfig: () => ({
                model: 'test-model',
                authType: 'oauth-personal',
            }),
            getToolRegistry: () => mockToolRegistry,
        };
        const scheduler = new CoreToolScheduler({
            config: mockConfig,
            onAllToolCallsComplete,
            onToolCallsUpdate,
            getPreferredEditor: () => 'vscode',
            onEditorClose: vi.fn(),
        });
        const abortController = new AbortController();
        const request = {
            callId: '1',
            name: 'mockEditTool',
            args: {},
            isClientInitiated: false,
            prompt_id: 'prompt-id-1',
        };
        await scheduler.schedule([request], abortController.signal);
        const awaitingCall = (await waitForStatus(onToolCallsUpdate, 'awaiting_approval'));
        // Cancel the edit
        const confirmationDetails = awaitingCall.confirmationDetails;
        if (confirmationDetails) {
            await confirmationDetails.onConfirm(ToolConfirmationOutcome.Cancel);
        }
        expect(onAllToolCallsComplete).toHaveBeenCalled();
        const completedCalls = onAllToolCallsComplete.mock
            .calls[0][0];
        expect(completedCalls[0].status).toBe('cancelled');
        // Check that the diff is preserved
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        const cancelledCall = completedCalls[0];
        expect(cancelledCall.response.resultDisplay).toBeDefined();
        expect(cancelledCall.response.resultDisplay.fileDiff).toBe('--- test.txt\n+++ test.txt\n@@ -1,1 +1,1 @@\n-old content\n+new content');
        expect(cancelledCall.response.resultDisplay.fileName).toBe('test.txt');
    });
});
describe('CoreToolScheduler YOLO mode', () => {
    it('should execute tool requiring confirmation directly without waiting', async () => {
        // Arrange
        const mockTool = new MockTool();
        mockTool.executeFn.mockReturnValue({
            llmContent: 'Tool executed',
            returnDisplay: 'Tool executed',
        });
        // This tool would normally require confirmation.
        mockTool.shouldConfirm = true;
        const declarativeTool = mockTool;
        const mockToolRegistry = {
            getTool: () => declarativeTool,
            getToolByName: () => declarativeTool,
            // Other properties are not needed for this test but are included for type consistency.
            getFunctionDeclarations: () => [],
            tools: new Map(),
            discovery: {},
            registerTool: () => { },
            getToolByDisplayName: () => declarativeTool,
            getTools: () => [],
            discoverTools: async () => { },
            getAllTools: () => [],
            getToolsByServer: () => [],
        };
        const onAllToolCallsComplete = vi.fn();
        const onToolCallsUpdate = vi.fn();
        // Configure the scheduler for YOLO mode.
        const mockConfig = {
            getSessionId: () => 'test-session-id',
            getUsageStatisticsEnabled: () => true,
            getDebugMode: () => false,
            getApprovalMode: () => ApprovalMode.YOLO,
            getAllowedTools: () => [],
            getContentGeneratorConfig: () => ({
                model: 'test-model',
                authType: 'oauth-personal',
            }),
            getToolRegistry: () => mockToolRegistry,
        };
        const scheduler = new CoreToolScheduler({
            config: mockConfig,
            onAllToolCallsComplete,
            onToolCallsUpdate,
            getPreferredEditor: () => 'vscode',
            onEditorClose: vi.fn(),
        });
        const abortController = new AbortController();
        const request = {
            callId: '1',
            name: 'mockTool',
            args: { param: 'value' },
            isClientInitiated: false,
            prompt_id: 'prompt-id-yolo',
        };
        // Act
        await scheduler.schedule([request], abortController.signal);
        // Assert
        // 1. The tool's execute method was called directly.
        expect(mockTool.executeFn).toHaveBeenCalledWith({ param: 'value' });
        // 2. The tool call status never entered 'awaiting_approval'.
        const statusUpdates = onToolCallsUpdate.mock.calls
            .map((call) => call[0][0]?.status)
            .filter(Boolean);
        expect(statusUpdates).not.toContain('awaiting_approval');
        expect(statusUpdates).toEqual([
            'validating',
            'scheduled',
            'executing',
            'success',
        ]);
        // 3. The final callback indicates the tool call was successful.
        expect(onAllToolCallsComplete).toHaveBeenCalled();
        const completedCalls = onAllToolCallsComplete.mock
            .calls[0][0];
        expect(completedCalls).toHaveLength(1);
        const completedCall = completedCalls[0];
        expect(completedCall.status).toBe('success');
        if (completedCall.status === 'success') {
            expect(completedCall.response.resultDisplay).toBe('Tool executed');
        }
    });
});
describe('CoreToolScheduler cancellation during executing with live output', () => {
    it('sets status to cancelled and preserves last output', async () => {
        class StreamingInvocation extends BaseToolInvocation {
            getDescription() {
                return `Streaming tool ${this.params.id}`;
            }
            async execute(signal, updateOutput) {
                updateOutput?.('hello');
                // Wait until aborted to emulate a long-running task
                await new Promise((resolve) => {
                    if (signal.aborted)
                        return resolve();
                    const onAbort = () => {
                        signal.removeEventListener('abort', onAbort);
                        resolve();
                    };
                    signal.addEventListener('abort', onAbort, { once: true });
                });
                // Return a normal (non-error) result; scheduler should still mark cancelled
                return { llmContent: 'done', returnDisplay: 'done' };
            }
        }
        class StreamingTool extends BaseDeclarativeTool {
            constructor() {
                super('stream-tool', 'Stream Tool', 'Emits live output and waits for abort', Kind.Other, {
                    type: 'object',
                    properties: { id: { type: 'string' } },
                    required: ['id'],
                }, true, true);
            }
            createInvocation(params) {
                return new StreamingInvocation(params);
            }
        }
        const tool = new StreamingTool();
        const mockToolRegistry = {
            getTool: () => tool,
            getFunctionDeclarations: () => [],
            tools: new Map(),
            discovery: {},
            registerTool: () => { },
            getToolByName: () => tool,
            getToolByDisplayName: () => tool,
            getTools: () => [],
            discoverTools: async () => { },
            getAllTools: () => [],
            getToolsByServer: () => [],
        };
        const onAllToolCallsComplete = vi.fn();
        const onToolCallsUpdate = vi.fn();
        const mockConfig = {
            getSessionId: () => 'test-session-id',
            getUsageStatisticsEnabled: () => true,
            getDebugMode: () => false,
            getApprovalMode: () => ApprovalMode.DEFAULT,
            getContentGeneratorConfig: () => ({
                model: 'test-model',
                authType: 'oauth-personal',
            }),
            getToolRegistry: () => mockToolRegistry,
        };
        const scheduler = new CoreToolScheduler({
            config: mockConfig,
            onAllToolCallsComplete,
            onToolCallsUpdate,
            getPreferredEditor: () => 'vscode',
            onEditorClose: vi.fn(),
        });
        const abortController = new AbortController();
        const request = {
            callId: '1',
            name: 'stream-tool',
            args: { id: 'x' },
            isClientInitiated: true,
            prompt_id: 'prompt-stream',
        };
        const schedulePromise = scheduler.schedule([request], abortController.signal);
        // Wait until executing
        await vi.waitFor(() => {
            const calls = onToolCallsUpdate.mock.calls;
            const last = calls[calls.length - 1]?.[0][0];
            expect(last?.status).toBe('executing');
        });
        // Now abort
        abortController.abort();
        await schedulePromise;
        await vi.waitFor(() => {
            expect(onAllToolCallsComplete).toHaveBeenCalled();
        });
        const completedCalls = onAllToolCallsComplete.mock
            .calls[0][0];
        expect(completedCalls[0].status).toBe('cancelled');
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        const cancelled = completedCalls[0];
        expect(cancelled.response.resultDisplay).toBe('hello');
    });
});
describe('CoreToolScheduler request queueing', () => {
    it('should queue a request if another is running', async () => {
        let resolveFirstCall;
        const firstCallPromise = new Promise((resolve) => {
            resolveFirstCall = resolve;
        });
        const mockTool = new MockTool();
        mockTool.executeFn.mockImplementation(() => firstCallPromise);
        const declarativeTool = mockTool;
        const mockToolRegistry = {
            getTool: () => declarativeTool,
            getToolByName: () => declarativeTool,
            getFunctionDeclarations: () => [],
            tools: new Map(),
            discovery: {},
            registerTool: () => { },
            getToolByDisplayName: () => declarativeTool,
            getTools: () => [],
            discoverTools: async () => { },
            getAllTools: () => [],
            getToolsByServer: () => [],
        };
        const onAllToolCallsComplete = vi.fn();
        const onToolCallsUpdate = vi.fn();
        const mockConfig = {
            getSessionId: () => 'test-session-id',
            getUsageStatisticsEnabled: () => true,
            getDebugMode: () => false,
            getApprovalMode: () => ApprovalMode.YOLO, // Use YOLO to avoid confirmation prompts
            getAllowedTools: () => [],
            getContentGeneratorConfig: () => ({
                model: 'test-model',
                authType: 'oauth-personal',
            }),
            getToolRegistry: () => mockToolRegistry,
        };
        const scheduler = new CoreToolScheduler({
            config: mockConfig,
            onAllToolCallsComplete,
            onToolCallsUpdate,
            getPreferredEditor: () => 'vscode',
            onEditorClose: vi.fn(),
        });
        const abortController = new AbortController();
        const request1 = {
            callId: '1',
            name: 'mockTool',
            args: { a: 1 },
            isClientInitiated: false,
            prompt_id: 'prompt-1',
        };
        const request2 = {
            callId: '2',
            name: 'mockTool',
            args: { b: 2 },
            isClientInitiated: false,
            prompt_id: 'prompt-2',
        };
        // Schedule the first call, which will pause execution.
        scheduler.schedule([request1], abortController.signal);
        // Wait for the first call to be in the 'executing' state.
        await waitForStatus(onToolCallsUpdate, 'executing');
        // Schedule the second call while the first is "running".
        const schedulePromise2 = scheduler.schedule([request2], abortController.signal);
        // Ensure the second tool call hasn't been executed yet.
        expect(mockTool.executeFn).toHaveBeenCalledTimes(1);
        expect(mockTool.executeFn).toHaveBeenCalledWith({ a: 1 });
        // Complete the first tool call.
        resolveFirstCall({
            llmContent: 'First call complete',
            returnDisplay: 'First call complete',
        });
        // Wait for the second schedule promise to resolve.
        await schedulePromise2;
        // Let the second call finish.
        const secondCallResult = {
            llmContent: 'Second call complete',
            returnDisplay: 'Second call complete',
        };
        // Since the mock is shared, we need to resolve the current promise.
        // In a real scenario, a new promise would be created for the second call.
        resolveFirstCall(secondCallResult);
        await vi.waitFor(() => {
            // Now the second tool call should have been executed.
            expect(mockTool.executeFn).toHaveBeenCalledTimes(2);
        });
        expect(mockTool.executeFn).toHaveBeenCalledWith({ b: 2 });
        // Wait for the second completion.
        await vi.waitFor(() => {
            expect(onAllToolCallsComplete).toHaveBeenCalledTimes(2);
        });
        // Verify the completion callbacks were called correctly.
        expect(onAllToolCallsComplete.mock.calls[0][0][0].status).toBe('success');
        expect(onAllToolCallsComplete.mock.calls[1][0][0].status).toBe('success');
    });
    it('should auto-approve a tool call if it is on the allowedTools list', async () => {
        // Arrange
        const mockTool = new MockTool('mockTool');
        mockTool.executeFn.mockReturnValue({
            llmContent: 'Tool executed',
            returnDisplay: 'Tool executed',
        });
        // This tool would normally require confirmation.
        mockTool.shouldConfirm = true;
        const declarativeTool = mockTool;
        const toolRegistry = {
            getTool: () => declarativeTool,
            getToolByName: () => declarativeTool,
            getFunctionDeclarations: () => [],
            tools: new Map(),
            discovery: {},
            registerTool: () => { },
            getToolByDisplayName: () => declarativeTool,
            getTools: () => [],
            discoverTools: async () => { },
            getAllTools: () => [],
            getToolsByServer: () => [],
        };
        const onAllToolCallsComplete = vi.fn();
        const onToolCallsUpdate = vi.fn();
        // Configure the scheduler to auto-approve the specific tool call.
        const mockConfig = {
            getSessionId: () => 'test-session-id',
            getUsageStatisticsEnabled: () => true,
            getDebugMode: () => false,
            getApprovalMode: () => ApprovalMode.DEFAULT, // Not YOLO mode
            getAllowedTools: () => ['mockTool'], // Auto-approve this tool
            getToolRegistry: () => toolRegistry,
            getContentGeneratorConfig: () => ({
                model: 'test-model',
                authType: 'oauth-personal',
            }),
        };
        const scheduler = new CoreToolScheduler({
            config: mockConfig,
            onAllToolCallsComplete,
            onToolCallsUpdate,
            getPreferredEditor: () => 'vscode',
            onEditorClose: vi.fn(),
        });
        const abortController = new AbortController();
        const request = {
            callId: '1',
            name: 'mockTool',
            args: { param: 'value' },
            isClientInitiated: false,
            prompt_id: 'prompt-auto-approved',
        };
        // Act
        await scheduler.schedule([request], abortController.signal);
        // Assert
        // 1. The tool's execute method was called directly.
        expect(mockTool.executeFn).toHaveBeenCalledWith({ param: 'value' });
        // 2. The tool call status never entered 'awaiting_approval'.
        const statusUpdates = onToolCallsUpdate.mock.calls
            .map((call) => call[0][0]?.status)
            .filter(Boolean);
        expect(statusUpdates).not.toContain('awaiting_approval');
        expect(statusUpdates).toEqual([
            'validating',
            'scheduled',
            'executing',
            'success',
        ]);
        // 3. The final callback indicates the tool call was successful.
        expect(onAllToolCallsComplete).toHaveBeenCalled();
        const completedCalls = onAllToolCallsComplete.mock
            .calls[0][0];
        expect(completedCalls).toHaveLength(1);
        const completedCall = completedCalls[0];
        expect(completedCall.status).toBe('success');
        if (completedCall.status === 'success') {
            expect(completedCall.response.resultDisplay).toBe('Tool executed');
        }
    });
    it('should handle two synchronous calls to schedule', async () => {
        const mockTool = new MockTool();
        const declarativeTool = mockTool;
        const mockToolRegistry = {
            getTool: () => declarativeTool,
            getToolByName: () => declarativeTool,
            getFunctionDeclarations: () => [],
            tools: new Map(),
            discovery: {},
            registerTool: () => { },
            getToolByDisplayName: () => declarativeTool,
            getTools: () => [],
            discoverTools: async () => { },
            getAllTools: () => [],
            getToolsByServer: () => [],
        };
        const onAllToolCallsComplete = vi.fn();
        const onToolCallsUpdate = vi.fn();
        const mockConfig = {
            getSessionId: () => 'test-session-id',
            getUsageStatisticsEnabled: () => true,
            getDebugMode: () => false,
            getApprovalMode: () => ApprovalMode.YOLO,
            getAllowedTools: () => [],
            getContentGeneratorConfig: () => ({
                model: 'test-model',
                authType: 'oauth-personal',
            }),
            getToolRegistry: () => mockToolRegistry,
        };
        const scheduler = new CoreToolScheduler({
            config: mockConfig,
            onAllToolCallsComplete,
            onToolCallsUpdate,
            getPreferredEditor: () => 'vscode',
            onEditorClose: vi.fn(),
        });
        const abortController = new AbortController();
        const request1 = {
            callId: '1',
            name: 'mockTool',
            args: { a: 1 },
            isClientInitiated: false,
            prompt_id: 'prompt-1',
        };
        const request2 = {
            callId: '2',
            name: 'mockTool',
            args: { b: 2 },
            isClientInitiated: false,
            prompt_id: 'prompt-2',
        };
        // Schedule two calls synchronously.
        const schedulePromise1 = scheduler.schedule([request1], abortController.signal);
        const schedulePromise2 = scheduler.schedule([request2], abortController.signal);
        // Wait for both promises to resolve.
        await Promise.all([schedulePromise1, schedulePromise2]);
        // Ensure the tool was called twice with the correct arguments.
        expect(mockTool.executeFn).toHaveBeenCalledTimes(2);
        expect(mockTool.executeFn).toHaveBeenCalledWith({ a: 1 });
        expect(mockTool.executeFn).toHaveBeenCalledWith({ b: 2 });
        // Ensure completion callbacks were called twice.
        expect(onAllToolCallsComplete).toHaveBeenCalledTimes(2);
    });
    it('should auto-approve remaining tool calls when first tool call is approved with ProceedAlways', async () => {
        let approvalMode = ApprovalMode.DEFAULT;
        const mockConfig = {
            getSessionId: () => 'test-session-id',
            getUsageStatisticsEnabled: () => true,
            getDebugMode: () => false,
            getApprovalMode: () => approvalMode,
            getAllowedTools: () => [],
            setApprovalMode: (mode) => {
                approvalMode = mode;
            },
        };
        const testTool = new TestApprovalTool(mockConfig);
        const toolRegistry = {
            getTool: () => testTool,
            getFunctionDeclarations: () => [],
            getFunctionDeclarationsFiltered: () => [],
            registerTool: () => { },
            discoverAllTools: async () => { },
            discoverMcpTools: async () => { },
            discoverToolsForServer: async () => { },
            removeMcpToolsByServer: () => { },
            getAllTools: () => [],
            getToolsByServer: () => [],
            tools: new Map(),
            config: mockConfig,
            mcpClientManager: undefined,
            getToolByName: () => testTool,
            getToolByDisplayName: () => testTool,
            getTools: () => [],
            discoverTools: async () => { },
            discovery: {},
        };
        mockConfig.getToolRegistry = () => toolRegistry;
        const onAllToolCallsComplete = vi.fn();
        const onToolCallsUpdate = vi.fn();
        const pendingConfirmations = [];
        const scheduler = new CoreToolScheduler({
            config: mockConfig,
            onAllToolCallsComplete,
            onToolCallsUpdate: (toolCalls) => {
                onToolCallsUpdate(toolCalls);
                // Capture confirmation handlers for awaiting_approval tools
                toolCalls.forEach((call) => {
                    if (call.status === 'awaiting_approval') {
                        const waitingCall = call;
                        if (waitingCall.confirmationDetails?.onConfirm) {
                            const originalHandler = pendingConfirmations.find((h) => h === waitingCall.confirmationDetails.onConfirm);
                            if (!originalHandler) {
                                pendingConfirmations.push(waitingCall.confirmationDetails.onConfirm);
                            }
                        }
                    }
                });
            },
            getPreferredEditor: () => 'vscode',
            onEditorClose: vi.fn(),
        });
        const abortController = new AbortController();
        // Schedule multiple tools that need confirmation
        const requests = [
            {
                callId: '1',
                name: 'testApprovalTool',
                args: { id: 'first' },
                isClientInitiated: false,
                prompt_id: 'prompt-1',
            },
            {
                callId: '2',
                name: 'testApprovalTool',
                args: { id: 'second' },
                isClientInitiated: false,
                prompt_id: 'prompt-2',
            },
            {
                callId: '3',
                name: 'testApprovalTool',
                args: { id: 'third' },
                isClientInitiated: false,
                prompt_id: 'prompt-3',
            },
        ];
        await scheduler.schedule(requests, abortController.signal);
        // Wait for all tools to be awaiting approval
        await vi.waitFor(() => {
            const calls = onToolCallsUpdate.mock.calls.at(-1)?.[0];
            expect(calls?.length).toBe(3);
            expect(calls?.every((call) => call.status === 'awaiting_approval')).toBe(true);
        });
        expect(pendingConfirmations.length).toBe(3);
        // Approve the first tool with ProceedAlways
        const firstConfirmation = pendingConfirmations[0];
        firstConfirmation(ToolConfirmationOutcome.ProceedAlways);
        // Wait for all tools to be completed
        await vi.waitFor(() => {
            expect(onAllToolCallsComplete).toHaveBeenCalled();
            const completedCalls = onAllToolCallsComplete.mock.calls.at(-1)?.[0];
            expect(completedCalls?.length).toBe(3);
            expect(completedCalls?.every((call) => call.status === 'success')).toBe(true);
        });
        // Verify approval mode was changed
        expect(approvalMode).toBe(ApprovalMode.AUTO_EDIT);
    });
});
//# sourceMappingURL=coreToolScheduler.test.js.map