/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { useState, useCallback, useEffect } from 'react';
import { FolderTrustChoice } from '../components/FolderTrustDialog.js';
import { loadTrustedFolders, TrustLevel, isWorkspaceTrusted, } from '../../config/trustedFolders.js';
import * as process from 'node:process';
export const useFolderTrust = (settings, onTrustChange) => {
    const [isTrusted, setIsTrusted] = useState(undefined);
    const [isFolderTrustDialogOpen, setIsFolderTrustDialogOpen] = useState(false);
    const [isRestarting, setIsRestarting] = useState(false);
    const folderTrust = settings.merged.security?.folderTrust?.enabled;
    const folderTrustFeature = settings.merged.security?.folderTrust?.featureEnabled;
    useEffect(() => {
        const trusted = isWorkspaceTrusted({
            folderTrust,
            folderTrustFeature,
        });
        setIsTrusted(trusted);
        setIsFolderTrustDialogOpen(trusted === undefined);
        onTrustChange(trusted);
    }, [onTrustChange, folderTrust, folderTrustFeature]);
    const handleFolderTrustSelect = useCallback((choice) => {
        const trustedFolders = loadTrustedFolders();
        const cwd = process.cwd();
        let trustLevel;
        const wasTrusted = isTrusted ?? true;
        switch (choice) {
            case FolderTrustChoice.TRUST_FOLDER:
                trustLevel = TrustLevel.TRUST_FOLDER;
                break;
            case FolderTrustChoice.TRUST_PARENT:
                trustLevel = TrustLevel.TRUST_PARENT;
                break;
            case FolderTrustChoice.DO_NOT_TRUST:
                trustLevel = TrustLevel.DO_NOT_TRUST;
                break;
            default:
                return;
        }
        trustedFolders.setValue(cwd, trustLevel);
        const newIsTrusted = trustLevel === TrustLevel.TRUST_FOLDER ||
            trustLevel === TrustLevel.TRUST_PARENT;
        setIsTrusted(newIsTrusted);
        onTrustChange(newIsTrusted);
        const needsRestart = wasTrusted !== newIsTrusted;
        if (needsRestart) {
            setIsRestarting(true);
            setIsFolderTrustDialogOpen(true);
        }
        else {
            setIsFolderTrustDialogOpen(false);
        }
    }, [onTrustChange, isTrusted]);
    return {
        isTrusted,
        isFolderTrustDialogOpen,
        handleFolderTrustSelect,
        isRestarting,
    };
};
//# sourceMappingURL=useFolderTrust.js.map