﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/waf/WAFRequest.h>
#include <aws/waf/WAF_EXPORTS.h>
#include <aws/waf/model/GeoMatchSetUpdate.h>

#include <utility>

namespace Aws {
namespace WAF {
namespace Model {

/**
 */
class UpdateGeoMatchSetRequest : public WAFRequest {
 public:
  AWS_WAF_API UpdateGeoMatchSetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateGeoMatchSet"; }

  AWS_WAF_API Aws::String SerializePayload() const override;

  AWS_WAF_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The <code>GeoMatchSetId</code> of the <a>GeoMatchSet</a> that you want to
   * update. <code>GeoMatchSetId</code> is returned by <a>CreateGeoMatchSet</a> and
   * by <a>ListGeoMatchSets</a>.</p>
   */
  inline const Aws::String& GetGeoMatchSetId() const { return m_geoMatchSetId; }
  inline bool GeoMatchSetIdHasBeenSet() const { return m_geoMatchSetIdHasBeenSet; }
  template <typename GeoMatchSetIdT = Aws::String>
  void SetGeoMatchSetId(GeoMatchSetIdT&& value) {
    m_geoMatchSetIdHasBeenSet = true;
    m_geoMatchSetId = std::forward<GeoMatchSetIdT>(value);
  }
  template <typename GeoMatchSetIdT = Aws::String>
  UpdateGeoMatchSetRequest& WithGeoMatchSetId(GeoMatchSetIdT&& value) {
    SetGeoMatchSetId(std::forward<GeoMatchSetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The value returned by the most recent call to <a>GetChangeToken</a>.</p>
   */
  inline const Aws::String& GetChangeToken() const { return m_changeToken; }
  inline bool ChangeTokenHasBeenSet() const { return m_changeTokenHasBeenSet; }
  template <typename ChangeTokenT = Aws::String>
  void SetChangeToken(ChangeTokenT&& value) {
    m_changeTokenHasBeenSet = true;
    m_changeToken = std::forward<ChangeTokenT>(value);
  }
  template <typename ChangeTokenT = Aws::String>
  UpdateGeoMatchSetRequest& WithChangeToken(ChangeTokenT&& value) {
    SetChangeToken(std::forward<ChangeTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of <code>GeoMatchSetUpdate</code> objects that you want to insert
   * into or delete from an <a>GeoMatchSet</a>. For more information, see the
   * applicable data types:</p> <ul> <li> <p> <a>GeoMatchSetUpdate</a>: Contains
   * <code>Action</code> and <code>GeoMatchConstraint</code> </p> </li> <li> <p>
   * <a>GeoMatchConstraint</a>: Contains <code>Type</code> and <code>Value</code>
   * </p> <p>You can have only one <code>Type</code> and <code>Value</code> per
   * <code>GeoMatchConstraint</code>. To add multiple countries, include multiple
   * <code>GeoMatchSetUpdate</code> objects in your request.</p> </li> </ul>
   */
  inline const Aws::Vector<GeoMatchSetUpdate>& GetUpdates() const { return m_updates; }
  inline bool UpdatesHasBeenSet() const { return m_updatesHasBeenSet; }
  template <typename UpdatesT = Aws::Vector<GeoMatchSetUpdate>>
  void SetUpdates(UpdatesT&& value) {
    m_updatesHasBeenSet = true;
    m_updates = std::forward<UpdatesT>(value);
  }
  template <typename UpdatesT = Aws::Vector<GeoMatchSetUpdate>>
  UpdateGeoMatchSetRequest& WithUpdates(UpdatesT&& value) {
    SetUpdates(std::forward<UpdatesT>(value));
    return *this;
  }
  template <typename UpdatesT = GeoMatchSetUpdate>
  UpdateGeoMatchSetRequest& AddUpdates(UpdatesT&& value) {
    m_updatesHasBeenSet = true;
    m_updates.emplace_back(std::forward<UpdatesT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_geoMatchSetId;

  Aws::String m_changeToken;

  Aws::Vector<GeoMatchSetUpdate> m_updates;
  bool m_geoMatchSetIdHasBeenSet = false;
  bool m_changeTokenHasBeenSet = false;
  bool m_updatesHasBeenSet = false;
};

}  // namespace Model
}  // namespace WAF
}  // namespace Aws
