﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/resiliencehub/ResilienceHubRequest.h>
#include <aws/resiliencehub/ResilienceHub_EXPORTS.h>

#include <utility>

namespace Aws {
namespace ResilienceHub {
namespace Model {

/**
 */
class StartAppAssessmentRequest : public ResilienceHubRequest {
 public:
  AWS_RESILIENCEHUB_API StartAppAssessmentRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartAppAssessment"; }

  AWS_RESILIENCEHUB_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Amazon Resource Name (ARN) of the Resilience Hub application. The format for
   * this ARN is:
   * arn:<code>partition</code>:resiliencehub:<code>region</code>:<code>account</code>:app/<code>app-id</code>.
   * For more information about ARNs, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">
   * Amazon Resource Names (ARNs)</a> in the <i>Amazon Web Services General
   * Reference</i> guide.</p>
   */
  inline const Aws::String& GetAppArn() const { return m_appArn; }
  inline bool AppArnHasBeenSet() const { return m_appArnHasBeenSet; }
  template <typename AppArnT = Aws::String>
  void SetAppArn(AppArnT&& value) {
    m_appArnHasBeenSet = true;
    m_appArn = std::forward<AppArnT>(value);
  }
  template <typename AppArnT = Aws::String>
  StartAppAssessmentRequest& WithAppArn(AppArnT&& value) {
    SetAppArn(std::forward<AppArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version of the application.</p>
   */
  inline const Aws::String& GetAppVersion() const { return m_appVersion; }
  inline bool AppVersionHasBeenSet() const { return m_appVersionHasBeenSet; }
  template <typename AppVersionT = Aws::String>
  void SetAppVersion(AppVersionT&& value) {
    m_appVersionHasBeenSet = true;
    m_appVersion = std::forward<AppVersionT>(value);
  }
  template <typename AppVersionT = Aws::String>
  StartAppAssessmentRequest& WithAppVersion(AppVersionT&& value) {
    SetAppVersion(std::forward<AppVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name for the assessment.</p>
   */
  inline const Aws::String& GetAssessmentName() const { return m_assessmentName; }
  inline bool AssessmentNameHasBeenSet() const { return m_assessmentNameHasBeenSet; }
  template <typename AssessmentNameT = Aws::String>
  void SetAssessmentName(AssessmentNameT&& value) {
    m_assessmentNameHasBeenSet = true;
    m_assessmentName = std::forward<AssessmentNameT>(value);
  }
  template <typename AssessmentNameT = Aws::String>
  StartAppAssessmentRequest& WithAssessmentName(AssessmentNameT&& value) {
    SetAssessmentName(std::forward<AssessmentNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Used for an idempotency token. A client token is a unique, case-sensitive
   * string of up to 64 ASCII characters. You should not reuse the same client token
   * for other API requests.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartAppAssessmentRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags assigned to the resource. A tag is a label that you assign to an Amazon
   * Web Services resource. Each tag consists of a key/value pair.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  StartAppAssessmentRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  StartAppAssessmentRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_appArn;

  Aws::String m_appVersion;

  Aws::String m_assessmentName;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_appArnHasBeenSet = false;
  bool m_appVersionHasBeenSet = false;
  bool m_assessmentNameHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResilienceHub
}  // namespace Aws
