﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/wisdom/ConnectWisdomServiceRequest.h>
#include <aws/wisdom/ConnectWisdomService_EXPORTS.h>
#include <aws/wisdom/model/ExternalSourceConfiguration.h>
#include <aws/wisdom/model/ImportJobType.h>

#include <utility>

namespace Aws {
namespace ConnectWisdomService {
namespace Model {

/**
 */
class StartImportJobRequest : public ConnectWisdomServiceRequest {
 public:
  AWS_CONNECTWISDOMSERVICE_API StartImportJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartImportJob"; }

  AWS_CONNECTWISDOMSERVICE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The tags used to organize, track, or control access for this resource.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  StartImportJobRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration information of the external source that the resource data
   * are imported from.</p>
   */
  inline const ExternalSourceConfiguration& GetExternalSourceConfiguration() const { return m_externalSourceConfiguration; }
  inline bool ExternalSourceConfigurationHasBeenSet() const { return m_externalSourceConfigurationHasBeenSet; }
  template <typename ExternalSourceConfigurationT = ExternalSourceConfiguration>
  void SetExternalSourceConfiguration(ExternalSourceConfigurationT&& value) {
    m_externalSourceConfigurationHasBeenSet = true;
    m_externalSourceConfiguration = std::forward<ExternalSourceConfigurationT>(value);
  }
  template <typename ExternalSourceConfigurationT = ExternalSourceConfiguration>
  StartImportJobRequest& WithExternalSourceConfiguration(ExternalSourceConfigurationT&& value) {
    SetExternalSourceConfiguration(std::forward<ExternalSourceConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of the import job.</p> <ul> <li> <p>For importing quick response
   * resource, set the value to <code>QUICK_RESPONSES</code>.</p> </li> </ul>
   */
  inline ImportJobType GetImportJobType() const { return m_importJobType; }
  inline bool ImportJobTypeHasBeenSet() const { return m_importJobTypeHasBeenSet; }
  inline void SetImportJobType(ImportJobType value) {
    m_importJobTypeHasBeenSet = true;
    m_importJobType = value;
  }
  inline StartImportJobRequest& WithImportJobType(ImportJobType value) {
    SetImportJobType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the knowledge base. This should not be a QUICK_RESPONSES
   * type knowledge base if you're storing Wisdom Content resource to it. Can be
   * either the ID or the ARN. URLs cannot contain the ARN.</p> <ul> <li> <p>For
   * importing Wisdom quick responses, this should be a <code>QUICK_RESPONSES</code>
   * type knowledge base.</p> </li> </ul>
   */
  inline const Aws::String& GetKnowledgeBaseId() const { return m_knowledgeBaseId; }
  inline bool KnowledgeBaseIdHasBeenSet() const { return m_knowledgeBaseIdHasBeenSet; }
  template <typename KnowledgeBaseIdT = Aws::String>
  void SetKnowledgeBaseId(KnowledgeBaseIdT&& value) {
    m_knowledgeBaseIdHasBeenSet = true;
    m_knowledgeBaseId = std::forward<KnowledgeBaseIdT>(value);
  }
  template <typename KnowledgeBaseIdT = Aws::String>
  StartImportJobRequest& WithKnowledgeBaseId(KnowledgeBaseIdT&& value) {
    SetKnowledgeBaseId(std::forward<KnowledgeBaseIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The metadata fields of the imported Wisdom resources.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetMetadata() const { return m_metadata; }
  inline bool MetadataHasBeenSet() const { return m_metadataHasBeenSet; }
  template <typename MetadataT = Aws::Map<Aws::String, Aws::String>>
  void SetMetadata(MetadataT&& value) {
    m_metadataHasBeenSet = true;
    m_metadata = std::forward<MetadataT>(value);
  }
  template <typename MetadataT = Aws::Map<Aws::String, Aws::String>>
  StartImportJobRequest& WithMetadata(MetadataT&& value) {
    SetMetadata(std::forward<MetadataT>(value));
    return *this;
  }
  template <typename MetadataKeyT = Aws::String, typename MetadataValueT = Aws::String>
  StartImportJobRequest& AddMetadata(MetadataKeyT&& key, MetadataValueT&& value) {
    m_metadataHasBeenSet = true;
    m_metadata.emplace(std::forward<MetadataKeyT>(key), std::forward<MetadataValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A pointer to the uploaded asset. This value is returned by <a
   * href="https://docs.aws.amazon.com/wisdom/latest/APIReference/API_StartContentUpload.html">StartContentUpload</a>.</p>
   */
  inline const Aws::String& GetUploadId() const { return m_uploadId; }
  inline bool UploadIdHasBeenSet() const { return m_uploadIdHasBeenSet; }
  template <typename UploadIdT = Aws::String>
  void SetUploadId(UploadIdT&& value) {
    m_uploadIdHasBeenSet = true;
    m_uploadId = std::forward<UploadIdT>(value);
  }
  template <typename UploadIdT = Aws::String>
  StartImportJobRequest& WithUploadId(UploadIdT&& value) {
    SetUploadId(std::forward<UploadIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  ExternalSourceConfiguration m_externalSourceConfiguration;

  ImportJobType m_importJobType{ImportJobType::NOT_SET};

  Aws::String m_knowledgeBaseId;

  Aws::Map<Aws::String, Aws::String> m_metadata;

  Aws::String m_uploadId;
  bool m_clientTokenHasBeenSet = true;
  bool m_externalSourceConfigurationHasBeenSet = false;
  bool m_importJobTypeHasBeenSet = false;
  bool m_knowledgeBaseIdHasBeenSet = false;
  bool m_metadataHasBeenSet = false;
  bool m_uploadIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace ConnectWisdomService
}  // namespace Aws
