﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/customer-profiles/CustomerProfilesRequest.h>
#include <aws/customer-profiles/CustomerProfiles_EXPORTS.h>
#include <aws/customer-profiles/model/SegmentGroup.h>

#include <utility>

namespace Aws {
namespace CustomerProfiles {
namespace Model {

/**
 */
class CreateSegmentDefinitionRequest : public CustomerProfilesRequest {
 public:
  AWS_CUSTOMERPROFILES_API CreateSegmentDefinitionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateSegmentDefinition"; }

  AWS_CUSTOMERPROFILES_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique name of the domain.</p>
   */
  inline const Aws::String& GetDomainName() const { return m_domainName; }
  inline bool DomainNameHasBeenSet() const { return m_domainNameHasBeenSet; }
  template <typename DomainNameT = Aws::String>
  void SetDomainName(DomainNameT&& value) {
    m_domainNameHasBeenSet = true;
    m_domainName = std::forward<DomainNameT>(value);
  }
  template <typename DomainNameT = Aws::String>
  CreateSegmentDefinitionRequest& WithDomainName(DomainNameT&& value) {
    SetDomainName(std::forward<DomainNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique name of the segment definition.</p>
   */
  inline const Aws::String& GetSegmentDefinitionName() const { return m_segmentDefinitionName; }
  inline bool SegmentDefinitionNameHasBeenSet() const { return m_segmentDefinitionNameHasBeenSet; }
  template <typename SegmentDefinitionNameT = Aws::String>
  void SetSegmentDefinitionName(SegmentDefinitionNameT&& value) {
    m_segmentDefinitionNameHasBeenSet = true;
    m_segmentDefinitionName = std::forward<SegmentDefinitionNameT>(value);
  }
  template <typename SegmentDefinitionNameT = Aws::String>
  CreateSegmentDefinitionRequest& WithSegmentDefinitionName(SegmentDefinitionNameT&& value) {
    SetSegmentDefinitionName(std::forward<SegmentDefinitionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The display name of the segment definition.</p>
   */
  inline const Aws::String& GetDisplayName() const { return m_displayName; }
  inline bool DisplayNameHasBeenSet() const { return m_displayNameHasBeenSet; }
  template <typename DisplayNameT = Aws::String>
  void SetDisplayName(DisplayNameT&& value) {
    m_displayNameHasBeenSet = true;
    m_displayName = std::forward<DisplayNameT>(value);
  }
  template <typename DisplayNameT = Aws::String>
  CreateSegmentDefinitionRequest& WithDisplayName(DisplayNameT&& value) {
    SetDisplayName(std::forward<DisplayNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the segment definition.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateSegmentDefinitionRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the base segments and dimensions for a segment definition along
   * with their respective relationship.</p>
   */
  inline const SegmentGroup& GetSegmentGroups() const { return m_segmentGroups; }
  inline bool SegmentGroupsHasBeenSet() const { return m_segmentGroupsHasBeenSet; }
  template <typename SegmentGroupsT = SegmentGroup>
  void SetSegmentGroups(SegmentGroupsT&& value) {
    m_segmentGroupsHasBeenSet = true;
    m_segmentGroups = std::forward<SegmentGroupsT>(value);
  }
  template <typename SegmentGroupsT = SegmentGroup>
  CreateSegmentDefinitionRequest& WithSegmentGroups(SegmentGroupsT&& value) {
    SetSegmentGroups(std::forward<SegmentGroupsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The segment SQL query.</p>
   */
  inline const Aws::String& GetSegmentSqlQuery() const { return m_segmentSqlQuery; }
  inline bool SegmentSqlQueryHasBeenSet() const { return m_segmentSqlQueryHasBeenSet; }
  template <typename SegmentSqlQueryT = Aws::String>
  void SetSegmentSqlQuery(SegmentSqlQueryT&& value) {
    m_segmentSqlQueryHasBeenSet = true;
    m_segmentSqlQuery = std::forward<SegmentSqlQueryT>(value);
  }
  template <typename SegmentSqlQueryT = Aws::String>
  CreateSegmentDefinitionRequest& WithSegmentSqlQuery(SegmentSqlQueryT&& value) {
    SetSegmentSqlQuery(std::forward<SegmentSqlQueryT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags used to organize, track, or control access for this resource.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateSegmentDefinitionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateSegmentDefinitionRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_domainName;

  Aws::String m_segmentDefinitionName;

  Aws::String m_displayName;

  Aws::String m_description;

  SegmentGroup m_segmentGroups;

  Aws::String m_segmentSqlQuery;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_domainNameHasBeenSet = false;
  bool m_segmentDefinitionNameHasBeenSet = false;
  bool m_displayNameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_segmentGroupsHasBeenSet = false;
  bool m_segmentSqlQueryHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace CustomerProfiles
}  // namespace Aws
