
package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link IpamConfig}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableIpamConfig.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableIpamConfig implements IpamConfig {
  private final @Nullable String subnet;
  private final @Nullable String ipRange;
  private final @Nullable String gateway;

  private ImmutableIpamConfig(
      @Nullable String subnet,
      @Nullable String ipRange,
      @Nullable String gateway) {
    this.subnet = subnet;
    this.ipRange = ipRange;
    this.gateway = gateway;
  }

  /**
   * @return The value of the {@code subnet} attribute
   */
  @JsonProperty("Subnet")
  @Override
  public @Nullable String subnet() {
    return subnet;
  }

  /**
   * @return The value of the {@code ipRange} attribute
   */
  @JsonProperty("IPRange")
  @Override
  public @Nullable String ipRange() {
    return ipRange;
  }

  /**
   * @return The value of the {@code gateway} attribute
   */
  @JsonProperty("Gateway")
  @Override
  public @Nullable String gateway() {
    return gateway;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link IpamConfig#subnet() subnet} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for subnet (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableIpamConfig withSubnet(@Nullable String value) {
    if (Objects.equals(this.subnet, value)) return this;
    return new ImmutableIpamConfig(value, this.ipRange, this.gateway);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link IpamConfig#ipRange() ipRange} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for ipRange (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableIpamConfig withIpRange(@Nullable String value) {
    if (Objects.equals(this.ipRange, value)) return this;
    return new ImmutableIpamConfig(this.subnet, value, this.gateway);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link IpamConfig#gateway() gateway} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for gateway (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableIpamConfig withGateway(@Nullable String value) {
    if (Objects.equals(this.gateway, value)) return this;
    return new ImmutableIpamConfig(this.subnet, this.ipRange, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableIpamConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableIpamConfig
        && equalsByValue((ImmutableIpamConfig) another);
  }

  private boolean equalsByValue(ImmutableIpamConfig another) {
    return Objects.equals(subnet, another.subnet)
        && Objects.equals(ipRange, another.ipRange)
        && Objects.equals(gateway, another.gateway);
  }

  /**
   * Computes a hash code from attributes: {@code subnet}, {@code ipRange}, {@code gateway}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(subnet);
    h += (h << 5) + Objects.hashCode(ipRange);
    h += (h << 5) + Objects.hashCode(gateway);
    return h;
  }

  /**
   * Prints the immutable value {@code IpamConfig} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "IpamConfig{"
        + "subnet=" + subnet
        + ", ipRange=" + ipRange
        + ", gateway=" + gateway
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link IpamConfig} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable IpamConfig instance
   */
  public static ImmutableIpamConfig copyOf(IpamConfig instance) {
    if (instance instanceof ImmutableIpamConfig) {
      return (ImmutableIpamConfig) instance;
    }
    return ImmutableIpamConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableIpamConfig ImmutableIpamConfig}.
   * <pre>
   * ImmutableIpamConfig.builder()
   *    .subnet(String | null) // nullable {@link IpamConfig#subnet() subnet}
   *    .ipRange(String | null) // nullable {@link IpamConfig#ipRange() ipRange}
   *    .gateway(String | null) // nullable {@link IpamConfig#gateway() gateway}
   *    .build();
   * </pre>
   * @return A new ImmutableIpamConfig builder
   */
  public static ImmutableIpamConfig.Builder builder() {
    return new ImmutableIpamConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableIpamConfig ImmutableIpamConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements IpamConfig.Builder {
    private String subnet;
    private String ipRange;
    private String gateway;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code IpamConfig} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(IpamConfig instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable String subnetValue = instance.subnet();
      if (subnetValue != null) {
        subnet(subnetValue);
      }
      @Nullable String ipRangeValue = instance.ipRange();
      if (ipRangeValue != null) {
        ipRange(ipRangeValue);
      }
      @Nullable String gatewayValue = instance.gateway();
      if (gatewayValue != null) {
        gateway(gatewayValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link IpamConfig#subnet() subnet} attribute.
     * @param subnet The value for subnet (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Subnet")
    public final Builder subnet(@Nullable String subnet) {
      this.subnet = subnet;
      return this;
    }

    /**
     * Initializes the value for the {@link IpamConfig#ipRange() ipRange} attribute.
     * @param ipRange The value for ipRange (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("IPRange")
    public final Builder ipRange(@Nullable String ipRange) {
      this.ipRange = ipRange;
      return this;
    }

    /**
     * Initializes the value for the {@link IpamConfig#gateway() gateway} attribute.
     * @param gateway The value for gateway (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Gateway")
    public final Builder gateway(@Nullable String gateway) {
      this.gateway = gateway;
      return this;
    }

    /**
     * Builds a new {@link ImmutableIpamConfig ImmutableIpamConfig}.
     * @return An immutable instance of IpamConfig
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableIpamConfig build() {
      return new ImmutableIpamConfig(subnet, ipRange, gateway);
    }
  }
}
