//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.eventbased.equivalence;

import static org.eclipse.escet.common.java.Lists.last;
import static org.eclipse.escet.common.java.Lists.list;
import static org.eclipse.escet.common.java.Lists.listc;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.escet.cif.eventbased.automata.Automaton;
import org.eclipse.escet.cif.eventbased.automata.Edge;
import org.eclipse.escet.cif.eventbased.automata.Event;
import org.eclipse.escet.cif.eventbased.automata.Location;
import org.eclipse.escet.common.java.Assert;

/** Data of the language equivalence command. */
public class LangEquivCalculation extends BlockPartitioner {
    /**
     * Constructor of the {@link LangEquivCalculation} class.
     *
     * @param automs Automata to check for language equivalence.
     */
    public LangEquivCalculation(List<Automaton> automs) {
        super(automs, true);
        Assert.check(automs.size() == 2);
    }

    /**
     * Check whether both automata provided in the constructor, have the same language using a {@link BlockPartitioner
     * block partitioning algorithm}.
     *
     * <p>
     * Language equivalence exists if the initial locations of both automata are in the same block.
     * </p>
     *
     * @return A counter example if one was found, else {@code null}.
     */
    public CounterExample checkLanguageEquivalence() {
        return performBlockPartitioning();
    }

    @Override
    protected CounterExample constructCounterExample(Block block, Event finalEvent) {
        // Get a path from the initial location of one of the automata to the block.
        List<Event> path = getReversePath(block);
        Collections.reverse(path);

        // Get the initial locations of both automata.
        Assert.areEqual(2, automs.size());
        Location[] locs = new Location[2];
        locs[0] = automs.get(0).initial;
        locs[1] = automs.get(1).initial;

        // Execute the path from the initial locations towards the locations in the block (at the end of the path). If
        // the path can't be executed in both automata, we've found a counter example.
        Location[] newLocs = new Location[2];
        for (int pathIdx = 0; pathIdx < path.size(); pathIdx++) {
            // Get next locations for both automata, for the event of the path, if they exist.
            Event evt = path.get(pathIdx);
            newLocs[0] = getNextLocation(locs[0], evt);
            newLocs[1] = getNextLocation(locs[1], evt);

            // If both automata have a next location, proceed to the next event of the path.
            if (newLocs[0] != null && newLocs[1] != null) {
                locs[0] = newLocs[0];
                locs[1] = newLocs[1];
                newLocs[0] = null;
                newLocs[1] = null;
                continue;
            }

            // Sanity check: at least of the automata must have been able to take a transition for the event.
            Assert.check(newLocs[0] != null || newLocs[1] != null);

            // Path found that exists in one automaton but not in the other.
            return new CounterExample(path.subList(0, pathIdx), locs, evt);
        }

        // The path could be executed in both automata. If no final event is given, the reason is conclusive (because of
        // markings).
        if (finalEvent == null) {
            return new CounterExample(path, locs, null);
        }

        // We found two locations that are part of different blocks. Construct a counter example from them.
        return constructCounterExample(path, locs[0], locs[1]);
    }

    @Override
    protected CounterExample constructCounterExample(Location[] initLocs) {
        // Get the two initial locations.
        Assert.areEqual(2, initLocs.length);
        Location initLoc1 = initLocs[0];
        Location initLoc2 = initLocs[1];

        // Construct a counter example for the two initial locations, which are in different blocks.
        return constructCounterExample(List.of(), initLoc1, initLoc2);
    }

    /**
     * Construct a counter example for the given locations, of two different automata, that are in different blocks.
     *
     * @param prefix The path from the initial state to the two given locations.
     * @param loc1 The first location.
     * @param loc2 The second location.
     * @return A counter example.
     */
    private CounterExample constructCounterExample(List<Event> prefix, Location loc1, Location loc2) {
        // The two locations that are part of different blocks. Figure out why, by computing a split-explanation path.
        List<Event> splitExplanationPath = getSplitExplanationPath(loc1, loc2);

        // Execute the split-explanation path towards the conclusive states. The last event is excluded, as that
        // contains the conclusive reason.
        List<Event> path = listc(prefix.size() + splitExplanationPath.size() - 1);
        path.addAll(prefix);
        for (int pathIdx = 0; pathIdx < splitExplanationPath.size() - 1; pathIdx++) {
            // Add the event to the counter example path.
            Event evt = splitExplanationPath.get(pathIdx);
            path.add(evt);

            // Get next locations for both automata, along the path.
            loc1 = getNextLocation(loc1, evt);
            loc2 = getNextLocation(loc2, evt);

            // The event must be enabled in the locations, for both automata. Only the last event of the path may not be
            // enabled, but that one is excluded from this loop.
            Assert.notNull(loc1);
            Assert.notNull(loc2);
        }

        // Get the last explanation event to determine the conclusive reason.
        Event lastExplanationEvent = last(splitExplanationPath);

        // If no final event is given, the reason is already conclusive (because of markings).
        if (lastExplanationEvent == null) {
            return new CounterExample(path, new Location[] {loc1, loc2}, null);
        }

        // Determine for how many automata the final event is enabled in their current locations.
        int numAutsEnabled = (getNextLocation(loc1, lastExplanationEvent) == null ? 0 : 1)
                + (getNextLocation(loc2, lastExplanationEvent) == null ? 0 : 1);

        if (numAutsEnabled != 1) {
            // If the event is enabled in both automata, then the reason is not conclusive. That should not happen.
            // If the event is disabled in both automata, then the reason is not conclusive. That should not happen.
            Assert.fail();
        }

        // The event is enabled in exactly one of the automata. This is a conclusive reason.
        return new CounterExample(path, new Location[] {loc1, loc2}, lastExplanationEvent);
    }

    /**
     * Derive a path from locations in the block to the initial state.
     *
     * @param blk Block to walk from.
     * @return Path from the block to the initial state.
     */
    @SuppressWarnings("null")
    private List<Event> getReversePath(Block blk) {
        // Initialize the result. We'll derive the path from the block to an initial state in reverse.
        List<Event> reversePath = list();

        // Find the 'best' location in the block to start from. We take one closest to an initial location, so one with
        // the smallest depth. Selecting a location means that we'll create a path within the automaton that contains
        // that location, back to the initial location of that automaton.
        BlockLocation best = null;
        for (BlockLocation bl: blk.locs) {
            if (best == null || bl.depth < best.depth) {
                best = bl;
            }
        }

        // While we haven't yet reached the initial location (with depth zero), keep extending the path.
        while (best.depth != 0) {
            // Find the 'best' edge to take from the current location, within the automaton of that location. We take
            // the edge that gets us the closest to the automaton's initial location, by selecting the predecessor
            // location with the smallest depth we can reach in one step.
            BlockLocation bestPrev = null;
            Edge bestIncEdge = null;
            for (Edge incEdge: best.loc.getIncoming()) {
                BlockLocation prev = blockLocs.get(incEdge.srcLoc);
                if (bestPrev == null || prev.depth < bestPrev.depth) {
                    bestPrev = prev;
                    bestIncEdge = incEdge;
                }
            }

            // Add the event of the edge we're taking to the path.
            reversePath.add(bestIncEdge.event);

            // Take the edge, so that we can take another step (if needed) in the next iteration.
            best = bestPrev;
        }

        // Return the path.
        return reversePath;
    }

    /**
     * Get the next location in the path.
     *
     * @param loc Current location in the path.
     * @param evt Event to perform.
     * @return Next location in the path, or {@code null} if the event cannot be taken.
     */
    private Location getNextLocation(Location loc, Event evt) {
        Iterator<Edge> iter = loc.getOutgoing(evt);
        if (!iter.hasNext()) {
            return null;
        }
        return iter.next().dstLoc;
    }

    /**
     * Finds the path towards a location that explains why the two given locations are part of different blocks.
     *
     * @param loc1 The first location.
     * @param loc2 The second location.
     * @return The path, starting in {@code loc1} and {@code loc2} that explains why these locations are part of
     *     different blocks. The last event of the path may be enabled only from one the locations. The last event of
     *     the path may be {@code null} to indicate that the markings are different.
     */
    private List<Event> getSplitExplanationPath(Location loc1, Location loc2) {
        // Initialize split-explanation path.
        List<Event> splitPath = list();

        // Explore the path until a conclusive reason is found that explains why these locations are not part of the
        // same block.
        Location curLoc1 = loc1;
        Location curLoc2 = loc2;
        while (true) {
            // Get the blocks for the current locations of the automata.
            Block block1 = blocks.get(blockLocs.get(curLoc1).blockNumber);
            Block block2 = blocks.get(blockLocs.get(curLoc2).blockNumber);

            // Check that the locations are indeed part of two different blocks.
            Assert.check(block1 != block2);

            // Get first two blocks after the split point where 'curLoc1' and 'curLoc2' are no longer in the same block.
            while (block1.parent != block2.parent) {
                int maxDepth = Math.max(block1.depth, block2.depth);
                if (block1.depth == maxDepth) {
                    block1 = block1.parent;
                }
                if (block2.depth == maxDepth) {
                    block2 = block2.parent;
                }
            }

            // Ensure that the blocks are different children of the last common block from which they were split.
            Assert.check(block1 != block2);

            // Ensure that indeed both blocks are split because of the same event.
            Assert.areEqual(block1.splitEvent, block2.splitEvent);

            // Add an entry to the split-explanation path.
            splitPath.add(block1.splitEvent);

            // Find the reason why these two blocks where created. There are three possibilities:
            // 1) splitEvent == null: The splitting reason was because of markings. Conclusive.
            // 2) splitEvent != null, and the event is not enabled in one location. Conclusive.
            // 3) splitEvent != null, and the event is enabled in both locations. Not conclusive. Continue searching
            //    from the two new locations. The search will terminate, as the splitting tree has no cycles.
            if (block1.splitEvent == null) {
                // Possibility 1. The reason is conclusive, stop searching.
                break;
            } else {
                // Either possibility 2 or 3. See which possibility it is, by checking whether the split event is
                // enabled in the current locations.
                Location nextLoc1 = getNextLocation(curLoc1, block1.splitEvent);
                Location nextLoc2 = getNextLocation(curLoc2, block2.splitEvent);

                // The event must be enabled in at least one current location, as otherwise we've hit a dead end.
                Assert.check(nextLoc1 != null || nextLoc2 != null);

                // Distinguish between possibilities 2 and 3.
                if (nextLoc1 == null || nextLoc2 == null) {
                    // Possibility 2. The reason is conclusive, stop searching.
                    break;
                } else {
                    // Possibility 3. The reason is not conclusive, keep searching.
                    curLoc1 = nextLoc1;
                    curLoc2 = nextLoc2;
                }
            }
        }

        // Return the path that explains the splitting reason.
        return splitPath;
    }
}
