from __future__ import division

import os
import re


def generate():
    """Generates a dictionary of all the known CRC formats from:
    https://reveng.sourceforge.io/crc-catalogue/all.htm

    See pwnlib/data/crcsum.txt for more information.
    """

    curdir, _ = os.path.split(__file__)
    path = os.path.join(curdir, '..', '..', 'data', 'crcsums.txt')
    with open(path) as fd:
        data = fd.read()
    out = {}

    def fixup(s):
        if s == 'true':
            return True
        elif s == 'false':
            return False
        elif s.startswith('"'):
            assert re.match('"[^"]+"', s)
            return s[1:-1]
        elif s.startswith('0x'):
            assert re.match('0x[0-9a-fA-F]+', s)
            return int(s[2:], 16)
        else:
            assert re.match('[0-9]+', s)
            return int(s, 10)

    for l in data.strip().split('\n'):
        if not l or l[0] == '#':
            continue

        ref, l = l.split(' ', 1)

        cur = {}
        cur['link'] = 'https://reveng.sourceforge.io/crc-catalogue/all.htm#' + ref
        for key in ['width', 'poly', 'init', 'refin', 'refout', 'xorout', 'check', 'name']:
            cur[key] = fixup(re.findall(r'%s=(\S+)' % key, l)[0])

        cur['name'] = cur['name'].lower().replace('/', '_').replace('-', '_')
        assert cur['name'] not in out
        out[cur['name']] = cur

    return out


all_crcs = \
    {'crc_10_atm': {'check': 409,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-10-atm',
                    'name': 'crc_10_atm',
                    'poly': 563,
                    'refin': False,
                    'refout': False,
                    'width': 10,
                    'xorout': 0},
     'crc_10_cdma2000': {'check': 563,
                         'init': 1023,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-10-cdma2000',
                         'name': 'crc_10_cdma2000',
                         'poly': 985,
                         'refin': False,
                         'refout': False,
                         'width': 10,
                         'xorout': 0},
     'crc_10_gsm': {'check': 298,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-10-gsm',
                    'name': 'crc_10_gsm',
                    'poly': 373,
                    'refin': False,
                    'refout': False,
                    'width': 10,
                    'xorout': 1023},
     'crc_11_flexray': {'check': 1443,
                        'init': 26,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-11-flexray',
                        'name': 'crc_11_flexray',
                        'poly': 901,
                        'refin': False,
                        'refout': False,
                        'width': 11,
                        'xorout': 0},
     'crc_11_umts': {'check': 97,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-11-umts',
                     'name': 'crc_11_umts',
                     'poly': 775,
                     'refin': False,
                     'refout': False,
                     'width': 11,
                     'xorout': 0},
     'crc_12_cdma2000': {'check': 3405,
                         'init': 4095,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-12-cdma2000',
                         'name': 'crc_12_cdma2000',
                         'poly': 3859,
                         'refin': False,
                         'refout': False,
                         'width': 12,
                         'xorout': 0},
     'crc_12_dect': {'check': 3931,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-12-dect',
                     'name': 'crc_12_dect',
                     'poly': 2063,
                     'refin': False,
                     'refout': False,
                     'width': 12,
                     'xorout': 0},
     'crc_12_gsm': {'check': 2868,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-12-gsm',
                    'name': 'crc_12_gsm',
                    'poly': 3377,
                    'refin': False,
                    'refout': False,
                    'width': 12,
                    'xorout': 4095},
     'crc_12_umts': {'check': 3503,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-12-umts',
                     'name': 'crc_12_umts',
                     'poly': 2063,
                     'refin': False,
                     'refout': True,
                     'width': 12,
                     'xorout': 0},
     'crc_13_bbc': {'check': 1274,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-13-bbc',
                    'name': 'crc_13_bbc',
                    'poly': 7413,
                    'refin': False,
                    'refout': False,
                    'width': 13,
                    'xorout': 0},
     'crc_14_darc': {'check': 2093,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-14-darc',
                     'name': 'crc_14_darc',
                     'poly': 2053,
                     'refin': True,
                     'refout': True,
                     'width': 14,
                     'xorout': 0},
     'crc_14_gsm': {'check': 12462,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-14-gsm',
                    'name': 'crc_14_gsm',
                    'poly': 8237,
                    'refin': False,
                    'refout': False,
                    'width': 14,
                    'xorout': 16383},
     'crc_15_can': {'check': 1438,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-15-can',
                    'name': 'crc_15_can',
                    'poly': 17817,
                    'refin': False,
                    'refout': False,
                    'width': 15,
                    'xorout': 0},
     'crc_15_mpt1327': {'check': 9574,
                        'init': 0,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-15-mpt1327',
                        'name': 'crc_15_mpt1327',
                        'poly': 26645,
                        'refin': False,
                        'refout': False,
                        'width': 15,
                        'xorout': 1},
     'crc_16_arc': {'check': 47933,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-arc',
                    'name': 'crc_16_arc',
                    'poly': 32773,
                    'refin': True,
                    'refout': True,
                    'width': 16,
                    'xorout': 0},
     'crc_16_cdma2000': {'check': 19462,
                         'init': 65535,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-cdma2000',
                         'name': 'crc_16_cdma2000',
                         'poly': 51303,
                         'refin': False,
                         'refout': False,
                         'width': 16,
                         'xorout': 0},
     'crc_16_cms': {'check': 44775,
                    'init': 65535,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-cms',
                    'name': 'crc_16_cms',
                    'poly': 32773,
                    'refin': False,
                    'refout': False,
                    'width': 16,
                    'xorout': 0},
     'crc_16_dds_110': {'check': 40655,
                        'init': 32781,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-dds-110',
                        'name': 'crc_16_dds_110',
                        'poly': 32773,
                        'refin': False,
                        'refout': False,
                        'width': 16,
                        'xorout': 0},
     'crc_16_dect_r': {'check': 126,
                       'init': 0,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-dect-r',
                       'name': 'crc_16_dect_r',
                       'poly': 1417,
                       'refin': False,
                       'refout': False,
                       'width': 16,
                       'xorout': 1},
     'crc_16_dect_x': {'check': 127,
                       'init': 0,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-dect-x',
                       'name': 'crc_16_dect_x',
                       'poly': 1417,
                       'refin': False,
                       'refout': False,
                       'width': 16,
                       'xorout': 0},
     'crc_16_dnp': {'check': 60034,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-dnp',
                    'name': 'crc_16_dnp',
                    'poly': 15717,
                    'refin': True,
                    'refout': True,
                    'width': 16,
                    'xorout': 65535},
     'crc_16_en_13757': {'check': 49847,
                         'init': 0,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-en-13757',
                         'name': 'crc_16_en_13757',
                         'poly': 15717,
                         'refin': False,
                         'refout': False,
                         'width': 16,
                         'xorout': 65535},
     'crc_16_genibus': {'check': 54862,
                        'init': 65535,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-genibus',
                        'name': 'crc_16_genibus',
                        'poly': 4129,
                        'refin': False,
                        'refout': False,
                        'width': 16,
                        'xorout': 65535},
     'crc_16_gsm': {'check': 52796,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-gsm',
                    'name': 'crc_16_gsm',
                    'poly': 4129,
                    'refin': False,
                    'refout': False,
                    'width': 16,
                    'xorout': 65535},
     'crc_16_ibm_3740': {'check': 10673,
                         'init': 65535,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-ibm-3740',
                         'name': 'crc_16_ibm_3740',
                         'poly': 4129,
                         'refin': False,
                         'refout': False,
                         'width': 16,
                         'xorout': 0},
     'crc_16_ibm_sdlc': {'check': 36974,
                         'init': 65535,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-ibm-sdlc',
                         'name': 'crc_16_ibm_sdlc',
                         'poly': 4129,
                         'refin': True,
                         'refout': True,
                         'width': 16,
                         'xorout': 65535},
     'crc_16_iso_iec_14443_3_a': {'check': 48901,
                                  'init': 50886,
                                  'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-iso-iec-14443-3-a',
                                  'name': 'crc_16_iso_iec_14443_3_a',
                                  'poly': 4129,
                                  'refin': True,
                                  'refout': True,
                                  'width': 16,
                                  'xorout': 0},
     'crc_16_kermit': {'check': 8585,
                       'init': 0,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-kermit',
                       'name': 'crc_16_kermit',
                       'poly': 4129,
                       'refin': True,
                       'refout': True,
                       'width': 16,
                       'xorout': 0},
     'crc_16_lj1200': {'check': 48628,
                       'init': 0,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-lj1200',
                       'name': 'crc_16_lj1200',
                       'poly': 28515,
                       'refin': False,
                       'refout': False,
                       'width': 16,
                       'xorout': 0},
     'crc_16_m17': {'check': 30507,
                    'init': 65535,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-m17',
                    'name': 'crc_16_m17',
                    'poly': 22837,
                    'refin': False,
                    'refout': False,
                    'width': 16,
                    'xorout': 0},
     'crc_16_maxim_dow': {'check': 17602,
                          'init': 0,
                          'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-maxim-dow',
                          'name': 'crc_16_maxim_dow',
                          'poly': 32773,
                          'refin': True,
                          'refout': True,
                          'width': 16,
                          'xorout': 65535},
     'crc_16_mcrf4xx': {'check': 28561,
                        'init': 65535,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-mcrf4xx',
                        'name': 'crc_16_mcrf4xx',
                        'poly': 4129,
                        'refin': True,
                        'refout': True,
                        'width': 16,
                        'xorout': 0},
     'crc_16_modbus': {'check': 19255,
                       'init': 65535,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-modbus',
                       'name': 'crc_16_modbus',
                       'poly': 32773,
                       'refin': True,
                       'refout': True,
                       'width': 16,
                       'xorout': 0},
     'crc_16_nrsc_5': {'check': 41062,
                       'init': 65535,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-nrsc-5',
                       'name': 'crc_16_nrsc_5',
                       'poly': 2059,
                       'refin': True,
                       'refout': True,
                       'width': 16,
                       'xorout': 0},
     'crc_16_opensafety_a': {'check': 23864,
                             'init': 0,
                             'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-opensafety-a',
                             'name': 'crc_16_opensafety_a',
                             'poly': 22837,
                             'refin': False,
                             'refout': False,
                             'width': 16,
                             'xorout': 0},
     'crc_16_opensafety_b': {'check': 8446,
                             'init': 0,
                             'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-opensafety-b',
                             'name': 'crc_16_opensafety_b',
                             'poly': 30043,
                             'refin': False,
                             'refout': False,
                             'width': 16,
                             'xorout': 0},
     'crc_16_profibus': {'check': 43033,
                         'init': 65535,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-profibus',
                         'name': 'crc_16_profibus',
                         'poly': 7631,
                         'refin': False,
                         'refout': False,
                         'width': 16,
                         'xorout': 65535},
     'crc_16_riello': {'check': 25552,
                       'init': 45738,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-riello',
                       'name': 'crc_16_riello',
                       'poly': 4129,
                       'refin': True,
                       'refout': True,
                       'width': 16,
                       'xorout': 0},
     'crc_16_spi_fujitsu': {'check': 58828,
                            'init': 7439,
                            'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-spi-fujitsu',
                            'name': 'crc_16_spi_fujitsu',
                            'poly': 4129,
                            'refin': False,
                            'refout': False,
                            'width': 16,
                            'xorout': 0},
     'crc_16_t10_dif': {'check': 53467,
                        'init': 0,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-t10-dif',
                        'name': 'crc_16_t10_dif',
                        'poly': 35767,
                        'refin': False,
                        'refout': False,
                        'width': 16,
                        'xorout': 0},
     'crc_16_teledisk': {'check': 4019,
                         'init': 0,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-teledisk',
                         'name': 'crc_16_teledisk',
                         'poly': 41111,
                         'refin': False,
                         'refout': False,
                         'width': 16,
                         'xorout': 0},
     'crc_16_tms37157': {'check': 9905,
                         'init': 35308,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-tms37157',
                         'name': 'crc_16_tms37157',
                         'poly': 4129,
                         'refin': True,
                         'refout': True,
                         'width': 16,
                         'xorout': 0},
     'crc_16_umts': {'check': 65256,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-umts',
                     'name': 'crc_16_umts',
                     'poly': 32773,
                     'refin': False,
                     'refout': False,
                     'width': 16,
                     'xorout': 0},
     'crc_16_usb': {'check': 46280,
                    'init': 65535,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-usb',
                    'name': 'crc_16_usb',
                    'poly': 32773,
                    'refin': True,
                    'refout': True,
                    'width': 16,
                    'xorout': 65535},
     'crc_16_xmodem': {'check': 12739,
                       'init': 0,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-16-xmodem',
                       'name': 'crc_16_xmodem',
                       'poly': 4129,
                       'refin': False,
                       'refout': False,
                       'width': 16,
                       'xorout': 0},
     'crc_17_can_fd': {'check': 20227,
                       'init': 0,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-17-can-fd',
                       'name': 'crc_17_can_fd',
                       'poly': 92251,
                       'refin': False,
                       'refout': False,
                       'width': 17,
                       'xorout': 0},
     'crc_21_can_fd': {'check': 972865,
                       'init': 0,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-21-can-fd',
                       'name': 'crc_21_can_fd',
                       'poly': 1058969,
                       'refin': False,
                       'refout': False,
                       'width': 21,
                       'xorout': 0},
     'crc_24_ble': {'check': 12737110,
                    'init': 5592405,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-24-ble',
                    'name': 'crc_24_ble',
                    'poly': 1627,
                    'refin': True,
                    'refout': True,
                    'width': 24,
                    'xorout': 0},
     'crc_24_flexray_a': {'check': 7961021,
                          'init': 16702650,
                          'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-24-flexray-a',
                          'name': 'crc_24_flexray_a',
                          'poly': 6122955,
                          'refin': False,
                          'refout': False,
                          'width': 24,
                          'xorout': 0},
     'crc_24_flexray_b': {'check': 2040760,
                          'init': 11259375,
                          'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-24-flexray-b',
                          'name': 'crc_24_flexray_b',
                          'poly': 6122955,
                          'refin': False,
                          'refout': False,
                          'width': 24,
                          'xorout': 0},
     'crc_24_interlaken': {'check': 11858918,
                           'init': 16777215,
                           'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-24-interlaken',
                           'name': 'crc_24_interlaken',
                           'poly': 3312483,
                           'refin': False,
                           'refout': False,
                           'width': 24,
                           'xorout': 16777215},
     'crc_24_lte_a': {'check': 13494019,
                      'init': 0,
                      'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-24-lte-a',
                      'name': 'crc_24_lte_a',
                      'poly': 8801531,
                      'refin': False,
                      'refout': False,
                      'width': 24,
                      'xorout': 0},
     'crc_24_lte_b': {'check': 2355026,
                      'init': 0,
                      'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-24-lte-b',
                      'name': 'crc_24_lte_b',
                      'poly': 8388707,
                      'refin': False,
                      'refout': False,
                      'width': 24,
                      'xorout': 0},
     'crc_24_openpgp': {'check': 2215682,
                        'init': 11994318,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-24-openpgp',
                        'name': 'crc_24_openpgp',
                        'poly': 8801531,
                        'refin': False,
                        'refout': False,
                        'width': 24,
                        'xorout': 0},
     'crc_24_os_9': {'check': 2101157,
                     'init': 16777215,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-24-os-9',
                     'name': 'crc_24_os_9',
                     'poly': 8388707,
                     'refin': False,
                     'refout': False,
                     'width': 24,
                     'xorout': 16777215},
     'crc_30_cdma': {'check': 79907519,
                     'init': 1073741823,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-30-cdma',
                     'name': 'crc_30_cdma',
                     'poly': 540064199,
                     'refin': False,
                     'refout': False,
                     'width': 30,
                     'xorout': 1073741823},
     'crc_31_philips': {'check': 216654956,
                        'init': 2147483647,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-31-philips',
                        'name': 'crc_31_philips',
                        'poly': 79764919,
                        'refin': False,
                        'refout': False,
                        'width': 31,
                        'xorout': 2147483647},
     'crc_32_aixm': {'check': 806403967,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-aixm',
                     'name': 'crc_32_aixm',
                     'poly': 2168537515,
                     'refin': False,
                     'refout': False,
                     'width': 32,
                     'xorout': 0},
     'crc_32_autosar': {'check': 379048042,
                        'init': 4294967295,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-autosar',
                        'name': 'crc_32_autosar',
                        'poly': 4104977171,
                        'refin': True,
                        'refout': True,
                        'width': 32,
                        'xorout': 4294967295},
     'crc_32_base91_d': {'check': 2268157302,
                         'init': 4294967295,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-base91-d',
                         'name': 'crc_32_base91_d',
                         'poly': 2821953579,
                         'refin': True,
                         'refout': True,
                         'width': 32,
                         'xorout': 4294967295},
     'crc_32_bzip2': {'check': 4236843288,
                      'init': 4294967295,
                      'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-bzip2',
                      'name': 'crc_32_bzip2',
                      'poly': 79764919,
                      'refin': False,
                      'refout': False,
                      'width': 32,
                      'xorout': 4294967295},
     'crc_32_cd_rom_edc': {'check': 1858268612,
                           'init': 0,
                           'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-cd-rom-edc',
                           'name': 'crc_32_cd_rom_edc',
                           'poly': 2147581979,
                           'refin': True,
                           'refout': True,
                           'width': 32,
                           'xorout': 0},
     'crc_32_cksum': {'check': 1985902208,
                      'init': 0,
                      'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-cksum',
                      'name': 'crc_32_cksum',
                      'poly': 79764919,
                      'refin': False,
                      'refout': False,
                      'width': 32,
                      'xorout': 4294967295},
     'crc_32_iscsi': {'check': 3808858755,
                      'init': 4294967295,
                      'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-iscsi',
                      'name': 'crc_32_iscsi',
                      'poly': 517762881,
                      'refin': True,
                      'refout': True,
                      'width': 32,
                      'xorout': 4294967295},
     'crc_32_iso_hdlc': {'check': 3421780262,
                         'init': 4294967295,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-iso-hdlc',
                         'name': 'crc_32_iso_hdlc',
                         'poly': 79764919,
                         'refin': True,
                         'refout': True,
                         'width': 32,
                         'xorout': 4294967295},
     'crc_32_jamcrc': {'check': 873187033,
                       'init': 4294967295,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-jamcrc',
                       'name': 'crc_32_jamcrc',
                       'poly': 79764919,
                       'refin': True,
                       'refout': True,
                       'width': 32,
                       'xorout': 0},
     'crc_32_mef': {'check': 3535941457,
                    'init': 4294967295,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-mef',
                    'name': 'crc_32_mef',
                    'poly': 1947962583,
                    'refin': True,
                    'refout': True,
                    'width': 32,
                    'xorout': 0},
     'crc_32_mpeg_2': {'check': 58124007,
                       'init': 4294967295,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-mpeg-2',
                       'name': 'crc_32_mpeg_2',
                       'poly': 79764919,
                       'refin': False,
                       'refout': False,
                       'width': 32,
                       'xorout': 0},
     'crc_32_xfer': {'check': 3171672888,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-32-xfer',
                     'name': 'crc_32_xfer',
                     'poly': 175,
                     'refin': False,
                     'refout': False,
                     'width': 32,
                     'xorout': 0},
     'crc_3_gsm': {'check': 4,
                   'init': 0,
                   'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-3-gsm',
                   'name': 'crc_3_gsm',
                   'poly': 3,
                   'refin': False,
                   'refout': False,
                   'width': 3,
                   'xorout': 7},
     'crc_3_rohc': {'check': 6,
                    'init': 7,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-3-rohc',
                    'name': 'crc_3_rohc',
                    'poly': 3,
                    'refin': True,
                    'refout': True,
                    'width': 3,
                    'xorout': 0},
     'crc_40_gsm': {'check': 910907393606,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-40-gsm',
                    'name': 'crc_40_gsm',
                    'poly': 75628553,
                    'refin': False,
                    'refout': False,
                    'width': 40,
                    'xorout': 1099511627775},
     'crc_4_g_704': {'check': 7,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-4-g-704',
                     'name': 'crc_4_g_704',
                     'poly': 3,
                     'refin': True,
                     'refout': True,
                     'width': 4,
                     'xorout': 0},
     'crc_4_interlaken': {'check': 11,
                          'init': 15,
                          'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-4-interlaken',
                          'name': 'crc_4_interlaken',
                          'poly': 3,
                          'refin': False,
                          'refout': False,
                          'width': 4,
                          'xorout': 15},
     'crc_5_epc_c1g2': {'check': 0,
                        'init': 9,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-5-epc-c1g2',
                        'name': 'crc_5_epc_c1g2',
                        'poly': 9,
                        'refin': False,
                        'refout': False,
                        'width': 5,
                        'xorout': 0},
     'crc_5_g_704': {'check': 7,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-5-g-704',
                     'name': 'crc_5_g_704',
                     'poly': 21,
                     'refin': True,
                     'refout': True,
                     'width': 5,
                     'xorout': 0},
     'crc_5_usb': {'check': 25,
                   'init': 31,
                   'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-5-usb',
                   'name': 'crc_5_usb',
                   'poly': 5,
                   'refin': True,
                   'refout': True,
                   'width': 5,
                   'xorout': 31},
     'crc_64_ecma_182': {'check': 7800480153909949255,
                         'init': 0,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-64-ecma-182',
                         'name': 'crc_64_ecma_182',
                         'poly': 4823603603198064275,
                         'refin': False,
                         'refout': False,
                         'width': 64,
                         'xorout': 0},
     'crc_64_go_iso': {'check': 13333283586479230977,
                       'init': 18446744073709551615,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-64-go-iso',
                       'name': 'crc_64_go_iso',
                       'poly': 27,
                       'refin': True,
                       'refout': True,
                       'width': 64,
                       'xorout': 18446744073709551615},
     'crc_64_ms': {'check': 8490612747469246186,
                   'init': 18446744073709551615,
                   'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-64-ms',
                   'name': 'crc_64_ms',
                   'poly': 2710187085972792137,
                   'refin': True,
                   'refout': True,
                   'width': 64,
                   'xorout': 0},
     'crc_64_we': {'check': 7128171145767219210,
                   'init': 18446744073709551615,
                   'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-64-we',
                   'name': 'crc_64_we',
                   'poly': 4823603603198064275,
                   'refin': False,
                   'refout': False,
                   'width': 64,
                   'xorout': 18446744073709551615},
     'crc_64_xz': {'check': 11051210869376104954,
                   'init': 18446744073709551615,
                   'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-64-xz',
                   'name': 'crc_64_xz',
                   'poly': 4823603603198064275,
                   'refin': True,
                   'refout': True,
                   'width': 64,
                   'xorout': 18446744073709551615},
     'crc_6_cdma2000_a': {'check': 13,
                          'init': 63,
                          'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-6-cdma2000-a',
                          'name': 'crc_6_cdma2000_a',
                          'poly': 39,
                          'refin': False,
                          'refout': False,
                          'width': 6,
                          'xorout': 0},
     'crc_6_cdma2000_b': {'check': 59,
                          'init': 63,
                          'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-6-cdma2000-b',
                          'name': 'crc_6_cdma2000_b',
                          'poly': 7,
                          'refin': False,
                          'refout': False,
                          'width': 6,
                          'xorout': 0},
     'crc_6_darc': {'check': 38,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-6-darc',
                    'name': 'crc_6_darc',
                    'poly': 25,
                    'refin': True,
                    'refout': True,
                    'width': 6,
                    'xorout': 0},
     'crc_6_g_704': {'check': 6,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-6-g-704',
                     'name': 'crc_6_g_704',
                     'poly': 3,
                     'refin': True,
                     'refout': True,
                     'width': 6,
                     'xorout': 0},
     'crc_6_gsm': {'check': 19,
                   'init': 0,
                   'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-6-gsm',
                   'name': 'crc_6_gsm',
                   'poly': 47,
                   'refin': False,
                   'refout': False,
                   'width': 6,
                   'xorout': 63},
     'crc_7_mmc': {'check': 117,
                   'init': 0,
                   'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-7-mmc',
                   'name': 'crc_7_mmc',
                   'poly': 9,
                   'refin': False,
                   'refout': False,
                   'width': 7,
                   'xorout': 0},
     'crc_7_rohc': {'check': 83,
                    'init': 127,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-7-rohc',
                    'name': 'crc_7_rohc',
                    'poly': 79,
                    'refin': True,
                    'refout': True,
                    'width': 7,
                    'xorout': 0},
     'crc_7_umts': {'check': 97,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-7-umts',
                    'name': 'crc_7_umts',
                    'poly': 69,
                    'refin': False,
                    'refout': False,
                    'width': 7,
                    'xorout': 0},
     'crc_82_darc': {'check': 749237524598872659187218,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-82-darc',
                     'name': 'crc_82_darc',
                     'poly': 229256212191916381701137,
                     'refin': True,
                     'refout': True,
                     'width': 82,
                     'xorout': 0},
     'crc_8_autosar': {'check': 223,
                       'init': 255,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-autosar',
                       'name': 'crc_8_autosar',
                       'poly': 47,
                       'refin': False,
                       'refout': False,
                       'width': 8,
                       'xorout': 255},
     'crc_8_bluetooth': {'check': 38,
                         'init': 0,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-bluetooth',
                         'name': 'crc_8_bluetooth',
                         'poly': 167,
                         'refin': True,
                         'refout': True,
                         'width': 8,
                         'xorout': 0},
     'crc_8_cdma2000': {'check': 218,
                        'init': 255,
                        'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-cdma2000',
                        'name': 'crc_8_cdma2000',
                        'poly': 155,
                        'refin': False,
                        'refout': False,
                        'width': 8,
                        'xorout': 0},
     'crc_8_darc': {'check': 21,
                    'init': 0,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-darc',
                    'name': 'crc_8_darc',
                    'poly': 57,
                    'refin': True,
                    'refout': True,
                    'width': 8,
                    'xorout': 0},
     'crc_8_dvb_s2': {'check': 188,
                      'init': 0,
                      'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-dvb-s2',
                      'name': 'crc_8_dvb_s2',
                      'poly': 213,
                      'refin': False,
                      'refout': False,
                      'width': 8,
                      'xorout': 0},
     'crc_8_gsm_a': {'check': 55,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-gsm-a',
                     'name': 'crc_8_gsm_a',
                     'poly': 29,
                     'refin': False,
                     'refout': False,
                     'width': 8,
                     'xorout': 0},
     'crc_8_gsm_b': {'check': 148,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-gsm-b',
                     'name': 'crc_8_gsm_b',
                     'poly': 73,
                     'refin': False,
                     'refout': False,
                     'width': 8,
                     'xorout': 255},
     'crc_8_hitag': {'check': 180,
                     'init': 255,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-hitag',
                     'name': 'crc_8_hitag',
                     'poly': 29,
                     'refin': False,
                     'refout': False,
                     'width': 8,
                     'xorout': 0},
     'crc_8_i_432_1': {'check': 161,
                       'init': 0,
                       'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-i-432-1',
                       'name': 'crc_8_i_432_1',
                       'poly': 7,
                       'refin': False,
                       'refout': False,
                       'width': 8,
                       'xorout': 85},
     'crc_8_i_code': {'check': 126,
                      'init': 253,
                      'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-i-code',
                      'name': 'crc_8_i_code',
                      'poly': 29,
                      'refin': False,
                      'refout': False,
                      'width': 8,
                      'xorout': 0},
     'crc_8_lte': {'check': 234,
                   'init': 0,
                   'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-lte',
                   'name': 'crc_8_lte',
                   'poly': 155,
                   'refin': False,
                   'refout': False,
                   'width': 8,
                   'xorout': 0},
     'crc_8_maxim_dow': {'check': 161,
                         'init': 0,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-maxim-dow',
                         'name': 'crc_8_maxim_dow',
                         'poly': 49,
                         'refin': True,
                         'refout': True,
                         'width': 8,
                         'xorout': 0},
     'crc_8_mifare_mad': {'check': 153,
                          'init': 199,
                          'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-mifare-mad',
                          'name': 'crc_8_mifare_mad',
                          'poly': 29,
                          'refin': False,
                          'refout': False,
                          'width': 8,
                          'xorout': 0},
     'crc_8_nrsc_5': {'check': 247,
                      'init': 255,
                      'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-nrsc-5',
                      'name': 'crc_8_nrsc_5',
                      'poly': 49,
                      'refin': False,
                      'refout': False,
                      'width': 8,
                      'xorout': 0},
     'crc_8_opensafety': {'check': 62,
                          'init': 0,
                          'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-opensafety',
                          'name': 'crc_8_opensafety',
                          'poly': 47,
                          'refin': False,
                          'refout': False,
                          'width': 8,
                          'xorout': 0},
     'crc_8_rohc': {'check': 208,
                    'init': 255,
                    'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-rohc',
                    'name': 'crc_8_rohc',
                    'poly': 7,
                    'refin': True,
                    'refout': True,
                    'width': 8,
                    'xorout': 0},
     'crc_8_sae_j1850': {'check': 75,
                         'init': 255,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-sae-j1850',
                         'name': 'crc_8_sae_j1850',
                         'poly': 29,
                         'refin': False,
                         'refout': False,
                         'width': 8,
                         'xorout': 255},
     'crc_8_smbus': {'check': 244,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-smbus',
                     'name': 'crc_8_smbus',
                     'poly': 7,
                     'refin': False,
                     'refout': False,
                     'width': 8,
                     'xorout': 0},
     'crc_8_tech_3250': {'check': 151,
                         'init': 255,
                         'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-tech-3250',
                         'name': 'crc_8_tech_3250',
                         'poly': 29,
                         'refin': True,
                         'refout': True,
                         'width': 8,
                         'xorout': 0},
     'crc_8_wcdma': {'check': 37,
                     'init': 0,
                     'link': 'https://reveng.sourceforge.io/crc-catalogue/all.htm#crc.cat.crc-8-wcdma',
                     'name': 'crc_8_wcdma',
                     'poly': 155,
                     'refin': True,
                     'refout': True,
                     'width': 8,
                     'xorout': 0}}
