﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Parameter.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/ModifyDBParameterGroupMessage">AWS
 * API Reference</a></p>
 */
class ModifyDBParameterGroupRequest : public RDSRequest {
 public:
  AWS_RDS_API ModifyDBParameterGroupRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ModifyDBParameterGroup"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the DB parameter group.</p> <p>Constraints:</p> <ul> <li> <p>If
   * supplied, must match the name of an existing <code>DBParameterGroup</code>.</p>
   * </li> </ul>
   */
  inline const Aws::String& GetDBParameterGroupName() const { return m_dBParameterGroupName; }
  inline bool DBParameterGroupNameHasBeenSet() const { return m_dBParameterGroupNameHasBeenSet; }
  template <typename DBParameterGroupNameT = Aws::String>
  void SetDBParameterGroupName(DBParameterGroupNameT&& value) {
    m_dBParameterGroupNameHasBeenSet = true;
    m_dBParameterGroupName = std::forward<DBParameterGroupNameT>(value);
  }
  template <typename DBParameterGroupNameT = Aws::String>
  ModifyDBParameterGroupRequest& WithDBParameterGroupName(DBParameterGroupNameT&& value) {
    SetDBParameterGroupName(std::forward<DBParameterGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of parameter names, values, and the application methods for the
   * parameter update. At least one parameter name, value, and application method
   * must be supplied; later arguments are optional. A maximum of 20 parameters can
   * be modified in a single request.</p> <p>Valid Values (for the application
   * method): <code>immediate | pending-reboot</code> </p> <p>You can use the
   * <code>immediate</code> value with dynamic parameters only. You can use the
   * <code>pending-reboot</code> value for both dynamic and static parameters.</p>
   * <p>When the application method is <code>immediate</code>, changes to dynamic
   * parameters are applied immediately to the DB instances associated with the
   * parameter group.</p> <p>When the application method is
   * <code>pending-reboot</code>, changes to dynamic and static parameters are
   * applied after a reboot without failover to the DB instances associated with the
   * parameter group.</p>  <p>You can't use <code>pending-reboot</code> with
   * dynamic parameters on RDS for SQL Server DB instances. Use
   * <code>immediate</code>.</p>  <p>For more information on modifying DB
   * parameters, see <a
   * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithParamGroups.html">Working
   * with DB parameter groups</a> in the <i>Amazon RDS User Guide</i>.</p>
   */
  inline const Aws::Vector<Parameter>& GetParameters() const { return m_parameters; }
  inline bool ParametersHasBeenSet() const { return m_parametersHasBeenSet; }
  template <typename ParametersT = Aws::Vector<Parameter>>
  void SetParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters = std::forward<ParametersT>(value);
  }
  template <typename ParametersT = Aws::Vector<Parameter>>
  ModifyDBParameterGroupRequest& WithParameters(ParametersT&& value) {
    SetParameters(std::forward<ParametersT>(value));
    return *this;
  }
  template <typename ParametersT = Parameter>
  ModifyDBParameterGroupRequest& AddParameters(ParametersT&& value) {
    m_parametersHasBeenSet = true;
    m_parameters.emplace_back(std::forward<ParametersT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_dBParameterGroupName;

  Aws::Vector<Parameter> m_parameters;
  bool m_dBParameterGroupNameHasBeenSet = false;
  bool m_parametersHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
