/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.shardingsphere.test.it.sql.parser.internal.asserts.statement.dal.standard.type;

import lombok.AccessLevel;
import lombok.NoArgsConstructor;
import org.apache.shardingsphere.sql.parser.statement.core.statement.type.dal.ShowAlterTableStatement;
import org.apache.shardingsphere.test.it.sql.parser.internal.asserts.SQLCaseAssertContext;
import org.apache.shardingsphere.test.it.sql.parser.internal.asserts.segment.database.DatabaseAssert;
import org.apache.shardingsphere.test.it.sql.parser.internal.asserts.segment.limit.LimitClauseAssert;
import org.apache.shardingsphere.test.it.sql.parser.internal.asserts.segment.orderby.OrderByClauseAssert;
import org.apache.shardingsphere.test.it.sql.parser.internal.asserts.segment.where.WhereClauseAssert;
import org.apache.shardingsphere.test.it.sql.parser.internal.cases.parser.jaxb.statement.dal.ShowAlterTableStatementTestCase;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;

/**
 * Show alter table statement assert.
 */
@NoArgsConstructor(access = AccessLevel.PRIVATE)
public final class ShowAlterTableStatementAssert {
    
    /**
     * Assert show alter table statement is correct with expected parser result.
     *
     * @param assertContext assert context
     * @param actual actual show alter table statement
     * @param expected expected show alter table statement test case
     */
    public static void assertIs(final SQLCaseAssertContext assertContext, final ShowAlterTableStatement actual, final ShowAlterTableStatementTestCase expected) {
        assertAlterType(assertContext, actual, expected);
        assertDatabase(assertContext, actual, expected);
        assertWhere(assertContext, actual, expected);
        assertOrderBy(assertContext, actual, expected);
        assertLimit(assertContext, actual, expected);
    }
    
    private static void assertAlterType(final SQLCaseAssertContext assertContext, final ShowAlterTableStatement actual, final ShowAlterTableStatementTestCase expected) {
        if (null != expected.getAlterType()) {
            Assertions.assertNotNull(actual.getAlterType().orElse(null), assertContext.getText("Actual alter type should exist."));
            MatcherAssert.assertThat(assertContext.getText("Alter type assertion error: "), actual.getAlterType().get(), Matchers.is(expected.getAlterType()));
        } else {
            Assertions.assertNull(actual.getAlterType().orElse(null), assertContext.getText("Actual alter type should not exist."));
        }
    }
    
    private static void assertDatabase(final SQLCaseAssertContext assertContext, final ShowAlterTableStatement actual, final ShowAlterTableStatementTestCase expected) {
        if (null != expected.getDatabase()) {
            Assertions.assertNotNull(actual.getDatabase().orElse(null), assertContext.getText("Actual database should exist."));
            DatabaseAssert.assertIs(assertContext, actual.getDatabase().get(), expected.getDatabase());
        } else {
            Assertions.assertNull(actual.getDatabase().orElse(null), assertContext.getText("Actual database should not exist."));
        }
    }
    
    private static void assertWhere(final SQLCaseAssertContext assertContext, final ShowAlterTableStatement actual, final ShowAlterTableStatementTestCase expected) {
        if (null != expected.getWhere()) {
            Assertions.assertNotNull(actual.getWhere().orElse(null), assertContext.getText("Actual where segment should exist."));
            WhereClauseAssert.assertIs(assertContext, actual.getWhere().get(), expected.getWhere());
        } else {
            Assertions.assertNull(actual.getWhere().orElse(null), assertContext.getText("Actual where segment should not exist."));
        }
    }
    
    private static void assertOrderBy(final SQLCaseAssertContext assertContext, final ShowAlterTableStatement actual, final ShowAlterTableStatementTestCase expected) {
        if (null != expected.getOrderBy()) {
            Assertions.assertNotNull(actual.getOrderBy().orElse(null), assertContext.getText("Actual order by segment should exist."));
            OrderByClauseAssert.assertIs(assertContext, actual.getOrderBy().get(), expected.getOrderBy());
        } else {
            Assertions.assertNull(actual.getOrderBy().orElse(null), assertContext.getText("Actual order by segment should not exist."));
        }
    }
    
    private static void assertLimit(final SQLCaseAssertContext assertContext, final ShowAlterTableStatement actual, final ShowAlterTableStatementTestCase expected) {
        if (null != expected.getLimit()) {
            Assertions.assertNotNull(actual.getLimit().orElse(null), assertContext.getText("Actual limit segment should exist."));
            LimitClauseAssert.assertRowCount(assertContext, actual.getLimit().get().getRowCount().orElse(null), expected.getLimit().getRowCount());
        } else {
            Assertions.assertNull(actual.getLimit().orElse(null), assertContext.getText("Actual limit segment should not exist."));
        }
    }
}
